{-------------------------------------------------------------------------------

        Copyright:              Bernie Pope 2003

        Module:                 Opts

        Description:            Command line processing for Buddha.

        Primary Authors:        Bernie Pope

        Notes:                  See the file ../LICENCE for licence information

-------------------------------------------------------------------------------}

{-
    This file is part of buddha.

    buddha is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    buddha is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with buddha; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
-}

module Opts 
   ( processCmdLine
   , makeUsageInfo
   , CmdLine (..)
   , flagIsSet
   ) where

import System.Console.GetOpt 
   ( OptDescr (..)
   , ArgDescr (..)
   , getOpt
   , usageInfo
   , ArgOrder (..)
   )

import ChatIO 
   ( ChatLevel (..) )

--------------------------------------------------------------------------------


data Flag 
   = Dump String          -- what data to dump
   | Trans String         -- options changing the transformation
   | Help                 -- print help/usage message
   | Version	          -- print the version number
   | Chat String          -- make the transformation chatty, silent, normal, verbose 
   | Skip String          -- phases to skip  
   | Cpp                  -- use the CPP on the program
   | Define String        -- definition of CPP macros at the command line
                          -- XXX may need Undefine also

options :: [OptDescr Flag]

options
 = [
    Option ['h']    ["help"] (NoArg Help)       
                                   ("\n\n    displays a usage message\n\n"),

    Option ['v']    ["version"] (NoArg Version)       
                                   ("\n\n    displays the version number\n\n"),

    Option ['c']    ["chat"] (ReqArg Chat "LEVEL")       
                                   ("\n\n    diagnostics about what buddha is doing\n"
   ++ "    LEVEL can be: silent, normal, or verbose\n\n"),

    Option ['p']    ["cpp"] (NoArg Cpp)       
                                   ("\n\n    use the C pre-processor on the source files\n\n"),

    Option ['D']    ["define"] (ReqArg Define "MACRO")       
                                   ("\n\n    define MACRO for the C pre-processor\n\n"),

    Option ['s']    ["skip"] (ReqArg Skip "PHASE")
                                   ("\n\n    skip a particular PHASE\n"
   ++ "    case PHASE of\n"
   ++ "     \"init\"         -> initialisation\n"
   ++ "     \"depend\"       -> dependency analysis\n"
   ++ "     \"trans\"        -> transformation\n"
   ++ "     \"build\"        -> building the debugging program\n\n"),

    Option ['d']    ["dump"] (ReqArg Dump "SECTION")       
                                   ("\n\n    dumps some info on a given SECTION\n"
   ++ "    case SECTION of\n"
   ++ "     \"parse\"         -> abstract syntax tree after parsing\n"
   ++ "     \"idents\"        -> the identifier information for a module\n"
   ++ "     \"rename\"        -> source after renaming\n"
   ++ "     \"derives\"       -> code for class instance derivations\n"
   ++ "     \"desugar\"       -> source after desugaring\n"
   ++ "     \"imports\"       -> list what is imported from other modules\n"
   ++ "     \"assocs\"        -> tycon and class associations for expanding (..) syntax\n"
   ++ "                          in imports and exports\n"
   ++ "     \"infix\"         -> fixity information for all vars and cons in scope\n"
   ++ "     \"synonyms\"      -> all the type synonyms that are in scope\n"
   ++ "     \"transOpts\"     -> the transformation options found in Buddha/*.opt\n"
   ++ "     \"stats\"         -> statistics about the whole program\n"
   ++ "     \"statsByModule\" -> statistics about each module\n"
   ++ "     \"all\"           -> all the above data\n\n"),

    Option ['t'] ["trans"]  (ReqArg Trans "FLAG")
                                   ("\n\n    alter the transformation\n"
   ++ "    case FLAG of\n"
   ++ "     \"cheapHo\"       -> don't record the applications of higher-order\n"
   ++ "                          arguments or results\n"
   ++ "     \"noprelude\"     -> do not implicitly import the Haskell Prelude\n")
   ]

data CmdLine 
   = CmdLine 
     { fileNames   :: [FilePath]    -- file names of files to transform
     , dumpFlags   :: [String]      -- flags for dumping info 
     , transFlags  :: [String]      -- flags for the transformation 
     , defines     :: [String]      -- macro defines
     , needHelp    :: Bool          -- does the user want help?
     , needVersion :: Bool          -- does the user want the version number?
     , chatLevel   :: ChatLevel     -- how chatty is buddha? default is Normal
     , skipFlags   :: [String]      -- names of phases to skip 
     , needCpp     :: Bool          -- run the CPP over the program
     }
                         
type ErrMsg = String

processCmdLine :: String -> [String] -> Either CmdLine ErrMsg 
processCmdLine progName cmdLine 
   = case getOpt RequireOrder options cmdLine of
          (_, _, errs@(_:_))  -> Right $ concat errs 
          (opts, hsFiles, []) 
             -> findFlags initCmdLine opts
                where
                initCmdLine 
                   = CmdLine 
                     { fileNames   = hsFiles 
                     , dumpFlags   = []
                     , transFlags  = []
                     , defines     = []
                     , needHelp    = False
                     , needVersion = False
                     , chatLevel   = Normal
                     , skipFlags   = []
                     , needCpp     = False
                     }

findFlags :: CmdLine -> [Flag] -> Either CmdLine ErrMsg
findFlags cmdLine [] = Left cmdLine
findFlags cmdLine (Dump d : flags) 
   | validDump d = findFlags (cmdLine { dumpFlags = d : oldDumps }) flags
   | otherwise = Right $ "command line - unknown dump flag: " ++ d 
   where
   oldDumps = dumpFlags cmdLine
   validDump d 
      = d `elem` ["parse", "idents", "rename", "derives", "desugar", "imports", "assocs",
                  "stats", "statsByModule", "all"]
findFlags cmdLine (Trans t : flags) 
   | validTrans t = findFlags (cmdLine { transFlags = t : oldTrans }) flags
   | otherwise = Right $ "command line - unknown transformation flag: " ++ t
   where
   oldTrans = transFlags cmdLine
   validTrans t 
      = t `elem` ["noprelude", "cheapHo"]

findFlags cmdLine (Define d : flags)
   = findFlags ( cmdLine { defines = d : oldDefines }) flags
   where
   oldDefines = defines cmdLine
findFlags cmdLine (Help : flags) 
   = findFlags (cmdLine { needHelp = True }) flags

findFlags cmdLine (Chat level : flags) 
   = case readLevel level of
        Nothing -> Right $ "command line - unknown chat level: " ++ level
        Just l  -> findFlags (cmdLine { chatLevel = l }) flags
   where
   readLevel :: String -> Maybe ChatLevel 
   readLevel s
      = case s of
         "silent"  -> Just Silent
         "normal"  -> Just Normal
         "verbose" -> Just Verbose
         other     -> Nothing 
findFlags cmdLine (Version : flags) 
   = findFlags (cmdLine { needVersion = True }) flags
findFlags cmdLine (Skip p : flags)  
   | validSkip p = findFlags (cmdLine { skipFlags = p : oldSkips }) flags
   | otherwise 
        = Right $ "command line - unknown phase to skip: " ++ p
   where
   oldSkips = skipFlags cmdLine
   validSkip p = p `elem` ["init", "depend", "trans", "build"]
findFlags cmdLine (Cpp : flags)
   = findFlags (cmdLine { needCpp = True }) flags

-- construct an appropriate usage string
makeUsageInfo :: String -> String
makeUsageInfo progName
   = let header = "Usage: " ++ progName ++ " [options] [file1 file2 ...]"
         footer = "\nReport bugs to bjpop@cs.mu.oz.au\n\n"
     in (usageInfo header options ++ footer)

-- determine if a particular flag is set from a number of flags
-- is "all" is present then the result is always true
flagIsSet :: [String] -> String -> Bool
flagIsSet ss s = s `elem` ss || "all" `elem` ss

