{-# OPTIONS -cpp #-}
{-------------------------------------------------------------------------------

        Copyright:              Bernie Pope 2003

        Module:                 Record 

        Description:            various pieces for handling records 

        Primary Authors:        Bernie Pope

-------------------------------------------------------------------------------}

{-
    This file is part of buddha.

    buddha is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    buddha is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with buddha; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
-}

module Record 
   ( desugarRecords
   , RecordMap
   , emptyRecordMap
   , lookupRecord
   , showRecordMap
   , findUpdateCandidates
   , plusRecordMap
   , recordMapToList 
   , addListItemsToRecMap
   , mkSelectorSet
   ) where

import Data.FiniteMap

import Language.Haskell.Syntax 

import List 
   ( mapAccumL )

import Utils 
   ( isSubset )

import SyntaxUtils
   ( reQualify
   , updateName
   , dropQualifier
   )

import IfaceUtils 
   ( IdentQual (..) )

import Error 
   ( fatalError )

import Data.Set
   ( Set 
   , mkSet 
   )

--------------------------------------------------------------------------------


type RecordMap = FiniteMap HsQName [HsQName] 

emptyRecordMap :: RecordMap
emptyRecordMap = emptyFM

lookupRecord :: RecordMap -> HsQName -> Maybe [HsQName] 
lookupRecord = lookupFM

showRecordMap :: RecordMap -> String
showRecordMap sigMap
   = unlines $ map show mapList
   where
   mapList = recordMapToList sigMap

recordMapToList :: RecordMap -> [(HsQName, [HsQName])]
recordMapToList = fmToList

plusRecordMap :: RecordMap -> RecordMap -> RecordMap
plusRecordMap = plusFM

desugarRecords :: Module -> RecordMap -> [HsDecl] -> (RecordMap, [HsDecl], [HsDecl])
desugarRecords modName recMap decls
   = (newRecMap, newDecls, selectorDecls)
   where
   (newRecMap, newDecls, selectorDecls) = desugarRecords' recMap decls

   desugarRecords' :: RecordMap -> [HsDecl] -> (RecordMap, [HsDecl], [HsDecl])
   desugarRecords' recMap [] = (recMap, [], [])

   desugarRecords' recMap (decl@(HsDataDecl sloc cntxt tyName args conDecls derives) : rest)
      = (nextRecMap, thisNewDecl : nextDecls, thisSelectors ++ restSelectors)
      where
      thisSelectors = selectorDeclsFromConDecls conDecls 
      thisNewDecl = HsDataDecl sloc cntxt tyName args newConDecls derives
      (thisNewRecMap, newConDecls) = mapAccumL (recDeclToConDecl modName) recMap conDecls 
      (nextRecMap, nextDecls, restSelectors) = desugarRecords' thisNewRecMap rest

   desugarRecords' recMap (decl@(HsNewTypeDecl sloc cntxt tyName args conDecl derives) : rest)
      = (nextRecMap, thisNewDecl : nextDecls, thisSelectors ++ restSelectors)
      where
      thisSelectors = selectorDeclsFromConDecls [conDecl]
      thisNewDecl = HsNewTypeDecl sloc cntxt tyName args newConDecl derives
      (thisNewRecMap, newConDecl) = recDeclToConDecl modName recMap conDecl 
      (nextRecMap, nextDecls, restSelectors) = desugarRecords' thisNewRecMap rest

   desugarRecords' recMap (otherDecl : rest)
      = (nextRecMap, otherDecl : nextDecls, restSelectors)
      where
      (nextRecMap, nextDecls, restSelectors) = desugarRecords' recMap rest 

{-
   Convert a Record Decl to a ConDecl and compute a recordMap for it

   ... K { f,j :: Int, z :: Bool } ...
  
   ->

   ( ... (K, [f,j,z]) ... , ... K Int Int Bool ...)
-}

recDeclToConDecl :: Module -> RecordMap -> HsConDecl -> (RecordMap, HsConDecl)
recDeclToConDecl modName recMap (HsRecDecl sloc conName recs)
   = (newRecMap, HsConDecl sloc conName argTypes)
   where
   argNamess :: [[HsName]]
   argTypess :: [[HsBangType]]
   (argNamess, argTypess) = unzip $ map recNameTys recs
   argNames = concat argNamess
   argTypes = concat argTypess
   recNameTys :: ([HsName], HsBangType) -> ([HsName], [HsBangType])
   recNameTys (names, t) = (names, replicate (length names) t)
   newRecMapUnQual :: RecordMap
   newRecMapUnQual = addToFM recMap (UnQual conName) (map UnQual argNames) 
   newRecMap = addToFM newRecMapUnQual (Qual modName conName) (map (Qual modName) argNames)

recDeclToConDecl _modName recMap conDecl@(HsConDecl sloc conName argTypes)
   = (recMap, conDecl) 

--------------------------------------------------------------------------------

-- making selector functions from Records

data SelectorPat = SelPat {con :: HsName, pos, arity :: Int, sloc :: SrcLoc}
type SelectorInfo = (HsName, [SelectorPat])

selectorDeclsFromConDecls :: [HsConDecl] -> [HsDecl]
selectorDeclsFromConDecls conDecls
   = selectorDecls
   where
   selInfos = selectorInfosFromConDecls conDecls []
   selectorDecls = map selectorInfoToFunBind selInfos

selectorInfosFromConDecls :: [HsConDecl] ->  [SelectorInfo] -> [SelectorInfo]
selectorInfosFromConDecls [] sels = sels
selectorInfosFromConDecls (decl:decls) sels
   = selectorInfosFromConDecls decls newSels
   where
   newSels = selectorInfos decl sels

selectorInfos :: HsConDecl -> [SelectorInfo] -> [SelectorInfo]
selectorInfos (HsRecDecl recLoc conName records) infos
   = selectorInfosAcc infos 0 partialPat selNames 
   where
   partialPat = SelPat 
                { con = conName
                , arity = length selNames
                , pos = undefined
                , sloc = recLoc }
   selNames = concatMap fst records
   selectorInfosAcc :: [SelectorInfo] -> Int -> SelectorPat -> [HsName] -> [SelectorInfo]
   selectorInfosAcc oldSels _position _patTemplate [] = oldSels 
   selectorInfosAcc oldSels position patTemplate (sel:sels)
      = insertSelector sel newSelPat rest
      where
      newSelPat = patTemplate {pos = position}
      rest = selectorInfosAcc oldSels (position + 1) patTemplate sels
selectorInfos (HsConDecl _sloc _conName _conDecls) infos = infos

insertSelector :: HsName -> SelectorPat -> [SelectorInfo] -> [SelectorInfo]
insertSelector thisSelName pat [] = [(thisSelName, [pat])]
insertSelector thisSelName pat ((nextSelName, pats):sels)
   | thisSelName == nextSelName = (nextSelName, pat:pats) : sels
   | otherwise = (nextSelName, pats) : insertSelector thisSelName pat sels

selectorInfoToFunBind :: SelectorInfo -> HsDecl
selectorInfoToFunBind selInfo 
   = HsFunBind $ selectorInfoToMatches selInfo

selectorInfoToMatches :: SelectorInfo -> [HsMatch]
selectorInfoToMatches (_selName, []) = []
selectorInfoToMatches (selName, (pat:pats))
   = thisMatch : selectorInfoToMatches (selName, pats) 
   where
   thisMatch = HsMatch thisSloc selName [newPat] newRhs []
   newPat  = HsPApp (UnQual (con pat)) patArgs
   patArgs = take (arity pat) $ 
                  (replicate (pos pat) HsPWildCard) ++ 
                  [newVarPat] ++ 
                  repeat HsPWildCard
   newVar = HsIdent "v"
   newVarPat = HsPVar newVar
   newVarExp = HsVar $ UnQual newVar
   newRhs = HsUnGuardedRhs newVarExp
   thisSloc = sloc pat


-- find all the record constructors that contain all of the selectors
-- found in a particular update expression

findUpdateCandidates :: RecordMap -> [HsFieldUpdate] -> [(HsQName, [HsQName])]
findUpdateCandidates recMap updates
   = findUpdates fieldNamesToFind allRecords
   where
   fieldNamesToFind = map updateName updates
   allRecords :: [(HsQName, [HsQName])]
   allRecords = recordMapToList recMap
   findUpdates :: [HsQName] -> [(HsQName, [HsQName])] -> [(HsQName, [HsQName])]
   findUpdates _namesToFind [] = [] 
   findUpdates namesToFind (thisRecord@(con, sels) : rest)
      | namesToFind `isSubset` sels = thisRecord : restMatches 
      | otherwise = restMatches
      where
      restMatches = findUpdates namesToFind rest

-- for updating a record map based on an import
addListItemsToRecMap :: IdentQual -> Module -> [(HsQName, [HsQName])] -> RecordMap -> RecordMap
addListItemsToRecMap qualification moduleName recList recMap
   | qualification == Qualified = mapWithQualifiedOnly
   | qualification == QualAndUnQual = mapWithBothQualifications
   | otherwise = fatalError __FILE__ __LINE__ $ "addListItemsToRecMap : given UnQualified"
   where
   qualifiedList = map (reQualifyItem moduleName) recList
   unQualifiedList = map dropQualifyItem recList
   reQualifyItem modName (qName, qNames)
      = (reQualify modName qName, map (reQualify modName) qNames) 
   dropQualifyItem (qName, qNames)
      = (dropQualifier qName, map dropQualifier qNames) 
   mapWithQualifiedOnly = addListToFM recMap qualifiedList
   mapWithBothQualifications = addListToFM mapWithQualifiedOnly unQualifiedList 

-- set of all selectors in the record map
mkSelectorSet :: RecordMap -> Set HsQName
mkSelectorSet records = mkSet $ concatMap snd $ recordMapToList records
