{-------------------------------------------------------------------------------

        Copyright:              Bernie Pope 2003

        Module:                 Rename 

        Description:            Rename some variables to avoid a nameclash 

        Primary Authors:        Bernie Pope

   Ensuring that names are unique
   ------------------------------

   buddha must introduce new variables into the program but avoid
   clashing with existing names.

   Every new variable gets a prefix (say 'v') followed by a unique
   integer. So for example a new variable might be called:

      v12

   To avoid clashing with a variable of the same name, any variable that
   has the syntax: prefix int 
   will have a new suffix added to it, say a '_'

   so "v24" from the original program becomes: "v24_" everywhere
   
   However this may clash with a variable of this name somewhere
   in the original program.

   So any variable of name "v24_" will become "v24__"
   
   and "v24__" will become "v24___" and so on.
               
   In practice we don't expect many names will have to change.

   Whenever true source names are needed we must remember to un-rename them
   otherwise confusion could result.

-------------------------------------------------------------------------------}

{-
    This file is part of buddha.

    buddha is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    buddha is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with buddha; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
-}

module Rename 
   ( runRename
   , unRename 
   , defaultVarPrefix
   , defaultConPrefix
   , renameModule ) 
   where

import Language.Haskell.Syntax

import Monad                    
   ( liftM
   , liftM2
   , liftM3  
   , liftM4  
   , liftM5 )

import Char                     
   ( isDigit )

--------------------------------------------------------------------------------

-- the default prefix to put on the front of a new variable:
defaultVarPrefix :: String
defaultVarPrefix = "v"

-- reserved prefixes for new constructors 
defaultConPrefix :: String
defaultConPrefix = "F"

-- a monad to thread state, we don't really need it now
-- but perhaps we'll find a use for it later

data State = State { {- empty at the moment -} }

newtype Rename a = Rename (State -> (a, State))

instance Monad Rename where
    return a
        = Rename (\state -> (a, state))

    Rename comp >>= fun
        = Rename (\state ->
                        let (result, newState) = comp state
                            Rename comp' = fun result
                        in comp' newState)

runRename :: Rename a -> a
runRename (Rename comp) 
   = result
   where
   (result, _finalState) = comp (State { })

-- select a component of the state
select :: (State -> a) -> Rename a
select selector = Rename (\state -> (selector state, state))

--------------------------------------------------------------------------------

{- patterns -}

renamePat :: HsPat -> Rename HsPat
renamePat (HsPVar name) = liftM HsPVar $ renameVarName name
renamePat p@(HsPLit lit) = return p
renamePat (HsPNeg pat) = liftM HsPNeg $ renamePat pat
renamePat (HsPInfixApp p1 qName p2)
   = liftM3 HsPInfixApp (renamePat p1)
                        (renameQualConName qName)
                        (renamePat p2)
renamePat (HsPApp qName pats)
   = liftM2 HsPApp (renameQualConName qName) $ mapM renamePat pats
renamePat (HsPTuple pats)
   = liftM HsPTuple $ mapM renamePat pats
renamePat (HsPList pats)
   = liftM HsPList $ mapM renamePat pats
renamePat (HsPParen p)
   = liftM HsPParen $ renamePat p
renamePat (HsPRec qName patFields)
   = liftM2 HsPRec (renameQualConName qName) $ mapM renamePatField patFields
renamePat (HsPAsPat asName pat)
   = liftM2 HsPAsPat (renameVarName asName) $ renamePat pat
renamePat p@HsPWildCard = return p
renamePat (HsPIrrPat p) = liftM HsPIrrPat $ renamePat p

{- pat fields -}

renamePatField :: HsPatField -> Rename HsPatField
renamePatField (HsPFieldPat qName p)
   = liftM2 HsPFieldPat (renameQualConName qName) $ renamePat p

{- names -}

renameQualVarName :: HsQName -> Rename HsQName
renameQualVarName (Qual modName name) = liftM (Qual modName) $ renameVarName name
renameQualVarName (UnQual name)
   = liftM UnQual $ renameVarName name
renameQualVarName qName@(Special specCon) = return qName

renameQualConName :: HsQName -> Rename HsQName
renameQualConName (Qual modName name) = liftM (Qual modName) $ renameConName name
renameQualConName (UnQual name)
   = liftM UnQual $ renameConName name
renameQualConName qName@(Special specCon) = return qName

renameQOp :: HsQOp -> Rename HsQOp
renameQOp (HsQVarOp name) = liftM HsQVarOp $ renameQualVarName name
renameQOp (HsQConOp name) = liftM HsQConOp $ renameQualConName name

renameVarName :: HsName -> Rename HsName
renameVarName (HsIdent str)  = liftM HsIdent  $ renameVarStr str
renameVarName (HsSymbol str) = liftM HsSymbol $ renameVarStr str

renameConName :: HsName -> Rename HsName
renameConName (HsIdent str)  = liftM HsIdent  $ renameConStr str
renameConName (HsSymbol str) = liftM HsSymbol $ renameConStr str

-- these are special names that need to be renamed b/c
-- their original name must also appear in the program
-- this is neccessary for anything implicitly inserted into the program
-- by the compiler 
specialNames :: [(String, String)]
specialNames
   = [ ("fromInteger",  "fromInteger_bUdDhA")
     , ("fromRational", "fromRational_bUdDhA")
     , ("negate",       "negate_bUdDhA") 
     ]

-- this is the counterpart to specialNames (reverse mapping of)
renamedSpecialNames :: [(String, String)] 
renamedSpecialNames = map (\(a,b)->(b,a)) specialNames 

renameVarStr :: String -> Rename String
renameVarStr str
   = do case needsRenameStr defaultVarPrefix str of
           True  -> return $ str ++ "_" 
           False -> case lookup str specialNames of
                       Nothing      -> return str
                       Just newName -> return newName 

renameConStr :: String -> Rename String
renameConStr str
   = do case needsRenameStr defaultConPrefix str of
           True  -> return $ str ++ "_" 
           False -> return str

{- a string needs renaming of it begins with the right prefix,
   is followed by one or more digits, and is followed by ZERO
   or more underscores -}

needsRenameStr :: String -> String -> Bool
needsRenameStr prefix str
   = (pref == prefix) &&
     length suffDig > 0 &&
     all (== '_') suffRest   -- true if zero or more underscores
   where
   (pref, suff) = splitAt (length prefix) str
   (suffDig, suffRest) = span isDigit suff

renameOp :: HsOp -> Rename HsOp
renameOp (HsVarOp name) = liftM HsVarOp $ renameVarName name
renameOp (HsConOp name) = liftM HsConOp $ renameConName name

renameCName  :: HsCName -> Rename HsCName
renameCName (HsVarName name) = liftM HsVarName $ renameVarName name
renameCName (HsConName name) = liftM HsConName $ renameConName name

renameModule :: HsModule -> Rename HsModule
renameModule (HsModule sloc modName exportSpecs importDecls decls)
   = liftM3 (HsModule sloc modName) 
            (renameExportSpecs exportSpecs)
            (mapM renameImportDecl importDecls)
            (mapM renameDecl decls)

renameExportSpecs :: Maybe [HsExportSpec] -> Rename (Maybe [HsExportSpec])
renameExportSpecs Nothing = return Nothing
renameExportSpecs (Just specs)
   = liftM Just $ mapM renameExportSpec specs

renameExportSpec :: HsExportSpec -> Rename HsExportSpec
renameExportSpec (HsEVar qName) = liftM HsEVar $ renameQualVarName qName
renameExportSpec s@(HsEAbs qName) = return s 
renameExportSpec s@(HsEThingAll qName) = return s 
renameExportSpec (HsEThingWith qName cNames)
   = liftM (HsEThingWith qName) $ mapM renameCName cNames 
renameExportSpec s@(HsEModuleContents modName) = return s

renameImportDecl :: HsImportDecl -> Rename HsImportDecl
renameImportDecl (HsImportDecl sloc modName qualified asName imports)
   = liftM (HsImportDecl sloc modName qualified asName) $ renameImportSpecs imports

renameImportSpecs :: (Maybe (Bool, [HsImportSpec])) -> Rename (Maybe (Bool, [HsImportSpec]))
renameImportSpecs Nothing = return Nothing
renameImportSpecs (Just (hiding, specs))
   = do newSpecs <- mapM renameImportSpec specs 
        return $ Just (hiding, newSpecs)

renameImportSpec :: HsImportSpec -> Rename HsImportSpec
renameImportSpec (HsIVar name) = liftM HsIVar $ renameVarName name
renameImportSpec i@(HsIAbs name) = return i 
renameImportSpec i@(HsIThingAll name) = return i 
renameImportSpec (HsIThingWith name cNames) 
   = liftM (HsIThingWith name) $ mapM renameCName cNames
                         
renameDecl :: HsDecl -> Rename HsDecl
renameDecl (HsTypeDecl sloc tyName argsNames t)
   = liftM (HsTypeDecl sloc tyName argsNames) 
           (renameType t)

renameDecl (HsDataDecl sloc cntxt tyName argsNames conDecls derives)
   = liftM5 (HsDataDecl sloc) 
            (renameCntxt cntxt)
            (return tyName)
            (return argsNames)
            (mapM renameConDecl conDecls)
            (return derives) 

renameDecl (HsInfixDecl sloc assoc prec ops)
   = liftM (HsInfixDecl sloc assoc prec)
           (mapM renameOp ops)

renameDecl (HsNewTypeDecl sloc cntxt tyName argsNames conDecl derives)
   = liftM5 (HsNewTypeDecl sloc)
            (renameCntxt cntxt)
            (return tyName)
            (return argsNames)
            (renameConDecl conDecl)
            (return derives)
         
renameDecl (HsClassDecl sloc cntxt className argsNames decls)
   = liftM4 (HsClassDecl sloc)
            (renameCntxt cntxt)
            (return className)
            (return argsNames)
            (mapM renameDecl decls)

renameDecl (HsInstDecl sloc cntxt className tys decls)
   = liftM4 (HsInstDecl sloc)
            (renameCntxt cntxt)
            (return className)
            (mapM renameType tys)
            (mapM renameDecl decls)

renameDecl (HsDefaultDecl sloc tys)
   = liftM (HsDefaultDecl sloc) $ mapM renameType tys 

renameDecl (HsTypeSig sloc names qType)
   = liftM2 (HsTypeSig sloc) (mapM renameVarName names) $ renameQualType qType

renameDecl (HsFunBind matchs)
   = liftM HsFunBind $ mapM renameMatch matchs 

renameDecl (HsPatBind sloc pat rhs decls)
   = liftM3 (HsPatBind sloc) 
            (renamePat pat)
            (renameRhs rhs)
            (mapM renameDecl decls)

renameMatch :: HsMatch -> Rename HsMatch
renameMatch (HsMatch sloc name pats rhs decls)
   = liftM4 (HsMatch sloc)
            (renameVarName name)
            (mapM renamePat pats)
            (renameRhs rhs)
            (mapM renameDecl decls)

renameConDecl :: HsConDecl -> Rename HsConDecl
renameConDecl (HsConDecl sloc name bangTypes)
   = liftM2 (HsConDecl sloc) (renameConName name)
            (mapM renameBangType bangTypes)
renameConDecl (HsRecDecl sloc name recs)
   = liftM2 (HsRecDecl sloc) (renameConName name)
            (mapM renameRecord recs) 

renameRecord :: ([HsName], HsBangType) -> Rename ([HsName], HsBangType)
renameRecord (names, bangType)
   = do newNames <- mapM renameVarName names
        newBangType <- renameBangType bangType
        return (newNames, newBangType)

renameBangType :: HsBangType -> Rename HsBangType
renameBangType (HsBangedTy t) = liftM HsBangedTy $ renameType t
renameBangType (HsUnBangedTy t) = liftM HsUnBangedTy $ renameType t

renameRhs :: HsRhs -> Rename HsRhs
renameRhs (HsUnGuardedRhs e) = liftM HsUnGuardedRhs $ renameExp e 
renameRhs (HsGuardedRhss rhss) = liftM HsGuardedRhss $ mapM renameGuardedRhs rhss 

renameGuardedRhs :: HsGuardedRhs -> Rename HsGuardedRhs
renameGuardedRhs (HsGuardedRhs sloc e1 e2)
   = liftM2 (HsGuardedRhs sloc) (renameExp e1) (renameExp e2)

renameQualType :: HsQualType -> Rename HsQualType
renameQualType (HsQualType cntxt t)
   = liftM2 HsQualType (renameCntxt cntxt) (renameType t) 

-- don't need to rename anything in a type yet
renameType :: HsType -> Rename HsType
renameType t = return t

renameCntxt :: HsContext -> Rename HsContext
renameCntxt ctxt = mapM renameAsst ctxt

renameAsst :: HsAsst -> Rename HsAsst
renameAsst (qName,tys) 
   = do newTypes <- mapM renameType tys
        return (qName, newTypes) 

renameLiteral :: HsLiteral -> Rename HsLiteral
renameLiteral lit = return lit

renameExp :: HsExp -> Rename HsExp
renameExp (HsVar qName) = liftM HsVar $ renameQualVarName qName
renameExp (HsCon qName) = liftM HsCon $ renameQualConName qName
renameExp (HsLit lit) = liftM HsLit $ renameLiteral lit 
renameExp (HsInfixApp e1 qOp e2)
   = liftM3 HsInfixApp (renameExp e1) (renameQOp qOp) $ renameExp e2 
renameExp (HsApp e1 e2) = liftM2 HsApp (renameExp e1) $ renameExp e2
renameExp (HsNegApp e) = liftM HsNegApp $ renameExp e 
renameExp (HsLambda sloc pats e)
   = liftM2 (HsLambda sloc) (mapM renamePat pats) $ renameExp e
renameExp (HsLet decls e)
   = liftM2 HsLet (mapM renameDecl decls) $ renameExp e 
renameExp (HsIf e1 e2 e3)
   = liftM3 HsIf (renameExp e1) (renameExp e2) $ renameExp e3 
renameExp (HsCase e alts) 
   = liftM2 HsCase (renameExp e) $ mapM renameAlt alts
renameExp (HsDo stmts) 
   = liftM HsDo $ mapM renameStmt stmts
renameExp (HsTuple es) = liftM HsTuple $ mapM renameExp es 
renameExp (HsList es) = liftM HsList $ mapM renameExp es 
renameExp (HsParen es) = liftM HsParen $ renameExp es 
renameExp (HsLeftSection e qOp) = liftM2 HsLeftSection (renameExp e) $ renameQOp qOp 
renameExp (HsRightSection qOp e) = liftM2 HsRightSection (renameQOp qOp) $ renameExp e 
renameExp (HsRecConstr qName fieldUpdates)
   = liftM2 HsRecConstr (renameQualConName qName) $ mapM renameFieldUpdate fieldUpdates 
renameExp (HsRecUpdate e fieldUpdates)
   = liftM2 HsRecUpdate (renameExp e) $ mapM renameFieldUpdate fieldUpdates 
renameExp (HsEnumFrom e) = liftM HsEnumFrom $ renameExp e
renameExp (HsEnumFromTo e1 e2) = liftM2 HsEnumFromTo (renameExp e1) (renameExp e2) 
renameExp (HsEnumFromThen e1 e2) = liftM2 HsEnumFromThen (renameExp e1) (renameExp e2) 
renameExp (HsEnumFromThenTo e1 e2 e3) 
   = liftM3 HsEnumFromThenTo (renameExp e1) (renameExp e2) $ renameExp e3
renameExp (HsListComp e stmts)
   = liftM2 HsListComp (renameExp e) $ mapM renameStmt stmts 
renameExp (HsExpTypeSig sloc e qType)
   = liftM2 (HsExpTypeSig sloc) (renameExp e) $ renameQualType qType
renameExp (HsAsPat name e)
   = liftM2 HsAsPat (renameVarName name) $ renameExp e
renameExp HsWildCard = return HsWildCard
renameExp (HsIrrPat e) = liftM HsIrrPat $ renameExp e

renameStmt :: HsStmt -> Rename HsStmt
renameStmt (HsGenerator sloc pat e)
   = liftM2 (HsGenerator sloc) (renamePat pat) $ renameExp e 
renameStmt (HsQualifier e)
   = liftM HsQualifier $ renameExp e
renameStmt (HsLetStmt decls)
   = liftM HsLetStmt $ mapM renameDecl decls

renameFieldUpdate :: HsFieldUpdate -> Rename HsFieldUpdate 
renameFieldUpdate (HsFieldUpdate qName e)
   = liftM2 HsFieldUpdate (renameQualConName qName) $ renameExp e 

renameAlt :: HsAlt -> Rename HsAlt
renameAlt (HsAlt sloc pat gAlts decls)
   = liftM3 (HsAlt sloc) (renamePat pat) (renameGuardedAlts gAlts) $ mapM renameDecl decls 

renameGuardedAlts :: HsGuardedAlts -> Rename HsGuardedAlts 
renameGuardedAlts (HsUnGuardedAlt e) = liftM HsUnGuardedAlt $ renameExp e 
renameGuardedAlts (HsGuardedAlts gAlts) = liftM HsGuardedAlts $ mapM renameGuardedAlt gAlts

renameGuardedAlt :: HsGuardedAlt -> Rename HsGuardedAlt
renameGuardedAlt (HsGuardedAlt sloc e1 e2)
   = liftM2 (HsGuardedAlt sloc) (renameExp e1) $ renameExp e2

--------------------------------------------------------------------------------

{- removing a renaming from a string -}

-- check if the string was renamed, if so drop the last character off 
-- or convert it back to its original name if it was a special name

unRename :: String -> String
unRename str
   | needsUnRenameStr defaultVarPrefix str = init str 
   | needsUnRenameStr defaultConPrefix str = init str 
   | otherwise = case lookup str renamedSpecialNames of
                    Nothing      -> str
                    Just newName -> newName 

{- a string should be un-renamed if it starts with the right prefix
   is followed by zero or more digits and concludes with one or more
   underscores -}
needsUnRenameStr :: String -> String -> Bool
needsUnRenameStr prefix str
   = (pref == prefix) &&
     length suffDig > 0 &&
     nonZeroAll (== '_') suffRest
   where
   (pref, suff) = splitAt (length prefix) str
   (suffDig, suffRest) = span isDigit suff
   nonZeroAll :: (a -> Bool) -> [a] -> Bool
   nonZeroAll p [] = False
   nonZeroAll p (x:xs) = all p (x:xs)
