{-# OPTIONS -cpp #-}
{-------------------------------------------------------------------------------

        Copyright:              Bernie Pope 2003

        Module:                 Synonym 

        Description:            This module contains code for removing type
 				synonyms from types.

        Primary Authors:        Bernie Pope

-------------------------------------------------------------------------------}

{-
    This file is part of buddha.

    buddha is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    buddha is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with buddha; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
-}

module Synonym 
   ( SynonymMap  
   , Synonym (..)  
   , showSynonymMap 
   , synonymMapToList 
   , emptySynonymMap 
   , remSynsFromType 
   , remSynsFromQualType 
   , remSynsFromBangType 
   , remSynsFromTypeSig 
   , insertModuleSyns 
   , synonymMapToDecls 
   , lookupSynonymMap
   , declsToSynonymMap 
   , plusSynonymMap
   ) where

import Language.Haskell.Syntax 

import Data.FiniteMap

import SyntaxUtils 
   ( bogusSrcLoc
   , unQualify 
   )

import Error
   ( fatalError )

--------------------------------------------------------------------------------

-- the details of a synonym other than its name 
data Synonym 
   = Synonym 
     { synonym_args :: [HsName]
     , synonym_rhs  :: HsType 
     }
     deriving Show

-- a map of all synonyms, indexed by the name of the synonym
type SynonymMap = FiniteMap HsQName Synonym 

-- look for a typeconstructor in the synonym map
lookupSynonymMap :: SynonymMap -> HsQName -> Maybe Synonym
lookupSynonymMap = lookupFM

plusSynonymMap :: SynonymMap -> SynonymMap -> SynonymMap
plusSynonymMap = plusFM

emptySynonymMap :: SynonymMap
emptySynonymMap = emptyFM

synonymMapToList :: SynonymMap -> [(HsQName, Synonym)]
synonymMapToList = fmToList

-- pretty crude printing of synonym map 
showSynonymMap :: SynonymMap -> String
showSynonymMap synonymMap 
   = unlines $ map show mapList
   where
   mapList = synonymMapToList synonymMap 

--------------------------------------------------------------------------------

remSynsFromType :: SynonymMap -> HsType -> HsType
remSynsFromType synMap (HsTyFun t1 t2)
   = HsTyFun (remSynsFromType synMap t1) (remSynsFromType synMap t2)
remSynsFromType synMap (HsTyTuple args)
   = HsTyTuple $ map (remSynsFromType synMap) args
remSynsFromType synMap t@(HsTyVar varName) = t
remSynsFromType synMap t@(HsTyCon conName)
   = case lookupSynonymMap synMap conName of
        Nothing -> t
        -- there should not be any arguments for this contrsutor
        Just synonym 
           -> case null (synonym_args synonym) of
                 True  -> remSynsFromType synMap (synonym_rhs synonym)  
                 False -> fatalError __FILE__ __LINE__ $ 
                              "remSynsFromType: partially applied type synonym: " ++ show t
remSynsFromType synMap t@(HsTyApp t1 t2)
   = remSynsFromFlatApp synMap $ flatTypeApp t

remSynsFromFlatApp :: SynonymMap -> [HsType] -> HsType
remSynsFromFlatApp synMap []
   = fatalError __FILE__ __LINE__ $ "remSynsFromFlatApp: empty type application"
remSynsFromFlatApp synMap t@(HsTyCon conName : args)
   = case lookupSynonymMap synMap conName of
        Nothing -> foldl HsTyApp (HsTyCon conName) $ map (remSynsFromType synMap) args
        Just synonym 
        -- check that the synoym is applied to the right number of arguments
           -> let synArgs = synonym_args synonym 
              in case (length synArgs) == (length args) of
                    False -> fatalError __FILE__ __LINE__ $ 
                                "remSynsFromFlatApp: synonym applied to the wrong number of arguments: " ++ show (foldl1 HsTyApp t)
                    True -> remSynsFromType synMap $ replaceTypes (zip synArgs args)
                                                                  (synonym_rhs synonym)
remSynsFromFlatApp synMap t@(HsTyVar varName : args)
   = foldl HsTyApp (HsTyVar varName) $ map (remSynsFromType synMap) args 

replaceTypes :: [(HsName, HsType)] -> HsType -> HsType
replaceTypes table (HsTyFun t1 t2)
   = HsTyFun (replaceTypes table t1) (replaceTypes table t2)
replaceTypes table (HsTyTuple args)
   = HsTyTuple $ map (replaceTypes table) args
replaceTypes table (HsTyApp t1 t2)
   = HsTyApp (replaceTypes table t1) (replaceTypes table t2)
replaceTypes table t@(HsTyVar v)
   = case lookup v table of
        -- all type variables should have a replacement!
        Nothing -> fatalError __FILE__ __LINE__ $ 
                      "replaceTypes: free type variable in synonym: " ++ show t 
        Just newT -> newT 
replaceTypes table t@(HsTyCon name) = t 

-- flatten a type application tree into a list of types
-- the leftmost type must be a type constructor or variable
flatTypeApp :: HsType -> [HsType]
flatTypeApp t
   = flatTypeAppAcc t []
   where
   flatTypeAppAcc (HsTyApp t1 t2) acc = flatTypeAppAcc t1 (t2:acc)
   flatTypeAppAcc t@(HsTyCon conName) acc = t : acc 
   flatTypeAppAcc t@(HsTyVar varName) acc = t : acc 
   flatTypeAppAcc otherType acc
      = fatalError __FILE__ __LINE__ $ 
           "flatTypeApp: leftmost type is not a constructor or var, in type: " ++ show t

remSynsFromQualType :: SynonymMap -> HsQualType -> HsQualType
remSynsFromQualType synMap (HsQualType cntxt t)
   = HsQualType cntxt $ remSynsFromType synMap t

remSynsFromBangType :: SynonymMap -> HsBangType -> HsBangType
remSynsFromBangType synMap (HsBangedTy t)
   = HsBangedTy $ remSynsFromType synMap t
remSynsFromBangType synMap (HsUnBangedTy t)
   = HsUnBangedTy $ remSynsFromType synMap t

remSynsFromTypeSig :: SynonymMap -> HsDecl -> HsDecl 
remSynsFromTypeSig synMap (HsTypeSig sloc names qType)
   = HsTypeSig sloc names $ remSynsFromQualType synMap qType 
-- just in case this is applied to a non sig decl
remSynsFromTypeSig _synMap otherdecl = otherdecl

--------------------------------------------------------------------------------

insertModuleSyns :: Module -> SynonymMap -> [HsDecl] -> SynonymMap
insertModuleSyns modName synMap [] = synMap
insertModuleSyns modName synMap (HsTypeDecl _sloc name args rhs : decls)
   = let thisSynonym = Synonym { synonym_args = args
                               , synonym_rhs  = rhs }
         map1 = addToFM synMap (UnQual name) thisSynonym
         map2 = addToFM map1 (Qual modName name) thisSynonym
     in insertModuleSyns modName map2 decls
insertModuleSyns modName synMap (otherDecl : decls) 
   = insertModuleSyns modName synMap decls


-- similar to above, but does not qualify the synonmys
declsToSynonymMap :: [HsDecl] -> SynonymMap
declsToSynonymMap [] = emptySynonymMap
declsToSynonymMap (HsTypeDecl _sloc name args rhs : decls)
   = let thisSynonym = Synonym { synonym_args = args
                               , synonym_rhs  = rhs }
         restMap = declsToSynonymMap decls
     in addToFM restMap (UnQual name) thisSynonym
declsToSynonymMap (otherDecl : decls) 
   = declsToSynonymMap decls

--------------------------------------------------------------------------------

-- convert a synonym map back into a list of HsDecls
-- and expand all the types in the synonym
-- this enures that the synonyms are in a canonical form
-- and do not refer to any other synonyms

synonymMapToDecls :: SynonymMap -> SynonymMap -> [HsDecl]
synonymMapToDecls allSyns synMap 
   = map (toSynDecl allSyns) $ synonymMapToList synMap
   where
   toSynDecl :: SynonymMap -> (HsQName, Synonym) -> HsDecl
   toSynDecl synMap (qName, synonym)
      = HsTypeDecl SyntaxUtils.bogusSrcLoc 
                   (SyntaxUtils.unQualify qName)
                   (synonym_args synonym)
                   (remSynsFromType synMap $ synonym_rhs synonym)
