{-# OPTIONS -cpp #-}
{-------------------------------------------------------------------------------

        Copyright:              Bernie Pope 2003

        Module:                 SyntaxUtils

        Description:            Generic syntax related utilities that don't 
                                have a good home anywhere else.

        Primary Authors:        Bernie Pope

-------------------------------------------------------------------------------}

{-
    This file is part of buddha.

    buddha is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    buddha is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with buddha; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
-}

module SyntaxUtils 
   ( leftTyConName
   , bogusSrcLoc
   , isBogusSrcLoc
   , madeUpSrcLoc
   , isMadeUpSrcLoc
   , trustedSrcLoc
   , isTrustedSrcLoc
   , fromHsQName
   , fromHsQNameQual
   , hsSpecialConToString
   , fromHsName
   , numArgsInQualType
   , numArgsInType
   , opExp
   , mkPreludeName
   , mkParen
   , mkParenPat
   , dropParens
   , foldApp
   , unfoldApp
   , unQualify
   , reQualify
   , dropQualifier
   , varsFromPat
   , prefixName
   , isQualified
   , maybeIsQualified 
   , isQualifiedType
   , thisModQualified
   , findImports
   , isImportAliased
   , isImportAll
   , isImportQualified
   , isDataDecl
   , isNewTypeDecl
   , updateName
   , patContainsLit
   , patContainsRec
   , decomposeQualType
   , qVarsFromContext
   , varsFromType 
   , litStringExp
   , isSimplePat
   , patToString
   ) where

import Language.Haskell.Syntax
import Language.Haskell.Pretty

import Error ( fatalError )

import List ( nub )

--------------------------------------------------------------------------------

data SrcLocEnum 
   = Bogus | MadeUp | Trusted  
   deriving (Enum)

-- an intentionally bogus source location
bogusSrcLoc :: SrcLoc
bogusSrcLoc 
   = SrcLoc 
     { srcFilename = []
     , srcLine = bogusNum
     , srcColumn = bogusNum
     }
   where
   bogusNum = negate $ fromEnum Bogus

isBogusSrcLoc :: SrcLoc -> Bool
isBogusSrcLoc sloc = sloc == bogusSrcLoc

-- a src location for trusted introduced pattern bindings
trustedSrcLoc :: SrcLoc
trustedSrcLoc
   = SrcLoc 
     { srcFilename = []
     , srcLine     = trustNum 
     , srcColumn   = trustNum  
     }
   where
   trustNum = negate $ fromEnum Trusted

isTrustedSrcLoc :: SrcLoc -> Bool
isTrustedSrcLoc sloc = sloc == trustedSrcLoc 

-- a made up srcLoc that is not supposed to be bogus, ie not the same as bogusSrcLoc
madeUpSrcLoc :: SrcLoc
madeUpSrcLoc 
   = SrcLoc 
     { srcFilename = []
     , srcLine = 0
     , srcColumn = 0
     }
   where
   madeUpNum = negate $ fromEnum MadeUp

isMadeUpSrcLoc :: SrcLoc -> Bool
isMadeUpSrcLoc sloc = sloc == bogusSrcLoc

-- get the string out of a qualified name
-- Foo.bar ---> "bar"
fromHsQName :: HsQName -> String
fromHsQName (Qual _mod name)  = fromHsName name
fromHsQName (UnQual name)     = fromHsName name
fromHsQName (Special specCon) = hsSpecialConToString specCon

-- keep the module qualifier in the name
fromHsQNameQual :: HsQName -> String
fromHsQNameQual (Qual (Module modName) name)  = modName ++ "." ++ fromHsName name
fromHsQNameQual (UnQual name)     = fromHsName name
fromHsQNameQual (Special specCon) = hsSpecialConToString specCon

-- get the string out of a name
fromHsName :: HsName -> String
fromHsName (HsIdent s)   = s
fromHsName (HsSymbol s)  = s

-- produce a string from the special constructors
hsSpecialConToString :: HsSpecialCon -> String
hsSpecialConToString HsUnitCon = "()"
hsSpecialConToString HsListCon = "[]"
hsSpecialConToString HsFunCon = "->"
hsSpecialConToString (HsTupleCon i)
   = "(" ++ (replicate (i - 1) ',') ++ ")"
hsSpecialConToString HsCons = "(:)" 

-- count the number of outermost arrows in the type
numArgsInQualType :: HsQualType -> Int
numArgsInQualType (HsQualType _cntxt t)
   = numArgsInType t

numArgsInType :: HsType -> Int
numArgsInType (HsTyFun t1 t2)
   = 1 + numArgsInType t2
numArgsInType otherType = 0

-- turn a qualified operator into an expression
opExp :: HsQOp -> HsExp
opExp (HsQVarOp name) = HsVar name
opExp (HsQConOp name) = HsCon name

-- make a name for a prelude identifier
mkPreludeName :: String -> HsQName
mkPreludeName s = Qual prelude_mod $ HsIdent s

-- foldApp [f, e1, e2, e3 ...] ---> (... (((f e1) e2) e3) ...)
foldApp :: [HsExp] -> HsExp
foldApp exps@(_:_) = foldl1 HsApp exps
foldApp [] = fatalError __FILE__ __LINE__ $ "foldApp: empty expression list"

-- unfoldApp (... (((f e1) e2) e3) ...) --->  [f, e1, e2, e3 ...]
unfoldApp :: HsExp -> [HsExp]
unfoldApp (HsApp e1 e2)
   = unfoldApp e1 ++ [e2]
-- get rid of redundant left parens
unfoldApp (HsParen e) = unfoldApp e
unfoldApp anythingElse = [anythingElse]

-- put parens around an expression if necessary
-- helps reduce output of redundant parens
mkParen :: HsExp -> HsExp
mkParen exp@(HsVar e) = exp
mkParen exp@(HsCon c) = exp
-- negative numbers need parens
mkParen exp@(HsLit lit) 
   = case lit of
        HsInt i -> if i < 0 then HsParen exp else exp
        HsFrac f -> HsParen exp
        other -> exp
mkParen exp@(HsTuple exps) = exp
mkParen exp@(HsList exps) = exp
mkParen exp@(HsParen e) = exp
mkParen exp@(HsEnumFrom e) = exp
mkParen exp@(HsEnumFromThen e1 e2) = exp
mkParen exp@(HsEnumFromTo e1 e2) = exp
mkParen exp@(HsEnumFromThenTo e1 e2 e3) = exp
mkParen exp@(HsListComp e stmts) = exp
mkParen otherExp = HsParen otherExp

-- put parens around a pattern if necessary
-- helps reduce output of redundant parens
mkParenPat :: HsPat -> HsPat
mkParenPat pat@(HsPVar v) = pat
mkParenPat pat@(HsPLit lit) = pat 
mkParenPat pat@(HsPNeg p) = HsPParen pat
mkParenPat pat@(HsPInfixApp p1 con p2) = HsPParen pat
mkParenPat pat@(HsPApp con []) = pat
mkParenPat pat@(HsPApp con ps@(_:_)) = HsPParen pat
mkParenPat pat@(HsPTuple ps) = pat
mkParenPat pat@(HsPList ps) = pat
mkParenPat pat@(HsPParen p) = pat
mkParenPat pat@(HsPRec con fields) = HsPParen pat
mkParenPat pat@(HsPAsPat name p) = pat
mkParenPat HsPWildCard = HsPWildCard
mkParenPat pat@(HsPIrrPat p) = HsPParen pat

-- drop all parens surrounding an expression
dropParens :: HsExp -> HsExp
dropParens (HsParen e) = dropParens e
dropParens otherExp = otherExp

-- collect the names of all the variables that occur in a pattern
varsFromPat :: HsPat -> [HsName]
varsFromPat (HsPVar name) = [name]
varsFromPat (HsPNeg pat) = varsFromPat pat
varsFromPat (HsPInfixApp pat1 _con pat2)
   = varsFromPat pat1 ++ varsFromPat pat2
varsFromPat (HsPApp _con args)
   = concatMap varsFromPat args
varsFromPat (HsPTuple args)
   = concatMap varsFromPat args
varsFromPat (HsPList args)
   = concatMap varsFromPat args
varsFromPat (HsPParen pat)
   = varsFromPat pat
varsFromPat (HsPRec _recName patFields)
   = concatMap varsFromPatField patFields
varsFromPat (HsPAsPat name pat)
   = name : varsFromPat pat
varsFromPat (HsPIrrPat pat)
   = varsFromPat pat
varsFromPat _anythingElse
   = []

varsFromPatField :: HsPatField -> [HsName]
varsFromPatField (HsPFieldPat _qname pat) = varsFromPat pat

-- prefix Name adds a string prefix to an existing name
prefixName :: String -> HsName -> HsName
prefixName prefix name
   = case name of
        HsIdent s   -> HsIdent   $ prefix ++ s
        HsSymbol s  -> HsSymbol  $ prefix ++ s

-- true if the name is qualified to a module
isQualified :: HsQName -> Bool
isQualified (Qual _ _) = True 
isQualified _ = False 

-- if a name is qualifed return the qualifier, else nothing
maybeIsQualified :: HsQName -> Maybe Module 
maybeIsQualified (Qual q _) = Just q 
maybeIsQualified _ = Nothing 

-- turn a qualified HsQName into an unqualified HsName
unQualify :: HsQName -> HsName
unQualify (Qual _mod name) = name
unQualify (UnQual name)    = name
unQualify (Special _specCon)
   = fatalError __FILE__ __LINE__ $ "unQualify : applied to special qualified name"

-- drop the qualification from a name but keep it as a HsQName 
dropQualifier :: HsQName -> HsQName
dropQualifier (Qual _mod name) = UnQual name
dropQualifier name@(UnQual _n) = name
dropQualifier name@(Special _specCon) = name

-- true if a name is qualified to a particular module
thisModQualified :: Module -> HsQName -> Bool
thisModQualified mod1 (Qual mod2 _name) = mod1 == mod2
thisModQualified _mod1 (UnQual _name) = False

-- check if a module is imported from a list of import declarations
-- true if it is imported by itself or as an alias
findImports :: Module -> [HsImportDecl] -> [HsImportDecl]
findImports modName [] = [] 
findImports modName (decl@(HsImportDecl _sloc importedModule _qual _alias _items) : imports)
   | modName == importedModule = decl : findImports modName imports
   | otherwise = findImports modName imports

-- is an import aliased?
isImportAliased :: HsImportDecl -> Bool
isImportAliased (HsImportDecl _sloc _importedModule _qual (Just _alias) _items) = True
isImportAliased other = False

-- does an import include everything from a module?
isImportAll :: HsImportDecl -> Bool
isImportAll (HsImportDecl _sloc _importedModule _qual _alias Nothing) = True
isImportAll other = False 

-- is an import qualified
isImportQualified :: HsImportDecl -> Bool
isImportQualified (HsImportDecl _sloc _importedModule True _alias _items) = True
isImportQualified other = False


{- conjure up a string name for the leftmost type constructor in a type

   t1 -> == "->"
   (a,b,c) == "(,,)"
-}

leftTyConName :: HsType -> String
leftTyConName (HsTyFun _t1 _t2) = "->"
leftTyConName (HsTyTuple args) = '(' : (replicate (length args - 1) ',' ++ ")")
leftTyConName (HsTyApp t1 _t2) = leftTyConName t1
leftTyConName (HsTyVar name) = fromHsName name
leftTyConName (HsTyCon qName) = fromHsQNameQual qName

isQualifiedType :: HsQualType -> Bool
isQualifiedType (HsQualType cntxt _t) = not $ null $ cntxt

-- predicates on declaration type
isDataDecl :: HsDecl -> Bool
isDataDecl (HsDataDecl _sloc _cntxt _tyName _args _condecls _derives) = True
isDataDecl otherDecl = False

isNewTypeDecl :: HsDecl -> Bool
isNewTypeDecl (HsNewTypeDecl _sloc _cntxt _tyName _args _condecl _derives) = True
isNewTypeDecl otherDecl = False

-- get the names of the selectors from a fieldUpdate
updateName :: HsFieldUpdate -> HsQName
updateName (HsFieldUpdate name _exp) = name

{- True if the pattern contains a literal -}
patContainsLit :: HsPat -> Bool
patContainsLit (HsPVar _v) = False
patContainsLit (HsPLit _lit) = True
patContainsLit (HsPNeg pat) = patContainsLit pat
patContainsLit (HsPInfixApp p1 op p2)
   = patContainsLit p1 || patContainsLit p2
patContainsLit (HsPApp con pats)
   = any patContainsLit pats
patContainsLit (HsPTuple pats)
   = any patContainsLit pats
patContainsLit (HsPList pats)
   = any patContainsLit pats
patContainsLit (HsPParen pat)
   = patContainsLit pat
patContainsLit pat@(HsPRec _qname fields)
   = any patFieldContainsLit fields
   where
   patFieldContainsLit (HsPFieldPat _conName pat) = patContainsLit pat
patContainsLit (HsPAsPat _name pat)
   = patContainsLit pat
patContainsLit HsPWildCard = False
patContainsLit (HsPIrrPat pat) = patContainsLit pat

{- True if the pattern contains a record -}
patContainsRec :: HsPat -> Bool
patContainsRec (HsPVar _v) = False
patContainsRec (HsPLit _lit) = False 
patContainsRec (HsPNeg pat) = patContainsRec pat
patContainsRec (HsPInfixApp p1 op p2)
   = patContainsRec p1 || patContainsRec p2
patContainsRec (HsPApp con pats)
   = any patContainsRec pats
patContainsRec (HsPTuple pats)
   = any patContainsRec pats
patContainsRec (HsPList pats)
   = any patContainsRec pats
patContainsRec (HsPParen pat)
   = patContainsRec pat
patContainsRec pat@(HsPRec _qname fields) = True 
patContainsRec (HsPAsPat _name pat)
   = patContainsRec pat
patContainsRec HsPWildCard = False
patContainsRec (HsPIrrPat pat) = patContainsRec pat

reQualify :: Module -> HsQName -> HsQName
reQualify newModName (Qual oldModName name) = Qual newModName name
reQualify newModName (UnQual name) = Qual newModName name
reQualify _newModName name@(Special _specCon) = name

decomposeQualType :: HsQualType -> (HsContext, [HsType], HsType)
decomposeQualType (HsQualType ctxt t)
   = (ctxt, reverse argTypes, resultType)
   where
   (argTypes, resultType) = decomposeType [] t
   decomposeType :: [HsType] -> HsType -> ([HsType], HsType)
   decomposeType args (HsTyFun t1 t2)
      = decomposeType (t1:args) t2
   decomposeType args otherType = (args, otherType)    

qVarsFromContext :: HsContext -> [HsType]
qVarsFromContext cntxt 
   = nub $ filter isTypeVar $ concatMap tyFromAsst cntxt
   where
   isTypeVar :: HsType -> Bool
   isTypeVar (HsTyVar _varName) = True
   isTypeVar other = False
   tyFromAsst :: HsAsst -> [HsType]
   tyFromAsst (qName, types) = types 

varsFromType :: HsType -> [HsType] 
varsFromType (HsTyFun t1 t2) 
   = let ts1 = varsFromType t1
         ts2 = varsFromType t2
     in ts1 ++ ts2
varsFromType (HsTyTuple ts)
   = concatMap varsFromType ts
varsFromType (HsTyApp t1 t2) 
   = let ts1 = varsFromType t1
         ts2 = varsFromType t2
     in ts1 ++ ts2
varsFromType t@(HsTyVar var) = [t]
varsFromType (HsTyCon c) = []

-- make a literal string expression
litStringExp :: String -> HsExp
litStringExp s = HsLit $ HsString s

-- is this pattern simple?
isSimplePat :: HsPat -> Bool
isSimplePat (HsPVar v) = True
isSimplePat other = False 

-- nice printable version of a pattern
patToString :: HsPat -> String 
patToString p = prettyPrint p
