{-------------------------------------------------------------------------------

        Copyright:              Bernie Pope 2003

        Module:                 TransOpts 

        Description:            Options for transforming each module 

        Primary Authors:        Bernie Pope

-------------------------------------------------------------------------------}

{-
    This file is part of buddha.

    buddha is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    buddha is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with buddha; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
-}

module TransOpts 
   ( ContextMap  
   , Context  
   , TransOpt (..)
   , defaultTransOpt
   , showContextMap
   , lookupContextMap
   , addToContextMap 
   , emptyContextMap 
   , readTransOpts 
   )
   where

import Data.FiniteMap

import Text.ParserCombinators.Parsec

import Language.Haskell.Syntax 
   ( Module (..) )

import IO                      
   ( hGetContents
   , isDoesNotExistError 
   , try
   , openFile
   , IOMode (ReadMode) 
   )

import Error                   
   ( abortWithError
   , ErrorCode (FileIOError) 
   )

--------------------------------------------------------------------------------

data TransOpt 
   = Trust
   | Suspect
   | Empty 
   | NoKids
   | ReEval 
   deriving (Show, Eq)

defaultTransOpt :: TransOpt
defaultTransOpt = Suspect 

type Context = [String]

type ContextMap = FiniteMap Context TransOpt

showContextMap :: ContextMap -> String
showContextMap contextMap
   = unlines $ map show mapList
     where
     mapList = fmToList contextMap 

lookupContextMap :: ContextMap -> Context -> Maybe TransOpt
lookupContextMap = lookupFM

emptyContextMap :: ContextMap
emptyContextMap = emptyFM

addToContextMap :: ContextMap -> Context -> TransOpt -> ContextMap
addToContextMap = addToFM

--------------------------------------------------------------------------------

-- the parser

processFile :: String -> ContextMap
processFile fileContents
   = foldr processLine emptyContextMap $ lines fileContents

processLine :: String -> ContextMap -> ContextMap
processLine line contextMap
   = case parse parseLine "" line of
        -- parse failed don't alter the context map
        Left err -> contextMap
        Right (context, opt) -> addToFM contextMap context opt

parseLine :: Parser (Context, TransOpt)
parseLine 
   = do spaces 
        context <- parseIdents
        spaces 
        semiColon
        spaces
        opt <- parseOpt
        -- reverse the context to form a more convenient stack
        return (reverse context, opt)

semiColon :: Parser Char
semiColon = char ';'
        
parseIdents :: Parser Context
parseIdents 
   = many1 $ do i <- ident 
                spaces 
                return i 

parseOpt :: Parser TransOpt
parseOpt =     do { string "Trust";   return Trust   }
           <|> do { string "Suspect"; return Suspect }
           <|> do { string "Empty";   return Empty   }
           <|> do { string "NoKids";  return NoKids  }
           <|> do { string "ReEval";  return ReEval  }

-- XXX we should rename here!
ident :: Parser String
ident = many1 identChar

identChar :: Parser Char
identChar = alphaNum <|> 
            satisfy (`elem` ",._'*+-/&^%$#@!~()><=?|:[]\\")

--------------------------------------------------------------------------------

{- read the transform options from a file
   if the module name is Foo then the options file is
   Buddha/Foo.opt
-}
readTransOpts :: Module -> IO ContextMap
readTransOpts (Module modName)
   = do let filePath = "Buddha/" ++ modName ++ ".opt"
        ioResult <- IO.try $ openFile filePath ReadMode 
        case ioResult of
           Left err 
              -> case isDoesNotExistError err of
                    True  -> return emptyContextMap 
                    False -> abortWithError FileIOError $ 
                                "could not open file: " ++ show filePath
           Right handle -> do contents <- hGetContents handle 
                              return $ processFile contents 
