{-# OPTIONS -cpp #-}
{-------------------------------------------------------------------------------

        Copyright:              Bernie Pope 2003

        Module:                 Transform 

        Description:            Debugging Source transformation for a 
                                single source file/module. 

        Primary Authors:        Bernie Pope

-------------------------------------------------------------------------------}

{-
    This file is part of buddha.

    buddha is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    buddha is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with buddha; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
-}

module Transform 
   (transformSrcFiles) 
   where

import Language.Haskell.Syntax  

import Language.Haskell.Pretty  

import Monad                    
   ( when
   , unless
   , foldM
   , liftM
   , liftM2
   , liftM3 
   , liftM4
   )

import ParserUtils              
   ( parseHsSource )

import SyntaxUtils              
   ( bogusSrcLoc
   , isBogusSrcLoc
   , trustedSrcLoc
   , isTrustedSrcLoc
   , fromHsName
   , fromHsQNameQual
   , hsSpecialConToString
   , mkParen
   , mkParenPat
   , dropParens
   , varsFromPat
   , foldApp
   , unfoldApp
   , varsFromPat
   , findImports 
   , leftTyConName 
   , litStringExp
   , patToString
   , isSimplePat
   )

import IfaceUtils               
   ( IdentQual (..)
   , addListItemsToMap 
   )

import Data.FiniteMap

import IO                       
   ( stderr 
   , stdout
   , hPutStrLn 
   , hPutStr
   , hClose
   , hGetContents
   , IOMode (..) 
   )

import Directory                
   ( getCurrentDirectory )

import Iface                    
   ( processImports
   , ModIface (..)
   , AssocMap
   , mkModIface
   , IfaceCache
   , showImportMap
   , interfaceDirectory
   , emptyIfaceCache
   , assocFromDecls
   , plusAssocMap
   , showAssocMap
   , writeModIface 
   , exportSelectors
   , importSelectors
   )

import Ident                    
   ( IdentMap
   , IdentInfo (..)
   , IdentBind (..)
   , Arity
   , lookupIdent
   , constructorIdents
   , joinIdentMap
   , showIdentMap
   , declIdents  
   , isLetBound 
   , isPatBound )

import FileIO                   
   ( tryOpenFile
   , createFileOverWrite 
   , copyFile
   , directorySep
   , baseName
   )

import List                     
   ( isSuffixOf )

import Desugar                  
   ( runDesugar
   , desugarDecl 
   , isMarkedSrcLoc 
   )

import TransMonad               
   ( runTrans
   , isFlagSet
   , Trans
   , getModName
   , context
   , getTransOpt 
   , freshVar  
   , nFreshVars 
   , pushConstantStack
   , popConstantStack
   , recordConstant
   , getRecentSrcLoc 
   , lookupTypeSigT
   , location
   , incPartialStatByT
   , incSaturatedStatByT
   , incOverSatStatByT
   , incLambdaStatByT
   , incConstantsStatByT
   , incMiscApsStatByT
   , incPatVarApStatByT
   )

import Error                    
   ( fatalError 
   , abortWithError
   , ErrorCode (..) )

import Infix                    
   ( infixer
   , insertTopInfixDecls
   , showInfixMap 
   ) 

import Synonym                  
   ( showSynonymMap
   , insertModuleSyns 
   )

import Unlit                    
   ( unlit )

import Rename                   
   ( runRename
   , defaultVarPrefix
   , unRename 
   , renameModule 
   )

import TransOpts		
   ( readTransOpts
   , TransOpt (..)  
   , showContextMap 
   ) 

import TypeSigMap               
   ( typeSigMapFromDecls
   , showTypeSigMap 
   )

import Derive                   
   ( deriveInstances )

import Opts                     
   ( flagIsSet
   , CmdLine (..) 
   )

import Record                   
   ( desugarRecords
   , showRecordMap
   , mkSelectorSet
   )

import Data.Set                 
   ( Set )

import Statistics               
   ( showStats 
   , Statistics 
   , initStats 
   , addStats
   )

import System
   ( ExitCode (..) 
   , system
   )

import ChatIO
   ( ifChat
   , ChatLevel (..)
   )

import BuddhaName
   ( buddhaNameVersion )

--------------------------------------------------------------------------------
-- transform all the source files 
-- thread the interface cache through each transformation to avoid re-reading
-- module interface files on each import
transformSrcFiles :: FilePath
                  -> CmdLine 
                  -> [FilePath]
                  -> IO Statistics 
transformSrcFiles ifacePath cmdLine files
   = transformSrcFiles' cmdLine emptyIfaceCache initStats files 
   where
   -- a bit of tedium to thread the interface cache through all the transformations
   transformSrcFiles' _cmdLine _cache stats [] = return stats
   transformSrcFiles' cmdLine cache stats (file:files) 
      = do (newCache, newTotalStats) <- transformSrcFile ifacePath cmdLine cache stats file
           transformSrcFiles' cmdLine newCache newTotalStats files

-- transform a single source file
transformSrcFile :: FilePath         -- the dir where the libs are
                 -> CmdLine          -- options pertaining to how the transformation works
                 -> IfaceCache       -- cache of module interfaces 
                 -> Statistics       -- accumulated stats for whole program
                 -> FilePath         -- the path of the file to transform 
                 -> IO (IfaceCache, Statistics)
transformSrcFile ifacePath cmdLine ifaceCache totalStats filePath 
   = do let dumps = dumpFlags cmdLine
        fileContents <- readFileAndPreProcess cmdLine filePath
        -- perhaps unlit a literate module
        let srcText = maybeUnlit filePath fileContents 
        moduleSyntax <- parseHsSource filePath srcText
        -- dump the parse tree if requested on the command line
        when (flagIsSet dumps "parse") $ hPutStrLn stdout $ show moduleSyntax
        -- rename the module
        let modRenamed = runRename (renameModule moduleSyntax)
        -- dump the renamed module if requested 
        when (flagIsSet dumps "rename") $ hPutStrLn stdout $ 
                                          prettyPrintWithMode printMode modRenamed
        -- break the module into components
        let (HsModule srcLoc modName exports imports decls) = modRenamed 
        -- possibly add an implicit import of the Prelude to the import decls
        let importsMaybeWithPrelude = importPrelude cmdLine imports
        -- process the import decls 
        (newIfaceCache, importMap, importModIface)
           <- processImports ifacePath importsMaybeWithPrelude ifaceCache 
        let (importIdents, importAssocs, importInfixMap, importSynonyms, importRecordMap)
               = ( modIface_idents     importModIface
                 , modIface_assocs     importModIface
                 , modIface_infixMap   importModIface
                 , modIface_synonymMap importModIface
                 , modIface_recordMap  importModIface )  
        -- possibly print the import map
        when (flagIsSet dumps "imports") $ hPutStrLn stdout $ showImportMap importMap
        -- collect toplevel infix decls from this module and include them with the imported infix rules 
        let totalInfixMap = insertTopInfixDecls modName importInfixMap decls
        when (flagIsSet dumps "infix") $ hPutStrLn stdout $ showInfixMap totalInfixMap
        -- reparse the infix applications according to the known fixity information
        let declsInfixed = infixer totalInfixMap decls

        -- get the tycon and class associations for this module
        -- this must be done before records are desugared so that field names can
        -- be gethered for record constructors 
        let moduleAssocs = assocFromDecls modName declsInfixed 
        -- join the module associations with those from imports
        let totalAssocs = importAssocs `plusAssocMap` moduleAssocs
        -- possibly print the association info out
        when (flagIsSet dumps "assocs") $ hPutStrLn stdout $ showAssocMap totalAssocs

        -- convert records to constructors and get record map for this module
        let (totalRecordMap, declsNoRecs, selectorDecls) 
             = desugarRecords modName importRecordMap declsInfixed 
        when (flagIsSet dumps "records") $ hPutStrLn stdout $ showRecordMap totalRecordMap
        when (flagIsSet dumps "selectors") $ hPutStrLn stdout 
                                           $ unlines $ map prettyPrint selectorDecls 
        let declsAndSelectors = declsNoRecs ++ selectorDecls
        -- get derived instance declarations
        let derivedInstanceDecls = deriveInstances totalInfixMap declsAndSelectors 
        when (flagIsSet dumps "derives") $ hPutStrLn stdout 
                                         $ unlines $ map prettyPrint derivedInstanceDecls
        let declsAndDerives = declsAndSelectors ++ derivedInstanceDecls
        -- desugar all the decls
        -- collect all synonyms from the module 
        let totalSynonyms = insertModuleSyns modName importSynonyms declsAndDerives 
        when (flagIsSet dumps "synonyms") $ hPutStrLn stdout $ showSynonymMap totalSynonyms
        -- we start at 1 because 0 is reserved for the module identifier
        -- raise this number if you want any more special identifiers
        let desugarResult 
               = runDesugar 1 totalSynonyms totalRecordMap $ mapM desugarDecl declsAndDerives
        (declsNoSugar, count1) 
           <- case desugarResult of
                 Left err -> abortWithError DesugarError err
                 Right result -> return result
        when (flagIsSet dumps "desugar") $ 
                do let mod = HsModule srcLoc modName exports importsMaybeWithPrelude declsNoSugar
                   hPutStrLn stdout $ prettyPrintWithMode printMode mod 
        -- collect all the top-level type sigs in the module
        let typeSigs = typeSigMapFromDecls declsNoSugar
        when (flagIsSet dumps "signatures") $ hPutStrLn stdout $ showTypeSigMap typeSigs
        -- collect the ident infos of the constructors in this module
        let consIdents = constructorIdents (Qual modName) declsNoSugar 
        -- collect the ident infos of top level vars in this module 
        let varsIdents = declIdents True (Qual modName) QualAndUnQual typeSigs declsNoSugar 
        -- no overlap between vars and cons, so plusFM is sufficient
        let moduleIdents = varsIdents `plusFM` consIdents
        -- combine imported infos with constructor infos 
        -- we use joinIdentMap to catch any overlapping values
        let initialIdentMap = moduleIdents `joinIdentMap` importIdents
        -- possibly print the identifier info 
        when (flagIsSet dumps "idents") $ hPutStrLn stdout $ showIdentMap initialIdentMap
        -- read the transformation options
        contextMap <- readTransOpts modName
        when (flagIsSet dumps "transOpts") $ hPutStrLn stdout $ showContextMap contextMap
        -- run the transformation
        let transResult
               = runTrans modName cmdLine count1 contextMap typeSigs $ 
                       mapM (transDecl initialIdentMap) declsNoSugar
        (transDecls, count2, stats)
           <- case transResult of
                 Left err -> abortWithError DesugarError err
                 Right result -> return result
        -- possibly print stats about the module
        when (flagIsSet dumps "statsByModule") 
             ( do hPutStrLn stdout $ show modName ++ ", stats"
                  hPutStrLn stdout $ showStats stats )
        -- create an additional decl to name the module and the file
        -- so that they can be referenced by each EDT node
        let modIdDecl = mkModId modName 
        -- transform the export declarations to ensure all names are transformed correctly
        let transExports 
               = case exports of
                    Nothing    -> Nothing 
                    Just specs -> Just $ map transExportSpec specs 
        let selectorSet = mkSelectorSet totalRecordMap
        let exportsWithSelectors
               = exportSelectors selectorSet totalAssocs transExports
        -- transform the imports to make sure each module as _Buddha appendix
        let transImports = transformImports selectorSet totalAssocs importsMaybeWithPrelude 
        -- make the new name for the module with the _Buddha appendix
        let transModName = transModuleName modName 
        -- rebuild the new module
        let transMod = HsModule srcLoc transModName exportsWithSelectors transImports 
                               $ modIdDecl : (concat transDecls)
        -- write the transformed module
        writeTransformedMod transMod 
        -- create a module interface
        modIface <- mkModIface 
                       modName
                       importMap
                       ModIface 
                          { modIface_idents     = initialIdentMap 
                          , modIface_assocs     = totalAssocs 
                          , modIface_infixMap   = totalInfixMap 
                          , modIface_synonymMap = totalSynonyms 
                          , modIface_recordMap  = totalRecordMap
                          }
                       exports
        -- write the interface file for this module
        writeModIface modName totalSynonyms modIface 
        return (newIfaceCache, addStats totalStats stats)

-- decide whether to unlit a file based on its extension 
maybeUnlit :: FilePath -> String -> String
maybeUnlit filePath contents
   = case List.isSuffixOf ".lhs" filePath of
        True -> unlit filePath contents
        False -> contents

--------------------------------------------------------------------------------

-- read the source file and possibly apply a pre-processor to it, such as CPP
-- if requested on the command line

buddhaDir :: String
buddhaDir = "Buddha"

-- XXX hacky definition of cpp
cpp = "gcc -xc -E -P -traditional-cpp -undef"

definedMacros :: CmdLine -> String
definedMacros cmdLine
   = unwords $ map (\opt -> "-D" ++ escapeQuotes opt) $ defines cmdLine 

escapeQuotes :: String -> String
escapeQuotes []       = []
escapeQuotes ('"':cs) = '\\' : '"' : escapeQuotes cs
escapeQuotes (c:cs)   = c : escapeQuotes cs

-- assumes gcc is in the path
cppCmd :: String -> FilePath -> FilePath -> String
cppCmd defs from to = unwords [cpp, defs, from, "-o " ++ to]

readFileAndPreProcess :: CmdLine -> FilePath -> IO String
readFileAndPreProcess cmdLine filePath
   = do if needCpp cmdLine 
           then do let fileCpp = buddhaDir ++ directorySep ++ baseName filePath ++ ".cpp" 
                   let command = cppCmd (definedMacros cmdLine) filePath fileCpp
                   ifChat Verbose $ putStrLn $ buddhaNameVersion ++ ": executing: " ++  command
                   status <- system command 
                   unless (status == ExitSuccess) $ do
                      abortWithError FileIOError $
                                     "pre-processing failed on file: " ++ filePath
                   handle <- tryOpenFile ReadMode [fileCpp]
                   hGetContents handle
           else do handle <- tryOpenFile ReadMode [filePath]
                   hGetContents handle
        -- don't close the file b/c we are reading it with hGetContents
        -- it will be semi-closed already

--------------------------------------------------------------------------------

-- possibly add an implicit import of the Prelude
-- if there is not an explicit import of it
-- the command line can override the implicit import with
-- the '-d noprelude' option
importPrelude :: CmdLine -> [HsImportDecl] -> [HsImportDecl]
importPrelude cmdLine imports
     -- command line option to not import the prelude
     -- useful when transforming the prelude itself
   | "noprelude" `elem` (transFlags cmdLine) = imports
   -- check to see if the prelude is already completely imported.
   -- if not add an explicit qualified import
   | otherwise 
        = if null preludeImports
             -- the prelude is not imported at all so we should
             -- import the whole thing
             then preludeImport : imports
             -- the prelude is imported at least once already
             -- so we should just import it qualified plus the
             -- needed bits (currently: fromInteger, fromRational)
             else preludeQualImport : preludeExtras : imports
   where
   preludeImports = findImports prelude_mod imports 
   preludeImport :: HsImportDecl
   preludeImport 
      = HsImportDecl bogusSrcLoc 
                     prelude_mod
                     False       -- not qualified
                     Nothing     -- no alias for it
                     Nothing     -- import everything
   preludeQualImport :: HsImportDecl
   preludeQualImport  
      = HsImportDecl bogusSrcLoc 
                     prelude_mod
                     True        -- qualified
                     Nothing     -- no alias 
                     Nothing     -- import everything
   preludeExtras 
      = HsImportDecl bogusSrcLoc
                     prelude_mod
                     False       -- not qualified
                     Nothing     -- no alias for it
                     (Just (False, neededPreludeThings)) -- essentials from the prelude 
                     
-- these things are absolutely necessary to have in scope
neededPreludeThings :: [HsImportSpec]
neededPreludeThings
   = [HsIVar $ HsIdent "fromInteger", HsIVar $ HsIdent "fromRational"]  


--------------------------------------------------------------------------------

-- the prefix to put at the front of a transformed modules name
transModuleAppendix = "_B" 

-- write out the transformed module
writeTransformedMod :: HsModule -> IO ()
writeTransformedMod mod@(HsModule sloc (Module modName) exports imports decls)
   = do currentDir <- getCurrentDirectory
        let fileName = modName ++ ".hs"
        handle <- createFileOverWrite (currentDir ++ "/" ++ interfaceDirectory)
                                      fileName
        let moduleStr 
             = prettyPrintWithMode printMode  
                                   (HsModule sloc 
                                       (Module modName) 
                                       exports 
                                       imports 
                                       decls)
        hPutStr handle moduleStr
        hClose handle
 
-- transform the imported module names and add the standard 
-- debugging libraries
transformImports :: Set HsQName -> AssocMap -> [HsImportDecl] -> [HsImportDecl]
transformImports selectorSet assocs imps 
   = debugLibImports ++ transImports' imps
   where
   debugLibImports = [debugLibImport]
   debugLibImport = HsImportDecl bogusSrcLoc buddhaLibFullName True (Just buddhaLibName) Nothing 
   transImports' [] = []
   transImports' ((HsImportDecl sloc modName isQual asName things):imps)
      = thisImportDecl : restImports
      where
      thisImportDecl
         = HsImportDecl sloc 
                    (transModuleName modName) 
                    isQual 
                    (transAsName asName) 
                    thingsWithSelectors
      thingsWithSelectors = importSelectors selectorSet assocs things
      restImports = transImports' imps 

-- we have the imports:
-- import Buddha as B 
-- the name of the standard Buddha library
buddhaLibName = Module "B"
buddhaLibFullName = Module "Buddha"

-- some standard names introduced into the program
valConName    = Qual buddhaLibName $ HsIdent "V" 
applyName     = HsIdent "app"
applyQName    = Qual buddhaLibName applyName

packStringQName = Qual buddhaLibName $ HsIdent "ps"
mkPackedString :: String -> HsExp
mkPackedString str = foldApp [HsVar packStringQName, HsLit $ HsString str]

nodeName      = HsIdent "node"
nodeQName     = Qual buddhaLibName nodeName

-- for functions that are recorded in the function table
funStr  = "f"
mkFunQName :: Int -> HsQName 
mkFunQName i = Qual buddhaLibName $ HsIdent $ funStr ++ show i

-- for functions that are unrecorded in the funciton table
funUnRecStr  = "fu"
mkFunUnRecQName :: Int -> HsQName 
mkFunUnRecQName i = Qual buddhaLibName $ HsIdent $ funUnRecStr ++ show i

-- for constants
refName  = HsIdent "ref"
refQName = Qual buddhaLibName refName

constantName  = HsIdent "con"
constantQName = Qual buddhaLibName constantName


-- encoding functions
fun :: Int -> HsExp -> Trans HsExp
fun arity exp 
   = do cheapFlag <- isFlagSet "cheapHo"
        if cheapFlag 
           then return $ mkParen $ foldApp [HsVar $ mkFunUnRecQName arity, mkParen exp] 
           else return $ mkParen $ foldApp [HsVar $ mkFunQName arity, mkParen exp]

funUnRecorded :: Int -> HsExp -> HsExp
funUnRecorded arity exp = mkParen $ foldApp [HsVar $ mkFunUnRecQName arity, mkParen exp]

justName = HsIdent "Just"
justQName = Qual buddhaLibName $ justName
justConExp = HsCon $ UnQual justName
nothingName = HsIdent "Nothing"
nothingQName = Qual buddhaLibName $ nothingName
nothingConExp = HsCon $ UnQual nothingName

-- what sort of expression
data WhatIsExp 
   = EVar 
   | ECon 
   | EApp 
   | ELam 
   | ELet 
   | ECase 
   | EOther
   deriving (Eq, Show)

whatIsExp :: HsExp -> WhatIsExp
whatIsExp (HsVar _)        = EVar
whatIsExp (HsCon _)        = ECon
whatIsExp (HsApp _ _)      = EApp
whatIsExp (HsLambda _ _ _) = ELam
whatIsExp (HsLet _ _)      = ELet 
whatIsExp (HsCase _ _)     = ECase
whatIsExp _anythingElse    = EOther 

-- the actual transformation

transDecl :: IdentMap -> HsDecl -> Trans [HsDecl]
transDecl idents decl = arityThenTrans idents decl

{- variable declarations -}

-- we return a list of decls due to the annoying
-- type sig situation, such that more than one ident
-- can be bound to a type sig and also because we need to add
-- new declarations to the module for non simple pattern 
-- bindings
transDecl' :: IdentMap -> HsDecl -> Trans [HsDecl]
transDecl' idents (HsFunBind matches@(m:ms)) 
   = context [nameStr] $
        do newMatches <- mapM (transMatch idents) matches
           return [HsFunBind newMatches]
   where
   HsMatch _sloc name _pats _rhs _wheres = m
   nameStr = fromHsName name  
transDecl' idents (HsFunBind []) = return $ [HsFunBind []]

{- simple pattern binding -}

-- pat binds introduced by earlier phases of buddha (namely desugaring)
-- should be transformed as "trusted"
transDecl' idents decl@(HsPatBind srcLoc pat (HsUnGuardedRhs rhs) wheres)
   | SyntaxUtils.isTrustedSrcLoc srcLoc 
     = location srcLoc $
          do let whereSigs   = typeSigMapFromDecls wheres 
                 whereIdents = declIdents False UnQual UnQualified whereSigs wheres 
                 newIdents   = idents `plusFM` whereIdents 
             newWheres <- mapM (transDecl newIdents) wheres
             (_, parentPVar, parentVar) <- freshVar
             newRhs <- debugRhsConst newIdents Trust (patToString pat) rhs 
             return [HsPatBind srcLoc pat (HsUnGuardedRhs newRhs) (concat newWheres)]
             -- return [HsPatBind srcLoc pat (HsUnGuardedRhs newRhs) (concat newWheres)]

transDecl' idents decl@(HsPatBind srcLoc pat (HsUnGuardedRhs rhs) wheres)
   | not (isSimplePat pat) 
     = context [patToString pat] $ location srcLoc $
          do -- pushConstantStack 
             let whereSigs   = typeSigMapFromDecls wheres
                 whereIdents = declIdents False UnQual UnQualified whereSigs wheres
                 newIdents   = idents `plusFM` whereIdents
             newWheres <- mapM (transDecl newIdents) wheres
             transOpt <- getTransOpt
             newRhs <- debugRhsConst newIdents transOpt (patToString pat) rhs
             (_, simplePVar, simpleExpVar) <- freshVar
             let simpleDecl
                    = HsPatBind srcLoc simplePVar (HsUnGuardedRhs newRhs) (concat newWheres)
                 patVars = varsFromPat pat
             selectors <- selectorDecls simpleExpVar patVars 
             -- popConstantStack
             return (simpleDecl:selectors)  
   where
   selectorDecls :: HsExp -> [HsName] -> Trans [HsDecl]
   selectorDecls _ [] = return []
   selectorDecls exp (name:names)
      = do (_, countPVar, countExp) <- freshVar 
           let selector = HsLambda bogusSrcLoc [mkParenPat pat] (HsVar $ UnQual name)
               body     = HsLambda bogusSrcLoc [countPVar] 
                                   (foldApp [ mkParen selector
                                            , mkParen $ foldApp [exp, countExp]])
               decl     = HsPatBind bogusSrcLoc (HsPVar name) (HsUnGuardedRhs body) []
           nextDecls <- selectorDecls exp names
           return (decl:nextDecls) 
 

transDecl' idents decl@(HsPatBind srcLoc pat (HsUnGuardedRhs rhs) wheres)
   = context [patToString pat] $ location srcLoc $
        do -- pushConstantStack 
           let whereSigs   = typeSigMapFromDecls wheres 
               whereIdents = declIdents False UnQual UnQualified whereSigs wheres 
               newIdents   = idents `plusFM` whereIdents 
           newWheres <- mapM (transDecl newIdents) wheres
           transOpt <- getTransOpt
           newRhs <- debugRhsConst newIdents transOpt (patToString pat) rhs 
           -- popConstantStack
           return [HsPatBind srcLoc pat (HsUnGuardedRhs newRhs) (concat newWheres)]

{-
transDecl' idents decl@(HsPatBind srcLoc pat (HsUnGuardedRhs rhs) wheres)
   = location srcLoc $ 
        do -- pushConstantStack 
           let whereSigs   = typeSigMapFromDecls wheres 
               whereIdents = declIdents False UnQual UnQualified whereSigs wheres 
               newIdents   = idents `plusFM` whereIdents 
           newWheres <- mapM (transDecl newIdents) wheres
           (_, newVarPat, newVarExp) <- freshVar
           newRhs <- transExp newIdents newVarExp rhs 
           let newLambda = HsLambda bogusSrcLoc [newVarPat] newRhs
           (_, newConstantPat, newConstantExp) <- freshVar
           let newConstantDecl
                  = HsPatBind bogusSrcLoc newConstantPat (HsUnGuardedRhs newLambda) 
                                                         (concat newWheres)
           let patVars = varsFromPat pat
           newPat <- transPat pat
           newDecls <- extraDecls newPat newConstantExp patVars
           return $ newConstantDecl : newDecls
           -- popConstantStack
      where
      extraDecls :: HsPat -> HsExp -> [HsName] -> Trans [HsDecl]
      extraDecls pat rhsExp [] = return []
      extraDecls pat rhsExp (name:names) 
         = do let selector = HsLambda bogusSrcLoc [mkParenPat pat] (HsVar $ UnQual name)  
                  newBody  = foldApp [mkParen selector, rhsExp]
                  decl     = HsPatBind 
                               bogusSrcLoc (HsPVar name)
                               (HsUnGuardedRhs newBody) []
              newDecls <- transDecl idents decl
              nextDecls <- extraDecls pat rhsExp names
              return (newDecls ++ nextDecls)
-}

{- type signatures are arity dependent, so we have to split them apart -}
transDecl' idents sig@(HsTypeSig sloc names qType)
   = location sloc $
        do newSig <- transTypeSig idents sig
           return [newSig]

{- data type declarations -}
transDecl' idents (HsDataDecl sloc cntxt tyName tyArgs condecls derives)
   = location sloc $
        do newConDecls <- mapM transConDecl condecls 
           newContext <- mapM transAssociation cntxt
           return [HsDataDecl sloc newContext tyName tyArgs newConDecls derives]

{- type synonyms possibly need qualified names transformed in the type -}
transDecl' idents (HsTypeDecl sloc name args t)
   = location sloc $
        do newT <- transType t
           return [HsTypeDecl sloc name args newT]

{- newType declarations -}
transDecl' idents (HsNewTypeDecl sloc cntxt tyName tyArgs condecl derives)
   = location sloc $ 
        do newConDecl <- transConDecl condecl
           newContext <- mapM transAssociation cntxt
           return [HsNewTypeDecl sloc newContext tyName tyArgs newConDecl derives] 

{- class declarations

   1) transform all type signatures in the class
   
   2) transform any default methods making sure to adjust
      the number of pattern arguments to match the arity
-}
transDecl' idents (HsClassDecl sloc cntxt name args decls)
   = context [fromHsName name] $ location sloc $
        do newClassDecls <- mapM (arityThenTrans idents) decls
           newContext <- mapM transAssociation cntxt
           return [HsClassDecl sloc newContext name args (concat newClassDecls)]

{- instance declarations

   1) transform the type in the head

   2) transform all declarations in the body making sure to adjust
      the number of pattern arguments to match the arity

-}
transDecl' idents (HsInstDecl sloc cntxt className types decls)
   = context (fromHsQNameQual className : tyConNames) $ location sloc $
        do newDecls <- mapM (arityThenTrans idents) decls
           newTypes <- mapM transType types
           newClassName <- transQualNameM className
           newContext <- mapM transAssociation cntxt
           return [HsInstDecl sloc newContext newClassName newTypes (concat newDecls)]
   where
   tyConNames :: [String]
   tyConNames = map leftTyConName types
   
transDecl' _idents anythingElse 
   = return [anythingElse]

-- adjust the arity then transform the declaration
arityThenTrans :: IdentMap -> HsDecl -> Trans [HsDecl]
arityThenTrans idents decl
   = do newDecl <- adjustPatsToArity idents decl
        transDecl' idents newDecl 

-- make the number of patterns the same as the arity for this binding 
adjustPatsToArity :: IdentMap -> HsDecl -> Trans HsDecl

   {- simple pattern binding -}
adjustPatsToArity idents decl@(HsPatBind sloc (HsPVar name) rhs wheres)
   = do case lookupIdent idents (UnQual name) of
           Just info -> let numArgs = identInfo_arity info
                        in case numArgs > 0 of
                              True  -> do let initMatch = HsMatch sloc name [] rhs wheres 
                                          newMatch <- addPats numArgs initMatch
                                          return $ HsFunBind [newMatch]
                              False -> return decl
           Nothing -> identNotInScope (UnQual name)

   {- non simple pattern binding cannot ever have any more arguments -}
adjustPatsToArity _idents decl@(HsPatBind _sloc _otherPattern _rhs _wheres)
   = return decl

{- after desugaring there will not be any guarded rhs equations
   so adding or removing arguments is easier -}

adjustPatsToArity idents decl@(HsFunBind matches)
   = liftM HsFunBind $ mapM (adjustPats idents) matches

adjustPatsToArity _idents otherDecl 
   = return otherDecl

adjustPats :: IdentMap -> HsMatch -> Trans HsMatch
adjustPats idents match@(HsMatch sloc name pats rhs wheres) 
   = do neededPats <- case lookupIdent idents (UnQual name) of
                              Just info -> return $ (identInfo_arity info) - length pats
                              Nothing   -> identNotInScope (UnQual name) 
        case compare neededPats 0 of
           GT -> addPats neededPats match
           LT -> return $ removePats (abs neededPats) match
           EQ -> return match

-- add a number of patterns to a match and variables to its rhs 
-- if the rhs is a lambda, just move the right number of patterns accross the '='
-- foo x y = \w z -> e  ---> foo x y w = \z -> e

addPats :: Int -> HsMatch -> Trans HsMatch
addPats 0 match = return match
addPats num (HsMatch sloc name pats rhs@(HsUnGuardedRhs e) wheres)
   | isLambda eNoParens
        = addPats (num - (numLamPats eNoParens)) 
                  (HsMatch sloc name (pats ++ (lamPats eNoParens)) 
                                     (HsUnGuardedRhs $ lamBody eNoParens) wheres)
   | otherwise 
        = do (_, newPatVars, newExpVars) <- nFreshVars num
             let newRhs = addVarsToRhs newExpVars rhs
                 newPats = pats ++ newPatVars 
             return $ HsMatch sloc name newPats newRhs wheres
   where
   isLambda :: HsExp -> Bool
   isLambda e = whatIsExp e == ELam
   numLamPats (HsLambda _sloc ps _e) = length ps
   lamBody (HsLambda _sloc _ps e) = e
   lamPats (HsLambda _sloc ps _e) = ps
   eNoParens = dropParens e

addVarsToRhs :: [HsExp] -> HsRhs -> HsRhs
addVarsToRhs vars (HsUnGuardedRhs e)
   = HsUnGuardedRhs $ applyExpToVars vars e
addVarsToRhs vars rhs@(HsGuardedRhss rhss)
   = fatalError __FILE__ __LINE__
        ("addVarsToRhs: applied to a guarded rhs: " ++ show rhs)

applyExpToVars :: [HsExp] -> HsExp -> HsExp
applyExpToVars vars (HsLet decls e) 
   = HsLet decls $ applyExpToVars vars e
applyExpToVars vars (HsIf e1 e2 e3) 
   = HsIf e1 (applyExpToVars vars e2) (applyExpToVars vars e3)
applyExpToVars vars (HsCase discrim alts)
   = HsCase discrim $ map (applyAltToVars vars) alts 
applyExpToVars vars (HsParen e)
   = HsParen $ applyExpToVars vars e
applyExpToVars vars e 
   = foldApp (mkParen e : vars)

applyAltToVars :: [HsExp] -> HsAlt -> HsAlt
applyAltToVars vars (HsAlt sloc pat gAlts decls)  
   = HsAlt sloc pat (applyGAltToVars vars gAlts) decls
   where
   applyGAltToVars :: [HsExp] -> HsGuardedAlts -> HsGuardedAlts
   applyGAltToVars vars (HsUnGuardedAlt e) = HsUnGuardedAlt $ applyExpToVars vars e 
   applyGAltToVars vars g@(HsGuardedAlts alts)
      = fatalError __FILE__ __LINE__
        ("applyGAltToVars : applied to guarded alts: " ++ show g)

{- shift a number of patterns to the rhs of a match
   
   removePats 2 "f x y z = rhs"

   => f x = \y z -> rhs 
-}

removePats :: Int -> HsMatch -> HsMatch
removePats num (HsMatch sloc name pats rhs wheres)
   = HsMatch sloc name keepPats newRhs wheres
   where
   (keepPats, losePats) = splitAt ((length pats) - num) pats
   newRhs = case losePats of
               []    -> rhs
               (_:_) -> lamIntroduceVarsRhs losePats rhs

lamIntroduceVarsRhs :: [HsPat] -> HsRhs -> HsRhs
lamIntroduceVarsRhs pats (HsUnGuardedRhs e)
   = HsUnGuardedRhs $ HsLambda trustedSrcLoc pats e
lamIntroduceVarsRhs _pats rhs@(HsGuardedRhss rhss)
   = fatalError __FILE__ __LINE__
        ("lamIntroduceVarsRhs: applied to a guarded rhs: " ++ show rhs)

{- a constructor declaration -}
transConDecl :: HsConDecl -> Trans HsConDecl
transConDecl (HsConDecl sloc conName bangTypes)
   = liftM (HsConDecl sloc conName) $ mapM transBangType bangTypes
transConDecl (HsRecDecl sloc recName fields)
   = liftM (HsRecDecl sloc recName) $ mapM transField fields

{- bang types -}
transBangType :: HsBangType -> Trans HsBangType
transBangType (HsBangedTy t) = liftM HsBangedTy $ transType t
transBangType (HsUnBangedTy t) = liftM HsUnBangedTy $ transType t

{- fields of a record constructor -}
transField :: ([HsName],HsBangType) -> Trans ([HsName],HsBangType)
transField (names, bangType)
   = do newBangType <- transBangType bangType
        return (names, newBangType) 

-- add the identifier information for each pattern
-- var plus the identifier information for the
-- vars bound in where clauses, before we enter the
-- where clauses and the rhs
transMatch :: IdentMap -> HsMatch -> Trans HsMatch
transMatch idents (HsMatch sloc funName pats (HsUnGuardedRhs rhs) wheres)
   = location sloc $ 
        do -- pushConstantStack 
           let identMapWithPats = patsToIdentMap pats idents
               whereSigs        = typeSigMapFromDecls wheres
               whereIdents      = declIdents False UnQual UnQualified whereSigs wheres
               newIdents        = identMapWithPats `plusFM` whereIdents 
           newPats   <- mapM transPatAs pats
           newWheres <- mapM (transDecl newIdents) wheres
           (_, parentPVar, parentVar) <- freshVar
           transOpt <- getTransOpt
           newRhs <- debugRhs newIdents transOpt parentVar (fromHsName funName) newPats rhs 
           -- popConstantStack
           return $ HsMatch sloc funName (parentPVar : newPats) 
                                         (HsUnGuardedRhs newRhs) 
                                         (concat newWheres)
transMatch _ m 
   = fatalError __FILE__ __LINE__ $ 
        "transMatch: does not support this match: " ++ show m

{- 
   build the call to make the edt, this will depend on what 
   kind of EDT is needed for this function
-}

debugRhs :: IdentMap -> TransOpt -> HsExp -> String -> [HsPat] -> HsExp -> Trans HsExp
debugRhs idents transOpt parentId name pats exp 
   = do case transOpt of
                      -- XXX these should change
           Empty   -> node  idents parentId name pats exp  
           Trust   -> trust idents parentId name exp 
           Suspect -> node  idents parentId name pats exp 
           NoKids  -> node  idents parentId name pats exp 
           ReEval  -> node  idents parentId name pats exp 

debugRhsConst :: IdentMap -> TransOpt -> String -> HsExp -> Trans HsExp
debugRhsConst idents transOpt name exp 
   = do case transOpt of
                      -- XXX these should change
           Empty   -> nodeConst idents name exp  
           Trust   -> trustConst idents name exp 
           Suspect -> nodeConst idents name exp 
           NoKids  -> nodeConst idents name exp 
           ReEval  -> nodeConst idents name exp 

nodeConst :: IdentMap -> String -> HsExp -> Trans HsExp
nodeConst idents name exp
   = do (_, countPat, countExp) <- freshVar
        (_, valPat, valExp) <- freshVar
        sloc <- getRecentSrcLoc
        (Module modName) <- getModName
        newExp <- transExp idents countExp exp
        let tuplePat = HsPTuple [valPat, countPat]
            tupleExp = HsTuple [valExp, countExp]
            newRhs = foldApp [ HsVar constantQName
                             , mkParen $ mkPackedString name
                             , HsList []
                             , HsLit $ HsInt $ fromIntegral $ srcLine sloc
                             , HsVar $ globalModQVar (Module modName) 
                             , mkParen newExp
                             ] 
            letDecl  = HsPatBind bogusSrcLoc tuplePat (HsUnGuardedRhs $ mkParen newRhs) []
            letBind  = HsLet [letDecl] tupleExp
        return $ foldApp [HsVar refQName, mkParen letBind]


node :: IdentMap -> HsExp -> String -> [HsPat] -> HsExp -> Trans HsExp
node idents parentId name pats exp
   = do (_, countPat, countExp) <- freshVar
        sloc <- getRecentSrcLoc
        (Module modName) <- getModName
        newExp <- transExp idents countExp exp
        return $ foldApp [ HsVar nodeQName
                         , mkParen $ mkPackedString name 
                         , HsList $ map patToValExp pats
                         , parentId
                         , mkParen $ HsLit $ HsInt $ fromIntegral $ srcLine sloc
                         , HsVar $ globalModQVar (Module modName) 
                         , mkParen (lambdaBody countPat newExp)
                         ]
   where
   patToValExp :: HsPat -> HsExp
   patToValExp pat = HsApp (HsVar valConName) $ patToExp pat
   lambdaBody :: HsPat -> HsExp -> HsExp
   lambdaBody pat exp = HsLambda trustedSrcLoc [pat] exp

trust :: IdentMap -> HsExp -> String -> HsExp -> Trans HsExp
trust idents parentId name exp
   = transExp idents parentId exp

trustConst :: IdentMap -> String -> HsExp -> Trans HsExp
trustConst idents name exp
   = do (_, countPat, countExp) <- freshVar
        newExp <- transExp idents countExp exp
        return $ HsLambda bogusSrcLoc [countPat] newExp

--------------------------------------------------------------------------------

-- converting HsType to an HsExp representation of a Meta.Type
-- and for qualified types too

{-
qTypeToRep :: HsQualType -> HsExp
qTypeToRep (HsQualType context t) 
   = HsTuple [contextToRep context, typeToRep t]

contextToRep :: HsContext -> HsExp
contextToRep constraints
   = HsList $ map constraintToRep constraints

constraintToRep :: HsAsst -> HsExp
constraintToRep (className, tyArgs)
   = HsTuple [litStringExp $ fromHsQNameQual className, HsList $ map typeToRep tyArgs]

tFunName    = HsIdent "TFun"
tFunQName   = Qual buddhaLibName tFunName

tTupleName  = HsIdent "TTuple"
tTupleQName = Qual buddhaLibName tTupleName

tListName   = HsIdent "TList"
tListQName  = Qual buddhaLibName tListName 

tAppName    = HsIdent "TApp"
tAppQName   = Qual buddhaLibName tAppName

tVarName    = HsIdent "TVar"
tVarQName   = Qual buddhaLibName tVarName

tConName    = HsIdent "TCon"
tConQName   = Qual buddhaLibName tConName

typeToRep :: HsType -> HsExp
typeToRep (HsTyFun t1 t2)
   = foldApp [HsCon tFunQName, mkParen $ typeToRep t1, mkParen$ typeToRep t2]
typeToRep (HsTyTuple ts)
   = foldApp [HsCon tTupleQName, HsList $ map typeToRep ts]
typeToRep (HsTyVar name) 
   = foldApp [HsCon tVarQName, litStringExp $ fromHsName name]
typeToRep (HsTyCon qName)
   = foldApp [HsCon tConQName, litStringExp $ fromHsQNameQual qName]
-- check for a list
typeToRep (HsTyApp t1 t2)
   | isList t1 = foldApp [HsCon tListQName, mkParen $ typeToRep t2]
   | otherwise = foldApp [HsCon tAppQName, mkParen $ typeToRep t1, mkParen $ typeToRep t2]   
   where
   isList :: HsType -> Bool
   isList (HsTyCon (Special HsListCon)) = True
   isList other = False
-}

--------------------------------------------------------------------------------

transExp :: IdentMap -> HsExp -> HsExp -> Trans HsExp
{- variable expressions -}
transExp idents parentId exp@(HsVar qName)
   = do newQName <- transQualNameM qName 
        let transVar = HsVar newQName 
        case lookupIdent idents qName of
           Nothing -> identNotInScope qName 
           Just identInfo 
                   -> let bindType = identInfo_bindType identInfo
                          arity    = identInfo_arity identInfo
                      in case bindType of
                            Pat        -> return $ transVar 
                            LetFun     -> do incPartialStatByT 1
                                             fun arity $ foldApp [transVar, parentId] 
                            LetConst   -> do incConstantsStatByT 1 
                                             return $ mkParen $ foldApp [transVar, parentId]
                                             {- seenBefore <- recordConstant qName
                                             if seenBefore 
                                                then return $ mkParen $ foldApp [HsVar pruneQName,                                                                                 transVar]
                                                else return $ transVar
                                             -}
               
                            LetConstOL -> do incConstantsStatByT 1
                                             return $ mkParen $ foldApp [transVar, parentId] 

{- constructor expressions -}
transExp idents parentId exp@(HsCon qName)
   = do newQName <- transQualNameM qName 
        let transCon = HsCon newQName 
        case lookupIdent idents qName of
           Nothing -> identNotInScope qName 
           Just identInfo 
              -> let arity = identInfo_arity identInfo
                 in  case compare arity 0 of
                        EQ -> return transCon 
                        GT -> do incPartialStatByT 1 
                                 fun arity $ transCon 
                        LT  -> fatalError __FILE__ __LINE__ $ 
                                          "transExp: constructor: " ++ show qName ++ 
                                          " has negative arity"

{- literals -}
transExp idents parentId exp@(HsLit _lit)
   = return exp

{- applications -}

-- unfold the application
transExp idents parentId exp@(HsApp _e1 _e2)
   = transUnfoldedApp idents parentId $ unfoldApp exp 

{- let expressions -}
transExp idents parentId (HsLet decls e)
   = do let letSigs = typeSigMapFromDecls decls 
            newIdents = idents `plusFM` (declIdents False UnQual UnQualified letSigs decls)
        newDecls <- mapM (transDecl newIdents) decls 
        newExp <- transExp newIdents parentId e
        return $ HsLet (concat newDecls) newExp 

{- lambdas introduced by previous parts of buddha - these ones should not
   get recorded in the function map -}
transExp idents parentId exp@(HsLambda sloc pats e)
   | SyntaxUtils.isTrustedSrcLoc sloc = 
        do incLambdaStatByT 1
           -- pushConstantStack 
           let identMapWithPats = patsToIdentMap pats idents
           newPats <- mapM transPatAs pats
           newExp  <- transExp identMapWithPats parentId e
           return $ funUnRecorded (length pats) $ HsLambda sloc newPats newExp

{- lambdas -}
transExp idents parentId exp@(HsLambda sloc pats e)
   = location sloc $ 
        do incLambdaStatByT 1
           -- pushConstantStack 
           let identMapWithPats = patsToIdentMap pats idents
           newPats <- mapM transPatAs pats
           newExp  <- transExp identMapWithPats parentId e
           fun (length pats) $ HsLambda sloc newPats newExp

{- case expressions -}
transExp idents parentId exp@(HsCase e alts)
   = do newAlts <- mapM (transAlt idents parentId) alts 
        newExp <- transExp idents parentId e
        return $ HsCase newExp newAlts

{- parenthesised expressions -}
transExp idents parentId (HsParen e)
   = do newExp <- transExp idents parentId e
        return $ HsParen newExp 

{- lists -}
transExp idents parentId exp@(HsList exps)
   = do incSaturatedStatByT $ length exps
        liftM HsList $ mapM (transExp idents parentId) exps 

{- tuples -}
transExp idents parentId exp@(HsTuple exps)
   = do incSaturatedStatByT 1 
        liftM HsTuple $ mapM (transExp idents parentId) exps 

{- explicitly type annotated expressions -}
transExp idents parentId (HsExpTypeSig sloc e (HsQualType cntxt t))
   = location sloc $
        do newExp <- transExp idents parentId e
           newType <- transType t
           newContext <- mapM transAssociation cntxt
           return $ HsExpTypeSig sloc newExp $ HsQualType newContext newType

{- anything else is unsupported -}
transExp _ _ e 
   = fatalError __FILE__ __LINE__ $ 
        "transExp: does not support this expression: " ++ show e

{- [e1, e2, e3] --> apply (apply e1 e2) e3 -}
leftApply :: [HsExp] -> HsExp
leftApply 
   = foldl1 (\acc next -> mkParen $ foldApp [HsVar applyQName, mkParen acc, next])

{- transform an unfolded application -} 
transUnfoldedApp :: IdentMap -> HsExp -> [HsExp] -> Trans HsExp
transUnfoldedApp idents parentId exps@(function:args)
   = case whatIsExp function of
           -- the function is a variable
           EVar  -> transVarApp idents parentId exps 
           -- the function is a constructor
           ECon  -> transConApp idents parentId exps 
           -- the function was not a variable nor a constructor
           -- must therfore have been let/case/lambda
           other -> transMiscApp idents parentId exps 
transUnfoldedApp _idents _parentId [] 
   = fatalError __FILE__ __LINE__ $ "transUnfoldedApp: empty application"

{- variable application -}
transVarApp :: IdentMap -> HsExp -> [HsExp] -> Trans HsExp
transVarApp idents parentId exps@(function@(HsVar funName) : args) 
   = do newFunName <- transQualNameM funName 
        functionInfo <- getIdentInfo idents function 
        let numArgs = length args
            transFunction = HsVar $ newFunName 
            functionBindType = identInfo_bindType functionInfo 
        newArgs <- mapM (transExp idents parentId) args
        case isLetBound functionBindType of
           -- the variable is let bound
           True -> do let funArity = identInfo_arity functionInfo
                      case compare funArity numArgs of
                         -- the application is saturated
                         EQ -> do incSaturatedStatByT 1
                                  return $ mkParen $ foldApp (transFunction : parentId : newArgs)
                         -- the application is partial
                         GT -> do incPartialStatByT 1
                                  let argDiff = funArity - numArgs
                                  fun argDiff $ foldApp (transFunction : parentId : newArgs) 
                         -- the application is over saturated
                         LT -> do incOverSatStatByT 1
                                  let (satArgs, restArgs) = splitAt funArity newArgs
                                  let satApp 
                                        = mkParen $ foldApp (transFunction : parentId : satArgs)
                                  return $ leftApply (satApp : restArgs)
           False -> case isPatBound functionBindType of
                       -- the variable was pattern bound
                       True -> do incPatVarApStatByT 1 
                                  return $ leftApply (transFunction : newArgs)
                       -- a constructor
                       False -> fatalError __FILE__ __LINE__ $ 
                                   "transVarApp: found a constructor, expecting a variable" 

transVarApp idents _parentId [] 
   = fatalError __FILE__ __LINE__ $ "transVarApp: empty application"

{- constructor application -}
transConApp :: IdentMap -> HsExp -> [HsExp] -> Trans HsExp 
transConApp idents parentId exps@(function@(HsCon conName) : args) 
   = do newConName <- transQualNameM conName 
        functionInfo <- getIdentInfo idents function 
        let numArgs = length args
            conArity = identInfo_arity functionInfo
            transFunction = HsCon newConName 
        -- transform the argument expressions
        newArgs <- mapM (transExp idents parentId) args
        case compare conArity numArgs of
           -- the application is saturated
           EQ -> do incSaturatedStatByT 1
                    return $ foldApp (transFunction : newArgs)
           -- the application is partial
           GT -> do incPartialStatByT 1
                    fun (conArity - numArgs) $ foldApp (transFunction : newArgs)

transConApp _idents _parentId [] 
  = fatalError __FILE__ __LINE__ $ "transConApp: empty application"

{- miscellaneous application -}
transMiscApp :: IdentMap -> HsExp -> [HsExp] -> Trans HsExp
transMiscApp idents parentId exp@(function:args) 
   = do incMiscApsStatByT 1 
        let numArgs = length args
        -- transform all the expressions
        newExps <- mapM (transExp idents parentId) exp
           -- the function was not a variable nor a constructor
           -- must therfore have been let/case/lambda
        return $ leftApply newExps

transMiscApp _idents _parentId [] 
   = fatalError __FILE__ __LINE__ $ "transMiscApp: empty application"
 
-- get the ident info about a var or con function
getIdentInfo :: IdentMap -> HsExp -> Trans IdentInfo
getIdentInfo idents exp 
   = case lookupIdent idents (name exp) of
        Nothing   -> identNotInScope (name exp) 
        Just info -> return info
   where
   name (HsVar n) = n
   name (HsCon n) = n
   name anythingElse 
      = fatalError __FILE__ __LINE__ $ 
           "getIdentInfo: attempt to get name of non identifier: " ++ show anythingElse

{- alternatives

   A { p -> e } ---->

       p -> E { e }

   -- if the srlLoc is marked by the desugaring stage then we
      wrp a call to D.alt around the rhs of the alternative
-}
transAlt :: IdentMap -> HsExp -> HsAlt -> Trans HsAlt
transAlt idents parentId (HsAlt sloc pat (HsUnGuardedAlt e) wheres)
   = do let whereSigs        = typeSigMapFromDecls wheres 
            whereIdents      = declIdents False UnQual UnQualified whereSigs wheres
            identMapWithPats = patsToIdentMap [pat] idents
            newIdents        = identMapWithPats `plusFM` whereIdents 
        newPat <- transPat pat
        newWheres <- mapM (transDecl newIdents) wheres
        newExp <- transExp newIdents parentId e
        return $ HsAlt sloc newPat (HsUnGuardedAlt newExp) (concat newWheres) 

transAlt _idents _parentId anythingElse
   = fatalError __FILE__ __LINE__ $ 
        "transAlt: this type of alternative is not supported: " ++ show anythingElse 

-- construct a call to apply
mkApply :: HsExp -> HsExp -> HsExp
mkApply e1 e2
   = foldApp [applyExp, e1, e2]
   where
   applyExp = HsVar $ applyQName

-- convert a pattern into an expresion
-- XXX not finished
patToExp :: HsPat -> HsExp
patToExp (HsPVar name) = HsVar $ UnQual name 
patToExp (HsPLit literal) = HsLit literal
patToExp (HsPNeg pat) = foldApp [HsVar $ UnQual $ HsIdent "negate", mkParen $ patToExp pat]
patToExp (HsPInfixApp pat1 conName pat2)
   = foldApp [HsCon conName, patToExp pat1, patToExp pat2]
patToExp (HsPApp con pats) = foldApp ((HsCon $ con) : map patToExp pats)
patToExp (HsPTuple pats) = HsTuple $ map patToExp pats
patToExp (HsPList pats) = HsList $ map patToExp pats
patToExp (HsPParen p) = HsParen $ patToExp p
-- this one is important for the correctness of the EDT construction
patToExp (HsPAsPat name pat) = HsVar $ UnQual name 
patToExp other 
   = fatalError __FILE__ __LINE__ $ "patToExp: does not support this pattern: " ++ show other 

{- transform a pattern
   1. make a pattern an as pattern, if it is not already an as pattern or variable 
   2. make sure qualified names have the _Buddha appendix
-}
transPatAs :: HsPat -> Trans HsPat
transPatAs pat@(HsPVar name) = return pat
transPatAs (HsPAsPat name pat) 
   = liftM (HsPAsPat name) $ transPatQNames pat
transPatAs otherPat
   = do newPat <- transPatQNames otherPat
        asPat newPat 

-- same as above but do not make it into an as pattern
transPat :: HsPat -> Trans HsPat
transPat pat@(HsPVar name) = return pat
transPat (HsPAsPat name pat) 
   = liftM (HsPAsPat name) $ transPatQNames pat
transPat otherPat = transPatQNames otherPat

-- transform all the qualified names in a pattern
transPatQNames :: HsPat -> Trans HsPat
transPatQNames pat@(HsPVar name) = return pat
transPatQNames pat@(HsPLit literal) = return pat
transPatQNames (HsPNeg pat) 
   = liftM HsPNeg $ transPatQNames pat
transPatQNames (HsPInfixApp pat1 qName pat2)
   = liftM3 HsPInfixApp (transPatQNames pat1) (transQualNameM qName) (transPatQNames pat2)
transPatQNames (HsPApp qName pats)
   = liftM2 HsPApp (transQualNameM qName) $ mapM transPatQNames pats
transPatQNames (HsPTuple pats)
   = liftM HsPTuple $ mapM transPatQNames pats
transPatQNames (HsPList pats)
   = liftM HsPList $ mapM transPatQNames pats
transPatQNames (HsPParen pat) 
   = liftM HsPParen $ transPatQNames pat
transPatQNames (HsPRec qName patFields)
   = liftM2 HsPRec (transQualNameM qName) (mapM transPatFieldQNames patFields)
transPatQNames (HsPAsPat name pat)
   = liftM (HsPAsPat name) $ transPatQNames pat
transPatQNames HsPWildCard = return HsPWildCard
transPatQNames (HsPIrrPat pat)
   = liftM HsPIrrPat $ transPatQNames pat

-- transform all the qualified names in a patfield
transPatFieldQNames :: HsPatField -> Trans HsPatField
transPatFieldQNames (HsPFieldPat qName pat)
   = liftM2 HsPFieldPat (transQualNameM qName) $ transPatQNames pat

-- as-name a pattern with a fresh name
asPat :: HsPat -> Trans HsPat
asPat pat = do (newPatStr, _, _) <- freshVar
               return $ HsPAsPat (HsIdent newPatStr) (mkParenPat pat)

-- insert each variable in a list of patterns into
-- the identmap one at a time
patsToIdentMap :: [HsPat] -> IdentMap -> IdentMap
patsToIdentMap [] idents = idents
patsToIdentMap (pat:pats) idents
   = patsToIdentMap pats newMap
   where
   newMap = addListItemsToMap UnQualified patIdents idents
   patIdents = [(UnQual name, 
                 IdentInfo { identInfo_bindType = Pat, 
                             identInfo_arity = 0 }) 
                | name <- varsFromPat pat] 
   

-- transforming types


-- the name of the F type from the standard debugging library
fType = HsTyCon $ Qual buddhaLibName $ HsIdent "F"

{- transform a type 
 
   U { v } = v
   U { t1 -> t2 } = F U{ t1 } U{ t2 }
   U { f } = f
   U { t1 t2 } = U{ t1 } U{ t2 }

-}
transType :: HsType -> Trans HsType
transType t
   = case t of
        HsTyFun t1 t2 -> do newT1 <- transType t1
                            newT2 <- transType t2
                            return $ foldTypeApp [fType, newT1, newT2]
        HsTyTuple ts  -> liftM HsTyTuple $ mapM transType ts
        HsTyApp t1 t2 -> liftM2 HsTyApp (transType t1) (transType t2) 
        HsTyVar v     -> return $ HsTyVar v 
        HsTyCon qName -> liftM HsTyCon $ transQualNameM qName 

foldTypeApp :: [HsType] -> HsType
foldTypeApp ts@(_:_) = foldl1 HsTyApp ts
foldTypeApp [] = fatalError __FILE__ __LINE__ $ "foldTypeApp: empty type list"

transQNamesInType :: HsType -> Trans HsType
transQNamesInType t
   = case t of
        HsTyFun t1 t2 -> liftM2 HsTyFun (transQNamesInType t1) (transQNamesInType t2)
        HsTyTuple ts  -> liftM HsTyTuple $ mapM transQNamesInType ts
        HsTyApp t1 t2 -> liftM2 HsTyApp (transType t1) (transQNamesInType t2)
        HsTyVar v     -> return $ HsTyVar v
        HsTyCon qName -> liftM HsTyCon $ transQualNameM qName

-- the name of the Debug type from the standard debugging library
debugTypeWrapper = HsTyCon $ Qual buddhaLibName $ HsIdent "D"

-- make a new type by applying the S constructor to it
appDebugTypeWrapper :: HsType -> HsType
appDebugTypeWrapper t = HsTyApp debugTypeWrapper t 

-- transform a type that is part of a function annotation

transTypeAnn :: HsType -> Trans HsType
transTypeAnn t 
   = liftM appDebugTypeWrapper $ transTypeAnnFuns t
   where
   transTypeAnnFuns :: HsType -> Trans HsType
   transTypeAnnFuns (HsTyFun t1 t2) 
      = liftM2 HsTyFun (transType t1) (transTypeAnnFuns t2)
   transTypeAnnFuns otherType
      = transType otherType
{-
   transTypeAnnFuns otherType
      = transQNamesInType otherType
-}

transTypeSig :: IdentMap -> HsDecl -> Trans HsDecl
transTypeSig idents sig@(HsTypeSig sloc names qType)
   = liftM (HsTypeSig sloc names) $ transSigQualType qType

transSigQualType :: HsQualType -> Trans HsQualType
transSigQualType (HsQualType cntxt t)
   = do newCntxt <- mapM transAssociation cntxt
        newType <- transTypeAnn t
        return $ HsQualType newCntxt newType

--------------------------------------------------------------------------------

{-
-- compute all the free variables in an expression
-- XXX not complete
freeVarsExp :: [HsName] -> HsExp -> [HsQName]
freeVarsExp boundVars exp
   = case exp of
        -- a qualified name must be free, b/c it can't be bound in a pattern
        -- XXX is this correct?
        HsVar var@(Qual _mod name) -> [var]
        HsVar var@(UnQual name) 
           -> case name `elem` boundVars of
                 True -> []
                 False -> [var]
        HsCon _ -> []
        HsLit _ -> []
        HsInfixApp e1 op e2 -> freeVarsExp boundVars e1 ++
                               freeVarsExp boundVars e2
        HsApp e1 e2 -> freeVarsExp boundVars e1 ++
                       freeVarsExp boundVars e2
        HsNegApp e -> freeVarsExp boundVars e
        HsLambda _sloc pats e 
           -> let patVars = concatMap varsFromPat pats
              in freeVarsExp (patVars ++ boundVars) e
        HsLet decls e -> concatMap (freeVarsDecl boundVars) decls ++ freeVarsExp boundVars e
        HsIf e1 e2 e3 
           -> freeVarsExp boundVars e1 ++
              freeVarsExp boundVars e2 ++
              freeVarsExp boundVars e3
        HsCase e alts
           -> freeVarsExp boundVars e ++ concatMap (freeVarsAlt boundVars) alts
        HsDo stmts -> []
        HsTuple exps -> concatMap (freeVarsExp boundVars) exps
        HsList exps -> concatMap (freeVarsExp boundVars) exps
        HsParen e -> freeVarsExp boundVars e
        HsLeftSection e op  -> freeVarsExp boundVars e
        HsRightSection op e -> freeVarsExp boundVars e
        HsExpTypeSig _sloc e _qt -> freeVarsExp boundVars e
        other -> []
     
freeVarsDecl :: [HsName] -> HsDecl -> [HsQName]
freeVarsDecl boundVars decl
   = case decl of
        HsFunBind matches -> concatMap (freeVarsMatch boundVars) matches       
        HsPatBind _sloc pat rhs wheres
           -> let patVars = varsFromPat pat
                  newBoundVars = patVars ++ boundVars
              in freeVarsRhs newBoundVars rhs ++
                 concatMap (freeVarsDecl newBoundVars) wheres 
        other -> []

freeVarsMatch :: [HsName] -> HsMatch -> [HsQName]
freeVarsMatch boundVars (HsMatch _sloc name pats rhs wheres)
   = let patVats = concatMap varsFromPat pats
         newBoundVars = name : patVats ++ boundVars
     in freeVarsRhs newBoundVars rhs ++
        concatMap (freeVarsDecl newBoundVars) wheres

-- XXX not finished
freeVarsRhs :: [HsName] -> HsRhs -> [HsQName]
freeVarsRhs boundVars (HsUnGuardedRhs exp)
   = freeVarsExp boundVars exp
freeVarsRhs boundVars (HsGuardedRhss rhss)
   = fatalError __FILE__ __LINE__ $ "boundVars: HsGuardedRhss unsupported"

-- XXX not finished
freeVarsAlt :: [HsName] -> HsAlt -> [HsQName]
freeVarsAlt boundVars (HsAlt _sloc pat gAlts wheres)
   = let patVars = varsFromPat pat 
     in freeVarsGuardedAlts (patVars ++ boundVars) gAlts

-- XXX not finished
freeVarsGuardedAlts ::  [HsName] -> HsGuardedAlts -> [HsQName]
freeVarsGuardedAlts boundVars (HsUnGuardedAlt e)
   = freeVarsExp boundVars e
freeVarsGuardedAlts boundVars (HsGuardedAlts alts)
   = fatalError __FILE__ __LINE__ $ "freeVarsGuardedAlts: HsGuardedAlts not supported"
-}

-- determine if a variable is pattern bound
-- if we can't find the variable in the ident map then
-- we presume it is false
{-
isPatVar :: IdentMap -> HsQName -> Bool
isPatVar idents name
   = case lookupIdent idents name of
        Nothing -> False
        Just info -> case identInfo_bindType info of
                        Pat   -> True
                        other -> False
-}

--------------------------------------------------------------------------------

{- create a binding for the module name and filename
   ie if the module is called Foo and the file is called foo.hs
   then create:

   bUdDhA_MoD_fIlE = ("Foo", "foo.hs")
-}

globalModVar  = HsIdent $ defaultVarPrefix ++ "0" 
globalModQVar :: Module -> HsQName
globalModQVar mod = transQualName $ Qual mod globalModVar 

mkModId :: Module -> HsDecl
mkModId (Module modName) 
   = HsPatBind 
        trustedSrcLoc
        (HsPVar globalModVar)
        (HsUnGuardedRhs $ mkPackedString modName) 
        []

--------------------------------------------------------------------------------

{- 
  add the _B suffix onto the module part of qualified names
-}
transQualName :: HsQName -> HsQName
transQualName (Qual modName name) 
   = Qual (transModuleName modName) name 
transQualName other = other

-- this version in the Trans monad for times when we want to call it in that context

transQualNameM :: HsQName -> Trans HsQName
transQualNameM qName
   = return $ transQualName qName

-- add the _B appendix to a module name
transModuleName :: Module -> Module
transModuleName (Module modName) 
   = Module $ modName ++ transModuleAppendix

-- add the _B appendix to as name (import Foo as F)
transAsName :: Maybe Module -> Maybe Module
transAsName Nothing = Nothing
transAsName (Just modName)
   = Just $ transModuleName modName

-- transform an export specification to make sure
-- all qualified names point to the right module (ie with _Buddha appendix)
transExportSpec :: HsExportSpec -> HsExportSpec
transExportSpec (HsEVar qName) 
   =  HsEVar $ transQualName qName 
transExportSpec (HsEAbs qName) 
   = HsEAbs $ transQualName qName
transExportSpec (HsEThingAll qName) 
   = HsEThingAll $ transQualName qName
transExportSpec (HsEThingWith qName items) 
   = HsEThingWith (transQualName qName) items
transExportSpec (HsEModuleContents moduleName)
   = HsEModuleContents $ transModuleName moduleName 

--------------------------------------------------------------------------------

-- transform the qualified name in an association
-- transform the types also
transAssociation :: HsAsst -> Trans HsAsst
transAssociation (className, types) 
   = do newTypes <- mapM transType types 
        newClassName <- transQualNameM className
        return (newClassName, newTypes)

--------------------------------------------------------------------------------

-- mode for pretty printing the code
printMode :: PPHsMode
printMode
   = PPHsMode{ classIndent = 3,
               doIndent = 3,
               caseIndent = 3,
               letIndent = 3,
               whereIndent = 3,
               onsideIndent = 2,
               spacing = True,
               layout = PPOffsideRule,
               linePragmas = False,
               comments = True
             }

--------------------------------------------------------------------------------

identNotInScope :: HsQName -> Trans a
identNotInScope name
   = do sloc <- getRecentSrcLoc
        fail $ errStr sloc
   where 
   errStr sloc = "identifier '" ++ fromHsQNameQual name ++ "' not in scope" ++ location sloc
   location sloc = " in "  ++ srcFilename sloc ++
                   ", near line " ++ (show $ srcLine sloc)

--------------------------------------------------------------------------------

-- transform all the qualified names in an expression, pattern etc...

{-
transQNameExp :: HsExp -> Trans HsExp
transQNameExp (HsVar qName) = liftM HsVar $ transQualNameM qName
transQNameExp (HsCon qName) = liftM HsCon $ transQualNameM qName 
transQNameExp e@(HsLit lit) = return e
transQNameExp (HsInfixApp e1 qOp e2)
   = liftM3 HsInfixApp (transQNameExp e1) (transQNameQOp qOp) (transQNameExp e2)
transQNameExp (HsApp e1 e2) = liftM2 HsApp (transQNameExp e1) (transQNameExp e2)
transQNameExp (HsNegApp e) = liftM HsNegApp $ transQNameExp e
transQNameExp (HsLambda sloc pats e)
   = liftM2 (HsLambda sloc) (mapM transPatQNames pats) (transQNameExp e)
transQNameExp (HsLet decls e) 
   = liftM2 HsLet (mapM transQNameDecl decls) (transQNameExp e)
transQNameExp (HsIf e1 e2 e3)
   = liftM3 HsIf (transQNameExp e1) (transQNameExp e2) (transQNameExp e3)
transQNameExp (HsCase e alts)
   = liftM2 HsCase (transQNameExp e) (mapM transQNameAlt alts)  
transQNameExp (HsDo stmts) = liftM HsDo (mapM transQNameStmt stmts)
transQNameExp (HsTuple exps) = liftM HsTuple (mapM transQNameExp exps)
transQNameExp (HsList exps) = liftM HsList (mapM transQNameExp exps)
transQNameExp (HsParen e) = liftM HsParen (transQNameExp e)
transQNameExp (HsLeftSection e qOp) 
   = liftM2 HsLeftSection (transQNameExp e) (transQNameQOp qOp)
transQNameExp (HsRightSection qOp e) 
   = liftM2 HsRightSection (transQNameQOp qOp) (transQNameExp e)
transQNameExp (HsRecConstr qName fieldUpdates)
   = liftM2 HsRecConstr (transQualNameM qName) (mapM transQNameFieldUpdate fieldUpdates)
transQNameExp (HsRecUpdate e fieldUpdates)
   = liftM2 HsRecUpdate (transQNameExp e) (mapM transQNameFieldUpdate fieldUpdates) 
transQNameExp (HsEnumFrom e)
   = liftM HsEnumFrom $ transQNameExp e
transQNameExp (HsEnumFromTo e1 e2)
   = liftM2 HsEnumFromTo (transQNameExp e1) (transQNameExp e2)
transQNameExp (HsEnumFromThen e1 e2)
   = liftM2 HsEnumFromThen (transQNameExp e1) (transQNameExp e2)
transQNameExp (HsEnumFromThenTo e1 e2 e3)
   = liftM3 HsEnumFromThenTo (transQNameExp e1) (transQNameExp e2) (transQNameExp e3)
transQNameExp (HsListComp e stmts) 
   = liftM2 HsListComp (transQNameExp e) (mapM transQNameStmt stmts)
transQNameExp (HsExpTypeSig sloc e qType)
   = liftM2 (HsExpTypeSig sloc) (transQNameExp e) (transQNameQType qType)

transQNameQOp :: HsQOp -> Trans HsQOp
transQNameQOp (HsQVarOp qName) = liftM HsQVarOp $ transQualNameM qName
transQNameQOp (HsQConOp qName) = liftM HsQConOp $ transQualNameM qName

transQNameFieldUpdate :: HsFieldUpdate -> Trans HsFieldUpdate
transQNameFieldUpdate (HsFieldUpdate qName e)
   = liftM2 HsFieldUpdate (transQualNameM qName) (transQNameExp e)

transQNameDecl :: HsDecl -> Trans HsDecl
transQNameDecl (HsTypeDecl sloc tName tArgs t)
   = liftM (HsTypeDecl sloc tName tArgs) $ transQNameType t
transQNameDecl (HsDataDecl sloc cntxt tyName tArgs conDecls derives)
   = liftM2 (HsDataDecl sloc cntxt tyName tArgs) (mapM transQNameConDecl conDecls)
            (mapM transQualNameM derives)
transQNameDecl decl@(HsInfixDecl sloc assoc i ops) = return decl 
transQNameDecl (HsNewTypeDecl sloc cntxt tName tArgs conDecl derives)
   = liftM2 (HsNewTypeDecl sloc cntxt tName tArgs) (transQNameConDecl conDecl) 
            (mapM transQualNameM derives)
transQNameDecl (HsClassDecl sloc cntxt cName cArgs decls)
   = liftM (HsClassDecl sloc cntxt cName cArgs) $ mapM transQNameDecl decls 
transQNameDecl (HsInstDecl sloc cntxt cName cArgs decls)
   = liftM4 (HsInstDecl sloc) (transQNameCntxt cntxt) (transQualNameM cName)
            (mapM transQNameType cArgs) (mapM transQNameDecl decls)
transQNameDecl (HsDefaultDecl sloc ts)
   = liftM (HsDefaultDecl sloc) $ mapM transQNameType ts
transQNameDecl (HsTypeSig sloc tNames qType)
   = liftM (HsTypeSig sloc tNames) $ transQNameQType qType
transQNameDecl (HsFunBind matches)
   = liftM HsFunBind $ mapM transQNameMatch matches
transQNameDecl (HsPatBind sloc pat rhs decls)
   = liftM3 (HsPatBind sloc) (transPatQNames pat)
            (transQNameRhs rhs) (mapM transQNameDecl decls) 

transQNameMatch :: HsMatch -> Trans HsMatch
transQNameMatch (HsMatch sloc name pats rhs decls)
   = liftM3 (HsMatch sloc name) (mapM transPatQNames pats)
            (transQNameRhs rhs) (mapM transQNameDecl decls)

transQNameRhs :: HsRhs -> Trans HsRhs
transQNameRhs (HsUnGuardedRhs e) = liftM HsUnGuardedRhs $ transQNameExp e
transQNameRhs (HsGuardedRhss gRhss)
   = liftM HsGuardedRhss $ mapM transQNameGRhs gRhss

transQNameGRhs :: HsGuardedRhs -> Trans HsGuardedRhs
transQNameGRhs (HsGuardedRhs sloc e1 e2)
   = liftM2 (HsGuardedRhs sloc) (transQNameExp e1) (transQNameExp e2)

transQNameConDecl :: HsConDecl -> Trans HsConDecl
transQNameConDecl (HsConDecl sloc name ts)
   = liftM (HsConDecl sloc name) $ mapM transQNameBangType ts 
transQNameConDecl (HsRecDecl sloc name namesTs) 
   = liftM (HsRecDecl sloc name) $ mapM transNameT namesTs 
   where
   transNameT :: ([HsName],HsBangType) -> Trans ([HsName],HsBangType)
   transNameT (name, bt) = liftM (\t -> (name, t)) $ transQNameBangType bt

transQNameBangType :: HsBangType -> Trans HsBangType
transQNameBangType (HsBangedTy t) = liftM HsBangedTy $ transQNameType t
transQNameBangType (HsUnBangedTy t) = liftM HsUnBangedTy $ transQNameType t
   
transQNameAlt :: HsAlt -> Trans HsAlt
transQNameAlt (HsAlt sloc pat gAlts decls) 
   = liftM3 (HsAlt sloc) (transPatQNames pat) (transQNameGuardedAlts gAlts) 
                         (mapM transQNameDecl decls)

transQNameStmt :: HsStmt -> Trans HsStmt
transQNameStmt (HsGenerator sloc pat e)
   = liftM2 (HsGenerator sloc) (transPatQNames pat) (transQNameExp e)
transQNameStmt (HsQualifier e) = liftM HsQualifier $ transQNameExp e 
transQNameStmt (HsLetStmt decls) = liftM HsLetStmt $ mapM transQNameDecl decls

transQNameGuardedAlts :: HsGuardedAlts -> Trans HsGuardedAlts
transQNameGuardedAlts (HsUnGuardedAlt e) 
   = liftM HsUnGuardedAlt $ transQNameExp e
transQNameGuardedAlts (HsGuardedAlts gAlts)
   = liftM HsGuardedAlts $ mapM transQNameGAlts gAlts
   where
   transQNameGAlts :: HsGuardedAlt -> Trans HsGuardedAlt
   transQNameGAlts (HsGuardedAlt sloc e1 e2)
      = liftM2 (HsGuardedAlt sloc) (transQNameExp e1) (transQNameExp e2)
 
transQNameQType :: HsQualType -> Trans HsQualType
transQNameQType (HsQualType cntxt t)
   = liftM2 HsQualType (transQNameCntxt cntxt) (transQNameType t)

transQNameType :: HsType -> Trans HsType
transQNameType (HsTyFun t1 t2)
   = liftM2 HsTyFun (transQNameType t1) (transQNameType t2)
transQNameType (HsTyTuple ts)
   = liftM HsTyTuple $ mapM transQNameType ts
transQNameType (HsTyApp t1 t2)
   = liftM2 HsTyApp (transQNameType t1) (transQNameType t2)
transQNameType t@(HsTyVar v) = return t
transQNameType (HsTyCon qName) = liftM HsTyCon $ transQualNameM qName

transQNameCntxt :: HsContext -> Trans HsContext
transQNameCntxt cntxt = mapM transQNameAsst cntxt

transQNameAsst :: HsAsst -> Trans HsAsst
transQNameAsst (qName, ts)
   = liftM2 (,) (transQualNameM qName) (mapM transQNameType ts)

-}
