//===========================================================================
//  @(#) $Name: cflowd-2-1-b1 $
//  @(#) $Id: CflowdCisco.hh,v 1.20 1999/08/28 08:53:02 dwm Exp $
//===========================================================================
//  CAIDA Copyright Notice
//
//  By accessing this software, cflowd++, you are duly informed
//  of and agree to be bound by the conditions described below in this
//  notice:
//
//  This software product, cflowd++, is developed by Daniel W. McRobb, and
//  copyrighted(C) 1998 by the University of California, San Diego
//  (UCSD), with all rights reserved.  UCSD administers the CAIDA grant,
//  NCR-9711092, under which part of this code was developed.
//
//  There is no charge for cflowd++ software. You can redistribute it
//  and/or modify it under the terms of the GNU General Public License,
//  v.  2 dated June 1991 which is incorporated by reference herein.
//  cflowd++ is distributed WITHOUT ANY WARRANTY, IMPLIED OR EXPRESS, OF
//  MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE or that the use
//  of it will not infringe on any third party's intellectual property
//  rights.
//
//  You should have received a copy of the GNU GPL along with cflowd++.
//  Copies can also be obtained from:
//
//    http://www.gnu.org/copyleft/gpl.html
//
//  or by writing to:
//
//    University of California, San Diego
//
//    SDSC/CAIDA
//    9500 Gilman Dr., MS-0505
//    La Jolla, CA 92093 - 0505  USA
//
//  Or contact:
//
//    info@caida.org
//===========================================================================

#ifndef _CFLOWDCISCO_HH_
#define _CFLOWDCISCO_HH_

extern "C" {
#include <sys/types.h>

#include "caida_t.h"
}

#include "CflowdRawFlow.hh"
#include "CflowdRawFlowLogger.hh"
#include "CflowdCiscoFlowInterfaceMap.hh"
#include "CflowdCiscoFlowEngineMap.hh"

#include "snmp++/address.h"
#include "snmp++/target.h"
#include "snmp++/snmp_pp.h"

//---------------------------------------------------------------------------
//  class CflowdCisco
//---------------------------------------------------------------------------
//  This class abstracts a single Cisco router in cflowd++.  
//---------------------------------------------------------------------------
class CflowdCisco
{
public:
  
  //  bitmasks for the table index.  Use these to determine if
  //  a bit is set in the table index (and hence if a particular
  //  piece of data is being collected for the Cisco).
  
  static const uint16_t  k_cflowdProtocolTableMask;
  static const uint16_t  k_cflowdPortTableMask;
  static const uint16_t  k_cflowdNetMatrixMask;
  static const uint16_t  k_cflowdAsMatrixMask;
  static const uint16_t  k_cflowdRawFlowMask;
  static const uint16_t  k_cflowdPortMatrixMask;
  static const uint16_t  k_cflowdInterfaceMatrixMask;
  static const uint16_t  k_cflowdNextHopTableMask;
  static const uint16_t  k_cflowdTosTableMask;
  
  //-------------------------------------------------------------------------
  //                              CflowdCisco()
  //.........................................................................
  //  constructor
  //-------------------------------------------------------------------------
  CflowdCisco();

  //--------------------------------------------------------------------------
  //              CflowdCisco(const CflowdCisco & cflowdCisco) 
  //..........................................................................
  //  copy constructor
  //--------------------------------------------------------------------------
  CflowdCisco(const CflowdCisco & cflowdCisco);
  
  //-------------------------------------------------------------------------
  //                             ~CflowdCisco()
  //.........................................................................
  //  destructor
  //-------------------------------------------------------------------------
  ~CflowdCisco();
  
  //-------------------------------------------------------------------------
  //                      ipv4addr_t IpAddress() const
  //.........................................................................
  //  Returns the IP address of the Cisco.
  //-------------------------------------------------------------------------
  ipv4addr_t IpAddress() const;
  
  //-------------------------------------------------------------------------
  //                 ipv4addr_t IpAddress(ipv4addr_t ipAddr)
  //.........................................................................
  //  Sets and returns the IP address of the Cisco.
  //-------------------------------------------------------------------------
  ipv4addr_t IpAddress(ipv4addr_t ipAddr);
  
  //-------------------------------------------------------------------------
  //                        uint16_t FlowPort() const
  //.........................................................................
  //  Returns the port number that we'll use to listen for flow-export
  //  packets from the Cisco.  This correpsonds to the port number
  //  specified in the Cisco configuration, and the CFDATAPORT setting
  //  in the CISCOEXPORTER stanza of cflowd.conf.
  //-------------------------------------------------------------------------
  uint16_t FlowPort() const;
  
  //-------------------------------------------------------------------------
  //                  uint16_t FlowPort(uint16_t flowPort)
  //.........................................................................
  //  Sets and returns the port number that we'll use to listen for
  //  flow-export packets from the Cisco.
  //-------------------------------------------------------------------------
  uint16_t FlowPort(uint16_t flowPort);

  //--------------------------------------------------------------------------
  //                  const string & SnmpCommunity() const 
  //..........................................................................
  //  Returns the SNMP community of the Cisco, as specified with
  //  SNMPCOMM in the CISCOEXPORTER stanza of cflowd.conf
  //--------------------------------------------------------------------------
  const string & SnmpCommunity() const;
  
  //--------------------------------------------------------------------------
  //       const string & SnmpCommunity(const string & snmpCommunity) 
  //..........................................................................
  //  Sets and returns the SNMP community for the Cisco.
  //--------------------------------------------------------------------------
  const string & SnmpCommunity(const string & snmpCommunity);
  
  //-------------------------------------------------------------------------
  //                        uint16_t LocalAS() const
  //.........................................................................
  //  Returns the local AS of the Cisco, as specified with LOCALAS in
  //  the CISCOEXPORTER stanza of cflowd.conf
  //-------------------------------------------------------------------------
  uint16_t LocalAS() const;
  
  //-------------------------------------------------------------------------
  //                   uint16_t LocalAS(uint16_t localAS)
  //.........................................................................
  //  Sets and returns the local AS of the Cisco.
  //-------------------------------------------------------------------------
  uint16_t LocalAS(uint16_t localAS);
  
  //-------------------------------------------------------------------------
  //                       uint16_t TableIndex() const
  //.........................................................................
  //  Returns the table index bitfield for the Cisco.
  //-------------------------------------------------------------------------
  uint16_t TableIndex() const;
  
  //-------------------------------------------------------------------------
  //                uint16_t TableIndex(uint16_t tableIndex)
  //.........................................................................
  //  Sets and returns the table index bitfield for the Cisco.
  //-------------------------------------------------------------------------
  uint16_t TableIndex(uint16_t tableIndex);

  //-------------------------------------------------------------------------
  //          inline CflowdCiscoFlowEngineMap & FlowEngines() const          
  //.........................................................................
  //  Returns a reference to the CflowdCiscoFlowEngineMap contained
  //  in the CflowdCisco object.  The CflowdCiscoFlowEngineMap is
  //  used to track sequence numbers for each flow-export engine on
  //  the Cisco (VIP adapters and RSPs).
  //-------------------------------------------------------------------------
  inline CflowdCiscoFlowEngineMap & FlowEngines() const
  {
    return(this->_flowEngines);
  }
  
  //-------------------------------------------------------------------------
  //         inline CflowdCiscoFlowInterfaceMap & Interfaces() const         
  //.........................................................................
  //  Returns a reference to the CflowdCiscoFlowInterfaceMap contained
  //  in the CflowdCisco object.  The CflowdCiscoFlowInterfaceMap is
  //  used to hold tabular data on a per-input-interface basis.
  //-------------------------------------------------------------------------
  inline CflowdCiscoFlowInterfaceMap & Interfaces() const
  {
    return(this->_interfaces);
  }

  //-------------------------------------------------------------------------
  //           inline CflowdRawFlowLogger * FlowLogger() const
  //.........................................................................
  //  Returns a pointer to the CflowdRawFlowLogger in the CflowdCisco
  //  object.  May return (CflowdRawFlowLogger *)NULL if there is no
  //  CflowdRawFlowLogger in the CflowdCisco object.
  //-------------------------------------------------------------------------
  inline CflowdRawFlowLogger * FlowLogger() const
  {
    return(this->_flowLogger);
  }
  
  //-------------------------------------------------------------------------
  //                   inline uint32_t LastCleared() const                   
  //.........................................................................
  //  Returns the last time we cleared the tabular data for the Cisco,
  //  in seconds since the UNIX epoch.
  //-------------------------------------------------------------------------
  inline uint32_t LastCleared() const
  {
    return(this->_lastCleared);
  }

  //-------------------------------------------------------------------------
  //            inline uint32_t LastCleared(uint32_t lastCleared)            
  //.........................................................................
  //  Sets and returns the last time we cleared the tabular data for
  //  the Cisco, in seconds since the UNIX epoch.
  //-------------------------------------------------------------------------
  inline uint32_t LastCleared(uint32_t lastCleared)
  {
    this->_lastCleared = lastCleared;
    return(this->_lastCleared);
  }
  
  //-------------------------------------------------------------------------
  //                   inline uint32_t LastUpdated() const                   
  //.........................................................................
  //  Returns the last time we updated the tabular data for the Cisco,
  //  in seconds since the UNIX epoch.
  //-------------------------------------------------------------------------
  inline uint32_t LastUpdated() const
  {
    return(this->_lastUpdated);
  }
  
  //-------------------------------------------------------------------------
  //            inline uint32_t LastUpdated(uint32_t lastUpdated)            
  //.........................................................................
  //  Sets and returns the last time we updated the tabular data for
  //  the Cisco, in seconds since the UNIX epoch.
  //-------------------------------------------------------------------------
  inline uint32_t LastUpdated(uint32_t lastUpdated)
  {
    this->_lastUpdated = lastUpdated;
    return(this->_lastUpdated);
  }

  //-------------------------------------------------------------------------
  //               inline uint32_t MissedFlowsThreshold() const              
  //.........................................................................
  //  
  //-------------------------------------------------------------------------
  inline uint32_t MissedFlowsThreshold() const
  {
    return(this->_missedFlowsThreshold);
  }

  //-------------------------------------------------------------------------
  //   inline uint32_t MissedFlowsThreshold(uint32_t missedFlowsThreshold)   
  //.........................................................................
  //  
  //-------------------------------------------------------------------------
  inline uint32_t MissedFlowsThreshold(uint32_t missedFlowsThreshold)
  {
    this->_missedFlowsThreshold = missedFlowsThreshold;
    return(this->_missedFlowsThreshold);
  }
  
  //-------------------------------------------------------------------------
  //                 int AddFlow(const CflowdRawFlow & flow)
  //.........................................................................
  //  Adds data from a CflowdRawFlow to the Cisco's tabular data
  //  (per-input-interface).  Always returns 0.
  //-------------------------------------------------------------------------
  int AddFlow(const CflowdRawFlow & flow);
  
  //-------------------------------------------------------------------------
  //                      istream & read(istream & is)
  //.........................................................................
  //  Reads the contents of a CflowdCisco from an istream.  Returns
  //  the istream.
  //-------------------------------------------------------------------------
  istream & read(istream & is);
  
  //-------------------------------------------------------------------------
  //                            int read(int fd)
  //.........................................................................
  //  Reads the contents of a CflowdCisco from a file descriptor.
  //  Returns the number of bytes read on success, -1 on failure.
  //-------------------------------------------------------------------------
  int read(int fd);
  
  //-------------------------------------------------------------------------
  //                   ostream & write(ostream & os) const
  //.........................................................................
  //  Writes the contents of a CflowdCisco to an ostream.  Returns the
  //  ostream.
  //-------------------------------------------------------------------------
  ostream & write(ostream & os) const;
  
  //-------------------------------------------------------------------------
  //                         int write(int fd) const
  //.........................................................................
  //  Writes the contents of a CflowdCisco to a file descriptor.
  //  Returns the number of bytes written on success, -1 on failure.
  //-------------------------------------------------------------------------
  int write(int fd) const;
  
  //-------------------------------------------------------------------------
  //                          int ClearTableData()
  //.........................................................................
  //  Clears the tabular data for the Cisco.  Always returns 0.
  //-------------------------------------------------------------------------
  int ClearTableData();

  int CreateFlowLogger(const string & flowDirectory, int numLogs,
                       int logSize);

  int GetInterfaceInfo();

  //--------------------------------------------------------------------------
  //                  bool HaveRecentInterfaceInfo() const 
  //..........................................................................
  //  Returns true if we've polled for interface information in the last
  //  30 minutes, else returns false.  A return of true doesn't really
  //  mean we actually got data, hence this function should be renamed.
  //--------------------------------------------------------------------------
  bool HaveRecentInterfaceInfo() const;

  //--------------------------------------------------------------------------
  //                bool HaveRecentInterfaceInfo(bool value) 
  //..........................................................................
  //  
  //--------------------------------------------------------------------------
  bool HaveRecentInterfaceInfo(bool value);
  
private:
  uint32_t                              _ipAddress;
  uint16_t                              _flowPort;
  uint16_t                              _localAS;
  string                                _snmpCommunity;
  uint32_t                              _lastCleared;
  uint32_t                              _lastUpdated;
  mutable CflowdCiscoFlowEngineMap      _flowEngines;
  mutable CflowdCiscoFlowInterfaceMap   _interfaces;
  uint32_t                              _missedFlowsThreshold;
  mutable CflowdRawFlowLogger          *_flowLogger;
  uint16_t                              _tableIndex;
  bool                                  _haveInterfaceInfo;
  uint32_t                              _lastSnmpQueryTime;
  
  void GetInterfaceAddresses(Snmp & session, CTarget & target);
  void GetInterfaceDescriptions(Snmp & session, CTarget & target);
};

#endif  // _CFLOWDCISCO_HH_
