//===========================================================================
//  $Name: cflowd-2-1-b1 $
//  $Id: CflowdCiscoFlowInterface.hh,v 1.10 1999/08/06 14:57:54 dwm Exp $
//===========================================================================
//  CAIDA Copyright Notice
//
//  By accessing this software, cflowd++, you are duly informed
//  of and agree to be bound by the conditions described below in this
//  notice:
//
//  This software product, cflowd++, is developed by Daniel W. McRobb, and
//  copyrighted(C) 1998 by the University of California, San Diego
//  (UCSD), with all rights reserved.  UCSD administers the CAIDA grant,
//  NCR-9711092, under which part of this code was developed.
//
//  There is no charge for cflowd++ software. You can redistribute it
//  and/or modify it under the terms of the GNU General Public License,
//  v.  2 dated June 1991 which is incorporated by reference herein.
//  cflowd++ is distributed WITHOUT ANY WARRANTY, IMPLIED OR EXPRESS, OF
//  MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE or that the use
//  of it will not infringe on any third party's intellectual property
//  rights.
//
//  You should have received a copy of the GNU GPL along with cflowd++.
//  Copies can also be obtained from:
//
//    http://www.gnu.org/copyleft/gpl.html
//
//  or by writing to:
//
//    University of California, San Diego
//
//    SDSC/CAIDA
//    9500 Gilman Dr., MS-0505
//    La Jolla, CA 92093 - 0505  USA
//
//  Or contact:
//
//    info@caida.org
//===========================================================================

#ifndef _CFLOWDCISCOFLOWINTERFACE_HH_
#define _CFLOWDCISCOFLOWINTERFACE_HH_

#include <string>

#include "CflowdAsMatrix.hh"
#include "CflowdNetMatrix.hh"
#include "CflowdPortMatrix.hh"
#include "CflowdProtocolTable.hh"
#include "CflowdInterfaceMatrix.hh"
#include "CflowdNextHopTable.hh"
#include "CflowdTosTable.hh"

#ifndef INADDR_NONE
  #define INADDR_NONE 0xffffffff
#endif

//---------------------------------------------------------------------------
//  class CflowdCiscoFlowInterface
//---------------------------------------------------------------------------
//  This class abstracts an input interface doing flow-switching
//  on a Cisco.  It is used to hold tabular data for an input interface.
//  In cflowd++, we keep one of these objects per input interface so
//  that tables and reports may be input interface specific.
//---------------------------------------------------------------------------
class CflowdCiscoFlowInterface
{
public:
  //--------------------------------------------------------------------------
  //                    inline CflowdCiscoFlowInterface() 
  //..........................................................................
  //  constructor
  //--------------------------------------------------------------------------
  inline CflowdCiscoFlowInterface()
  {
    _ipAddr = INADDR_NONE;
    _tableIndex = 0;
  }

  //--------------------------------------------------------------------------
  //                    inline ~CflowdCiscoFlowInterface() 
  //..........................................................................
  //  destructor
  //--------------------------------------------------------------------------
  inline ~CflowdCiscoFlowInterface()
  {
    if (this->_asMatrix.size() > 0)
      this->_asMatrix.erase(this->_asMatrix.begin(),this->_asMatrix.end());
    if (this->_netMatrix.size() > 0)
      this->_netMatrix.erase(this->_netMatrix.begin(),this->_netMatrix.end());
    if (this->_portMatrix.size() > 0)
      this->_portMatrix.erase(this->_portMatrix.begin(),
                              this->_portMatrix.end());
    if (this->_protocolTable.size() > 0)
      this->_protocolTable.erase(this->_protocolTable.begin(),
                                 this->_protocolTable.end());
    if (this->_intfMatrix.size() > 0)
      this->_intfMatrix.erase(this->_intfMatrix.begin(),
                              this->_intfMatrix.end());
    if (this->_nextHopTable.size() > 0)
      this->_nextHopTable.erase(this->_nextHopTable.begin(),
                                this->_nextHopTable.end());
    if (this->_tosTable.size() > 0)
      this->_tosTable.erase(this->_tosTable.begin(),this->_tosTable.end());
  }
  
  //--------------------------------------------------------------------------
  //                  inline const string & IfDescr() const 
  //..........................................................................
  //  Returns the ifDescr for the interface.  If ifDescr has not been set,
  //  returns string("").
  //--------------------------------------------------------------------------
  inline const string & IfDescr() const
  {
    return(this->_ifDescr);
  }

  //--------------------------------------------------------------------------
  //          inline const string & IfDescr(const string & ifDescr) 
  //..........................................................................
  //  Sets and returns the ifDescr for the interface.
  //--------------------------------------------------------------------------
  inline const string & IfDescr(const string & ifDescr)
  {
    this->_ifDescr = ifDescr;
    return(this->_ifDescr);
  }
  
  //--------------------------------------------------------------------------
  //                     inline ipv4addr_t IpAddr() const 
  //..........................................................................
  //  Returns the IP address of the interface.  If the IP address has not
  //  been set, returns 0.
  //--------------------------------------------------------------------------
  inline ipv4addr_t IpAddr() const
  {
    return(this->_ipAddr);
  }

  //--------------------------------------------------------------------------
  //               inline ipv4addr_t IpAddr(ipv4addr_t ipAddr) 
  //..........................................................................
  //  Sets and returns the IP address of the interface.
  //--------------------------------------------------------------------------
  inline ipv4addr_t IpAddr(ipv4addr_t ipAddr)
  {
    this->_ipAddr = ipAddr;
    return(this->_ipAddr);
  }
  
  //-------------------------------------------------------------------------
  //                    inline uint8_t TableIndex() const                    
  //.........................................................................
  //  Returns the table index bitfield for the interface.  Currently
  //  this is unused, since we keep our table index one level up
  //  in the containment heirarchy (in the CflowdCisco class, i.e.
  //  per-Cisco and not per-interface).
  //-------------------------------------------------------------------------
  inline uint8_t TableIndex() const
  {
    return(this->_tableIndex);
  }
  
  //-------------------------------------------------------------------------
  //              inline uint8_t TableIndex(uint8_t tableIndex)              
  //.........................................................................
  //  Sets and returns the table index bitfield for the interface.
  //  Currently this is unused, since we keep our table index one level
  //  up in the containment heirarchy (in the CflowdCisco class, i.e.
  //  per-Cisco and not per-interface).
  //-------------------------------------------------------------------------
  inline uint8_t TableIndex(uint8_t tableIndex)
  {
    this->_tableIndex = tableIndex;
    return(this->_tableIndex);
  }
  
  //-------------------------------------------------------------------------
  //                 inline CflowdAsMatrix & AsMatrix() const                
  //.........................................................................
  //  Returns a reference to the AS matrix for the interface.
  //-------------------------------------------------------------------------
  inline CflowdAsMatrix & AsMatrix() const
  {
    return(this->_asMatrix);
  }
  
  //-------------------------------------------------------------------------
  //                inline CflowdNetMatrix & NetMatrix() const               
  //.........................................................................
  //  Returns a reference to the net matrix for the interface.
  //-------------------------------------------------------------------------
  inline CflowdNetMatrix & NetMatrix() const
  {
    return(this->_netMatrix);
  }
  
  //-------------------------------------------------------------------------
  //               inline CflowdPortMatrix & PortMatrix() const              
  //.........................................................................
  //  Returns a reference to the port matrix for the interface.
  //-------------------------------------------------------------------------
  inline CflowdPortMatrix & PortMatrix() const
  {
    return(this->_portMatrix);
  }
  
  //-------------------------------------------------------------------------
  //            inline CflowdProtocolTable & ProtocolTable() const           
  //.........................................................................
  //  Returns a reference to the protocol table for the interface.
  //-------------------------------------------------------------------------
  inline CflowdProtocolTable & ProtocolTable() const
  {
    return(this->_protocolTable);
  }

  //-------------------------------------------------------------------------
  //          inline CflowdInterfaceMatrix & InterfaceMatrix() const         
  //.........................................................................
  //  Returns a reference to the interface matrix for the interface.
  //-------------------------------------------------------------------------
  inline CflowdInterfaceMatrix & InterfaceMatrix() const
  {
    return(this->_intfMatrix);
  }

  //-------------------------------------------------------------------------
  //             inline CflowdNextHopTable & NextHopTable() const            
  //.........................................................................
  //  Returns a reference to the IP nexthop table for the interface.
  //-------------------------------------------------------------------------
  inline CflowdNextHopTable & NextHopTable() const
  {
    return(this->_nextHopTable);
  }

  //--------------------------------------------------------------------------
  //                 inline CflowdTosTable & TosTable() const 
  //..........................................................................
  //  Returns a reference to the IP TOS table for the interface.
  //--------------------------------------------------------------------------
  inline CflowdTosTable & TosTable() const
  {
    return(this->_tosTable);
  }
  
  //-------------------------------------------------------------------------
  //                 int AddFlow(const CflowdRawFlow & flow)
  //.........................................................................
  //  Adds the contents of a raw flow to the interface's tabular
  //  data.  Always returns 0.
  //-------------------------------------------------------------------------
  int AddFlow(const CflowdRawFlow & flow);
  
  
private:
  string                         _ifDescr;
  ipv4addr_t                     _ipAddr;
  uint8_t                        _tableIndex;
  mutable CflowdAsMatrix         _asMatrix;
  mutable CflowdNetMatrix        _netMatrix;
  mutable CflowdPortMatrix       _portMatrix;
  mutable CflowdProtocolTable    _protocolTable;
  mutable CflowdInterfaceMatrix  _intfMatrix;
  mutable CflowdNextHopTable     _nextHopTable;
  mutable CflowdTosTable         _tosTable;
};

#endif  // _CFLOWDCISCOFLOWINTERFACE_HH_
