//===========================================================================
//  @(#) $Name: cflowd-2-1-b1 $
//  @(#) $Id: CflowdConfig.hh,v 1.1 1999/01/25 09:37:45 dwm Exp $
//===========================================================================
//  CAIDA Copyright Notice
//
//  By accessing this software, cflowd++, you are duly informed
//  of and agree to be bound by the conditions described below in this
//  notice:
//
//  This software product, cflowd++, is developed by Daniel W. McRobb, and
//  copyrighted(C) 1998 by the University of California, San Diego
//  (UCSD), with all rights reserved.  UCSD administers the CAIDA grant,
//  NCR-9711092, under which part of this code was developed.
//
//  There is no charge for cflowd++ software. You can redistribute it
//  and/or modify it under the terms of the GNU General Public License,
//  v.  2 dated June 1991 which is incorporated by reference herein.
//  cflowd++ is distributed WITHOUT ANY WARRANTY, IMPLIED OR EXPRESS, OF
//  MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE or that the use
//  of it will not infringe on any third party's intellectual property
//  rights.
//
//  You should have received a copy of the GNU GPL along with cflowd++.
//  Copies can also be obtained from:
//
//    http://www.gnu.org/copyleft/gpl.html
//
//  or by writing to:
//
//    University of California, San Diego
//
//    SDSC/CAIDA
//    9500 Gilman Dr., MS-0505
//    La Jolla, CA 92093 - 0505  USA
//
//  Or contact:
//
//    info@caida.org
//===========================================================================

#ifndef _CFLOWDCONFIG_HH_
#define _CFLOWDCONFIG_HH_

#include <string>

#include "CflowdCiscoMap.hh"
#include "CflowdFlowPortList.hh"
#include "CflowdCollectorMap.hh"

//---------------------------------------------------------------------------
//  class CflowdConfig
//---------------------------------------------------------------------------
//  This class is used to hold cflowd configuration information.
//---------------------------------------------------------------------------
class CflowdConfig
{
public:
  static const string k_defaultCflowdConfigFile;
  
  //-------------------------------------------------------------------------
  //                            CflowdConfig()
  //.........................................................................
  //  constructor
  //-------------------------------------------------------------------------
  CflowdConfig();
  
  //-------------------------------------------------------------------------
  //                           ~CflowdConfig()
  //.........................................................................
  //  destructor
  //-------------------------------------------------------------------------
  ~CflowdConfig();
  
  //-------------------------------------------------------------------------
  //                         int LogFacility() const
  //.........................................................................
  //  Returns the syslog facility for cflowd.  This corresponds to the
  //  LOGFACILITY setting in the OPTIONS stanza in cflowd.conf.
  //-------------------------------------------------------------------------
  int LogFacility() const;
  
  //-------------------------------------------------------------------------
  //                    int LogFacility(int logFacility)
  //.........................................................................
  //  Sets and returns the syslog facility for cflowd.
  //-------------------------------------------------------------------------
  int LogFacility(int logFacility);
  
  //-------------------------------------------------------------------------
  //                       int TcpCollectPort() const
  //.........................................................................
  //  Returns the TCP port number on which cflowd is listening for
  //  collector connections from cfdcollect.  This corresponds to the
  //  TCPCOLLECTPORT setting in the OPTIONS stanza in cflowd.conf.
  //-------------------------------------------------------------------------
  int TcpCollectPort() const;
  
  //-------------------------------------------------------------------------
  //                 int TcpCollectPort(int tcpCollectPort)
  //.........................................................................
  //  Sets and returns the TCP port number on which cflowd will listen
  //  for collector connections from cfdcollect.
  //-------------------------------------------------------------------------
  int TcpCollectPort(int tcpCollectPort);

  //-------------------------------------------------------------------------
  //                      size_t PacketBufSize() const
  //.........................................................................
  //  Returns the size of the shared memory segment being used for packet
  //  buffering.
  //-------------------------------------------------------------------------
  size_t PacketBufSize() const;
  
  //-------------------------------------------------------------------------
  //                 size_t PacketBufSize(size_t pktBufSize)
  //.........................................................................
  //  Sets and returns the size of the shared memory segment to be used
  //  for packet buffering.
  //-------------------------------------------------------------------------
  size_t PacketBufSize(size_t pktBufSize);
  
  //-------------------------------------------------------------------------
  //                   const string & MapDirectory() const
  //.........................................................................
  //  UNUSED
  //-------------------------------------------------------------------------
  const string & MapDirectory() const;
  
  //-------------------------------------------------------------------------
  //        const string & MapDirectory(const string & mapDirectory)
  //.........................................................................
  //  UNUSED
  //-------------------------------------------------------------------------
  const string & MapDirectory(const string & mapDirectory);

  //-------------------------------------------------------------------------
  //                 const string & RawFlowSockFile() const
  //.........................................................................
  //  Returns the path to the named stream socket on which cflowdmux
  //  will listen for connections from raw flow clients.  This
  //  corresponds to the RAWFLOWSOCKFILE setting in the OPTIONS
  //  stanza in cflowd.conf.
  //-------------------------------------------------------------------------
  const string & RawFlowSockFile() const;

  //-------------------------------------------------------------------------
  //     const string & RawFlowSockFile(const string & rawFlowSockFile)
  //.........................................................................
  //  Sets and returns the path to the named stream socket on which
  //  cflowdmux will listen for connections from raw flow clients.
  //-------------------------------------------------------------------------
  const string & RawFlowSockFile(const string & rawFlowSockFile);
  
  //-------------------------------------------------------------------------
  //                  const string & FlowDirectory() const
  //.........................................................................
  //  Returns the directory in which raw flow files are stored.
  //-------------------------------------------------------------------------
  const string & FlowDirectory() const;
  
  //-------------------------------------------------------------------------
  //       const string & FlowDirectory(const string & flowDirectory)
  //.........................................................................
  //  Sets and returns the directory in which to store raw flow files.
  //-------------------------------------------------------------------------
  const string & FlowDirectory(const string & flowDirectory);
  
  //-------------------------------------------------------------------------
  //                    uint32_t  FlowFileLength() const
  //.........................................................................
  //  Returns the size of the raw flow files to be used per router.
  //-------------------------------------------------------------------------
  uint32_t  FlowFileLength() const;
  
  //-------------------------------------------------------------------------
  //            uint32_t FlowFileLength(uint32_t flowFileLength)
  //.........................................................................
  //  Sets and returns the size of the raw flow files to be used per
  //  router.
  //-------------------------------------------------------------------------
  uint32_t FlowFileLength(uint32_t flowFileLength);
  
  //-------------------------------------------------------------------------
  //                      uint32_t NumFlowFiles() const
  //.........................................................................
  //  Returns the number of raw flow files to keep per router.
  //-------------------------------------------------------------------------
  uint32_t NumFlowFiles() const;
  
  //-------------------------------------------------------------------------
  //              uint32_t NumFlowFiles(uint32_t numFlowFiles)
  //.........................................................................
  //  Sets and returns the number of raw flow files to keep per router.
  //-------------------------------------------------------------------------
  uint32_t NumFlowFiles(uint32_t numFlowFiles);

  //-------------------------------------------------------------------------
  //                      uint32_t MinLogMissed() const
  //.........................................................................
  //  Returns the threshold at which cflowd will syslog a message about
  //  missed flows.  cflowd only does this when contacted by cfdcollect.
  //-------------------------------------------------------------------------
  uint32_t MinLogMissed() const;

  //-------------------------------------------------------------------------
  //              uint32_t MinLogMissed(uint32_t minLogMissed)
  //.........................................................................
  //  Sets and returns the threshold at which cflowd will syslog a
  //  message about missed flows.  cflowd only does this when contacted
  //  by cfdcollect.
  //-------------------------------------------------------------------------
  uint32_t MinLogMissed(uint32_t minLogMissed);

  //-------------------------------------------------------------------------
  //                  const string & TableSockFile() const
  //.........................................................................
  //  Returns the path to the named stream socket on which cflowd
  //  will listen for connections from local clients requesting tabular
  //  data (cfdases, cfdnets, et. al.).  This corresponds to the
  //  TABLESOCKFILE setting in the OPTIONS stanza in cflowd.conf.
  //-------------------------------------------------------------------------
  const string & TableSockFile() const;
  
  //-------------------------------------------------------------------------
  //       const string & TableSockFile(const string & tableSockFile)
  //.........................................................................
  //  Sets and returns the path to the named stream socket on which
  //  cflowd will listen for connections from local clients requesting
  //  tabular data (cfdases, cfdnets, et. al.).
  //-------------------------------------------------------------------------
  const string & TableSockFile(const string & tableSockFile);

  //-------------------------------------------------------------------------
  //                 inline CflowdCiscoMap & CiscoMap() const                
  //.........................................................................
  //  Returns a reference to the map of CflowdCisco pointers contained
  //  in the cflowd configuration.  This is a map keyed by IP address.
  //  Each CflowdCisco corresponds to a CISCOEXPORTER stanza in
  //  cflowd.conf, but our map may contain multiple pointers to the same
  //  CflowdCisco (we normally have keys for each interface IP address,
  //  pointing to the same CflowdCisco).
  //-------------------------------------------------------------------------
  inline CflowdCiscoMap & CiscoMap() const
  {
    return(this->_ciscoMap);
  }
  
  //-------------------------------------------------------------------------
  //                CflowdFlowPortList & FlowPortList() const                
  //.........................................................................
  //  Returns a reference to the contained CflowdFlowPortList.  The
  //  CflowdFlowPortList holds a list of the UDP ports on which
  //  cflowdmux is listening for flow-export packets.
  //-------------------------------------------------------------------------
  CflowdFlowPortList & FlowPortList() const
  {
    return(this->_portList);
  }
  
  //-------------------------------------------------------------------------
  //                CflowdCollectorMap & CollectorMap() const                
  //.........................................................................
  //  Returns a reference to the contained CflowdCollectorMap.  The
  //  CflowdCollectorMap holds pointers to multiple CflowdCollector
  //  objects and is keyed by IP address.
  //-------------------------------------------------------------------------
  CflowdCollectorMap & CollectorMap() const
  {
    return(this->_collectorMap);
  }

  //-------------------------------------------------------------------------
  //                              void Clear()
  //.........................................................................
  //  Clears the CflowdConfig object.
  //-------------------------------------------------------------------------
  void Clear();
  
private:
  int                          _logFacility;
  int                          _tcpCollectPort;
  size_t                       _pktBufSize;
  string                       _mapDirectory;
  string                       _rawFlowSockFile;
  string                       _tableSockFile;
  string                       _flowDirectory;
  uint32_t                     _flowFileLength;
  uint32_t                     _numFlowFiles;
  uint32_t                     _minLogMissed;
  mutable CflowdCiscoMap       _ciscoMap;
  mutable CflowdFlowPortList   _portList;
  mutable CflowdCollectorMap   _collectorMap;
};

#endif  // _CFLOWDCONFIG_HH_
