//===========================================================================
//  $Name: cflowd-2-1-b1 $
//  $Id: CflowdRawFlow.hh,v 1.15 2000/02/29 19:32:20 dwm Exp $
//===========================================================================
//  CAIDA Copyright Notice
//
//  By accessing this software, cflowd++, you are duly informed
//  of and agree to be bound by the conditions described below in this
//  notice:
//
//  This software product, cflowd++, is developed by Daniel W. McRobb, and
//  copyrighted(C) 1998 by the University of California, San Diego
//  (UCSD), with all rights reserved.  UCSD administers the CAIDA grant,
//  NCR-9711092, under which part of this code was developed.
//
//  There is no charge for cflowd++ software. You can redistribute it
//  and/or modify it under the terms of the GNU General Public License,
//  v.  2 dated June 1991 which is incorporated by reference herein.
//  cflowd++ is distributed WITHOUT ANY WARRANTY, IMPLIED OR EXPRESS, OF
//  MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE or that the use
//  of it will not infringe on any third party's intellectual property
//  rights.
//
//  You should have received a copy of the GNU GPL along with cflowd++.
//  Copies can also be obtained from:
//
//    http://www.gnu.org/copyleft/gpl.html
//
//  or by writing to:
//
//    University of California, San Diego
//
//    SDSC/CAIDA
//    9500 Gilman Dr., MS-0505
//    La Jolla, CA 92093 - 0505  USA
//
//  Or contact:
//
//    info@caida.org
//===========================================================================

#ifndef _CFLOWDRAWFLOW_HH_
#define _CFLOWDRAWFLOW_HH_

extern "C" {
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <sys/uio.h>
#include <assert.h>
#include <unistd.h>
#include <time.h>
#include <errno.h>
#include <syslog.h>
  
#include "caida_t.h"
}

#include <iostream.h>

#include "CflowdFlowPdu.h"

//---------------------------------------------------------------------------
//  class CflowdRawFlow
//---------------------------------------------------------------------------
//  This class abstracts a raw flow from Cisco's flow-export.  It is
//  used by cflowdmux to send raw flows to clients, including cflowd.
//
//  The motivation for the multiple constructors: provide a means of
//  dealing with multiple versions of Cisco flow-export that's not too
//  painful for the class library user.  As new version of flow-export
//  come along, we should be able to just add new constructors and
//  possibly new data and access members (if the new flow-export version
//  has additional information not contained in current versions).
//---------------------------------------------------------------------------
class CflowdRawFlow
{
public:
  typedef uint32_t index_type;

  //  Define some masks for checking the bits in the flow index.
  //  These are used to detect the presence of data fields by bitwise
  //  ANDing them with the flow index; we set a single bit for
  //  each field present in the data.
  
  static const index_type k_routerMask        = 0x00000001;
  static const index_type k_srcIpAddrMask     = 0x00000002;
  static const index_type k_dstIpAddrMask     = 0x00000004;
  static const index_type k_inputIfIndexMask  = 0x00000008;
  static const index_type k_outputIfIndexMask = 0x00000010;
  static const index_type k_srcPortMask       = 0x00000020;
  static const index_type k_dstPortMask       = 0x00000040;
  static const index_type k_pktsMask          = 0x00000080;
  static const index_type k_bytesMask         = 0x00000100;
  static const index_type k_ipNextHopMask     = 0x00000200;
  static const index_type k_startTimeMask     = 0x00000400;
  static const index_type k_endTimeMask       = 0x00000800;
  static const index_type k_protocolMask      = 0x00001000;
  static const index_type k_tosMask           = 0x00002000;
  static const index_type k_srcAsMask         = 0x00004000;
  static const index_type k_dstAsMask         = 0x00008000;
  static const index_type k_srcMaskLenMask    = 0x00010000;
  static const index_type k_dstMaskLenMask    = 0x00020000;
  static const index_type k_tcpFlagsMask      = 0x00040000;
  static const index_type k_inputEncapMask    = 0x00080000;
  static const index_type k_outputEncapMask   = 0x00100000;
  static const index_type k_peerNextHopMask   = 0x00200000;
  static const index_type k_engineTypeMask    = 0x00400000;
  static const index_type k_engineIdMask      = 0x00800000;

  typedef struct {
    index_type  _index;
    ipv4addr_t  _router;
    ipv4addr_t  _srcIpAddr;
    ipv4addr_t  _dstIpAddr;
    uint16_t    _inputIfIndex;
    uint16_t    _outputIfIndex;
    uint16_t    _srcPort;
    uint16_t    _dstPort;
    uint32_t    _pkts;
    uint32_t    _bytes;
    ipv4addr_t  _ipNextHop;
    uint32_t    _startTime;
    uint32_t    _endTime;
    uint8_t     _protocol;
    uint8_t     _tos;
    uint16_t    _srcAs;
    uint16_t    _dstAs;
    uint8_t     _srcMaskLen;
    uint8_t     _dstMaskLen;
    uint8_t     _tcpFlags;
    uint8_t     _inputEncap;
    uint8_t     _outputEncap;
    ipv4addr_t  _peerNextHop;
    uint8_t     _engineType;
    uint8_t     _engineId;
    bool        _isHostOrder;
    uint8_t     _version;
  } _data_t;

  //-------------------------------------------------------------------------
  //                             CflowdRawFlow()
  //.........................................................................
  //  constructor
  //-------------------------------------------------------------------------
  CflowdRawFlow();
  
  //-------------------------------------------------------------------------
  //            CflowdRawFlow(ipv4addr_t ciscoIp,                    
  //                          const CiscoFlowHeaderV1_t * flowHeader, 
  //                          const CiscoFlowEntryV1_t * flowEntry)
  //.........................................................................
  //  Constructs from an IP address (of the Cisco sending the flow
  //  data) and a pointer to a v1 flow header and a v1 flow entry.
  //-------------------------------------------------------------------------
  CflowdRawFlow(ipv4addr_t ciscoIp,
                const CiscoFlowHeaderV1_t * flowHeader,
                const CiscoFlowEntryV1_t * flowEntry);

  //-------------------------------------------------------------------------
  //          void Init(ipv4addr_t ciscoIp,                  
  //                    const CiscoFlowHeaderV1_t * flowHeader, 
  //                    const CiscoFlowEntryV1_t * flowEntry)
  //.........................................................................
  //  
  //-------------------------------------------------------------------------
  void Init(ipv4addr_t ciscoIp,
            const CiscoFlowHeaderV1_t * flowHeader,
            const CiscoFlowEntryV1_t * flowEntry);

  //-------------------------------------------------------------------------
  //          CflowdRawFlow(ipv4addr_t ciscoIp,                    
  //                        const CiscoFlowHeaderV5_t * flowHeader, 
  //                        const CiscoFlowEntryV5_t * flowEntry)
  //.........................................................................
  //  Constructs from an IP address (of the Cisco sending the flow
  //  data) and a pointer to a v5 flow header and a v5 flow entry.
  //-------------------------------------------------------------------------
  inline CflowdRawFlow(ipv4addr_t ciscoIp,
                       const CiscoFlowHeaderV5_t * flowHeader,
                       const CiscoFlowEntryV5_t * flowEntry)
  {
    this->data._index = 0;
    this->data._isHostOrder = true;
    
    this->Version(ntohs(flowHeader->version));
    assert(this->data._version == 5);
    
    this->Router(ciscoIp);
    this->StartTime(ntohl(flowHeader->unix_secs) +
      (ntohl(flowEntry->first)/1000 - ntohl(flowHeader->sysUptime)/1000));
    this->EndTime(ntohl(flowHeader->unix_secs) +
      (ntohl(flowEntry->last)/1000 - ntohl(flowHeader->sysUptime)/1000));
    this->IpNextHop(flowEntry->nexthop);
    this->SrcIpAddr(flowEntry->srcaddr);
    this->DstIpAddr(flowEntry->dstaddr);
    this->InputIfIndex(ntohs(flowEntry->input));
    this->OutputIfIndex(ntohs(flowEntry->output));
    this->SrcPort(ntohs(flowEntry->srcport));
    this->DstPort(ntohs(flowEntry->dstport));
    this->Protocol(flowEntry->prot);
    this->Tos(flowEntry->tos);
    this->TcpFlags(flowEntry->tcp_flags);
    this->SrcAs(ntohs(flowEntry->src_as));
    this->DstAs(ntohs(flowEntry->dst_as));
    this->SrcMaskLen(flowEntry->src_mask);
    this->DstMaskLen(flowEntry->dst_mask);
    this->Pkts(ntohl(flowEntry->pkts));
    this->Bytes(ntohl(flowEntry->bytes));
    this->EngineType(flowHeader->engine_type);
    this->EngineId(flowHeader->engine_id);
  }

  //-------------------------------------------------------------------------
  //        inline void Init(ipv4addr_t ciscoIp,                     
  //                         const CiscoFlowHeaderV5_t * flowHeader, 
  //                         const CiscoFlowEntryV5_t * flowEntry)   
  //.........................................................................
  //  
  //-------------------------------------------------------------------------
  inline void Init(ipv4addr_t ciscoIp,
                   const CiscoFlowHeaderV5_t * flowHeader,
                   const CiscoFlowEntryV5_t * flowEntry)
  {
    this->data._index = 0;
    this->data._isHostOrder = true;
    
    this->Version(ntohs(flowHeader->version));
    assert(this->data._version == 5);
    
    this->Router(ciscoIp);
    this->StartTime(ntohl(flowHeader->unix_secs) +
      (ntohl(flowEntry->first)/1000 - ntohl(flowHeader->sysUptime)/1000));
    this->EndTime(ntohl(flowHeader->unix_secs) +
      (ntohl(flowEntry->last)/1000 - ntohl(flowHeader->sysUptime)/1000));
    this->IpNextHop(flowEntry->nexthop);
    this->SrcIpAddr(flowEntry->srcaddr);
    this->DstIpAddr(flowEntry->dstaddr);
    this->InputIfIndex(ntohs(flowEntry->input));
    this->OutputIfIndex(ntohs(flowEntry->output));
    this->SrcPort(ntohs(flowEntry->srcport));
    this->DstPort(ntohs(flowEntry->dstport));
    this->Protocol(flowEntry->prot);
    this->Tos(flowEntry->tos);
    this->TcpFlags(flowEntry->tcp_flags);
    this->SrcAs(ntohs(flowEntry->src_as));
    this->DstAs(ntohs(flowEntry->dst_as));
    this->SrcMaskLen(flowEntry->src_mask);
    this->DstMaskLen(flowEntry->dst_mask);
    this->Pkts(ntohl(flowEntry->pkts));
    this->Bytes(ntohl(flowEntry->bytes));
    this->EngineType(flowHeader->engine_type);
    this->EngineId(flowHeader->engine_id);

    return;
  }
  
  //-------------------------------------------------------------------------
  //          CflowdRawFlow(ipv4addr_t ciscoIp,                    
  //                        const CiscoFlowHeaderV6_t * flowHeader, 
  //                        const CiscoFlowEntryV6_t * flowEntry)
  //.........................................................................
  //  Constructs from an IP address (of the Cisco sending the flow
  //  data) and a pointer to a v6 flow header and a v6 flow entry.
  //-------------------------------------------------------------------------
  CflowdRawFlow(ipv4addr_t ciscoIp,
                const CiscoFlowHeaderV6_t * flowHeader,
                const CiscoFlowEntryV6_t * flowEntry);

  //-------------------------------------------------------------------------
  //           void Init(ipv4addr_t ciscoIp,                        
  //                     const CiscoFlowHeaderV6_t * flowHeader,    
  //                     const CiscoFlowEntryV6_t * flowEntry)
  //.........................................................................
  //  
  //-------------------------------------------------------------------------
  void Init(ipv4addr_t ciscoIp,
            const CiscoFlowHeaderV6_t * flowHeader,
            const CiscoFlowEntryV6_t * flowEntry);
  
  //--------------------------------------------------------------------------
  //        CflowdRawFlow(ipv4addr_t ciscoIp,                    
  //                      const CiscoFlowHeaderV8_t * flowHeader, 
  //                      const CiscoFlowEntryV8AsAggV2_t * flowEntry)
  //..........................................................................
  //  Constructs from an IP address (of the Cisco sending the flow data),
  //  a v8 flow header and a v8 AS aggregation flow entry.
  //--------------------------------------------------------------------------
  inline CflowdRawFlow(ipv4addr_t ciscoIp,
                       const CiscoFlowHeaderV8_t * flowHeader,
                       const CiscoFlowEntryV8AsAggV2_t * flowEntry)
  {
    this->Init(ciscoIp,flowHeader,flowEntry);
  }

  //--------------------------------------------------------------------------
  //        void Init(ipv4addr_t ciscoIp,                      
  //                  const CiscoFlowHeaderV8_t * flowHeader,  
  //                  const CiscoFlowEntryV8AsAggV2_t * flowEntry)
  //..........................................................................
  //  
  //--------------------------------------------------------------------------
  void Init(ipv4addr_t ciscoIp,
            const CiscoFlowHeaderV8_t * flowHeader,
            const CiscoFlowEntryV8AsAggV2_t * flowEntry);

  //--------------------------------------------------------------------------
  //   CflowdRawFlow(ipv4addr_t ciscoIp,                    
  //                 const CiscoFlowHeaderV8_t * flowHeader, 
  //                 const CiscoFlowEntryV8ProtocolPortAggV2_t * flowEntry)
  //..........................................................................
  //  Constructs from an IP address (of the Cisco sending the flow data),
  //  a v8 flow header and a v8 protocol/port aggregation flow entry.
  //--------------------------------------------------------------------------
  inline CflowdRawFlow(ipv4addr_t ciscoIp,
                       const CiscoFlowHeaderV8_t * flowHeader,
                       const CiscoFlowEntryV8ProtocolPortAggV2_t * flowEntry)
  {
    this->Init(ciscoIp,flowHeader,flowEntry);
  }
  
  //--------------------------------------------------------------------------
  //     void Init(ipv4addr_t ciscoIp,                      
  //               const CiscoFlowHeaderV8_t * flowHeader,  
  //               const CiscoFlowEntryV8ProtocolPortAggV2_t * flowEntry)
  //..........................................................................
  //  
  //--------------------------------------------------------------------------
  void Init(ipv4addr_t ciscoIp,
            const CiscoFlowHeaderV8_t * flowHeader,
            const CiscoFlowEntryV8ProtocolPortAggV2_t * flowEntry);

  //--------------------------------------------------------------------------
  //    inline CflowdRawFlow(ipv4addr_t ciscoIp,                    
  //                  const CiscoFlowHeaderV8_t * flowHeader, 
  //                  const CiscoFlowEntryV8NetMatrixAggV2_t * flowEntry)
  //..........................................................................
  //  Constructs from an IP address (of the Cisco sending the flow data),
  //  a v8 flow header and a v8 net matrix aggregation flow entry.
  //--------------------------------------------------------------------------
  inline CflowdRawFlow(ipv4addr_t ciscoIp,
                       const CiscoFlowHeaderV8_t * flowHeader,
                       const CiscoFlowEntryV8NetMatrixAggV2_t * flowEntry)
  {
    this->Init(ciscoIp,flowHeader,flowEntry);
  }

  //--------------------------------------------------------------------------
  //     void Init(ipv4addr_t ciscoIp,                      
  //               const CiscoFlowHeaderV8_t * flowHeader,  
  //               const CiscoFlowEntryV8NetMatrixAggV2_t * flowEntry)
  //..........................................................................
  //  
  //--------------------------------------------------------------------------
  void Init(ipv4addr_t ciscoIp,
            const CiscoFlowHeaderV8_t * flowHeader,
            const CiscoFlowEntryV8NetMatrixAggV2_t * flowEntry);
  

  //-------------------------------------------------------------------------
  //                           inline void Clear()                           
  //.........................................................................
  //  
  //-------------------------------------------------------------------------
  inline void Clear()
  {
    this->data._index = 0;
    this->data._isHostOrder = true;
    return;
  }
  
  //-------------------------------------------------------------------------
  //                      inline index_type Index() const                      
  //.........................................................................
  //  Returns the bitfield index.  The index has a bit set for each
  //  of the pieces of data contained in the flow.  If the bit is set,
  //  the corresponding data is present.
  //-------------------------------------------------------------------------
  inline index_type Index() const
  {
    return(this->data._index);
  }

  //-------------------------------------------------------------------------
  //                      inline uint32_t Length() const                     
  //.........................................................................
  //  
  //-------------------------------------------------------------------------
  inline uint32_t Length() const
  {
    uint32_t  length;

    length = sizeof(this->data._index);

    if (this->data._index & CflowdRawFlow::k_routerMask)
      length += sizeof(this->data._router);
    if (this->data._index & CflowdRawFlow::k_srcIpAddrMask)
      length += sizeof(this->data._srcIpAddr);
    if (this->data._index & CflowdRawFlow::k_dstIpAddrMask)
      length += sizeof(this->data._dstIpAddr);
    if (this->data._index & CflowdRawFlow::k_inputIfIndexMask)
      length += sizeof(this->data._inputIfIndex);
    if (this->data._index & CflowdRawFlow::k_outputIfIndexMask)
      length += sizeof(this->data._outputIfIndex);
    if (this->data._index & CflowdRawFlow::k_srcPortMask)
      length += sizeof(this->data._srcPort);
    if (this->data._index & CflowdRawFlow::k_dstPortMask)
      length += sizeof(this->data._dstPort);
    if (this->data._index & CflowdRawFlow::k_pktsMask)
      length += sizeof(this->data._pkts);
    if (this->data._index & CflowdRawFlow::k_bytesMask)
      length += sizeof(this->data._bytes);
    if (this->data._index & CflowdRawFlow::k_ipNextHopMask)
      length += sizeof(this->data._ipNextHop);
    if (this->data._index & CflowdRawFlow::k_startTimeMask)
      length += sizeof(this->data._startTime);
    if (this->data._index & CflowdRawFlow::k_endTimeMask)
      length += sizeof(this->data._endTime);
    if (this->data._index & CflowdRawFlow::k_protocolMask)
      length += sizeof(this->data._protocol);
    if (this->data._index & CflowdRawFlow::k_tosMask)
      length += sizeof(this->data._tos);
    if (this->data._index & CflowdRawFlow::k_srcAsMask)
      length += sizeof(this->data._srcAs);
    if (this->data._index & CflowdRawFlow::k_dstAsMask)
      length += sizeof(this->data._dstAs);
    if (this->data._index & CflowdRawFlow::k_srcMaskLenMask)
      length += sizeof(this->data._srcMaskLen);
    if (this->data._index & CflowdRawFlow::k_dstMaskLenMask)
      length += sizeof(this->data._dstMaskLen);
    if (this->data._index & CflowdRawFlow::k_tcpFlagsMask)
      length += sizeof(this->data._tcpFlags);
    if (this->data._index & CflowdRawFlow::k_inputEncapMask)
      length += sizeof(this->data._inputEncap);
    if (this->data._index & CflowdRawFlow::k_outputEncapMask)
      length += sizeof(this->data._outputEncap);
    if (this->data._index & CflowdRawFlow::k_peerNextHopMask)
      length += sizeof(this->data._peerNextHop);
    if (this->data._index & CflowdRawFlow::k_engineTypeMask)
      length += sizeof(this->data._engineType);
    if (this->data._index & CflowdRawFlow::k_engineIdMask)
      length += sizeof(this->data._engineId);

    return(length);
  }
  
  //-------------------------------------------------------------------------
  //                     inline ipv4addr_t Router() const                    
  //.........................................................................
  //  Returns the IP address of the router that sent the flow data.
  //-------------------------------------------------------------------------
  inline ipv4addr_t Router() const
  {
    return(this->data._router);
  }

  //-------------------------------------------------------------------------
  //               inline ipv4addr_t Router(ipv4addr_t router)               
  //.........................................................................
  //  Sets and returns the IP address of the router that sent the flow
  //  data.
  //-------------------------------------------------------------------------
  inline ipv4addr_t Router(ipv4addr_t router)
  {
    this->data._router = router;
    this->data._index |= k_routerMask;
    return(this->data._router);
  }
  
  //-------------------------------------------------------------------------
  //                   inline ipv4addr_t SrcIpAddr() const                   
  //.........................................................................
  //  Returns the IP address of the host that sourced the flow.
  //-------------------------------------------------------------------------
  inline ipv4addr_t SrcIpAddr() const
  {
    return(this->data._srcIpAddr);
  }

  //-------------------------------------------------------------------------
  //            inline ipv4addr_t SrcIpAddr(ipv4addr_t srcIpAddr)            
  //.........................................................................
  //  Sets and returns the IP address of the host that sourced the flow.
  //-------------------------------------------------------------------------
  inline ipv4addr_t SrcIpAddr(ipv4addr_t srcIpAddr)
  {
    this->data._srcIpAddr = srcIpAddr;
    this->data._index |= k_srcIpAddrMask;
    return(this->data._srcIpAddr);
  }
  
  //-------------------------------------------------------------------------
  //                   inline ipv4addr_t DstIpAddr() const                   
  //.........................................................................
  //  Returns the IP address of the host that sinked the flow.
  //-------------------------------------------------------------------------
  inline ipv4addr_t DstIpAddr() const
  {
    return(this->data._dstIpAddr);
  }

  //-------------------------------------------------------------------------
  //            inline ipv4addr_t DstIpAddr(ipv4addr_t dstIpAddr)            
  //.........................................................................
  //  Sets and returns the IP address of the host that sinked the flow.
  //-------------------------------------------------------------------------
  inline ipv4addr_t DstIpAddr(ipv4addr_t dstIpAddr)
  {
    this->data._dstIpAddr = dstIpAddr;
    this->data._index |= k_dstIpAddrMask;
    return(this->data._dstIpAddr);
  }
  
  //-------------------------------------------------------------------------
  //                   inline uint16_t InputIfIndex() const                  
  //.........................................................................
  //  Returns the ifIndex of the interface which received the flow.
  //-------------------------------------------------------------------------
  inline uint16_t InputIfIndex() const
  {
    return(this->data._inputIfIndex);
  }

  //-------------------------------------------------------------------------
  //           inline uint16_t InputIfIndex(uint16_t inputIfIndex)           
  //.........................................................................
  //  Sets and returns the ifIndex of the interface which received the
  //  flow.
  //-------------------------------------------------------------------------
  inline uint16_t InputIfIndex(uint16_t inputIfIndex)
  {
    this->data._inputIfIndex = inputIfIndex;
    this->data._index |= k_inputIfIndexMask;
    return(this->data._inputIfIndex);
  }

  //-------------------------------------------------------------------------
  //                  inline uint16_t OutputIfIndex() const                  
  //.........................................................................
  //  Returns the ifIndex of the interface which transmitted the flow
  //  to the destination.
  //-------------------------------------------------------------------------
  inline uint16_t OutputIfIndex() const
  {
    return(this->data._outputIfIndex);
  }

  //-------------------------------------------------------------------------
  //          inline uint16_t OutputIfIndex(uint16_t outputIfIndex)          
  //.........................................................................
  //  Sets and returns the ifIndex of the interface which transmitted the
  //  flow to the destination.
  //-------------------------------------------------------------------------
  inline uint16_t OutputIfIndex(uint16_t outputIfIndex)
  {
    this->data._outputIfIndex = outputIfIndex;
    this->data._index |= k_outputIfIndexMask;
    return(this->data._outputIfIndex);
  }
  
  //-------------------------------------------------------------------------
  //                     inline uint16_t SrcPort() const                     
  //.........................................................................
  //  Returns the transport source port number.
  //-------------------------------------------------------------------------
  inline uint16_t SrcPort() const
  {
    return(this->data._srcPort);
  }

  //-------------------------------------------------------------------------
  //                inline uint16_t SrcPort(uint16_t srcPort)                
  //.........................................................................
  //  Sets and returns the transport source port number.
  //-------------------------------------------------------------------------
  inline uint16_t SrcPort(uint16_t srcPort)
  {
    this->data._srcPort = srcPort;
    this->data._index |= k_srcPortMask;
    return(this->data._srcPort);
  }
  
  //-------------------------------------------------------------------------
  //                     inline uint16_t DstPort() const                     
  //.........................................................................
  //  Returns the transport destination port number.
  //-------------------------------------------------------------------------
  inline uint16_t DstPort() const
  {
    return(this->data._dstPort);
  }

  //-------------------------------------------------------------------------
  //                inline uint16_t DstPort(uint16_t dstPort)                
  //.........................................................................
  //  Sets and returns the transport destination port number.
  //-------------------------------------------------------------------------
  inline uint16_t DstPort(uint16_t dstPort)
  {
    this->data._dstPort = dstPort;
    this->data._index |= k_dstPortMask;
    return(this->data._dstPort);
  }
  
  //-------------------------------------------------------------------------
  //                       inline uint32_t Pkts() const                      
  //.........................................................................
  //  Returns the number of packets in the flow.
  //-------------------------------------------------------------------------
  inline uint32_t Pkts() const
  {
    return(this->data._pkts);
  }

  //-------------------------------------------------------------------------
  //                   inline uint32_t Pkts(uint32_t pkts)                   
  //.........................................................................
  //  Sets and returns the number of packets in the flow.
  //-------------------------------------------------------------------------
  inline uint32_t Pkts(uint32_t pkts)
  {
    this->data._pkts = pkts;
    this->data._index |= k_pktsMask;
    return(this->data._pkts);
  }
  
  //-------------------------------------------------------------------------
  //                      inline uint32_t Bytes() const                      
  //.........................................................................
  //  Returns the number of bytes in the flow.
  //-------------------------------------------------------------------------
  inline uint32_t Bytes() const
  {
    return(this->data._bytes);
  }

  //-------------------------------------------------------------------------
  //                  inline uint32_t Bytes(uint32_t bytes)                  
  //.........................................................................
  //  Sets and returns the number of bytes in the flow.
  //-------------------------------------------------------------------------
  inline uint32_t Bytes(uint32_t bytes)
  {
    this->data._bytes = bytes;
    this->data._index |= k_bytesMask;
    return(this->data._bytes);
  }
  
  //-------------------------------------------------------------------------
  //                   inline ipv4addr_t IpNextHop() const                   
  //.........................................................................
  //  Returns the IP next hop for the flow.  This is relative to the
  //  router.
  //-------------------------------------------------------------------------
  inline ipv4addr_t IpNextHop() const
  {
    return(this->data._ipNextHop);
  }

  //-------------------------------------------------------------------------
  //            inline ipv4addr_t IpNextHop(ipv4addr_t ipNextHop)            
  //.........................................................................
  //  Sets and returns the IP next hop for the flow.  This is relative to
  //  the router.
  //-------------------------------------------------------------------------
  inline ipv4addr_t IpNextHop(ipv4addr_t ipNextHop)
  {
    this->data._ipNextHop = ipNextHop;
    this->data._index |= k_ipNextHopMask;
    return(this->data._ipNextHop);
  }
  
  //-------------------------------------------------------------------------
  //                    inline uint32_t StartTime() const                    
  //.........................................................................
  //  Returns the start time of the flow.
  //-------------------------------------------------------------------------
  inline uint32_t StartTime() const
  {
    return(this->data._startTime);
  }

  //-------------------------------------------------------------------------
  //              inline uint32_t StartTime(uint32_t startTime)              
  //.........................................................................
  //  Sets and returns the start time of the flow.
  //-------------------------------------------------------------------------
  inline uint32_t StartTime(uint32_t startTime)
  {
    this->data._startTime = startTime;
    this->data._index |= k_startTimeMask;
    return(this->data._startTime);
  }
  
  //-------------------------------------------------------------------------
  //                     inline uint32_t EndTime() const                     
  //.........................................................................
  //  Returns the end time of the flow.
  //-------------------------------------------------------------------------
  inline uint32_t EndTime() const
  {
    return(this->data._endTime);
  }

  //-------------------------------------------------------------------------
  //                inline uint32_t EndTime(uint32_t endTime)                
  //.........................................................................
  //  Sets and returns the end time of the flow.
  //-------------------------------------------------------------------------
  inline uint32_t EndTime(uint32_t endTime)
  {
    this->data._endTime = endTime;
    this->data._index |= k_endTimeMask;
    return(this->data._endTime);
  }
  
  //-------------------------------------------------------------------------
  //                     inline uint8_t Protocol() const                     
  //.........................................................................
  //  Returns the IP protocol (ICMP, UDP, TCP, etc.) of the flow.
  //-------------------------------------------------------------------------
  inline uint8_t Protocol() const
  {
    return(this->data._protocol);
  }

  //-------------------------------------------------------------------------
  //                inline uint8_t Protocol(uint8_t protocol)                
  //.........................................................................
  //  Sets and returns the IP protocol (ICMP, UDP, TCP, etc.) of the flow.
  //-------------------------------------------------------------------------
  inline uint8_t Protocol(uint8_t protocol)
  {
    this->data._protocol = protocol;
    this->data._index |= k_protocolMask;
    return(this->data._protocol);
  }
    
  //-------------------------------------------------------------------------
  //                        inline uint8_t Tos() const                       
  //.........................................................................
  //  Returns the TOS (type of service) of the flow.
  //-------------------------------------------------------------------------
  inline uint8_t Tos() const
  {
    return(this->data._tos);
  }

  //-------------------------------------------------------------------------
  //                     inline uint8_t Tos(uint8_t tos)                     
  //.........................................................................
  //  Sets and returns the TOS (type of service) of the flow.
  //-------------------------------------------------------------------------
  inline uint8_t Tos(uint8_t tos)
  {
    this->data._tos = tos;
    this->data._index |= k_tosMask;
    return(this->data._tos);
  }
  
  //-------------------------------------------------------------------------
  //                      inline uint16_t SrcAs() const                      
  //.........................................................................
  //  Returns the source AS of the flow.
  //-------------------------------------------------------------------------
  inline uint16_t SrcAs() const
  {
    return(this->data._srcAs);
  }

  //-------------------------------------------------------------------------
  //                  inline uint16_t SrcAs(uint16_t srcAs)                  
  //.........................................................................
  //  Sets and returns the source AS of the flow.
  //-------------------------------------------------------------------------
  inline uint16_t SrcAs(uint16_t srcAs)
  {
    this->data._srcAs = srcAs;
    this->data._index |= k_srcAsMask;
    return(this->data._srcAs);
  }
  
  //-------------------------------------------------------------------------
  //                      inline uint16_t DstAs() const                      
  //.........................................................................
  //  Returns the destination AS of the flow.
  //-------------------------------------------------------------------------
  inline uint16_t DstAs() const
  {
    return(this->data._dstAs);
  }

  //-------------------------------------------------------------------------
  //                  inline uint16_t DstAs(uint16_t dstAs)                  
  //.........................................................................
  //  Sets and returns the destination AS of the flow.
  //-------------------------------------------------------------------------
  inline uint16_t DstAs(uint16_t dstAs)
  {
    this->data._dstAs = dstAs;
    this->data._index |= k_dstAsMask;
    return(this->data._dstAs);
  }
  
  //-------------------------------------------------------------------------
  //                    inline uint8_t SrcMaskLen() const                    
  //.........................................................................
  //  Returns the source network mask length of the flow.
  //-------------------------------------------------------------------------
  inline uint8_t SrcMaskLen() const
  {
    return(this->data._srcMaskLen);
  }

  //-------------------------------------------------------------------------
  //              inline uint8_t SrcMaskLen(uint8_t srcMaskLen)              
  //.........................................................................
  //  Sets and returns the source network mask length of the flow.
  //-------------------------------------------------------------------------
  inline uint8_t SrcMaskLen(uint8_t srcMaskLen)
  {
    this->data._srcMaskLen = srcMaskLen;
    this->data._index |= k_srcMaskLenMask;
    return(this->data._srcMaskLen);
  }
  
  //-------------------------------------------------------------------------
  //                    inline uint8_t DstMaskLen() const                    
  //.........................................................................
  //  Returns the destination network mask length of the flow.
  //-------------------------------------------------------------------------
  inline uint8_t DstMaskLen() const
  {
    return(this->data._dstMaskLen);
  }

  //-------------------------------------------------------------------------
  //              inline uint8_t DstMaskLen(uint8_t dstMaskLen)              
  //.........................................................................
  //  Sets and returns the destination network mask length of the flow.
  //-------------------------------------------------------------------------
  inline uint8_t DstMaskLen(uint8_t dstMaskLen)
  {
    this->data._dstMaskLen = dstMaskLen;
    this->data._index |= k_dstMaskLenMask;
    return(this->data._dstMaskLen);
  }
  
  //-------------------------------------------------------------------------
  //                     inline uint8_t TcpFlags() const                     
  //.........................................................................
  //  Returns the bitwise OR of all the TCP flags seen in the flow.
  //  Not applicable if the flow was not a TCP flow.
  //-------------------------------------------------------------------------
  inline uint8_t TcpFlags() const
  {
    return(this->data._tcpFlags);
  }

  //-------------------------------------------------------------------------
  //                inline uint8_t TcpFlags(uint8_t tcpFlags)                
  //.........................................................................
  //  Sets and returns the bitwise OR of all the TCP flags seen in the
  //  flow.  Not applicable if the flow was not a TCP flow.
  //-------------------------------------------------------------------------
  inline uint8_t TcpFlags(uint8_t tcpFlags)
  {
    this->data._tcpFlags = tcpFlags;
    this->data._index |= k_tcpFlagsMask;
    return(this->data._tcpFlags);
  }
  
  //-------------------------------------------------------------------------
  //                    inline uint8_t InputEncap() const                    
  //.........................................................................
  //  Returns the input encapsulation of the flow.
  //-------------------------------------------------------------------------
  inline uint8_t InputEncap() const
  {
    return(this->data._inputEncap);
  }

  //-------------------------------------------------------------------------
  //              inline uint8_t InputEncap(uint8_t inputEncap)              
  //.........................................................................
  //  Sets and returns the input encapsulation of the flow.
  //-------------------------------------------------------------------------
  inline uint8_t InputEncap(uint8_t inputEncap)
  {
    this->data._inputEncap = inputEncap;
    this->data._index |= k_inputEncapMask;
    return(this->data._inputEncap);
  }
  
  //-------------------------------------------------------------------------
  //                    inline uint8_t OutputEncap() const                   
  //.........................................................................
  //  Returns the output encapsulation of the flow.
  //-------------------------------------------------------------------------
  inline uint8_t OutputEncap() const
  {
    return(this->data._outputEncap);
  }

  //-------------------------------------------------------------------------
  //             inline uint8_t OutputEncap(uint8_t outputEncap)             
  //.........................................................................
  //  Sets and returns the output encapsulation of the flow.
  //-------------------------------------------------------------------------
  inline uint8_t OutputEncap(uint8_t outputEncap)
  {
    this->data._outputEncap = outputEncap;
    this->data._index |= k_outputEncapMask;
    return(this->data._outputEncap);
  }
  
  //-------------------------------------------------------------------------
  //                  inline ipv4addr_t PeerNextHop() const                  
  //.........................................................................
  //  Returns the peer next hop (border router) IP address of the flow.
  //-------------------------------------------------------------------------
  inline ipv4addr_t PeerNextHop() const
  {
    return(this->data._peerNextHop);
  }

  //-------------------------------------------------------------------------
  //          inline ipv4addr_t PeerNextHop(ipv4addr_t peerNextHop)          
  //.........................................................................
  //  Sets and returns the peer next hop (border router) IP address of
  //  the flow.
  //-------------------------------------------------------------------------
  inline ipv4addr_t PeerNextHop(ipv4addr_t peerNextHop)
  {
    this->data._peerNextHop = peerNextHop;
    this->data._index |= k_peerNextHopMask;
    return(this->data._peerNextHop);
  }

  //-------------------------------------------------------------------------
  //                    inline uint8_t EngineType() const                    
  //.........................................................................
  //  Returns the engine type for the flow.  If set to 0, the flow did
  //  not come from a VIP adapter.  If set to 1, the flow came from a
  //  IP adapter.
  //-------------------------------------------------------------------------
  inline uint8_t EngineType() const
  {
    return(this->data._engineType);
  }

  //-------------------------------------------------------------------------
  //              inline uint8_t EngineType(uint8_t engineType)              
  //.........................................................................
  //  Sets and returns the engine type for the flow.  If set to 0, the
  //  flow did not come from a VIP adapter.  If set to 1, the flow came
  //  from a VIP adapter.
  //-------------------------------------------------------------------------
  inline uint8_t EngineType(uint8_t engineType)
  {
    this->data._engineType = engineType;
    this->data._index |= k_engineTypeMask;
    return(this->data._engineType);
  }

  //-------------------------------------------------------------------------
  //                     inline uint8_t EngineId() const                     
  //.........................................................................
  //  Returns the flow-export engine identifier.  This is really only
  //  applicable in the presence of VIP adapters, which are capable of
  //  sending their own flows.
  //-------------------------------------------------------------------------
  inline uint8_t EngineId() const
  {
    return(this->data._engineId);
  }
  
  //-------------------------------------------------------------------------
  //                inline uint8_t EngineId(uint8_t engineId)                
  //.........................................................................
  //  Sets and returns the flow-export engine identifier.  This is
  //  really only applicable in the presence of VIP adapters, which are
  //  capable of sending their own flows.
  //-------------------------------------------------------------------------
  inline uint8_t EngineId(uint8_t engineId)
  {
    this->data._engineId = engineId;
    this->data._index |= k_engineIdMask;
    return(this->data._engineId);
  }

  //-------------------------------------------------------------------------
  //                      inline uint8_t Version() const                     
  //.........................................................................
  //  Returns the flow-export version for the flow.
  //-------------------------------------------------------------------------
  inline uint8_t Version() const
  {
    return(this->data._version);
  }

  //-------------------------------------------------------------------------
  //                 inline uint8_t Version(uint8_t version)                 
  //.........................................................................
  //  Sets and returns the flow-export version for the flow.
  //-------------------------------------------------------------------------
  inline uint8_t Version(uint8_t version)
  {
    this->data._version = version;
    return(this->data._version);
  }
  
  //-------------------------------------------------------------------------
  //                      istream & Read(istream & is)
  //.........................................................................
  //  Reads a flow from an istream.  Returns the istream.
  //-------------------------------------------------------------------------
  istream & Read(istream & is);

  //-------------------------------------------------------------------------
  //                            int Read(int fd)
  //.........................................................................
  //  Reads a flow from a file descriptor.  Probably more I/O efficient
  //  than Read(istream & is), since we use scatter reads.  Also your
  //  only reasonable option for reading from a socket.  Returns the
  //  number of bytes read on success, -1 on failure.
  //-------------------------------------------------------------------------
  int Read(int fd);

  //-------------------------------------------------------------------------
  //                      ostream & Write(ostream & os) const
  //.........................................................................
  //  Writes a flow to an ostream.  Returns the ostream.
  //-------------------------------------------------------------------------
  ostream & Write(ostream & os) const;

  //-------------------------------------------------------------------------
  //                            int Write(int fd) const
  //.........................................................................
  //  Writes a flow to a file descriptor.  Probably more I/O efficient
  //  than Write(ostream & os), since we use gather writes.  Also your
  //  only reasonable option for writing to a socket.  Returns the
  //  number of bytes written on success, -1 on failure.
  //-------------------------------------------------------------------------
  int Write(int fd) const;

  //-------------------------------------------------------------------------
  //                     int Write(caddr_t & memoryAddr) const
  //.........................................................................
  //  Writes a flow to a raw memory address.  Doesn't check the target
  //  address, so be careful with this member function!  Returns the
  //  bytes written on success, -1 on failure.
  //-------------------------------------------------------------------------
  int Write(caddr_t & memoryAddr) const;
  
  //-------------------------------------------------------------------------
  //        friend ostream& operator << (ostream& os,                
  //                                     const CflowdRawFlow & flow)
  //.........................................................................
  //  Overloaded ostream '<<' operator to dump the contents of a
  //  CflowdRawFlow object to an ostream in a human-readable form.
  //  Returns the ostream.
  //-------------------------------------------------------------------------
  friend ostream& operator << (ostream& os,
                               const CflowdRawFlow & flow);

  //-------------------------------------------------------------------------
  //                  inline const _data_t *DataPtr() const                  
  //.........................................................................
  //  
  //-------------------------------------------------------------------------
  inline const _data_t *DataPtr() const
  {
    return(&(this->data));
  }

  //-------------------------------------------------------------------------
  //   inline CflowdRawFlow & operator = (const CflowdRawFlow & rawFlow)
  //.........................................................................
  //  
  //-------------------------------------------------------------------------
  inline CflowdRawFlow & operator = (const CflowdRawFlow & rawFlow)
  {
    memcpy(&(this->data),rawFlow.DataPtr(),sizeof(_data_t));
    return(*this);
  }
  
private:
  //-------------------------------------------------------------------------
  //                         void ToHostByteOrder()
  //.........................................................................
  //  Converts all fields to host byte order.
  //-------------------------------------------------------------------------
  void ToHostByteOrder();
  
  //-------------------------------------------------------------------------
  //                        void ToNetworkByteOrder()
  //.........................................................................
  //  Converts all fields to network byte order.
  //-------------------------------------------------------------------------
  void ToNetworkByteOrder();

  mutable _data_t  data;
  
};

#endif  // _CFLOWDRAWFLOW_HH_
