//===========================================================================
//  @(#) $Name: cflowd-2-1-b1 $
//  @(#) $Id: CflowdRawFlowLogger.hh,v 1.3 1999/08/11 16:11:35 dwm Exp $
//===========================================================================
//  CAIDA Copyright Notice
//
//  By accessing this software, cflowd++, you are duly informed
//  of and agree to be bound by the conditions described below in this
//  notice:
//
//  This software product, cflowd++, is developed by Daniel W. McRobb, and
//  copyrighted(C) 1998 by the University of California, San Diego
//  (UCSD), with all rights reserved.  UCSD administers the CAIDA grant,
//  NCR-9711092, under which part of this code was developed.
//
//  There is no charge for cflowd++ software. You can redistribute it
//  and/or modify it under the terms of the GNU General Public License,
//  v.  2 dated June 1991 which is incorporated by reference herein.
//  cflowd++ is distributed WITHOUT ANY WARRANTY, IMPLIED OR EXPRESS, OF
//  MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE or that the use
//  of it will not infringe on any third party's intellectual property
//  rights.
//
//  You should have received a copy of the GNU GPL along with cflowd++.
//  Copies can also be obtained from:
//
//    http://www.gnu.org/copyleft/gpl.html
//
//  or by writing to:
//
//    University of California, San Diego
//
//    SDSC/CAIDA
//    9500 Gilman Dr., MS-0505
//    La Jolla, CA 92093 - 0505  USA
//
//  Or contact:
//
//    info@caida.org
//===========================================================================

#ifndef _CFLOWDRAWFLOWLOGGER_HH_
#define _CFLOWDRAWFLOWLOGGER_HH_

//---------------------------------------------------------------------------
//  class CflowdRawFlowLogger
//---------------------------------------------------------------------------
//  This class encapsulates raw flow logging.  It provides a simple
//  interface for log creation/configuration and the actual logging
//  (via the AddFlow() member).
//
//  Note that underneath the hood we're using mmap() for the open log
//  file.  Hence the meaning of this->MapAddr() is overloaded; if it's
//  set to (caddr_t)(-1), the log is not open.
//---------------------------------------------------------------------------
class CflowdRawFlowLogger
{
public:
  //-------------------------------------------------------------------------
  //                          CflowdRawFlowLogger()
  //.........................................................................
  //  constructor
  //-------------------------------------------------------------------------
  CflowdRawFlowLogger();

  //-------------------------------------------------------------------------
  //            CflowdRawFlowLogger(const string & flowDirectory,            
  //                                const string & flowFilePrefix,           
  //                                int numLogs, int logSize)
  //.........................................................................
  //  constructor
  //  This is the constructor we normally use.  This will automatically
  //  set the flow directory, lfow file prefix, number of log files and
  //  log file size.  It will also open the logger; if opening the
  //  logger fails, this->MapAddr() will return (caddr_t)(-1).
  //-------------------------------------------------------------------------
  CflowdRawFlowLogger(const string & flowDirectory,
                      const string & flowFilePrefix,
                      int numLogs, int logSize);
  
  //-------------------------------------------------------------------------
  //                         ~CflowdRawFlowLogger()
  //.........................................................................
  //  destructor
  //-------------------------------------------------------------------------
  ~CflowdRawFlowLogger();

  //-------------------------------------------------------------------------
  //               inline const string & FlowDirectory() const               
  //.........................................................................
  //  Returns the name of the directory in which log files are to be
  //  stored.
  //-------------------------------------------------------------------------
  inline const string & FlowDirectory() const
  {
    return(this->_flowDirectory);
  }

  //-------------------------------------------------------------------------
  //       const string & FlowDirectory(const string & flowDirectory)
  //.........................................................................
  //  Sets and returns the name of the directory in which to store
  //  log files.
  //-------------------------------------------------------------------------
  const string & FlowDirectory(const string & flowDirectory);
  
  //-------------------------------------------------------------------------
  //               inline const string & FlowFilePrefix() const              
  //.........................................................................
  //  Returns the flow file prefix being used to name log files.
  //-------------------------------------------------------------------------
  inline const string & FlowFilePrefix() const
  {
    return(this->_flowFilePrefix);
  }
  
  //-------------------------------------------------------------------------
  //      const string & FlowFilePrefix(const string & flowFilePrefix)
  //.........................................................................
  //  Sets and returns the flow file prefix to be used to name log
  //  files.
  //-------------------------------------------------------------------------
  const string & FlowFilePrefix(const string & flowFilePrefix);
  
  //-------------------------------------------------------------------------
  //                       inline int LogSize() const
  //.........................................................................
  //  Returns the size (in bytes) to use for each log file.  If the
  //  log size has not been set, returns -1.
  //-------------------------------------------------------------------------
  inline int LogSize() const
  {
    return(this->_logSize);
  }
  
  //-------------------------------------------------------------------------
  //                        int LogSize(int logSize)
  //.........................................................................
  //  Sets and returns the size to use for each log file.
  //-------------------------------------------------------------------------
  int LogSize(int logSize);
    
  //-------------------------------------------------------------------------
  //                       inline int NumLogs() const
  //.........................................................................
  //  Returns the number of log files being used.  If the number of
  //  log files has not been set, returns -1.
  //-------------------------------------------------------------------------
  inline int NumLogs() const
  {
    return(this->_numLogs);
  }
  
  //-------------------------------------------------------------------------
  //                        int NumLogs(int numLogs)
  //.........................................................................
  //  Sets and returns the number of log files to use.
  //-------------------------------------------------------------------------
  int NumLogs(int numLogs);
  
  //-------------------------------------------------------------------------
  //                               int Open()
  //.........................................................................
  //  Opens the logger.  Returns 0 on success, -1 on failure.
  //-------------------------------------------------------------------------
  int Open();
  
  //-------------------------------------------------------------------------
  //                               int Close()
  //.........................................................................
  //  Closes the logger.  Returns 0 on success, -1 on failure.  Note
  //  that if this function fails, neither _mapAddr or _writePtr are
  //  reset.
  //-------------------------------------------------------------------------
  int Close();

  //-------------------------------------------------------------------------
  //                               int Roll()
  //.........................................................................
  //  Rolls the logger's log files.  Returns the result of this->Open()
  //  that is used to open the log after rolling.
  //-------------------------------------------------------------------------
  int Roll();
  
  //-------------------------------------------------------------------------
  //              int AddFlow(const CflowdRawFlow & rawFlow)
  //.........................................................................
  //  Adds a CflowdRawFlow to the log.  Returns the result of the
  //  underlying rawFlow.Write(caddr_t &) call (number of bytes written
  //  to log).
  //-------------------------------------------------------------------------
  int AddFlow(const CflowdRawFlow & rawFlow);

  //-------------------------------------------------------------------------
  //                 inline const caddr_t MapAddr() const
  //.........................................................................
  //  Returns the address of the logger's memory map.  This is normally
  //  only used to look for the special value (caddr_t)(-1), which
  //  indicates that the log is not open.
  //-------------------------------------------------------------------------
  inline const caddr_t MapAddr() const
  {
    return(this->_mapAddr);
  }
  
private:
  string           _flowDirectory;
  string           _flowFilePrefix;
  int              _logSize;
  int              _numLogs;
  caddr_t          _mapAddr;
  caddr_t          _writePtr;
};

#endif  // _CFLOWDRAWFLOWLOGGER_HH_
