//===========================================================================
//  $Name: cflowd-2-1-b1 $
//  $Id: CflowdUint64TrafficCounter.hh,v 1.2 1999/02/07 18:43:03 dwm Exp $
//===========================================================================
//  CAIDA Copyright Notice
//
//  By accessing this software, cflowd++, you are duly informed
//  of and agree to be bound by the conditions described below in this
//  notice:
//
//  This software product, cflowd++, is developed by Daniel W. McRobb, and
//  copyrighted(C) 1998 by the University of California, San Diego
//  (UCSD), with all rights reserved.  UCSD administers the CAIDA grant,
//  NCR-9711092, under which part of this code was developed.
//
//  There is no charge for cflowd++ software. You can redistribute it
//  and/or modify it under the terms of the GNU General Public License,
//  v.  2 dated June 1991 which is incorporated by reference herein.
//  cflowd++ is distributed WITHOUT ANY WARRANTY, IMPLIED OR EXPRESS, OF
//  MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE or that the use
//  of it will not infringe on any third party's intellectual property
//  rights.
//
//  You should have received a copy of the GNU GPL along with cflowd++.
//  Copies can also be obtained from:
//
//    http://www.gnu.org/copyleft/gpl.html
//
//  or by writing to:
//
//    University of California, San Diego
//
//    SDSC/CAIDA
//    9500 Gilman Dr., MS-0505
//    La Jolla, CA 92093 - 0505  USA
//
//  Or contact:
//
//    info@caida.org
//===========================================================================

#ifndef _CFLOWDUINT64TRAFFICCOUNTER_HH_
#define _CFLOWDUINT64TRAFFICCOUNTER_HH_

extern "C" {
#include <sys/types.h>
#include <syslog.h>
  
#include "caida_t.h"
}

#include "ArtsPrimitive.hh"

extern ArtsPrimitive g_CfdArtsPrimitive;

//---------------------------------------------------------------------------
//  class CflowdUint64TrafficCounter
//---------------------------------------------------------------------------
//  This class is used to hold traffic (packet and byte) counters for
//  our various tables and matrices.  Note that I've inlined everything;
//  this class' members get called frequently and we don't ever want
//  function call overhead where we can avoid it.
//---------------------------------------------------------------------------
class CflowdUint64TrafficCounter
{
public:
  
  //-------------------------------------------------------------------------
  //                  inline CflowdUint64TrafficCounter()                  
  //.........................................................................
  //  constructor
  //-------------------------------------------------------------------------
  inline CflowdUint64TrafficCounter()
  {
    this->_pkts = 0;
    this->_bytes = 0;
  }
  
  //-------------------------------------------------------------------------
  //                       inline uint64_t Pkts() const
  //.........................................................................
  //  Returns the packet counter.
  //-------------------------------------------------------------------------
  inline uint64_t Pkts() const
  {
    return(this->_pkts);
  }
  
  //-------------------------------------------------------------------------
  //                   inline uint64_t Pkts(uint64_t pkts)                   
  //.........................................................................
  //  Sets and returns the packet counter.
  //-------------------------------------------------------------------------
  inline uint64_t Pkts(uint64_t pkts)
  {
    this->_pkts = pkts;
    return(this->_pkts);
  }

  //-------------------------------------------------------------------------
  //                  inline uint64_t AddPkts(uint64_t pkts)                 
  //.........................................................................
  //  Adds pkts to the packet counter.  Returns the new value of the
  //  packet counter.
  //-------------------------------------------------------------------------
  inline uint64_t AddPkts(uint64_t pkts)
  {
    this->_pkts += pkts;
    return(this->_pkts);
  }
  
  //-------------------------------------------------------------------------
  //                      inline uint64_t Bytes() const
  //.........................................................................
  //  Returns the byte counter.
  //-------------------------------------------------------------------------
  inline uint64_t Bytes() const
  {
    return(this->_bytes);
  }
  
  //-------------------------------------------------------------------------
  //                  inline uint64_t Bytes(uint64_t bytes)                  
  //.........................................................................
  //  Sets and returns the byte counter.
  //-------------------------------------------------------------------------
  inline uint64_t Bytes(uint64_t bytes)
  {
    this->_bytes = bytes;
    return(this->_bytes);
  }

  //-------------------------------------------------------------------------
  //                  inline uint64_t Bytes(uint64_t bytes)                  
  //.........................................................................
  //  Adds bytes to the byte counter.  Returns the new value of the byte
  //  counter.
  //-------------------------------------------------------------------------
  inline uint64_t AddBytes(uint64_t bytes)
  {
    this->_bytes += bytes;
    return(this->_bytes);
  }
  
  //-------------------------------------------------------------------------
  //                   inline istream & read(istream & is)                   
  //.........................................................................
  //  Reads the counter from an istream.  Returns the istream.
  //-------------------------------------------------------------------------
  inline istream & read(istream & is)
  {
    g_CfdArtsPrimitive.ReadUint64(is,this->_pkts,sizeof(this->_pkts));
    g_CfdArtsPrimitive.ReadUint64(is,this->_bytes,sizeof(this->_bytes));
    return(is);
  }
  
  //-------------------------------------------------------------------------
  //                         inline int read(int fd)                         
  //.........................................................................
  //  Reads the counter from a file descriptor.  Returns the number of
  //  bytes read on success, -1 on failure.
  //-------------------------------------------------------------------------
  inline int read(int fd)
  {
    int  rc;
    int  bytesRead = 0;
    
    rc = g_CfdArtsPrimitive.ReadUint64(fd,this->_pkts,sizeof(this->_pkts));
    if (rc < (int)sizeof(this->_pkts)) {
      syslog(LOG_ERR,
             "[E] ArtsPrimitive.ReadUint64(%d,%d,%d) failed: %m {%s:%d}",
             fd,this->_pkts,sizeof(this->_pkts),__FILE__,__LINE__);
      return(-1);
    }
    bytesRead += rc;
    
    rc = g_CfdArtsPrimitive.ReadUint64(fd,this->_bytes,sizeof(this->_bytes));
    if (rc < (int)sizeof(this->_bytes)) {
      syslog(LOG_ERR,
             "[E] ArtsPrimitive.ReadUint64(%d,%d,%d) failed: %m {%s:%d}",
             fd,this->_bytes,sizeof(this->_bytes),__FILE__,__LINE__);
      return(-1);
    }
    bytesRead += rc;
    return(bytesRead);
  }
  
  //-------------------------------------------------------------------------
  //                inline ostream & write(ostream & os) const               
  //.........................................................................
  //  Writes the counter to an ostream.  Returns the ostream.
  //-------------------------------------------------------------------------
  inline ostream & write(ostream & os) const
  {
    g_CfdArtsPrimitive.WriteUint64(os,this->_pkts,sizeof(this->_pkts));
    g_CfdArtsPrimitive.WriteUint64(os,this->_bytes,sizeof(this->_bytes));
    return(os);
  }
  
  //-------------------------------------------------------------------------
  //                      inline int write(int fd) const                     
  //.........................................................................
  //  Writes the counter to a file descriptor.  Returns the number of
  //  bytes written on success, -1 on failure.
  //-------------------------------------------------------------------------
  inline int write(int fd) const
  {
    int  rc;
    int  bytesWritten = 0;

    rc = g_CfdArtsPrimitive.WriteUint64(fd,this->_pkts,sizeof(this->_pkts));
    if (rc < (int)sizeof(this->_pkts)) {
      syslog(LOG_ERR,
             "[E] ArtsPrimitive.WriteUint64(%d,%d,%d) failed: %m {%s:%d}",
             fd,this->_pkts,sizeof(this->_pkts),__FILE__,__LINE__);
      return(-1);
    }
    bytesWritten += rc;
    rc = g_CfdArtsPrimitive.WriteUint64(fd,this->_bytes,sizeof(this->_bytes));
    if (rc < (int)sizeof(this->_bytes)) {
      syslog(LOG_ERR,
             "[E] ArtsPrimitive.WriteUint64(%d,%d,%d) failed: %m {%s:%d}",
             fd,this->_bytes,sizeof(this->_bytes),__FILE__,__LINE__);
      return(-1);
    }
    bytesWritten += rc;
    return(bytesWritten);
  }

  //--------------------------------------------------------------------------
  //                        static uint8_t Length()
  //..........................................................................
  //  Returns the bytes required to read/write the object from/to a file
  //  descriptor.  Note we just return a static member; the I/O size of
  //  the CflowdUint64TrafficCounter class is constant.
  //--------------------------------------------------------------------------
  static uint8_t Length()
  {
    return(_ioLength);
  }
  
private:
  uint64_t            _pkts;
  uint64_t            _bytes;
  static uint8_t      _ioLength;
};

#endif  // _CFLOWDUINT64TRAFFICCOUNTER_HH_
