//===========================================================================
//  @(#) $Name: cflowd-2-1-b1 $
//  @(#) $Id: CflowdNetMatrix.cc,v 1.17 1999/08/28 08:52:02 dwm Exp $
//===========================================================================
//  CAIDA Copyright Notice
//
//  By accessing this software, cflowd++, you are duly informed
//  of and agree to be bound by the conditions described below in this
//  notice:
//
//  This software product, cflowd++, is developed by Daniel W. McRobb, and
//  copyrighted(C) 1998 by the University of California, San Diego
//  (UCSD), with all rights reserved.  UCSD administers the CAIDA grant,
//  NCR-9711092, under which part of this code was developed.
//
//  There is no charge for cflowd++ software. You can redistribute it
//  and/or modify it under the terms of the GNU General Public License,
//  v.  2 dated June 1991 which is incorporated by reference herein.
//  cflowd++ is distributed WITHOUT ANY WARRANTY, IMPLIED OR EXPRESS, OF
//  MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE or that the use
//  of it will not infringe on any third party's intellectual property
//  rights.
//
//  You should have received a copy of the GNU GPL along with cflowd++.
//  Copies can also be obtained from:
//
//    http://www.gnu.org/copyleft/gpl.html
//
//  or by writing to:
//
//    University of California, San Diego
//
//    SDSC/CAIDA
//    9500 Gilman Dr., MS-0505
//    La Jolla, CA 92093 - 0505  USA
//
//  Or contact:
//
//    info@caida.org
//===========================================================================

#include <string>
#include <strstream.h>

#include "ArtsPrimitive.hh"
#include "CflowdNetMatrix.hh"

#define k_flowNetMatrixV5FieldsMask (CflowdRawFlow::k_srcIpAddrMask|\
                                     CflowdRawFlow::k_dstIpAddrMask|\
                                     CflowdRawFlow::k_srcMaskLenMask|\
                                     CflowdRawFlow::k_dstMaskLenMask|\
                                     CflowdRawFlow::k_pktsMask|\
                                     CflowdRawFlow::k_bytesMask)

static const string rcsid = "@(#) $Name: cflowd-2-1-b1 $ $Id: CflowdNetMatrix.cc,v 1.17 1999/08/28 08:52:02 dwm Exp $";

//-------------------------------------------------------------------------
//        int CflowdNetMatrix::AddFlow(const CflowdRawFlow & flow)
//.........................................................................
//  
//-------------------------------------------------------------------------
int CflowdNetMatrix::AddFlow(const CflowdRawFlow & flow)
{
  CflowdNetMatrixKey  netmKey;
  ipv4addr_t          srcMask, dstMask, srcMaskLen, dstMaskLen;

  if ((flow.Index() & k_flowNetMatrixV5FieldsMask) ==
      k_flowNetMatrixV5FieldsMask) {
    if (flow.SrcMaskLen() == 0) {
      srcMaskLen = 32;
    }
    else {
      srcMaskLen = flow.SrcMaskLen();
    }
    if (flow.DstMaskLen() == 0) {
      dstMaskLen = 32;
    }
    else {
      dstMaskLen = flow.DstMaskLen();
    }
    srcMask = (~0 << (32 - srcMaskLen));
    netmKey.Src(flow.SrcIpAddr() & htonl(srcMask));
    netmKey.SrcMaskLen(srcMaskLen);
    dstMask = (~0 << (32 - dstMaskLen));
    netmKey.Dst(flow.DstIpAddr() & htonl(dstMask));
    netmKey.DstMaskLen(dstMaskLen);
    ((*this)[netmKey]).AddPkts(flow.Pkts());
    ((*this)[netmKey]).AddBytes(flow.Bytes());
    return(0);
  }
  else {
    if (flow.Version() != 8) {
      syslog(LOG_ERR,
             "[E] got a v%d flow without net matrix fields"
             " (index = %#x) {%s:%d}",
             flow.Version(),flow.Index(),__FILE__,__LINE__);
    }
    return(-1);
  }
}

//-------------------------------------------------------------------------
//               istream & CflowdNetMatrix::read(istream & is)              
//.........................................................................
//  
//-------------------------------------------------------------------------
istream & CflowdNetMatrix::read(istream & is)
{
  uint64_t                       numEntries, entryNum;
  CflowdNetMatrixKey             netKey;
  CflowdNetMatrixTrafficCounter  netTraffic;

  if ((*this).size() > 0) {
    (*this).erase((*this).begin(),(*this).end());
  }
  
  g_CfdArtsPrimitive.ReadUint64(is,numEntries,sizeof(numEntries));
  for (entryNum = 0; entryNum < numEntries; entryNum++) {
    netKey.read(is);
    netTraffic.read(is);
    (*this)[netKey] = netTraffic;
  }
  return(is);
}

//-------------------------------------------------------------------------
//                     int CflowdNetMatrix::read(int fd)                    
//.........................................................................
//  
//-------------------------------------------------------------------------
int CflowdNetMatrix::read(int fd)
{
  uint64_t                       numEntries, entryNum;
  CflowdNetMatrixKey             netKey;
  CflowdNetMatrixTrafficCounter  netTraffic;
  int                            rc;
  int                            bytesRead = 0;
  
  if ((*this).size() > 0) {
    (*this).erase((*this).begin(),(*this).end());
  }
  
  rc = g_CfdArtsPrimitive.ReadUint64(fd,numEntries,sizeof(numEntries));
  if (rc < (int)sizeof(numEntries)) {
    syslog(LOG_ERR,
           "[E] g_CfdArtsPrimitive.ReadUint64(%d,%d,%d) failed: %m {%s:%d}",
           fd,numEntries,sizeof(numEntries),__FILE__,__LINE__);
    return(-1);
  }
  bytesRead += rc;

  uint32_t expectedLength = 
    numEntries * (CflowdNetMatrixKey::Length() +
                  CflowdNetMatrixTrafficCounter::Length());
  char *buf = (char *)malloc(expectedLength);
  if (buf == (char *)0) {
    syslog(LOG_ERR,"[E] malloc(%u) failed: %m {%s:%d}",
           expectedLength,__FILE__,__LINE__);
    return(-1);
  }
  
  if (g_CfdArtsPrimitive.FdRead(fd,buf,expectedLength) < expectedLength) {
    syslog(LOG_ERR,"[E] dArtsPrimitive.FdRead(%d,%p,%u) failed: %m {%s:%d}",
           fd,buf,expectedLength,__FILE__,__LINE__);
    return(-1);
  }
  
  istrstream  inStream(buf,expectedLength);
  
  for (entryNum = 0; entryNum < numEntries; entryNum++) {
    netKey.read(inStream);
    netTraffic.read(inStream);
    bytesRead += CflowdNetMatrixKey::Length() + 
      CflowdNetMatrixTrafficCounter::Length();
    (*this)[netKey] = netTraffic;
  }
  bytesRead += expectedLength;
  free(buf);
  
  return(bytesRead);
}

//-------------------------------------------------------------------------
//           ostream & CflowdNetMatrix::write(ostream & os) const           
//.........................................................................
//  
//-------------------------------------------------------------------------
ostream & CflowdNetMatrix::write(ostream & os) const
{
  uint64_t                         numEntries;
  CflowdNetMatrix::const_iterator  netmIter;
  
  numEntries = (*this).size();
  
  g_CfdArtsPrimitive.WriteUint64(os,numEntries,sizeof(numEntries));

  for (netmIter = (*this).begin(); netmIter != (*this).end(); netmIter++) {
    (*netmIter).first.write(os);
    (*netmIter).second.write(os);
  }
  return(os);
}

//-------------------------------------------------------------------------
//                 int CflowdNetMatrix::write(int fd) const                 
//.........................................................................
//  
//-------------------------------------------------------------------------
int CflowdNetMatrix::write(int fd) const
{
  uint64_t                         numEntries;
  CflowdNetMatrix::const_iterator  netmIter;
  int                              rc;
  int                              bytesWritten = 0;
  
  numEntries = (*this).size();
  rc = g_CfdArtsPrimitive.WriteUint64(fd,numEntries,sizeof(numEntries));
  if (rc < (int)sizeof(numEntries)) {
    syslog(LOG_ERR,"[E] g_CfdArtsPrimitive.WriteUint64(%d,%u,%d) failed:"
           " %m {%s:%d}",fd,numEntries,sizeof(numEntries),
           __FILE__,__LINE__);
    return(-1);
  }
  bytesWritten += rc;
  for (netmIter = (*this).begin(); netmIter != (*this).end(); netmIter++) {
    rc = (*netmIter).first.write(fd);
    if (rc < 0) {
      syslog(LOG_ERR,"[E] (*netmIter).first.write(%d) failed: %m {%s:%d}",
             fd,__FILE__,__LINE__);
      return(-1);
    }
    bytesWritten += rc;
    rc = (*netmIter).second.write(fd);
    if (rc < 0) {
      syslog(LOG_ERR,"[E] (*netmIter).second.write(%d) failed: %m {%s:%d}",
             fd,__FILE__,__LINE__);
      return(-1);
    }
    bytesWritten += rc;
  }
  fsync(fd);
  
  return(bytesWritten);
}


//---------------------------------------------------------------------------
//         ostream& operator << (ostream& os,
//                               const CflowdNetMatrix & netMatrix)
//...........................................................................
//
//---------------------------------------------------------------------------
ostream& operator << (ostream& os,
                      const CflowdNetMatrix & netMatrix)
{
  CflowdNetMatrix::const_iterator  netIter;

  for (netIter = netMatrix.begin(); netIter != netMatrix.end();
       netIter++) {
    struct in_addr  srcInAddr, dstInAddr;

    srcInAddr.s_addr = (*netIter).first.Src();
    dstInAddr.s_addr = (*netIter).first.Dst();
    
    os << "NET MATRIX ENTRY" << endl
       << "  src net: " << inet_ntoa(srcInAddr) << "/"
       << (uint16_t)(*netIter).first.SrcMaskLen() << endl;
    os << "  dst net: " << inet_ntoa(dstInAddr) << "/"
       << (uint16_t)(*netIter).first.DstMaskLen() << endl;
    os << "  packets: " << (*netIter).second.Pkts() << endl
       << "    bytes: " << (*netIter).second.Bytes() << endl;
  }
  return(os);
}

uint8_t CflowdNetMatrixKey::_ioLength = ((sizeof(ipv4addr_t) * 2) +
                                         (sizeof(uint8_t) * 2));
uint8_t CflowdUint64TrafficCounter::_ioLength = sizeof(uint64_t) * 2;
