/*===================================================================

  Copyright (c) 1996
  Hewlett-Packard Company

  ATTENTION: USE OF THIS SOFTWARE IS SUBJECT TO THE FOLLOWING TERMS.
  Permission to use, copy, modify, distribute and/or sell this software 
  and/or its documentation is hereby granted without fee. User agrees 
  to display the above copyright notice and this license notice in all 
  copies of the software and any documentation of the software. User 
  agrees to assume all liability for the use of the software; Hewlett-Packard 
  makes no representations about the suitability of this software for any 
  purpose. It is provided "AS-IS without warranty of any kind,either express 
  or implied. User hereby grants a royalty-free license to any and all 
  derivatives based upon this software code base. 

 
  SNMP++ A D D R E S S . H

  ADDRESS CLASS DEFINITION

  VERSION:
  2.6

  RCS INFO:
  $Header: /caimis/cvs/cflowd/snmp++/classes/include/snmp++/address.h,v 1.2 1999/09/02 15:00:37 dwm Exp $

  DESIGN:
  Peter E Mellquist

  AUTHOR:
  Peter E Mellquist

  LANGUAGE:
  ANSI C++

  OPERATING SYSTEMS:
  MS-Windows Win32
  BSD UNIX

  DESCRIPTION:
  Address class definition. Encapsulates various network
  addresses into easy to use, safe and portable classes.

=====================================================================*/
#ifndef _ADDRESS
#define _ADDRESS


//----[ includes ]-----------------------------------------------------
extern "C"
{
#include <string.h>
#include <memory.h>
}

#include "smival.h"
#include "collect.h"

// include sockets header files
// for Windows16 and Windows32 include Winsock
// otherwise assume UNIX
#ifdef __unix__
// The /**/ stuff below is meant to fool MS C++ into skipping these 
// files when creating its makefile.  8-Dec-95 TM
#include /**/ <unistd.h>
#include /**/ <sys/socket.h>
#include /**/ <netinet/in.h>
#include /**/ <netdb.h>
#include /**/ <arpa/inet.h>
typedef in_addr IN_ADDR;
extern int h_errno;  // defined in WinSock header, but not for UX?!
#endif // __unix__


#ifdef WIN32
#ifndef __unix__         // __unix__ overrides WIN32 if both options are present
#include <winsock.h>
#endif
#endif

//----[ macros ]-------------------------------------------------------
#define BUFSIZE 40     // worst case of address lens
#define OUTBUFF 80     // worst case of output lens

#define IPLEN      4
#define UDPIPLEN   6
#define IPXLEN     10
#define IPXSOCKLEN 12
#define MACLEN     6
#define MAX_FRIENDLY_NAME 80
#define HASH0 19
#define HASH1 13
#define HASH2 7

#ifndef INADDR_NONE
  #define INADDR_NONE 0xFFFFFFFF
#endif

//----[ enumerated types for address types ]---------------------------
enum addr_type {
  type_ip,
  type_ipx,
  type_udp,
  type_ipxsock,
  type_mac,
  type_invalid
};

//---[ forward declarations ]-----------------------------------------
class GenAddress; 
class UdpAddress;
class IpxSockAddress;

//--------------------------------------------------------------------
//----[ Address class ]-----------------------------------------------
//--------------------------------------------------------------------
class DLLOPT Address: public  SnmpSyntax {

public:

  // allow destruction of derived classes
  virtual ~Address();

  // overloaded equivlence operator, are two addresses equal?
  DLLOPT friend int operator==( const Address &lhs,const Address &rhs);

  // overloaded not equivlence operator, are two addresses not equal?
  DLLOPT friend int operator!=( const Address &lhs,const Address &rhs);

  // overloaded > operator, is a1 > a2
  DLLOPT friend int operator>( const Address &lhs,const Address &rhs);

  // overloaded >= operator, is a1 >= a2
  DLLOPT friend int operator>=( const Address &lhs,const Address &rhs);

  // overloaded < operator, is a1 < a2
  DLLOPT friend int operator<( const Address &lhs,const Address &rhs);

  // overloaded <= operator, is a1 <= a2
  DLLOPT friend int operator<=( const Address &lhs,const Address &rhs);

  // equivlence operator overloaded, are an address and a string equal?
  DLLOPT friend int operator==( const Address &lhs,const char *rhs);

  // overloaded not equivlence operator, are an address and string not equal?
  DLLOPT friend int operator!=( const Address &lhs,const char *rhs);

  // overloaded < , is an address greater than a string?
  DLLOPT friend int operator>( const Address &lhs,const char *rhs);

  // overloaded >=, is an address greater than or equal to a string?
  DLLOPT friend int operator>=( const Address &lhs,const char *rhs);

  // overloaded < , is an address less than a string?
  DLLOPT friend int operator<( const Address &lhs,const char *rhs);

  // overloaded <=, is an address less than or equal to a string?
  DLLOPT friend int operator<=( const Address &lhs,const char *rhs);

  // const char * operator overloaded for streaming output
  virtual operator const char *() const = 0;

  // is the address object valid?
  virtual int valid() const;            

  // syntax type
  virtual SmiUINT32 get_syntax() = 0;

  // for non const [], allows reading and writing
  unsigned char& operator[]( const int position);

  // get a printable ASCII value
  virtual char *get_printable() = 0;

  // create a new instance of this Value
  virtual SnmpSyntax *clone() const = 0;

  // return the type of address
  virtual addr_type get_type() const = 0; 

  // overloaded assignment operator
  virtual SnmpSyntax& operator=( SnmpSyntax &val) = 0;

  // return a hash key
  virtual unsigned int hashFunction() const { return 0;};

protected:
  int valid_flag;
  unsigned char address_buffer[BUFSIZE]; // internal representation

  // parse the address string
  // redefined for each specific address subclass
  virtual int parse_address( const char * inaddr) =0;

  // format the output
  // redefined for each specific address subclass
  virtual void format_output() =0;

  // a reused trimm white space method
  void trim_white_space( char * ptr);

};


//-----------------------------------------------------------------------
//---------[ IP Address Class ]------------------------------------------
//-----------------------------------------------------------------------
class DLLOPT IpAddress : public Address {

public:
  // construct an IP address with no agrs
  IpAddress( void);

  // construct an IP address with a string
  IpAddress( const char *inaddr);

  // construct an IP address with another IP address
  IpAddress( const IpAddress  &ipaddr);

  // construct an IP address with a GenAddress
  IpAddress( const GenAddress &genaddr);

  // destructor (ensure that SnmpSyntax::~SnmpSyntax() is overridden)
  ~IpAddress();

  // copy an instance of this Value
  SnmpSyntax& operator=( SnmpSyntax &val);

  // assignment to another IpAddress object overloaded
  IpAddress& operator=( const IpAddress &ipaddress);

  // create a new instance of this Value
  SnmpSyntax *clone() const;

  // return the friendly name
  // returns a NULL string if there isn't one
  char *friendly_name(int &status);

  virtual char *get_printable() ;

  // const char * operator overloaded for streaming output
  virtual operator const char *() const;

  // logically AND two IPaddresses and
  // return the new one
  void mask( const IpAddress& ipaddr);

  // return the type
  virtual addr_type get_type() const;

  // syntax type
  virtual SmiUINT32 get_syntax();

protected:
  char output_buffer[OUTBUFF];           // output buffer

  // friendly name storage
  char iv_friendly_name[MAX_FRIENDLY_NAME];
  int  iv_friendly_name_status;

  // redefined parse address
  // specific to IP addresses
  virtual int parse_address( const char *inaddr);
   

  // redefined format output
  // specific to IP addresses
  virtual void format_output();

  // parse a dotted string
  int parse_dotted_ipstring( const char *inaddr);

  // using the currently defined address, do a DNS
  // and try to fill up the name
  int addr_to_friendly();

};

//------------------------------------------------------------------------
//---------[ UDP Address Class ]------------------------------------------
//------------------------------------------------------------------------
class DLLOPT UdpAddress : public IpAddress {

public:

  // constructor, no args
  UdpAddress( void);

  // constructor with a dotted string
  UdpAddress( const char *inaddr);

  // construct an Udp address with another Udp address
  UdpAddress( const UdpAddress &udpaddr);

  // construct a Udp address with a GenAddress
  UdpAddress( const GenAddress &genaddr);

  // construct a Udp address with an IpAddress
  // default port # to zero
  UdpAddress( const IpAddress &ipaddr);

  // destructor
  ~UdpAddress();

  // syntax type
  SmiUINT32 get_syntax();

  // copy an instance of this Value
  SnmpSyntax& operator=( SnmpSyntax &val);

  // assignment to another IpAddress object overloaded
  UdpAddress& operator=( const UdpAddress &udpaddr);

  // create a new instance of this Value
  SnmpSyntax *clone() const;

  virtual char *get_printable() ;

  // const char * operator overloaded for streaming output
  virtual operator const char *() const;

  // set the port number
  void set_port( const unsigned short p);

  // get the port number
  unsigned short get_port() const; 

  // return the type
  virtual addr_type get_type() const;

protected:
  char output_buffer[OUTBUFF];           // output buffer

  // redefined parse address
  // specific to IP addresses
  virtual int parse_address( const char *inaddr);

  // redefined format output
  // specific to IP addresses
  virtual void format_output();
};

//-------------------------------------------------------------------------
//---------[ 802.3 MAC Address Class ]-------------------------------------
//-------------------------------------------------------------------------
class DLLOPT MacAddress : public Address {

public:
  // constructor, no arguments
  MacAddress( void);

  // constructor with a string argument
  MacAddress( const char  *inaddr);

  // constructor with another MAC object
  MacAddress( const MacAddress  &macaddr);

  // construct a MacAddress with a GenAddress
  MacAddress( const GenAddress &genaddr);

  // destructor 
  ~MacAddress();

  // syntax type
  SmiUINT32 get_syntax();

  // copy an instance of this Value
  SnmpSyntax& operator=( SnmpSyntax &val);

  // assignment to another IpAddress object overloaded
  MacAddress& operator=( const MacAddress &macaddress);

  // create a new instance of this Value
  SnmpSyntax *clone() const; 

  virtual char *get_printable();

  // const char * operator overloaded for streaming output
  virtual operator const char *() const;

  // return the type
  virtual addr_type get_type() const;

  // return a hash key
  unsigned int hashFunction() const;


protected:
  char output_buffer[OUTBUFF];           // output buffer

  // redefined parse address for macs
  virtual int parse_address( const char *inaddr);

  // redefined format output for MACs
  virtual void format_output();
};

//------------------------------------------------------------------------
//---------[ IPX Address Class ]------------------------------------------
//------------------------------------------------------------------------
class DLLOPT IpxAddress : public Address {

public:
  // constructor no args
  IpxAddress( void);

  // constructor with a string arg
  IpxAddress( const char  *inaddr);

  // constructor with another ipx object
  IpxAddress( const IpxAddress  &ipxaddr);

  // construct with a GenAddress
  IpxAddress( const GenAddress &genaddr);

  // destructor 
  ~IpxAddress();

  // syntax type
  virtual SmiUINT32 get_syntax();

  // copy an instance of this Value
  SnmpSyntax& operator=( SnmpSyntax &val); 

  // assignment to another IpAddress object overloaded
  IpxAddress& operator=( const IpxAddress &ipxaddress);

  // get the host id portion of an ipx address
  int get_hostid( MacAddress& mac);

  // create a new instance of this Value
  SnmpSyntax *clone() const; 

  virtual char *get_printable();

  // const char * operator overloaded for streaming output
  virtual operator const char *() const;

  // return the type
  virtual addr_type get_type() const;

protected:
  // ipx format separator
  char separator;
  char output_buffer[OUTBUFF];           // output buffer

  // redefined parse address for ipx strings
  virtual int parse_address( const char  *inaddr);

  // redefined format output for ipx strings
  // uses same separator as when constructed
  virtual void format_output();
 
};



//------------------------------------------------------------------------
//---------[ IpxSock Address Class ]--------------------------------------
//------------------------------------------------------------------------
class DLLOPT IpxSockAddress : public IpxAddress {

public:
  // constructor, no args
  IpxSockAddress( void);

  // constructor with a dotted string
  IpxSockAddress( const char *inaddr);

  // construct an Udp address with another Udp address
  IpxSockAddress( const IpxSockAddress &ipxaddr);

  //constructor with a GenAddress
  IpxSockAddress( const GenAddress &genaddr);

  //constructor with a IpxAddress
  // default socket # is 0
  IpxSockAddress( const IpxAddress &ipxaddr);

  // destructor
  ~IpxSockAddress();

  // syntax type
  virtual SmiUINT32 get_syntax();

  // copy an instance of this Value
  SnmpSyntax& operator=( SnmpSyntax &val); 

  // assignment to another IpAddress object overloaded
  IpxSockAddress& operator=( const IpxSockAddress &ipxaddr);

  // create a new instance of this Value
  SnmpSyntax *clone() const;

  // set the socket number
  void set_socket( const unsigned short s);

  // get the socket number
  unsigned short get_socket() const;

  virtual char *get_printable();

  // const char * operator overloaded for streaming output
  virtual operator const char *() const;

  // return the type
  virtual addr_type get_type() const;

protected:
  char output_buffer[OUTBUFF];           // output buffer

  // redefined parse address for ipx strings
  virtual int parse_address( const char  *inaddr);

  // redefined format output
  // specific to IP addresses
  virtual void format_output();

};




//-------------------------------------------------------------------------
//--------[ Generic Address ]----------------------------------------------
//-------------------------------------------------------------------------
class DLLOPT GenAddress : public Address {

public:
  // constructor, no arguments
  GenAddress( void);

  // constructor with a string argument
  GenAddress( const char  *addr);

  // constructor with an Address
  GenAddress( const Address &addr);

  // constructor with another GenAddress
  GenAddress( const GenAddress &addr);

  // destructor
  ~GenAddress();

  // get the snmp syntax of the contained address
  SmiUINT32 get_syntax();

  // create a new instance of this Value
  SnmpSyntax *clone() const;

  // assignment of a GenAddress
  GenAddress& operator=( const GenAddress &addr);

  // copy an instance of this Value
  SnmpSyntax& operator=( SnmpSyntax &val);

  virtual char *get_printable();

  // const char * operator overloaded for streaming output
  virtual operator const char *() const;

  // return the type
  virtual addr_type get_type() const;

protected:
  // pointer to a a concrete address
  Address *address;
  char output_buffer[OUTBUFF];           // output buffer

  // redefined parse address for macs
  virtual int parse_address( const char *addr);

  // format output for a generic address
  virtual void format_output();

};

// create OidCollection type
typedef SnmpCollection <GenAddress> AddressCollection;

#endif  //_ADDRESS
