extern "C" {
#include <unistd.h>
#include <stdio.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
}

#include <string>
#include <map>

#include "address.h"
#include "target.h"
#include "snmp_pp.h"

typedef unsigned int  ipv4addr_t;

static Oid   g_ipAdEntIfIndexOid("1.3.6.1.2.1.4.20.1.2");
static Oid   g_ifDescrOid("1.3.6.1.2.1.2.2.1.2");

//----------------------------------------------------------------------------
//                           class InterfaceInfo 
//----------------------------------------------------------------------------
//  
//----------------------------------------------------------------------------
class InterfaceInfo
{
public:
  //--------------------------------------------------------------------------
  //                             InterfaceInfo() 
  //..........................................................................
  //  
  //--------------------------------------------------------------------------
  InterfaceInfo()
  {
    this->_ifDescr = string("");
    this->_ipAddr = INADDR_NONE;
  }
  
  //--------------------------------------------------------------------------
  //                     ipv4addr_t IpAddress() const 
  //..........................................................................
  //  
  //--------------------------------------------------------------------------
  ipv4addr_t IpAddress() const
  {
    return(this->_ipAddr);
  }

  //--------------------------------------------------------------------------
  //                const string & IpAddress(const Oid & oid) 
  //..........................................................................
  //  
  //--------------------------------------------------------------------------
  ipv4addr_t IpAddress(const Oid & oid)
  {
    this->_ipAddr = 0;
    
    for (int i = oid.len() - 4; i < oid.len(); i++) {
      this->_ipAddr |= ((ipv4addr_t)(oid[i]) << ((oid.len() - (i + 1)) * 8));
    }
    
    this->_ipAddr = htonl(this->_ipAddr);

    return(this->_ipAddr);
  }
  
  //--------------------------------------------------------------------------
  //                      const string & IfDescr() const 
  //..........................................................................
  //  
  //--------------------------------------------------------------------------
  const string & IfDescr() const
  {
    return(_ifDescr);
  }

  //--------------------------------------------------------------------------
  //                const string & IfDescr(const char * descr) 
  //..........................................................................
  //  
  //--------------------------------------------------------------------------
  const string & IfDescr(const char * descr)
  {
    _ifDescr = string(descr);
    return(_ifDescr);
  }
  
private:
  ipv4addr_t  _ipAddr;
  string      _ifDescr;
};

typedef map<unsigned short,InterfaceInfo,less<unsigned short> >
InterfaceInfoMap_t;

//----------------------------------------------------------------------------
//  static void GetInterfaceDescriptions(Snmp & session, CTarget & target, 
//                                       InterfaceInfoMap_t & ifMap) 
//............................................................................
//  
//----------------------------------------------------------------------------
static void GetInterfaceDescriptions(Snmp & session, CTarget & target,
                                     InterfaceInfoMap_t & ifMap)
{
  Vb   varBinding;
  Pdu  pdu;
  int  status;
  Oid  receivedOid;
  
  varBinding.set_oid(g_ifDescrOid);
  pdu += varBinding;

  while ((status = session.get_next(pdu,target)) == SNMP_CLASS_SUCCESS) {
    for (int i = 0; i < pdu.get_vb_count(); i++) {
      pdu.get_vb(varBinding,i);
      varBinding.get_oid(receivedOid);
      if (g_ifDescrOid.nCompare(g_ifDescrOid.len(),receivedOid) == 0) {
        ifMap[receivedOid[receivedOid.len() - 1]].IfDescr((const char *)(varBinding.get_printable_value()));
      } else {
        pdu.delete_vb(i);
      }
    }
    if (pdu.get_vb_count() < 1)
      break;
  }
  return;
}

//----------------------------------------------------------------------------
//   static void GetInterfaceAddresses(Snmp & session, CTarget & target, 
//                                     InterfaceInfoMap_t & ifMap) 
//............................................................................
//  
//----------------------------------------------------------------------------
static void GetInterfaceAddresses(Snmp & session, CTarget & target,
                                  InterfaceInfoMap_t & ifMap)
{
  Vb   varBinding;
  Pdu  pdu;
  int  status;
  Oid  receivedOid;
  
  varBinding.set_oid(g_ipAdEntIfIndexOid);
  pdu += varBinding;

  while ((status = session.get_next(pdu,target)) == SNMP_CLASS_SUCCESS) {
    for (int i = 0; i < pdu.get_vb_count(); i++) {
      pdu.get_vb(varBinding,i);
      varBinding.get_oid(receivedOid);
      if (g_ipAdEntIfIndexOid.nCompare(g_ipAdEntIfIndexOid.len(),
                                       receivedOid) == 0) {
        int interfaceIndex;
        varBinding.get_value(interfaceIndex);
        ifMap[interfaceIndex].IpAddress(receivedOid);
      } else {
        pdu.delete_vb(i);
      }
    }
    if (pdu.get_vb_count() < 1)
      break;
  }
  return;  
}

//----------------------------------------------------------------------------
//                     int main(int argc, char *argv[]) 
//............................................................................
//  
//----------------------------------------------------------------------------
int main(int argc, char *argv[])
{
  int           optChar;
  extern char  *optarg;
  GenAddress    targetAddress("0.0.0.0");
  string        community;
  Vb            vb;
  Pdu           pdu;
  int           status;
  InterfaceInfoMap_t  interfaceInfoMap;
  
  while ((optChar = getopt(argc,argv,"h:c:")) != EOF) {
    switch (optChar) {
      case 'h':
        targetAddress = optarg;
        break;
      case 'c':
        community = string(optarg);
        break;
      default:
        break;
    }
  }

  if (targetAddress == "0.0.0.0" || community.length() < 1) {
    cerr << "usage: " << argv[0] << " -h host -c community" << endl;
    return(1);
  }

  Snmp session(status);

  if (status) {
    cerr << "Failed to create SNMP session: " << status << endl;
    return(1);
  }
  
  CTarget  target(targetAddress,community.c_str(),community.c_str());
  target.set_timeout(200);
  target.set_retry(2);
  
  if (! target.valid()) {
    cerr << "Invalid target" << endl;
    return(1);
  }

  GetInterfaceDescriptions(session,target,interfaceInfoMap);
  GetInterfaceAddresses(session,target,interfaceInfoMap);

  InterfaceInfoMap_t::iterator ifMapIter;
  struct in_addr               inAddr;
  
  for (ifMapIter = interfaceInfoMap.begin();
       ifMapIter != interfaceInfoMap.end(); ifMapIter++) {
    if ((*ifMapIter).second.IpAddress() != INADDR_NONE) {
      cout << (*ifMapIter).first << " " << (*ifMapIter).second.IfDescr();
      inAddr.s_addr = (*ifMapIter).second.IpAddress();
      cout << " " << inet_ntoa(inAddr) << endl;
    }
  }
       
  exit(0);
}

  
  
  
