-----------------------------------------------------------------------------
-- CVS $Id: CheckRDF.hs,v 1.15 2003/03/03 11:29:25 obraun Exp $
-- CVS $Unsane: projects/checkrdf/CheckRDF.hs,v 1.13 2005/05/29 19:27:39 lizard Exp $
-- |
--
-- Module      : CheckRDF
-- Copyright   : (c) Oliver Braun 2002-2005
-- License     : BSD
--
-- Maintainer  : obraun@informatik.unibw-muenchen.de
-- Stability   : provisional
-- Portability : portable
--
--
-----------------------------------------------------------------------------

module CheckRDF where

import RDFConfig

import System (getArgs, exitWith, ExitCode(..),system,getEnv)
import IO ( catch, openFile, IOMode(..), hIsEOF, hGetLine
          , hPutStr, hClose, hFlush, stdin, Handle, hPutStrLn, hGetContents)
import Directory ( doesFileExist, getPermissions, readable, removeFile )
import List ( partition, intersperse, groupBy, sortBy)
import Time (getClockTime,toCalendarTime,calendarTimeToString)

import Text.XML.HaXml.Types
import Text.XML.HaXml.Parse         (xmlParse)
import Text.XML.HaXml.Xtract.Parse  (parseXtract)
import Text.PrettyPrint.HughesPJ    (render, vcat)
import Text.XML.HaXml.Pretty        (content)

-- * Type declarations
data RDF = RDF {ctitle  :: String
               ,clink   :: String
               ,citems  :: [RDFItem]
               }
               deriving (Read,Show)

data RDFItem = RDFItem {title :: String
                       ,link  :: String
                       }
                       deriving (Read,Show)

-- | Make a RDF list including all new RDFs from the files given at the command line
mkNewRDFList :: RDFConfig -> [String] -> IO [RDF]
mkNewRDFList config = mapM (\f -> do contents <- readFile f
                                     return $! mkNewRDF config f contents)

-- | Open file and extract channel title, link and items and return a RDF
mkNewRDF :: RDFConfig -> FilePath -> String -> RDF
mkNewRDF config filename contents =
     let elem = getElem $ xmlParse filename contents
         ct = render $ vcat $ map content $ dfilter (parseXtract "channel/title/-") elem
         cl = render $ vcat $ map content $ dfilter (parseXtract "channel/link/-")  elem
         len1 = length $ dfilter (parseXtract "//item/title") elem
         len2 = length $ dfilter (parseXtract "//item/link") elem
         items = if len1 == len2
           then (map (\ i -> (render $ vcat $ map content $
                                dfilter (parseXtract ("//item/title["++i++"]/-")) elem
                             ,render $ vcat $ map content $
                                dfilter (parseXtract ("//item/link["++i++"]/-" )) elem)
                     )
                     $ map show [0..len1-1])
           else []
         rdfitems = map mkRDFItem items
     in  RDF {ctitle = ct
             ,clink  = cl
             ,citems = rdfitems
             }
 where getElem (Document _ _ e _) = CElem e
       dfilter f = \ x -> f x x
       mkRDFItem (t,l) = RDFItem {title = substituteChar '\n' ' ' t
                                 ,link  = filter (/='\n') l
                                 }

substituteChar :: Char -> Char -> String -> String
substituteChar from to = map (\ c -> if c == from then to else c)

-- | Just return the old contents of the file or return []
getRDFList :: RDFConfig -> FilePath -> IO [RDF]
getRDFList config file =
  do cfex <- doesFileExist file
     if cfex
      then do
        perms <- getPermissions file
        if readable perms
         then do
           h <- openFile file ReadMode
           rdfs <- readRDFs h
           hClose h
           return rdfs
         else return []
      else return []

readRDFs :: Handle -> IO [RDF]
readRDFs h = do
    iseof <- hIsEOF h
    if iseof
     then return []
     else do line <- hGetLine h
             mrdf <- catch (do rdf <- (readIO line :: IO RDF)
                               return $ Just rdf)
                           (const $ return Nothing)
             maybe (return [])
                   (\rdf -> do rdfs <- readRDFs h
                               return $! rdf : rdfs)
                   mrdf

mkNewsAndState :: [RDF] -> [RDF] -> ([RDF],[RDF])
mkNewsAndState old new = (news,newState)
 where (notNewRDF,newRDF) = partition (\r -> (clink r) `elem` (map clink old)) new
       newsAndState'      = map (flip newsAndState notNewRDF) old
       newState           = map snd newsAndState' ++ newRDF
       news               = filter (\r -> not $ null $ citems r) $ map fst newsAndState'
                                   ++ newRDF

newsAndState :: RDF -> [RDF] -> (RDF,RDF)
newsAndState oldRDF newRDFs = (new_s_RDF,newRDF)
 where rdfTitle = ctitle oldRDF
       rdfLink  = clink  oldRDF
       newRDF'  = filter (\r -> clink r == rdfLink) newRDFs
       notShownYet' = filter (\i -> not ((title i) `elem` (map title (citems oldRDF))))
                            $ citems newRDF
       (notShownYet,newRDF) = if null newRDF'
                              then ([]          ,oldRDF      )
                              else (notShownYet',head newRDF')
       new_s_RDF = RDF {ctitle = rdfTitle
                       ,clink  = rdfLink
                       ,citems = notShownYet
                       }

mergeRDFLists :: [RDF] -> [RDF] -> [RDF]
mergeRDFLists r1 r2 = map mergeRDFList
                      $ groupBy  (\ a b -> ctitle a == ctitle b)
                      $ sortBy   (\ a b -> compare (ctitle a) (ctitle b))
                      $ r1 ++ r2

mergeRDFList :: [RDF] -> RDF
mergeRDFList []       = undefined
mergeRDFList rs@(r:_) = r {citems = concatMap citems rs}

putNews :: RDFConfig -> [RDF] -> [RDF] -> IO Bool
putNews _ []   _ = return False
putNews config news rdfs =
  do writeToPending <- doesFileExist $ pendingflagfile config
     writeRDFs (statefile config) rdfs
     if writeToPending
      then do
              writeRDFs (pendingfile config) news
              return False
      else do gt <- getClockTime
              ct <- toCalendarTime gt
              h <- openFile (outfile config) AppendMode
              hPutStrLn h $ "<!-- TIMESTAMP: " ++ calendarTimeToString ct ++ " -->"
              mapM_ (putNews' h) news
              hClose h
              writeFile (pendingfile config) "[]"
              return True

writeRDFs :: FilePath -> [RDF] -> IO ()
writeRDFs file rdfs = do
  fex <- doesFileExist file
  if fex then removeFile file else return ()
  h <- openFile file WriteMode
  writeRDFs' h rdfs
  hClose h

writeRDFs' :: Handle -> [RDF] -> IO ()
writeRDFs' h []     = return ()
writeRDFs' h (x:xs) = do hPutStrLn h $ show x
                         writeRDFs' h xs

putNews' :: Handle -> RDF -> IO ()
putNews' h news = do hPutStrLn h $ "<a href=\"" ++ clink news ++ "\" class=\"newstitlelink\">"
                     hPutStrLn h $ "<span class=\"newstitletext\">" ++ ctitle news ++ "</span>"
                     hPutStrLn h "</a>"
                     hPutStrLn h   "<ul class=\"newsitemlist\">"
                     mapM (putNewsItems h) $ citems news
                     hPutStrLn h   "</ul>"

putNewsItems :: Handle -> RDFItem -> IO ()
putNewsItems h item = do hPutStrLn h $ "<li class=\"newsitem\"><a href=\"" ++ link item ++ "\" class=\"newsitemlink\">"
                         hPutStrLn h $ "<span class=\"newsitemtext\">   " ++ title item ++ "</span>"
                         hPutStrLn h   "</a></li>"

-- vim:tw=9999:
