/*******************************************************************************
 *
 * colors.c
 *
 * Hash table for quick lookup of rgb values for color names. 
 *
 * Cheetah Web Browser
 * Copyright (C) 2001 Garett Spencley 
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA. 
 *
 *******************************************************************************/

#include <stdlib.h>
#include <ctype.h>
#include <string.h>
#include <stdlib.h>
#include <stdio.h>

#include "colors.h"
#include "str_util.h"
#include "error.h"

#define TOTAL_KEYWORDS 478
#define MIN_WORD_LENGTH 3
#define MAX_WORD_LENGTH 20
#define MIN_HASH_VALUE 41
#define MAX_HASH_VALUE 984

__inline static unsigned int color_hash(const char *str, const int len)
{
	static unsigned short asso_values[] = {
		985, 985, 985, 985, 985, 985, 985, 985, 985, 985,
      985, 985, 985, 985, 985, 985, 985, 985, 985, 985,
      985, 985, 985, 985, 985, 985, 985, 985, 985, 985,
      985, 985, 985, 985, 985, 985, 985, 985, 985, 985,
      985, 985, 985, 985, 985, 985, 985, 985,  75, 443,
      474,  60,  10,  45,  20,  55,  50,  40, 985, 985,
      985, 985, 985, 985, 985, 985, 985, 985, 985, 985,
      985, 985, 985, 985, 985, 985, 985, 985, 985, 985,
      985, 985, 985, 985, 985, 985, 985, 985, 985, 985,
      985, 985, 985, 985, 985, 985, 985, 459, 360, 470,
      348, 500,  35, 267, 190, 163, 985,  25,  40, 105,
      185, 425, 250, 985, 240, 148, 365,  15,  25,  28,
      985, 208, 985, 985, 985, 985, 985, 985, 985, 985,
      985, 985, 985, 985, 985, 985, 985, 985, 985, 985,
      985, 985, 985, 985, 985, 985, 985, 985, 985, 985,
      985, 985, 985, 985, 985, 985, 985, 985, 985, 985,
      985, 985, 985, 985, 985, 985, 985, 985, 985, 985,
      985, 985, 985, 985, 985, 985, 985, 985, 985, 985,
      985, 985, 985, 985, 985, 985, 985, 985, 985, 985,
      985, 985, 985, 985, 985, 985, 985, 985, 985, 985,
      985, 985, 985, 985, 985, 985, 985, 985, 985, 985,
      985, 985, 985, 985, 985, 985, 985, 985, 985, 985,
      985, 985, 985, 985, 985, 985, 985, 985, 985, 985,
      985, 985, 985, 985, 985, 985, 985, 985, 985, 985,
      985, 985, 985, 985, 985, 985, 985, 985, 985, 985,
      985, 985, 985, 985, 985, 985
	};
	return len + asso_values[(unsigned char) str[len - 1]] + asso_values[(unsigned char) str[0]];
}

__inline struct colors_t *color_lookup(const char *str, const int len)
{
	static struct colors_t wordlist[] = {
{"khaki4", "827839" },
      {"wheat4", "816f54"},
      {"violetred4", "7d0541"},
      {"lightblue4", "5e767e" },
      {"lightpink4", "7f4e52" },
      {"lightcyan4", "717d7d" },
      {"lightsalmon4", "7f462c" },
      {"lightyellow4", "827d6b" },
      {"lemonchiffon4", "827b60" },
      {"lightskyblue4", "566d7e" },
      {"lavenderblush4", "817679" },
      {"lightgoldenrod4", "817339" },
      {"lightsteelblue4", "646d7e" },
      {"firebrick", "800517" },
      {"lightpink", "faafba" },
      {"lightyellow", "fffedc" },
      {"lightgoldenrodyellow", "faf8cc" },
      {"lightcoral", "e77471" },
      {"khaki3", "c9be62" },
      {"wheat3", "c8b189"},
      {"violetred3", "c12869"},
      {"firebrick3", "c11b17" },
      {"lightblue3", "95b9c7" },
      {"lightpink3", "c48189" },
      {"lightcyan3", "afc7c7" },
      {"lightsalmon3", "c47451" },
      {"lightyellow3", "c9c7aa" },
      {"lemonchiffon3", "c9c299" },
      {"lightskyblue3", "87afc7" },
      {"lavenderblush3", "c8bbbe" },
      {"lightgoldenrod3", "c8b560" },
      {"lightsteelblue3", "9aadc7" },
      {"maroon4", "7d0552" },
      {"mistyrose4", "806f6c"},
      {"mediumorchid4", "6a287e" },
      {"mediumpurple4", "4e387e" },
      {"snow4", "817c7b"},
      {"salmon4", "7e3817"},
      {"sienna4", "7e3517"},
      {"skyblue4", "41627e"},
      {"seagreen4", "387c44"},
      {"seashell4", "817873"},
      {"slateblue4", "342d7e"},
      {"steelblue4", "2b547e"},
      {"slategray4", "616d7e"},
      {"springgreen4", "347c2c"},
      {"maroon3", "c12283" },
      {"magenta3", "c031c7" },
      {"mistyrose3", "c6afac"},
      {"mediumorchid3", "a74ac7" },
      {"mediumpurple3", "7a5dc7" },
      {"ivory4", "817d74" },
      {"snow", "fff9fa"},
      {"indianred4", "7e2217" },
      {"khaki", "ada96e" },
      {"seashell", "fef3eb"},
      {"navajowhite4", "806a4b"},
      {"hotpink4", "7d2252" },
      {"honeydew4", "7a7d74" },
      {"snow3", "c8c4c2"},
      {"salmon3", "c36241"},
      {"sienna3", "c35817"},
      {"skyblue3", "659ec7"},
      {"seagreen3", "54c571"},
      {"seashell3", "c8bfb6"},
      {"slateblue3", "574ec7"},
      {"steelblue3", "488ac7"},
      {"slategray3", "98afc7"},
      {"mintcream", "f5fff9"},
      {"springgreen3", "4cc552"},
      {"hotpink", "f660ab" },
      {"honeydew", "f0feee" },
      {"ivory3", "c9c7b9" },
      {"linen", "f9eee2" },
      {"forestgreen", "4e9258" },
      {"indianred3", "c24641" },
      {"lawngreen", "87f717" },
      {"limegreen", "41a317" },
      {"lightcyan", "e0ffff" },
      {"lightsalmon", "f9966b" },
      {"lemonchiffon", "fff8c6" },
      {"lightseagreen", "3ea99f" },
      {"yellow", "ffff00"},
      {"lavenderblush", "fdeef4" },
      {"navajowhite3", "c7aa7d"},
      {"hotpink3", "c25283" },
      {"honeydew3", "bcc7b9" },
      {"rosybrown4", "7f5a58"},
      {"royalblue4", "15317e"},
      {"lightslategray", "6d7b8d" },
      {"pink4", "7f525d"},
      {"plum4", "7e587e"},
      {"purple4", "461b7e"},
      {"peru", "c57726"},
      {"palegreen4", "4e7c41"},
      {"peachpuff4", "806752"},
      {"paleturquoise4", "5e7d7e"},
      {"palevioletred4", "7e354d"},
      {"pink", "faafbe"},
      {"gold4", "806517" },
      {"gray54", "807d7c" },
      {"gray44", "646060" },
      {"gray34", "4a4344" },
      {"gray24", "302226" },
      {"green4", "347c17" },
      {"gray94", "eeeeee" },
      {"gray84", "d2d2d1" },
      {"gray74", "b7b6b4" },
      {"gray64", "9a9998" },
      {"goldenrod4", "805817" },
      {"lavender", "e3e4fa" },
      {"gray56", "858381" },
      {"gray46", "696565" },
      {"gray36", "4e4848" },
      {"gray26", "34282c" },
      {"gray96", "f4f4f3" },
      {"gray86", "d7d7d7" },
      {"gray76", "bcbbba" },
      {"gray66", "a09f9d" },
      {"peachpuff", "fcd5b0"},
      {"maroon", "810541" },
      {"moccasin", "fde0ac"},
      {"mediumseagreen", "306754" },
      {"greenyellow", "b1fb17" },
      {"mediumforestgreen", "347235" },
      {"mediumspringgreen", "348017" },
      {"rosybrown3", "c5908e"},
      {"royalblue3", "2554c7"},
      {"gray59", "8d8b89" },
      {"gray49", "726e6d" },
      {"gray39", "565051" },
      {"gray29", "3e3535" },
      {"gray99", "fbfbfb" },
      {"gray89", "e0e0e0" },
      {"gray79", "c3c4c2" },
      {"gray69", "a9a8a6" },
      {"pink3", "c48793"},
      {"plum3", "c38ec7"},
      {"purple3", "6c2dc7"},
      {"gray55", "82807e" },
      {"gray45", "666362" },
      {"gray35", "4c4646" },
      {"gray25", "342826" },
      {"gray95", "f0f1f0" },
      {"gray85", "d5d5d4" },
      {"gray75", "b9b8b6" },
      {"gray65", "9e9c9b" },
      {"palegreen3", "7dc56c"},
      {"peachpuff3", "c6a688"},
      {"gray58", "8b8987" },
      {"gray48", "6e6a6b" },
      {"gray38", "544e4f" },
      {"gray28", "3b3131" },
      {"gray18", "250517" },
      {"gray98", "f9f9fa" },
      {"gray88", "dddddc" },
      {"gray78", "c1c1bf" },
      {"gray68", "a5a4a3" },
      {"paleturquoise3", "92c7c7"},
      {"palevioletred3", "c25a7c"},
      {"gray57", "878583" },
      {"gray47", "6d6968" },
      {"gray37", "504a4b" },
      {"gray27", "382d2c" },
      {"gray97", "f6f6f5" },
      {"gray87", "dbdbd9" },
      {"gray77", "bebebc" },
      {"gray67", "a3a2a0" },
      {"gold3", "c7a317" },
      {"gray53", "7c7979" },
      {"gray43", "625d5d" },
      {"gray23", "302217" },
      {"green3", "4cc417" },
      {"gray93", "ebebea" },
      {"gray83", "d0cfcf" },
      {"gray73", "b3b3b1" },
      {"gray63", "999795" },
      {"goldenrod3", "c68e17" },
      {"seagreen", "4e8975"},
      {"sandybrown", "ee9a4d"},
      {"springgreen", "4aa02c"},
      {"gray0", "150517" },
      {"gray60", "8f8e8d" },
      {"gray50", "747170" },
      {"gray40", "595454" },
      {"gray30", "413839" },
      {"gray90", "e2e3e1" },
      {"gray80", "c7c7c5" },
      {"gray70", "acaba9" },
      {"gray100", "ffffff" },
      {"plum", "b93b8f"},
      {"slategray", "657383"},
      {"deeppink4", "7d053f" },
      {"darkorange4", "7e3117" },
      {"dodgerblue4", "153e7e" },
      {"darkorchid4", "571b7e" },
      {"deepskyblue4", "25587e" },
      {"darkseagreen4", "617c58" },
      {"darkgoldenrod4", "7f5217" },
      {"darkslategray4", "4c7d7e" },
      {"darkolivegreen4", "667c26" },
      {"blue4", "151b7e" },
      {"ivory", "ffffee" },
      {"bisque4", "816e59" },
      {"burlywood4", "806341" },
      {"deeppink", "f52887" },
      {"violetred", "e9358a"},
      {"thistle4", "806d7e"},
      {"turquoise4", "30787e"},
      {"black", "000000" },
      {"violet", "8d38c9"},
      {"navy", "150567"},
      {"wheat", "f3daa9"},
      {"lightgoldenrod", "ecd872" },
      {"yellowgreen", "52d017"},
      {"deeppink3", "c12267" },
      {"darkorange3", "c35617" },
      {"dodgerblue3", "1569c7" },
      {"darkorchid3", "8b31c7" },
      {"deepskyblue3", "3090c7" },
      {"darkseagreen3", "99c68e" },
      {"darkgoldenrod3", "c58917" },
      {"darkslategray3", "78c7c7" },
      {"darkolivegreen3", "a0c544" },
      {"blue3", "1528c7" },
      {"brown3", "c22217" },
      {"bisque3", "c7af92" },
      {"burlywood3", "c6a06d" },
      {"tomato3", "c23e17"},
      {"thistle3", "c6aec7"},
      {"rosybrown", "b38481"},
      {"turquoise3", "43bfc7"},
      {"orange4", "7f4817"},
      {"orchid4", "7d387c"},
      {"palegreen", "79d867"},
      {"olivedrab4", "617c17"},
      {"orangered4", "7e0517"},
      {"green", "00ff00" },
      {"mediumorchid", "b048b5" },
      {"mediumgoldenrod", "ccb954" },
      {"mediumvioletred", "ca226b" },
      {"khaki1", "fff380" },
      {"azure4", "7a7d7d" },
      {"wheat1", "fee4b1"},
      {"violetred1", "f6358a"},
      {"gray", "736f6e" },
      {"aquamarine4", "417c64" },
      {"antiquewhite4", "817468" },
      {"cyan4", "307d7e" },
      {"coral4", "7e2817" },
      {"firebrick1", "f62817" },
      {"cornsilk4", "817a68" },
      {"cadetblue4", "4c787e" },
      {"chartreuse4", "437c17" },
      {"orange3", "c57717"},
      {"orchid3", "c160c3"},
      {"lightblue1", "bdedff" },
      {"lightpink1", "f9a7b0" },
      {"olivedrab3", "99c517"},
      {"orangered3", "c22817"},
      {"lightgoldenrod1", "ffe87c" },
      {"lightsteelblue1", "c6deff" },
      {"cornsilk", "fff7d7" },
      {"khaki2", "ede275" },
      {"wheat2", "ebd3a3"},
      {"violetred2", "e4317f"},
      {"papayawhip", "feeccf"},
      {"coral", "f76541" },
      {"firebrick2", "e42217" },
      {"indianred", "5e2217" },
      {"darkkhaki", "b7ad59" },
      {"lightblue2", "afdcec" },
      {"lightpink2", "e799a3" },
      {"lightcyan2", "cfecec" },
      {"azure3", "bcc7c7" },
      {"lightsalmon2", "e78a61" },
      {"lightyellow2", "edebcb" },
      {"lemonchiffon2", "ece5b6" },
      {"lightskyblue2", "a0cfec" },
      {"lavenderblush2", "ebdde2" },
      {"lightgoldenrod2", "ecd672" },
      {"lightsteelblue2", "b7ceec" },
      {"aquamarine3", "69c69f" },
      {"antiquewhite3", "c8b9a6" },
      {"white", "ffffff"},
      {"cyan3", "46c7c7" },
      {"coral3", "c34a2c" },
      {"cornsilk3", "c8c2a7" },
      {"cadetblue3", "77bfc7" },
      {"chartreuse3", "6cc417" },
      {"darkgreen", "254117" },
      {"darksalmon", "e18b6b" },
      {"darkseagreen", "8bb381" },
      {"floralwhite", "fff9ee" },
      {"darkolivegreen", "4a4117" },
      {"lightblue", "addfff" },
      {"brown", "980517" },
      {"lightskyblue", "82cafa" },
      {"tan", "d8af79"},
      {"lightslateblue", "736aff" },
      {"lightsteelblue", "728fce" },
      {"maroon1", "f535aa" },
      {"magenta1", "f43eff" },
      {"mediumorchid1", "d462ff" },
      {"mediumpurple1", "9e7bff" },
      {"dimgray", "463e41" },
      {"darkslategray", "25383c" },
      {"magenta", "ff00ff" },
      {"maroon2", "e3319d" },
      {"magenta2", "e238ec" },
      {"mistyrose2", "ead0cc"},
      {"red", "ff0000"},
      {"mediumorchid2", "c45aec" },
      {"mediumpurple2", "9172ec" },
      {"salmon1", "f88158"},
      {"sienna1", "f87431"},
      {"skyblue1", "82caff"},
      {"seagreen1", "6afb92"},
      {"slateblue1", "7369ff"},
      {"steelblue1", "5cb3ff"},
      {"slategray1", "c2dfff"},
      {"springgreen1", "5efb6e"},
      {"palegoldenrod", "ede49e"},
      {"palevioletred", "d16587"},
      {"sienna", "8a4117"},
      {"mistyrose", "fde1dd"},
      {"mediumblue", "152dc6" },
      {"indianred1", "f75d59" },
      {"mediumpurple", "8467d7" },
      {"midnightblue", "151b54"},
      {"gold", "d4a017" },
      {"mediumslateblue", "5e5a80" },
      {"mediumturquoise", "48cccd" },
      {"mediumaquamarine", "348781" },
      {"goldenrod", "edda74" },
      {"snow2", "ece7e6"},
      {"salmon2", "e67451"},
      {"sienna2", "e66c2c"},
      {"skyblue2", "79baec"},
      {"seagreen2", "64e986"},
      {"seashell2", "ebe2d9"},
      {"slateblue2", "6960ec"},
      {"steelblue2", "56a5ec"},
      {"slategray2", "b4cfec"},
      {"springgreen2", "57e964"},
      {"hotpink1", "f665ab" },
      {"ivory2", "ececdc" },
      {"indianred2", "e55451" },
      {"skyblue", "6698ff"},
      {"slateblue", "737ca1"},
      {"steelblue", "4863a0"},
      {"yellow1", "fffc17"},
      {"cyan", "00ffff" },
      {"navajowhite2", "eac995"},
      {"hotpink2", "e45e9d" },
      {"honeydew2", "deebdc" },
      {"red1", "f62217"},
      {"rosybrown1", "fbbbb9"},
      {"royalblue1", "306eff"},
      {"navajowhite", "fddaa3"},
      {"plum1", "f9b7ff"},
      {"purple1", "893bff"},
      {"gainsboro", "d8d9d7" },
      {"palegreen1", "a0fc8d"},
      {"darkorchid", "7d1b7e" },
      {"paleturquoise1", "bcfeff"},
      {"darkgoldenrod", "af7817" },
      {"gold1", "fdd017" },
      {"gray61", "939190" },
      {"gray51", "787473" },
      {"gray41", "5c5858" },
      {"gray31", "41383c" },
      {"gray21", "2b1b17" },
      {"green1", "5ffb17" },
      {"gray91", "e5e6e4" },
      {"gray81", "cacac9" },
      {"gray71", "aeadac" },
      {"red2", "e41b17"},
      {"goldenrod1", "fbb917" },
      {"blanchedalmond", "fee8c6" },
      {"darkviolet", "842dce" },
      {"rosybrown2", "e8adaa"},
      {"royalblue2", "2b65ec"},
      {"pink2", "e7a1b0"},
      {"plum2", "e6a9ec"},
      {"purple2", "7f38ec"},
      {"palegreen2", "94e981"},
      {"peachpuff2", "eac5a3"},
      {"blueviolet", "7931df" },
      {"paleturquoise2", "adebec"},
      {"palevioletred2", "e56e94"},
      {"gold2", "eac117" },
      {"gray52", "7a7777" },
      {"gray42", "5f5a59" },
      {"gray32", "463e3f" },
      {"green2", "59e817" },
      {"gray92", "e8e9e8" },
      {"gray82", "cccccb" },
      {"gray72", "b1b1af" },
      {"gray62", "959492" },
      {"royalblue", "2b60de"},
      {"goldenrod2", "e9ab17" },
      {"purple", "8e35ef"},
      {"powderblue", "addce3"},
      {"paleturquoise", "aeebec"},
      {"ghostwhite", "f7f7ff" },
      {"orchid", "e57ded"},
      {"orangered", "f63817"},
      {"olivedrab", "658017"},
      {"tomato", "f75431"},
      {"darkorange1", "f87217" },
      {"darkorchid1", "b041ff" },
      {"darkseagreen1", "c3fdb8" },
      {"darkgoldenrod1", "fbb117" },
      {"darkslategray1", "9afeff" },
      {"darkolivegreen1", "ccfb5d" },
      {"blue1", "1535ff" },
      {"brown1", "f63526" },
      {"tan1", "fa9b3c"},
      {"burlywood1", "fcce8e" },
      {"thistle1", "fcdfff"},
      {"turquoise1", "52f3ff"},
      {"deeppink2", "e4287c" },
      {"darkorange2", "e56717" },
      {"dodgerblue2", "157dec" },
      {"darkorchid2", "a23bec" },
      {"deepskyblue2", "38acec" },
      {"darkseagreen2", "b5eaaa" },
      {"darkgoldenrod2", "e8a317" },
      {"darkslategray2", "8eebec" },
      {"darkolivegreen2", "bce954" },
      {"blue2", "1531ec" },
      {"brown2", "e42d17" },
      {"bisque2", "ead0ae" },
      {"tan2", "e78e35"},
      {"burlywood2", "eabe83" },
      {"tomato2", "e54c2c"},
      {"thistle2", "e9cfec"},
      {"turquoise2", "4ee2ec"},
      {"darkorange", "f88017" },
      {"dodgerblue", "1589ff" },
      {"deepskyblue", "3bb9ff" },
      {"darkslateblue", "2b3856" },
      {"darkturquoise", "3b9c9c" },
      {"blue", "0000ff" },
      {"beige", "f5f3d7" },
      {"bisque", "fde0bc" },
      {"thistle", "d2b9d3"},
      {"turquoise", "43c6db"},
      {"orange1", "fa9b17"},
      {"orchid1", "f67dfa"},
      {"olivedrab1", "c3fb17"},
      {"orange2", "e78e17"},
      {"orchid2", "e473e7"},
      {"olivedrab2", "b5e917"},
      {"orangered2", "e43117"},
      {"aquamarine1", "87fdce" },
      {"antiquewhite1", "feedd6" },
      {"cyan1", "57feff" },
      {"cadetblue1", "99f3ff" },
      {"orange", "f87a17"},
      {"oldlace", "fcf3e2"},
      {"azure2", "deecec" },
      {"aquamarine2", "7deabe" },
      {"antiquewhite2", "ebdbc5" },
      {"cyan2", "50ebec" },
      {"coral2", "e55b3c" },
      {"cornsilk2", "ece5c6" },
      {"cadetblue2", "8ee2ec" },
      {"chartreuse2", "7fe817" },
      {"azure", "efffff" },
      {"aliceblue", "eff7ff" },
      {"aquamarine", "43b7ba" },
      {"antiquewhite", "9e8d2 "},
      {"cadetblue", "578693" },
      {"chocolate", "c85a17" },
      {"chartreuse", "8afb17" },
      {"cornflowerblue", "151b8d" }
	};

	static short lookup[] = {
		-1,    -1,    -1,    -1,    -1,    -1,    -1,
         -1,    -1,    -1,    -1,    -1,    -1,    -1,
         -1,    -1,    -1,    -1,    -1,    -1,    -1,
         -1,    -1,    -1,    -1,    -1,    -1,    -1,
         -1,    -1,    -1,    -1,    -1,    -1,    -1,
         -1,    -1,    -1,    -1,    -1,    -1,     0,
         -1,    -1,     1,     2,    -1,    -1,    -1,
         -1,    -1,    -1,    -1,    -1,    -1,    -1,
         -1,    -1,    -1,    -1,  -554,    -1,  -551,
       -549,    10,  -545,  -467,    -2,    -1,    13,
       -470,    -2,  -472,    -2,    14,  -475,    -3,
         -1,    -1,    15,    -1,    -1,    -1,    -1,
         -1,    -1,    -1,    -1,    16,    -1,    17,
         18,    -1,    -1,    19,    20,    -1,    -1,
         -1,    -1,    -1,    -1,    -1,    -1,    -1,
         21,    -1,    -1,    -1,    -1,  -602,    -1,
       -599,  -597,    29,  -595,  -448,    -2,  -451,
         -2,  -453,    -2,    32,  -456,    -3,    33,
         -1,    -1,  -608,  -444,    -2,    -1,    -1,
         -1,    -1,    -1,    -1,    -1,    -1,    -1,
         -1,    -1,    -1,    -1,    -1,    -1,    -1,
         -1,    -1,    -1,    -1,    -1,    -1,    -1,
         -1,    -1,    -1,    -1,    -1,    -1,    -1,
         -1,    -1,    36,    -1,  -665,    39,  -663,
       -655,    -1,    45,    -1,    46,    47,    -1,
         48,  -436,    -3,  -660,    51,    52,  -429,
         -2,    53,  -438,    -2,  -441,    -2,    -1,
         -1,    -1,    -1,    -1,    54,    -1,    -1,
         55,    -1,    -1,    -1,    -1,    -1,    -1,
         -1,    -1,    -1,    -1,    56,    57,    58,
         -1,    -1,    -1,    59,    -1,  -723,    62,
       -706,  -702,    68,    69,    -1,    70,  -413,
         -3,    -1,    71,  -415,    -2,    72,    73,
         74,    -1,    75,  -718,    -1,    79,    80,
         81,  -402,    -3,    -1,    82,    83,  -418,
         -2,    -1,    -1,    -1,    -1,    -1,    -1,
         -1,    -1,    -1,    -1,    -1,    84,    85,
         86,  -742,    -1,    89,  -391,    -2,  -756,
         -1,    92,    -1,    93,  -750,  -384,    -2,
         -1,  -754,  -382,    -2,  -388,    -2,    98,
         -1,    -1,    99,  -763,  -378,    -9,    -1,
        109,   110,    -1,    -1,    -1,    -1,  -778,
        119,    -1,   120,    -1,   121,  -367,    -8,
         -1,    -1,    -1,   122,    -1,   123,  -787,
       -354,    -2,  -790,  -352,    -2,  -833,    -1,
       -831,    -1,   138,  -824,    -1,  -800,  -331,
         -2,  -817,  -804,  -320,    -2,    -1,  -808,
       -318,    -8,    -1,   168,  -813,  -309,    -8,
         -1,   177,  -329,    -9,    -1,   178,    -1,
        179,   180,  -339,    -8,   181,  -829,   189,
       -296,    -7,  -342,    -2,  -350,    -8,    -1,
         -1,    -1,   190,    -1,    -1,    -1,    -1,
         -1,   191,    -1,   192,    -1,  -865,   196,
        197,  -857,   200,    -1,   201,   202,   203,
       -280,    -2,   204,   205,   206,   207,    -1,
        208,  -285,    -3,    -1,    -1,   209,    -1,
         -1,    -1,    -1,    -1,   210,   211,   212,
         -1,    -1,    -1,   213,    -1,   214,    -1,
         -1,    -1,    -1,    -1,    -1,    -1,    -1,
         -1,    -1,    -1,    -1,   215,    -1,  -915,
        219,   220,  -907,   223,    -1,   224,   225,
        226,  -257,    -2,   227,    -1,   228,   229,
        230,   231,  -262,    -3,    -1,    -1,    -1,
         -1,  -927,    -1,   234,  -925,  -243,    -2,
       -246,    -2,    -1,    -1,    -1,    -1,    -1,
         -1,    -1,   237,    -1,    -1,    -1,    -1,
         -1,    -1,    -1,   238,    -1,    -1,  -948,
       -239,    -2,    -1,    -1,    -1,   241,   242,
         -1,   243,   244,   245,   246,    -1,   247,
         -1,    -1,   248,   249,    -1,   250,   251,
        252,   253,  -985,  -980,    -1,  -975,  -220,
         -2,  -978,  -218,    -2,  -222,    -2,   262,
         -1,   263,  -224,    -2,   264,   265,   266,
         -1,    -1,    -1,    -1,   267,    -1,    -1,
         -1,   268, -1000,  -209,    -2,    -1, -1051,
        274, -1045, -1038,   279, -1016,   282,    -1,
        283,   284,    -1,   285,   286,  -198,    -2,
        287,   288,   289,   290,   291,    -1,   292,
        293,   294,    -1,   295,   296,    -1,   297,
        298, -1036,   301,   302,  -179,    -2,  -201,
         -2, -1043,    -1,   305,  -175,    -2,  -203,
         -2,    -1,   306,    -1,   307,  -207,    -3,
         -1,    -1,    -1,    -1,    -1,    -1,    -1,
         -1,    -1,    -1,    -1,    -1,   308,   309,
         -1,   310,    -1,   311, -1072,  -166,    -2,
         -1,    -1,    -1, -1085,   316,   317, -1083,
         -1,   321,  -160,    -3,  -164,    -2,    -1,
         -1,    -1, -1123,    -1,   324,   325,   326,
        327, -1104,    -1,   330, -1101,   333,  -147,
         -2,   334,  -150,    -2,   335,    -1, -1118,
        338, -1116, -1114,    -1,   344,  -137,    -3,
       -139,    -2,  -142,    -2,   345,    -1,   346,
       -156,    -2,    -1,   347,    -1,    -1,    -1,
         -1,    -1,    -1,    -1,   348,    -1, -1139,
        351,   352,  -129,    -2,    -1,    -1,    -1,
         -1,    -1,    -1,    -1,    -1,    -1,   353,
        354,   355,    -1,    -1,    -1,    -1,    -1,
         -1,    -1,    -1,    -1,    -1,    -1,    -1,
         -1,   356,    -1,    -1,    -1,    -1,    -1,
      -1173,  -121,    -2,   359,    -1,   360,    -1,
        361,   362,    -1,   363,    -1,    -1,   364,
        365,    -1,   366,    -1,    -1,    -1,    -1,
         -1,   367, -1206,    -1,   377,    -1,   378,
         -1,   379,   380, -1204,   -97,    -2,  -110,
         -9, -1211,    -1,   385,   -95,    -2, -1215,
        388,   -92,    -2, -1218,   -89,    -2,    -1,
         -1,    -1,    -1,    -1,   391, -1231,    -1,
        400,    -1,   401,   -86,    -8,    -1,    -1,
        402,    -1,    -1,    -1,   403,    -1,    -1,
        404,    -1,    -1,    -1,    -1,    -1,    -1,
         -1,    -1,    -1,    -1,    -1,    -1,    -1,
        405,    -1,   406,    -1,    -1,   407,    -1,
         -1,    -1,    -1,    -1,    -1,    -1,    -1,
         -1,    -1,    -1,   408,    -1,   409,    -1,
         -1,    -1,    -1,    -1, -1293,    -1,   412,
      -1289,   415,    -1,   416,   417,   -65,    -2,
        418,   419,   -68,    -2,   420,    -1,   421,
         -1,    -1,    -1,    -1,    -1,    -1,    -1,
         -1,    -1,    -1,    -1,    -1,   422,    -1,
      -1331,   426,   427, -1329,   430,    -1,   431,
        432,   433,    -1,   434,   435,    -1,   436,
        437,    -1,   438,   -50,    -2,   -55,    -3,
         -1,    -1,    -1,    -1, -1346,   441,    -1,
      -1341,   -36,    -2,   444,   445,   446,   -39,
         -2,    -1,    -1,    -1,   447,    -1,   448,
      -1355,   -29,    -2,   451,    -1,    -1,    -1,
         -1,    -1,    -1,    -1,    -1,    -1,    -1,
         -1,    -1,    -1,    -1,    -1,    -1,    -1,
         -1,    -1,    -1,    -1,    -1,    -1,    -1,
         -1,    -1,    -1, -1386,   -26,    -2, -1389,
        -24,    -2,    -1,   456,    -1,   457,    -1,
         -1,   458,    -1,    -1,    -1,    -1,   459,
         -1,    -1,    -1,    -1,    -1,    -1,    -1,
        460,   461,    -1,    -1,    -1,    -1,    -1,
         -1,   462,    -1,    -1,    -1,    -1,   463,
         -1,   464,    -1,    -1,   465,   466,    -1,
         -1,   467,   468,   469,    -1,    -1,    -1,
         -1,    -1,    -1,    -1,    -1,   470,    -1,
         -1,    -1,   471,   472,    -1,   473,    -1,
         -1,    -1,    -1,    -1,    -1,    -1, -1460,
        476,    -4,    -2,    -1,   477
	};

	if (len <= MAX_WORD_LENGTH && len >= MIN_WORD_LENGTH) {

		register int key = color_hash(str, len);

		if (key <= MAX_HASH_VALUE && key >= 0) {

			register int index = lookup[key];

			if (index >= 0) {

				register const char *s = wordlist[index].name;

				if (my_strcmp(str, s) == 0)
					return &wordlist[index];

			} else if (index < -TOTAL_KEYWORDS) {

				register int offset = -1 - TOTAL_KEYWORDS - index;
				register struct colors_t *wordptr = &wordlist[TOTAL_KEYWORDS + lookup[offset]];
				register struct colors_t *wordendptr = wordptr + -lookup[offset + 1];

				while (wordptr < wordendptr) {

					register const char *s = wordptr->name;

					if (my_strcmp(str, s) == 0)
						return wordptr;

					wordptr++;
				}
			}
		}
	}

	return NULL;
}

/*
 * get_hex_value() - returns the hex value for 
 * a color name (ie: azure2 = deecec) 
 */

char *color_get_hex_value(const char *rgb)
{
	struct colors_t *colors;
	char *lower;

	if (!rgb)
		return NULL;

	lower = str_to_lower(rgb);
	if(!lower)
		return NULL;

	colors = color_lookup(lower, strlen(lower));

	free(lower);

	if (!colors)
		return NULL;

	return strdup(colors->hex);
}

/*
 * hex_string_to_int() - converts a hex value (in a string)
 * to an integer
 */

__inline int hex_string_to_int(char *hex)
{
	int result;
	sscanf(hex, "%x", &result);
	return result;
}

gint32 get_color_value(const char *color)
{
	char *hex;
	gint32 result;

	if(!color)
		return -1;

	hex = *color == '#' ? strdup(++color) : color_get_hex_value(color);	
	if(!hex) 
		return 0;

	result = strtol(hex, NULL, 16);
	free(hex);

	return result;
}
