/*
 * File: dw_page.c
 *
 * Copyright (C) 1997 Raph Levien <raph@acm.org>
 * Copyright (C) 1999 Luca Rota <drake@freemail.it>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 */

/*
 * This module contains the Dw_page widget, which is the "back end" to
 * Web text widgets including html. 
 */

#define TRACE_GTK_METHODS
#undef NO_WINDOW
#define HUGE_WINDOW
#define DW_PAGE_MAX_WINDOW_SIZE 32760

#include <ctype.h>
#include <string.h>
#include <stdio.h>

#include <gtk/gtk.h>

#include "list.h"
#include "dw_page.h"


static void Dw_page_init(DwPage * page);
static void Dw_page_class_init(DwPageClass * klass);

static void Dw_page_destroy(GtkObject * object);

static void Dw_page_size_request(DwWidget * widget, DwRequisition * requisition);
static void Dw_page_size_allocate(DwWidget * widget, DwAllocation * allocation);
static void Dw_page_set_width(DwWidget * widget, gint32 width);
static void Dw_page_set_ascent(DwWidget * widget, gint32 ascent);
static void Dw_page_set_descent(DwWidget * widget, gint32 descent);
static void Dw_page_draw(DwWidget * page, DwRectangle * area, GdkEventExpose * event);
static void Dw_page_realize(DwWidget * widget);
static void Dw_page_unrealize(DwWidget * widget);
static gint Dw_page_button_press(DwWidget * widget, gint32 x, gint32 y, GdkEventButton * event);
static gint Dw_page_button_release(DwWidget * widget, gint32 x, gint32 y, GdkEventButton * event);
static gint Dw_page_motion_notify(DwWidget * widget, gint32 x, gint32 y, GdkEventMotion * event);


static void Dw_page_add(DwContainer * container, DwWidget * widget);
static void Dw_page_remove(DwContainer * container, DwWidget * widget);
static void Dw_page_forall(DwContainer * container, DwCallback callback, gpointer callback_data);

static void Dw_page_real_link_entered(DwPage * page, const char *url);

static void Dw_page_rewrap(DwPage * page);
static gint Dw_page_line_x_offset(DwPage * page, DwPageLine * line);

enum {
	LINK_ENTERED,
	LINK_PRESSED,
	LINK_RELEASED,
	LINK_CLICKED,
	LAST_SIGNAL
};

static DwContainerClass *parent_class;
static guint page_signals[LAST_SIGNAL] = { 0 };


/*
 * Return the type of DwPage
 */
GtkType a_Dw_page_get_type(void)
{
	static GtkType type = 0;

	if (!type) {
		GtkTypeInfo info = {
			"DwPage",
			sizeof(DwPage),
			sizeof(DwPageClass),
			(GtkClassInitFunc) Dw_page_class_init,
			(GtkObjectInitFunc) Dw_page_init,
			(GtkArgSetFunc) NULL,
			(GtkArgGetFunc) NULL,
		};

		type = gtk_type_unique(DW_TYPE_CONTAINER, &info);
	}

	return type;
}


/*
 * Create a new DwPage
 */
DwWidget *a_Dw_page_new(void)
{
	GtkObject *object;

	object = gtk_object_new(DW_TYPE_PAGE, NULL);
	return DW_WIDGET(object);
}


/*
 * Initialize a DwPage
 */
static void Dw_page_init(DwPage * page)
{
	page->num_lines_max = 16;
	page->num_lines = 0;
	page->lines = g_new(DwPageLine, page->num_lines_max);

	page->num_words_max = 512;
	page->num_words = 0;
	page->current_word = 0;
	page->words = g_new(DwPageWord, page->num_words_max);

	page->num_links_max = 16;
	page->num_links = 0;
	page->links = g_new(DwPageLink, page->num_links_max);

	page->last_line_max_width = 0;

	page->hover_link = -1;
	/* end stuff from gtk_page_init. */

	page->link_color    = 0x0000ff; /* blue */
	page->visited_color = 0x736f6e; /* gray */
	page->bgnd_color    = 0xffffff; /* white */

	page->x_click = -1;
	page->current_map = 0;
	page->num_shapes = 0;
	page->num_shapes_max = 8;	/* not a critical value */
	page->shapes = g_new(DwPageShape, page->num_shapes_max);

	/* random values */
	page->avail_width = 100;
	page->avail_ascent = 100;
	page->avail_descent = 0;

	page->wrapped_width = 100;	/* todo: ??? */
	page->wrapped_ascent = 100;
	page->wrapped_descent = 0;
}

/*
 * Initialize the DwPage's class
 */
static void Dw_page_class_init(DwPageClass * klass)
{
	GtkObjectClass *object_class;
	DwWidgetClass *widget_class;
	DwContainerClass *container_class;

	object_class = (GtkObjectClass *) klass;
	widget_class = (DwWidgetClass *) klass;
	container_class = (DwContainerClass *) klass;
	parent_class = gtk_type_class(DW_TYPE_CONTAINER);

	/* todo: the link_xxx signals should return boolean values, which are
	 * then returned by the event signal function of DwPage. But I don't
	 * know that much about Gtk+ signals :-(
	 * --SG 
	 */
	page_signals[LINK_ENTERED] = gtk_signal_new("link_entered", GTK_RUN_FIRST, object_class->type, GTK_SIGNAL_OFFSET(DwPageClass, link_entered), gtk_marshal_NONE__POINTER, GTK_TYPE_NONE, 1, GTK_TYPE_POINTER);
	page_signals[LINK_PRESSED] = gtk_signal_new("link_pressed", GTK_RUN_FIRST, object_class->type, GTK_SIGNAL_OFFSET(DwPageClass, link_pressed), gtk_marshal_NONE__POINTER_POINTER, GTK_TYPE_NONE, 2, GTK_TYPE_POINTER, GTK_TYPE_GDK_EVENT);
	page_signals[LINK_RELEASED] = gtk_signal_new("link_released", GTK_RUN_FIRST, object_class->type, GTK_SIGNAL_OFFSET(DwPageClass, link_released), gtk_marshal_NONE__POINTER_POINTER, GTK_TYPE_NONE, 2, GTK_TYPE_POINTER, GTK_TYPE_GDK_EVENT);
	page_signals[LINK_CLICKED] = gtk_signal_new("link_clicked", GTK_RUN_FIRST, object_class->type, GTK_SIGNAL_OFFSET(DwPageClass, link_clicked), gtk_marshal_NONE__POINTER_POINTER, GTK_TYPE_NONE, 2, GTK_TYPE_POINTER, GTK_TYPE_GDK_EVENT);
	gtk_object_class_add_signals(object_class, page_signals, LAST_SIGNAL);

	object_class->destroy = Dw_page_destroy;

	widget_class->size_request = Dw_page_size_request;
	widget_class->size_allocate = Dw_page_size_allocate;
	widget_class->set_width = Dw_page_set_width;
	widget_class->set_ascent = Dw_page_set_ascent;
	widget_class->set_descent = Dw_page_set_descent;
	widget_class->draw = Dw_page_draw;
	widget_class->realize = Dw_page_realize;
	widget_class->unrealize = Dw_page_unrealize;
	widget_class->button_press_event = Dw_page_button_press;
	widget_class->button_release_event = Dw_page_button_release;
	widget_class->motion_notify_event = Dw_page_motion_notify;

	container_class->add = Dw_page_add;
	container_class->remove = Dw_page_remove;
	container_class->forall = Dw_page_forall;

	klass->link_entered = Dw_page_real_link_entered;
	klass->link_pressed = NULL;
	klass->link_released = NULL;
	klass->link_clicked = NULL;
}


 /*
  * Destroy the page (standard Gtk+ function)
  */
static void Dw_page_destroy(GtkObject * object)
{
	DwPage *page;
	DwPageWord *word;
	gint word_index;
	gint link_index;

	page = DW_PAGE(object);

	for (word_index = 0; word_index < page->num_words; word_index++) {
		word = &page->words[word_index];
		if (word->content_type == DW_PAGE_CONTENT_WIDGET)
			gtk_object_destroy(GTK_OBJECT(word->content.widget.widget));
		else if (word->content_type == DW_PAGE_CONTENT_TEXT)
			g_free(word->content.text);
		else if (word->content_type == DW_PAGE_CONTENT_ANCHOR)
			g_free(word->content.anchor);

		a_Dw_style_unref(word->style);
	}

	g_free(page->lines);
	page->num_words = 0;	/* make sure we don't own widgets anymore */

	g_free(page->shapes);

	for (link_index = 0; link_index < page->num_links; link_index++) {
		g_free(page->links[link_index].url);
		g_free(page->links[link_index].alt);
	}
	g_free(page->links);

	(*(GTK_OBJECT_CLASS(parent_class)->destroy)) (object);
}


/*
 * Standard Dw function
 */
static void Dw_page_size_request(DwWidget * widget, DwRequisition * requisition)
{
	DwPage *page;
	DwPageWord *word;
	gboolean must_rewrap = FALSE;
	int word_index;

	page = DW_PAGE(widget);

	/* Rewrapping is only necessary, when the viewport size has changed,
	 * or a child widget needs resizing. In a third case, when new words have
	 * been added, there is no need to call Dw_page_rewrap.
	 *
	 * The code can be optimized: If in the first case the width did not
	 * change, and there are no child widgets with a relative height,
	 * rewrapping is neither necessary.
	 */

	if (page->wrapped_width != page->avail_width || page->wrapped_ascent != page->avail_ascent || page->wrapped_descent != page->avail_descent) {
		must_rewrap = TRUE;
		/*g_print(">>>    Dw_page_size_request: new size\n"); */
	} else {
		for (word_index = 0; !must_rewrap && word_index < page->num_words; word_index++) {
			word = &page->words[word_index];

			if (word->content_type == DW_PAGE_CONTENT_WIDGET && DW_WIDGET_NEEDS_RESIZE(word->content.widget.widget))
				must_rewrap = TRUE;
		}

		/*if (must_rewrap)
		   g_print(">>>    Dw_page_size_request: children\n"); */
	}

	if (must_rewrap) {
		Dw_page_rewrap(page);
		page->wrapped_width = page->avail_width;
		page->wrapped_ascent = page->avail_ascent;
		page->wrapped_descent = page->avail_descent;
	}

	requisition->width = page->real_width;

	if (page->num_lines > 0)
		requisition->ascent = page->lines[page->num_lines - 1].y_top + page->lines[page->num_lines - 1].y_ascent + page->lines[page->num_lines - 1].y_descent;
	else
		requisition->ascent = 0;

	/* todo: requisition->descent could get a useful value (as in TeX),
	   but I don't have any idea where this will really be needed ;-) */
	requisition->descent = 0;
}


/*
 * Standard Dw function
 */
static void Dw_page_size_allocate(DwWidget * widget, DwAllocation * allocation)
{
	DwPage *page;
	int line_index, word_index;
	DwPageLine *line;
	DwPageWord *word;
	int x_cursor;
	DwAllocation child_allocation;

	page = DW_PAGE(widget);

	//g_print ("Dw_page_size_allocate\n");

	for (line_index = 0; line_index < page->num_lines; line_index++) {
		line = &(page->lines[line_index]);
		x_cursor = Dw_page_line_x_offset(page, line);

		for (word_index = line->first_word; word_index < line->last_word; word_index++) {
			word = &(page->words[word_index]);

			if (word->content_type == DW_PAGE_CONTENT_WIDGET) {
				/*g_print ("   %20s: (%d, %d)\n",
				   gtk_type_name (GTK_OBJECT_TYPE
				   (word->content.widget.widget)),
				   x_cursor, line->y_top); */

				/* todo: justification within the line is done here! */
				child_allocation.x = x_cursor + widget->allocation.x;
				/* align=top:
				   child_allocation.y = line->y_top + widget->allocation.y;
				 */
				/* align=bottom (base line) */
				child_allocation.y = line->y_top + widget->allocation.y + (line->y_ascent - word->y_ascent);
				child_allocation.width = word->x_size;
				child_allocation.ascent = word->y_ascent;
				child_allocation.descent = word->y_descent;
				a_Dw_widget_size_allocate(word->content.widget.widget, &child_allocation);
			}

			x_cursor += (word->x_size + word->x_space);
		}
	}

	widget->allocation = *allocation;
}


/*
 * Standard Dw function
 */
static void Dw_page_set_width(DwWidget * widget, gint32 width)
{
	DwPage *page;

	page = DW_PAGE(widget);

	//g_print (">>> Dw_page_set_width: %d -> %d\n", page->avail_width, width);

	if (page->avail_width != width) {
		page->avail_width = width;
		Dw_widget_queue_resize(widget);
	}
}


/*
 * Standard Dw function
 */
static void Dw_page_set_ascent(DwWidget * widget, gint32 ascent)
{
	DwPage *page;

	page = DW_PAGE(widget);

	//g_print (">>> Dw_page_set_ascent: %d -> %d\n", page->avail_ascent, ascent);

	if (page->avail_ascent != ascent) {
		page->avail_ascent = ascent;
		Dw_widget_queue_resize(widget);
	}
}


/*
 * Standard Dw function
 */
static void Dw_page_set_descent(DwWidget * widget, gint32 descent)
{
	DwPage *page;

	page = DW_PAGE(widget);

	//g_print (">>> Dw_page_set_descent: %d -> %d\n", page->avail_descent,
	//descent);

	if (page->avail_descent != descent) {
		page->avail_descent = descent;
		Dw_widget_queue_resize(widget);
	}
}


/*
 * Standard Dw function
 */
static void Dw_page_realize(DwWidget * widget)
{
	DwPage *page = DW_PAGE(widget);

	page->hand_cursor = gdk_cursor_new(GDK_HAND2);
}


/*
 * Standard Dw function
 */
static void Dw_page_unrealize(DwWidget * widget)
{
	DwPage *page = DW_PAGE(widget);

	gdk_cursor_destroy(page->hand_cursor);
}


/*
 * Standard Dw function
 */
static void Dw_page_add(DwContainer * container, DwWidget * widget)
{
	/* todo */
}


/*
 * Standard Dw function
 */
static void Dw_page_remove(DwContainer * container, DwWidget * widget)
{
	/* todo */
}


/*
 * Standard Dw function
 */
static void Dw_page_forall(DwContainer * container, DwCallback callback, gpointer callback_data)
{
	DwPage *page;
	int word_index;
	DwPageWord *word;

	page = DW_PAGE(container);

	for (word_index = 0; word_index < page->num_words; word_index++) {
		word = &page->words[word_index];

		if (word->content_type == DW_PAGE_CONTENT_WIDGET)
			(*callback) (word->content.widget.widget, callback_data);
	}
}

static void Dw_page_real_link_entered(DwPage * page, const char *url)
{
	a_Dw_widget_set_cursor(DW_WIDGET(page), url ? page->hand_cursor : NULL);
}


/* todo:
 * 
 * + do request_repaint for added stuff. But what level of
 * granularity? reinstate the update_begin/update_end cycle? Do at
 * line granularity when making a new line (but risk paint
 * inconsistency)? Do at word granularity but take on region overhead
 * on every word (optimize away if outside visible rectangle)?
 * 
 * + better formatting, including justification & floats
 * 
 */


/*
 * Returns the x offset (the indentation plus any offset needed for
 * centering or right justification) for the line. The offset returned
 * is relative to the page (i.e. add allocation.x0 to get the dw
 * toplevel coordinate.
 */
static gint Dw_page_line_x_offset(DwPage * page, DwPageLine * line)
{
	gint x;

	if (line->first)
		x = page->words[line->first_word].style->left_indent_first;
	else
		x = page->words[line->first_word].style->left_indent_rest;
	return x;
}


/*
 * ?
 */
static void Dw_page_add_line(DwPage * page)
{
	gint num_lines;
	DwPageLine *line;

	num_lines = page->num_lines;
	a_List_add(page->lines, num_lines, sizeof(*page->lines), page->num_lines_max);
	line = &page->lines[num_lines];
	line->first_word = page->current_word;
	line->last_word = page->current_word;
	if (num_lines == 0) {
		line->y_top = 0;
	} else {
		line->y_top = page->lines[num_lines - 1].y_top + page->lines[num_lines - 1].y_ascent + page->lines[num_lines - 1].y_descent + page->lines[num_lines - 1].y_space;
	}
	line->x_size = 0;
	line->y_ascent = 6;
	line->y_descent = 3;
	line->y_space = 0;
	line->hard = FALSE;
	line->first = (num_lines == 0 || (page->lines[num_lines - 1].hard && page->lines[num_lines - 1].y_space > 0));
	page->num_lines++;
}

/*
 * Allocate a new word in a page structure. This routine is where word
 * wrapping happens. The newly allocated word has the x_size, x_space,
 * y_ascent, and y_descent fields filled in, but no others.
 */
static DwPageWord *Dw_page_new_word(DwPage * page, gint x_size, gint y_ascent, gint y_descent, DwStyle * style)
{
	DwPageLine *line;
	DwPageWord *word;
	DwStyle *style2;
	gboolean existing;
	gboolean new_line;
	gint width, set_width, nw, nl;

	line = NULL;
	word = NULL;
	new_line = FALSE;
	existing = FALSE;

	/* Is it an existing word? */
	if (x_size == -1)
		existing = TRUE;

	if (existing) {
		word = &page->words[page->current_word];
		style2 = word->style;
		x_size = word->x_size;
		y_ascent = word->y_ascent;
		y_descent = word->y_descent;
	} else {
		style2 = style;
	}

	nl = page->num_lines;
	if (nl == 0) {
		new_line = TRUE;
	} else {
		line = &page->lines[nl - 1];
		if (line->hard) {
			new_line = TRUE;
		} else {
			/* Calculate width of new line and see if it exceeds page width. */
			nw = line->last_word;
			if (nw > line->first_word) {
				set_width = page->avail_width - style2->right_indent;

				if (line->first)
					set_width -= style2->left_indent_first;
				else
					set_width -= style2->left_indent_rest;

				width = line->x_size + page->words[nw - 1].x_space + x_size;
				if (width > set_width)
					new_line = TRUE;
			}
		}

		if (new_line)
			page->real_width = MAX(page->real_width, line->x_size + Dw_page_line_x_offset(page, line));
	}

	if (new_line) {
		Dw_page_add_line(page);
		line = &page->lines[nl];
	}

	nw = line->last_word;

	/* Add a new word */
	if (!existing) {
		a_List_add(page->words, nw, sizeof(*page->words), page->num_words_max);
		word = &page->words[nw];
		word->x_size = x_size;
		word->x_space = 0;
		word->y_ascent = y_ascent;
		word->y_descent = y_descent;

		/* Just in case the page only has one word */
		if (page->num_words == 0)
			page->real_width = x_size;

		page->num_words++;
	}

	/* Apply the attributes to the line */
	if (nw - line->first_word)
		line->x_size += page->words[nw - 1].x_space;
	line->x_size += x_size;
	line->y_ascent = MAX(line->y_ascent, y_ascent);
	line->y_descent = MAX(line->y_descent, y_descent);

	/* update the width requisitions */
	if (nw == line->first_word && line->first)
		page->last_line_max_width = style2->left_indent_first;
	else if (nw == line->first_word)
		page->last_line_max_width += page->words[line->first_word].x_space;
	else
		page->last_line_max_width += page->words[nw - 1].x_space;
	page->last_line_max_width += x_size;

	page->current_word++;
	line->last_word++;

	return word;
}

/* Performance is not as good as it could be - it could only rewrap if
 * it needs to (should be a cheap test: the line is ok if it fits
 * within the page->width and either it's hard or the width plus the
 * width of the first word on the next line (plus the spacing of the
 * last word) exceed the page->width). This would save on memory
 * allocation too.
 * 
 * But hey, it works!
 */


static void Dw_page_calc_widget_size(DwPage * page, DwPageWord * word)
{
	DwRequisition requisition;
	gint req_height, alloc_height;

	a_Dw_widget_size_request(word->content.widget.widget, &requisition);

	if (word->content.widget.rel_width == -1)
		word->x_size = requisition.width;
	else
		word->x_size = word->content.widget.rel_width * page->avail_width;

	if (word->content.widget.rel_height == -1) {
		word->y_ascent = requisition.ascent;
		word->y_descent = requisition.descent;
	} else {
		/* todo: re-think about it */
		req_height = requisition.ascent + requisition.descent;
		if (req_height) {
			alloc_height = word->content.widget.rel_height * (page->avail_ascent + page->avail_descent);

			word->y_ascent = requisition.ascent * alloc_height / req_height;
			word->y_descent = requisition.descent * alloc_height / req_height;
		} else {
			word->y_ascent = 0;
			word->y_descent = 0;
		}
	}
}


/*
 * Rewrap the page.
 * There are basically two times we'll want to do this:
 * either when the viewport is resized, or when the size changes on one
 * of the child widgets.
 */
static void Dw_page_rewrap(DwPage * page)
{
	DwPageLine *old_lines;
	DwWidget *widget;
	gint old_num_lines;
	gint line_index, word_index;
	gint nl;
	DwPageLine *old_line;
	DwPageWord *word;

	widget = DW_WIDGET(page);

	old_lines = page->lines;
	old_num_lines = page->num_lines;

	//g_print(">>>       Dw_page_rewrap (%d x %d x %d)\n",
	//page->avail_width, page->avail_ascent, page->avail_descent);

	/* Initialize the lines structure of the page. */
	page->num_lines = 0;
	page->num_lines_max = 16;
	page->lines = g_new(DwPageLine, page->num_lines_max);

	page->current_word = 0;

	page->real_width = 0;

	/* Iterate over the old lines, adding the words to the page structure. */
	for (line_index = 0; line_index < old_num_lines; line_index++) {
		old_line = &(old_lines[line_index]);
		/* Here, we're actually relying on the invariant that a soft line
		 * is never followed by an empty hard line. */
		if (old_line->last_word == old_line->first_word && old_line->hard) {
			Dw_page_add_line(page);
		}
		for (word_index = old_line->first_word; word_index < old_line->last_word; word_index++) {
			word = &page->words[word_index];

			if (word->content_type == DW_PAGE_CONTENT_WIDGET)
				Dw_page_calc_widget_size(page, word);

			Dw_page_new_word(page, -1, -1, -1, NULL);

			if (word->content_type == DW_PAGE_CONTENT_ANCHOR) {
				Dw_widget_set_anchor(widget, word->content.anchor, page->lines[page->num_lines - 1].y_top);
			}
		}

		if (old_line->hard) {
			a_Dw_page_linebreak(page);
			page->lines[page->num_lines - 1].y_space = old_line->y_space;
		}
	}
	g_free(old_lines);

	/* The last line hasn't been compared to real_width yet */
	nl = page->num_lines;
	if (nl > 0)
		page->real_width = MAX(page->real_width, page->lines[nl - 1].x_size + Dw_page_line_x_offset(page, &(page->lines[nl - 1])));
}

/*
 * Paint a line
 * - x and y are toplevel dw coordinates (Question: what Dw? Changed. Test!)
 * - area is used always (ev. set it to event->area)
 * - event is only used when is_expose
 */
static void Dw_page_draw_line(DwPage * page, DwPageLine * line, DwRectangle * area, GdkEventExpose * event)
{
	DwWidget *widget;
	DwPageWord *word;
	gint word_index;
	gint x_cursor, y_cursor;
	gint diff;
	GdkColormap *colormap;
	gint uline_width;
	DwWidget *child;
	DwRectangle child_area;
	GdkWindow *window;

	/* Here's an idea on how to optimize this routine to minimize the number
	 * of calls to gdk_draw_string:
	 * 
	 * Copy the text from the words into a buffer, adding a new word
	 * only if: the attributes match, and the spacing is either zero or
	 * equal to the width of ' '. In the latter case, copy a " " into
	 * the buffer. Then draw the buffer. */

	widget = DW_WIDGET(page);
	window = widget->window;
	colormap = gtk_widget_get_colormap(widget->viewport);
	x_cursor = Dw_widget_x_world_to_viewport(widget, widget->allocation.x + Dw_page_line_x_offset(page, line));
	y_cursor = Dw_widget_y_world_to_viewport(widget, widget->allocation.y + line->y_top + line->y_ascent);

	for (word_index = line->first_word; word_index < line->last_word; word_index++) {
		word = &page->words[word_index];
		diff = 0;
		switch (word->content_type) {
		case DW_PAGE_CONTENT_TEXT:
			/* Adjust the text baseline if the word is <SUP>-ed or <SUB>-ed */
			if (word->style->SubSup == TEXT_SUB)
				diff = word->y_ascent / 2;
			else if (word->style->SubSup == TEXT_SUP)
				diff -= word->y_ascent / 3;

			gdk_draw_string(window, word->style->font->font, word->style->color->gc, x_cursor, y_cursor + diff, word->content.text);
			if (word->style->uline >= 0) {
				uline_width = word->x_size;
				if (word_index + 1 < line->last_word && word->style->uline == page->words[word_index + 1].style->uline)
					uline_width += word->x_space;
				gdk_draw_line(window, word->style->color->gc, x_cursor, y_cursor + 1 + diff, x_cursor + uline_width - 1, y_cursor + 1 + diff);
			}

			if (word->style->strike >= 0) {
				uline_width = word->x_size;
				if (word_index + 1 < line->last_word && word->style->strike == page->words[word_index + 1].style->strike)
					uline_width += word->x_space;
				gdk_draw_line(window, word->style->color->gc, x_cursor, y_cursor - word->y_ascent / 2 + diff, x_cursor + uline_width - 1, y_cursor - word->y_ascent / 2 + diff);
			}
			break;

		case DW_PAGE_CONTENT_WIDGET:
			child = word->content.widget.widget;
			if (Dw_widget_intersect(child, area, &child_area)) {
				//g_print ("Drawing widget %p\n", child);
				a_Dw_widget_draw(child, &child_area, event);
			}

			break;

		case DW_PAGE_CONTENT_ANCHOR:
			/* nothing - an anchor isn't seen */
			/* BUG: sometimes anchors have x_space; 
			 * we subtract that just in case --EG */
			x_cursor -= word->x_size + word->x_space;
			break;
		}
		x_cursor += word->x_size + word->x_space;
	}
}

/*
 * Find the first line index that includes y, relative to top of widget.
 */
static gint Dw_page_find_line_index(DwPage * page, gint y)
{
	gint max_index = page->num_lines - 1;
	gint step, index, low = 0;

	step = (page->num_lines + 1) >> 1;
	while (step > 1) {
		index = low + step;
		if (index <= max_index && page->lines[index].y_top < y)
			low = index;
		step = (step + 1) >> 1;
	}

	if (low < max_index && page->lines[low + 1].y_top < y)
		low++;

	/* This new routine returns the line number between
	 * (y_top) and (y_top + y_ascent + y_descent + y_space) : 
	 * the space _below_ the line is considered part of the line.
	 * Old routine returned line number between
	 * (y_top - previous_line->y_space) and (y_top + y_ascent + y_descent) :
	 * the space _above_ the line was considered part of the line.
	 * This is important for Dw_page_find_link()  --EG
	 */
	return low;
}

/*
 * Draw the actual lines, starting at (x, y) in toplevel Dw coords.
 * (former Dw_page_expose_lines)
 */
static void Dw_page_draw(DwWidget * widget, DwRectangle * area, GdkEventExpose * event)
{
	DwPage *page;
	gint line_index;
	DwPageLine *line;
	//gint x1;

	page = DW_PAGE(widget);
	line_index = Dw_page_find_line_index(page, area->y);

	for (; line_index < page->num_lines; line_index++) {
		line = &(page->lines[line_index]);
		if (line->y_top >= area->y + area->height)
			break;

		Dw_page_draw_line(page, line, area, event);
	}
}


/*
 * Find a link given a coordinate location relative to the window
 */
static gint Dw_page_find_link(DwPage * page, gint x, gint y)
{
	gint line_index, word_index;
	gint x_cursor, last_x_cursor;
	gint x1, y1;		/* coordinates relative to page */
	gint x0, y0;		/* coordinates relative to word */
	DwPageLine *line;
	DwPageWord *word;
	DwPageShape *shape;
	int dx, dy;
	int num_shape;
	guint map;

	/* x1 = x - DW_WIDGET(page)->allocation.x;
	 * y1 = y - DW_WIDGET(page)->allocation.y; */
	x1 = x;
	y1 = y;

	if ((line_index = Dw_page_find_line_index(page, y1)) >= page->num_lines)
		return -1;
	line = &page->lines[line_index];
	if (line->y_top + line->y_ascent + line->y_descent <= y1)
		return -1;

	x0 = x1;
	y0 = y1 - line->y_top;
	page->x_click = -1;

	x_cursor = Dw_page_line_x_offset(page, line);
	for (word_index = line->first_word; word_index < line->last_word; word_index++) {
		word = &page->words[word_index];
		last_x_cursor = x_cursor;
		x_cursor += word->x_size + word->x_space;
		if (last_x_cursor <= x1 && x_cursor > x1) {
			if (word->style->link >= 0) {
				/* todo: DW_STYLE_HAS_MAP will move elsewhere */
				if (word->style->flags & DW_STYLE_HAS_MAP) {
					page->x_click = x0;
					page->y_click = y0;
				} else
					page->x_click = -1;
				return word->style->link;

			} else if (word->style->flags & DW_STYLE_HAS_MAP) {
				map = word->style->map;
				for (num_shape = 0; num_shape < page->num_shapes; num_shape++) {
					shape = &(page->shapes[num_shape]);
					if (shape->map == map) {
						if (shape->type == DW_PAGE_SHAPE_CIRCLE) {
							dx = shape->data.circle.x - x0;
							dy = shape->data.circle.y - y0;
							if (shape->data.circle.r2 >= (dx * dx + dy * dy))
								return shape->link;
						} else if (shape->type == DW_PAGE_SHAPE_RECT) {
							if (x0 > shape->data.rect.left && x0 < shape->data.rect.right && y0 > shape->data.rect.top && y0 < shape->data.rect.bottom)
								return shape->link;
						} else if (shape->type == DW_PAGE_SHAPE_POLY) {
							if (gdk_region_point_in(shape->data.poly, x0, y0))
								return shape->link;
						}
					}
				}
			}
		}
	}
	return -1;
}


/*
 * Standard Dw function.
 */
static gint Dw_page_button_press(DwWidget * widget, gint32 x, gint32 y, GdkEventButton * event)
{
	DwPage *page = DW_PAGE(widget);

#ifdef VERBOSE
	g_print("Dw_page_button_press: button (%d, %d) +%d\n", x, y, button->button);
#endif

	page->link_pressed = Dw_page_find_link(page, x, y);
	if (page->link_pressed >= 0) {
		gtk_signal_emit(GTK_OBJECT(widget), page_signals[LINK_PRESSED], page->links[page->link_pressed].url, event);
		return TRUE;
	} else
		return FALSE;
}


/*
 * Standard Dw function.
 */
static gint Dw_page_button_release(DwWidget * widget, gint32 x, gint32 y, GdkEventButton * event)
{
	DwPage *page = DW_PAGE(widget);
	char full_url[1024];
	gint link_pressed, link_released; 
	/* gint i, j, changed; */

	link_pressed = page->link_pressed;
	link_released = Dw_page_find_link(page, x, y);
	page->link_pressed = -1;

	if (link_released >= 0) {

		gtk_signal_emit(GTK_OBJECT(widget), page_signals[LINK_RELEASED], page->links[link_released].url, event);

		if (link_pressed == link_released) {

			/*
			 * todo: How do we pass these datas to the server ?
			 * Url.c functions can't deal with that yet
			 */

			if (page->x_click > 0) {
				sprintf(full_url, "%s?%d,%d", page->links[link_pressed].url, page->x_click, page->y_click);
				gtk_signal_emit(GTK_OBJECT(widget), page_signals[LINK_CLICKED], full_url, event);
			} else
				gtk_signal_emit(GTK_OBJECT(widget), page_signals[LINK_CLICKED], page->links[link_released].url, event);

			/*
			 * This is only a workaround to visualize links opened in a
			 * new window. It will definitely change.
			 */

#if 0
			if (link_released >= 0)

				for (i = 0; i < page->num_lines; i++) {

					changed = FALSE;

					for (j = page->lines[i].first_word; j < page->lines[i].last_word; j++)

						if (page->words[j].style->link == link_released) {

							DwStyle style_attr = *(page->words[j].style);

							style_attr.color = a_Dw_style_color_new(page->visited_color, widget->window);
							a_Dw_style_unref(page->words[j].style);
							page->words[j].style = a_Dw_style_new(&style_attr, widget->window);
							changed = TRUE;

						}

					if (changed)
						Dw_widget_queue_draw_area(widget, 0, page->lines[i].y_top, widget->allocation.width, page->lines[i].y_ascent + page->lines[i].y_descent);
				}
#endif
			/* end workaround */
		}

		page->x_click = -1;
		return TRUE;
	} else
		return FALSE;
}


/*
 * Standard Dw function.
 */
static gint Dw_page_motion_notify(DwWidget * widget, gint32 x, gint32 y, GdkEventMotion * event)
{
	DwPage *page = DW_PAGE(widget);
	gint hover_link;
	char full_url[1024];

	hover_link = Dw_page_find_link(page, x, y);

	if (hover_link >= 0 && page->x_click >= 0) {
		sprintf(full_url, "%s?%d,%d", page->links[hover_link].url, page->x_click, page->y_click);
		gtk_signal_emit(GTK_OBJECT(widget), page_signals[LINK_ENTERED], full_url);
	}

	if (page->hover_link != hover_link) {
		if (hover_link >= 0) {
			/* If page->x_click >= 0, the signal has already been emitted. */
			if (page->x_click < 0) {
				if (page->links[hover_link].alt)
					gtk_signal_emit(GTK_OBJECT(widget), page_signals[LINK_ENTERED], page->links[hover_link].alt);
				else
					gtk_signal_emit(GTK_OBJECT(widget), page_signals[LINK_ENTERED], page->links[hover_link].url);
			}
		} else
			gtk_signal_emit(GTK_OBJECT(widget), page_signals[LINK_ENTERED], NULL);

		page->hover_link = hover_link;
		return TRUE;
	} else
		return FALSE;
}


/*
 * Create a new link, return the index.
 */
gint a_Dw_page_new_link(DwPage * page, const char *url, const char *alt)
{
	gint nl;

	nl = page->num_links;
	a_List_add(page->links, nl, sizeof(*page->links), page->num_links_max);
	page->links[nl].url = url ? g_strdup(url) : NULL;
	page->links[nl].alt = alt ? g_strdup(alt) : NULL;
	return page->num_links++;
}


/*
 * Add a word to the page structure. Stashes the argument pointer in
 * the page data structure so that it will be deallocated on destroy.
 */
void a_Dw_page_add_text(DwPage * page, char *text, DwStyle * style)
{
	DwPageWord *word;
	gint x_size, y_ascent, y_descent;

	/* todo: ???
	   if (attr >= page->num_attrs || page->attrs[attr].font >= page->num_fonts)
	   return;                   / *BUG: Should use default text! * /
	 */
	x_size = gdk_string_width(style->font->font, text);
	y_ascent = style->font->font->ascent;
	y_descent = style->font->font->descent;

	/* In case of a sub or super script we increase the word's height and
	 * potentially the line's height. 
	 */
	if (style->SubSup == TEXT_SUB)
		y_descent += (y_ascent / 2);
	else if (style->SubSup == TEXT_SUP)
		y_ascent += (y_ascent / 3);

	word = Dw_page_new_word(page, x_size, y_ascent, y_descent, style);
	word->content_type = DW_PAGE_CONTENT_TEXT;
	word->content.text = text;
	word->style = style;
	a_Dw_style_ref(style);
}

/*
 * Add a widget (word type) to the page. todo: comments.
 */
void a_Dw_page_add_widget(DwPage * page, DwWidget * widget, gfloat rel_width, gfloat rel_height, DwStyle * style)
{
	DwPageWord st_word, *word;
	gint x_size, y_ascent, y_descent;

	/* First, find the child widget's width. For now, we always
	 * give the minimum requisition, but that will change with table
	 * support (tables should probably get 100% of the set width - the
	 * margins, or the maximum requisition, whichever is smaller). */

	/* todo: a bit kludgy */
	st_word.content.widget.rel_width = rel_width;
	st_word.content.widget.rel_height = rel_height;
	st_word.content.widget.widget = widget;
	Dw_page_calc_widget_size(page, &st_word);
	x_size = st_word.x_size;
	y_ascent = st_word.y_ascent;
	y_descent = st_word.y_descent;

	word = Dw_page_new_word(page, x_size, y_ascent, y_descent, style);

	word->content_type = DW_PAGE_CONTENT_WIDGET;
	word->content.widget.widget = widget;
	word->content.widget.rel_width = rel_width;
	word->content.widget.rel_height = rel_height;
	word->style = style;
	a_Dw_style_ref(style);

	Dw_widget_set_parent(widget, DW_WIDGET(page));
	a_Dw_widget_set_style(widget, style);
}


/*
 * Add an anchor to the page. name is copied, so no strdup is neccessary for
 * the caller.
 */
void a_Dw_page_add_anchor(DwPage * page, char *name, DwStyle * style)
{
	DwPageWord *word;

	word = Dw_page_new_word(page, 0, 0, 0, style);
	word->content_type = DW_PAGE_CONTENT_ANCHOR;
	word->content.anchor = g_strdup(name);
	word->style = style;
	a_Dw_style_ref(style);

	Dw_widget_set_anchor(DW_WIDGET(page), word->content.anchor, page->lines[page->num_lines - 1].y_top);
}


/*
 * ?
 */
void a_Dw_page_add_space(DwPage * page, DwStyle * style)
{
	gint nl, nw;
	gint space;

	nl = page->num_lines - 1;
	if (nl >= 0) {
		nw = page->num_words - 1;
		if (nw >= 0) {
			space = style->font->space_width;
			page->words[nw].x_space = space;
		}
	}
}

/*
 * Cause a line break
 */
void a_Dw_page_linebreak(DwPage * page)
{
	gint i;

	i = page->num_lines;
	if (i == 0)
		return;
	page->lines[i - 1].hard = TRUE;
}

/*
 * Cause a paragraph break
 */
void a_Dw_page_parbreak(DwPage * page, gint space)
{
	gint i;

	i = page->num_lines;
	if (i == 0)
		return;
	page->lines[i - 1].hard = TRUE;
	if (space > page->lines[i - 1].y_space)
		page->lines[i - 1].y_space = space;
}

/*
 * Call this routine before updating the state. By wrapping all state
 * changes between these two routines, the internal state of the widget
 * always matches the display, without requiring each state change
 * operation to calculate updates on a fine-grained basis.
 */
void a_Dw_page_update_begin(DwPage * page)
{
}


/*
 * Call this routine after updating the state.
 */
void a_Dw_page_update_end(DwPage * page)
{
	Dw_widget_queue_resize(DW_WIDGET(page));
}




typedef struct {
	gint LineNum;
	gint Index;
} LineData;

typedef enum {
	F_NewKey,
	F_Read,
	F_Seek,
	F_Found,
	F_End
} FindState;

typedef struct {
	gchar *Key;
	guint KeyLen;
	DwPage *Page;
	gint LineNum;
	gint Matches;
	GSList *IList;
	GString *TextBuf;
	gchar StopSet[3];
	gint Eof;
	FindState State;
} FindData;

/*
 * Read lines into local buffer
 */
void Dw_page_read_lines(FindData * F, gint N)
{
	gint i, j, nw;
	LineData *LNode;

	for (j = 0; F->LineNum < F->Page->num_lines && j < N; F->LineNum++, j++) {
		LNode = g_new(LineData, 1);
		LNode->LineNum = F->LineNum;
		LNode->Index = F->TextBuf->len;
		F->IList = g_slist_append(F->IList, LNode);
		/* This can find strings that span over two or more lines */
		for (nw = i = F->Page->lines[F->LineNum].first_word; i < F->Page->lines[F->LineNum].last_word; i++) {
			if (F->Page->words[i].content_type != DW_PAGE_CONTENT_TEXT)
				continue;
			F->TextBuf = g_string_append(F->TextBuf, F->Page->words[i].content.text);
			F->TextBuf = g_string_append(F->TextBuf, " ");
			++nw;
		}
		if (nw == 0)	/* Empty line ? */
			F->IList = g_slist_remove(F->IList, LNode);
	}
	F->Eof = (F->LineNum == F->Page->num_lines);
}

/*
 * Erase the first N characters from local buffer
 */
void Dw_page_erase_chars(FindData * F, gint N)
{
	gint i;
	LineData *LNode, *LNext;

	for (i = 0; (LNode = g_slist_nth_data(F->IList, i)); i++)
		LNode->Index -= N;

	while ((LNode = g_slist_nth_data(F->IList, 0)) && LNode->Index <= 0) {
		LNode->Index = 0;
		if ((LNext = g_slist_nth_data(F->IList, 1)) && LNext->Index <= 0)
			F->IList = g_slist_remove(F->IList, LNode);
		else
			break;
	}
	F->TextBuf = g_string_erase(F->TextBuf, 0, N);
}

/*
 * Erase the local buffer
 */
void Dw_page_erase_buf(FindData * F)
{
	LineData *LNode;

	if (F->IList)
		while ((LNode = g_slist_nth_data(F->IList, 0)))
			F->IList = g_slist_remove(F->IList, LNode);

	if (F->TextBuf)
		F->TextBuf = g_string_truncate(F->TextBuf, 0);
}

/*
 * Find the text in the page.
 */
void a_Dw_page_find_text(DwPage * page, char *NewKey)
{
	gint i;
	gchar *Ptr = NULL;
	LineData *LNode;
	static FindData *F = NULL;
	DwWidget *widget = DW_WIDGET(page);

	g_return_if_fail(widget->viewport != NULL);

	if (!F)
		F = g_new0(FindData, 1);
	if (!F->Key || strcmp(F->Key, NewKey) || F->Page != page)
		F->State = F_NewKey;

	/* Let the FSM find the search string */
	while (1) {
		switch (F->State) {
		case F_NewKey:
			if (!F->TextBuf)
				F->TextBuf = g_string_sized_new(256);
			if (F->Key)
				g_free(F->Key);
			F->Key = g_strdup(NewKey);
			if (!(F->KeyLen = strlen(F->Key)))
				return;
			sprintf(F->StopSet, "%c%c", tolower(F->Key[0]), toupper(F->Key[0]));
			Dw_page_erase_buf(F);
			F->Page = page;
			F->Matches = F->LineNum = F->Eof = 0;
			F->State = F_Read;
			break;
		case F_Read:
			if (F->Eof)
				F->State = F_End;
			else {
				Dw_page_read_lines(F, 10);
				F->State = F_Seek;
			}
			// g_print("TextBuf: %s\n", F->TextBuf->str);
			break;
		case F_Seek:
			if (!(Ptr = strpbrk(F->TextBuf->str, F->StopSet))) {
				Dw_page_erase_buf(F);
				F->State = F_Read;
			} else if (strlen(Ptr) < F->KeyLen) {
				F->State = F_Read;
			} else if (g_strncasecmp(Ptr, F->Key, F->KeyLen) == 0) {
				Dw_page_erase_chars(F, Ptr - F->TextBuf->str);
				F->State = F_Found;
			} else
				Dw_page_erase_chars(F, Ptr - F->TextBuf->str + 1);
			break;
		case F_Found:
			F->Matches++;
			LNode = g_slist_nth_data(F->IList, 0);
			i = LNode->LineNum;
			a_Dw_widget_scroll_to(widget, page->lines[i].y_top);
			Dw_page_erase_chars(F, 1);
			F->State = F_Seek;
			return;
		case F_End:
			if (F->Matches)
				F->State = F_NewKey;
			else
				return;
		}
	}
}
