/*******************************************************************************
 *
 * document.c - routines for loading html documents
 *                   
 * Cheetah Web Browser
 * Copyright (C) 2001 Garett Spencley 
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA. 
 *
 *******************************************************************************/

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>

#include <pthread.h>

#include "http.h"
#include "file.h"
#include "error.h"
#include "image.h"
#include "debug.h"
#include "img_jpeg.h"
#include "img_png.h"
#include "img_gif.h"

/* FIXME: This should be more generic and be in something like mime.c and use a hash
 * table or something */

enum { IMG_JPEG, IMG_GIF, IMG_PNG, FILE_HTML, UNKOWN };

unsigned int decode_content_type(const char *content_type) 
{
	static const char *types[] = {
		"image/jpeg", "image/gif", "image/png", "text/html", NULL
	};

	int i;
	char content[256], *p;

	p = content;

	while(*content_type &&  *content_type != ';') 
		*p++ = *content_type++;

	*p = 0;

	for(i = 0; types[i]; i++) 
		if(strcmp(content, types[i]) == 0)
			break;	

	return i;
}

char *get_type_based_on_ext(const char *file_name) 
{
	/* FIXME: TEMPORARY AND CRAPPY!!!! */
		
	static char *ext[] = {
		"jpg", "gif", "png", "html", NULL
	};

	static char *types[] = {
		"image/jpeg", "image/gif", "image/png", "text/html", NULL
	};

	int i;

	char *p;

	p = strrchr(file_name, '.');
	if(!p)
		return NULL;

	for(i = 0; ext[i]; i++) 
		if(strcasecmp(p, ext[i]) == 0)
			break;

	return types[i];
}

/*
 * load_file() - loads a local file
 */

char *load_file(char *path)
{
	FILE *infile;
	char *result;
	int counter;
	struct stat fileStat;

	if(!path) {
		debug_print("load_file got NULL path");
		return NULL;
	}

	if(stat(path, &fileStat) < 0) {
		error("Unable to stat %s", path);
		return NULL;
	}

	result = (char *)malloc(fileStat.st_size);
	if(!result) {
		error("Out of memory.");
		return NULL;
	}

	infile = fopen(path, "r");
	if(!infile) {
		error("Unable to open %s", path);
		free(result);
		return NULL;
	}

	counter = fread(result, 1, fileStat.st_size, infile);
	if(counter != fileStat.st_size) {
		error("Unable to read file %s", path);
		free(result);
		return NULL;
	}

	/* NOTE: This will cause some bugs (and mask others). Note that not
	 * all files may be treated as text. What if a file to be loaded is
	 * an image? NULLs at the end of the file may screw things up! Code
	 * that uses this function should not rely on having a NULL terminated
	 * string as a 'file'. */
	result[counter] = 0;

	fclose(infile);
	return result;
}

/*
 * http_get_file() - retrieves a file through http
 * and loads it into a buffer
 */

char *http_get_file(uri_t *uri, GuiMessage *msg)
{
	char *buf;
	int result;
	HttpFile *file;

	if(!uri)
		return NULL;

	result = http_connect(uri->host, uri->port, msg);
	if(result < 0) {

		http_perror(uri->host, msg->text);

		/* remember this is running multi-threaded!
		   don't feel tempted to do msg->pop=++msg->seq ,
		   which is likely to screw up */
		msg->pop = msg->seq + 1;
		msg->seq++;

		return NULL; 
	}

	file = http_get(uri->abs_path, msg);
	if(!file) {
		http_perror(uri->abs_path, msg->text);
		msg->pop = msg->seq + 1;
		msg->seq++;
		return NULL;
	}

	http_close();

	/* Very cheap hack to NULL terminate html files (it prevents crashes) */

	if(decode_content_type(file->content_type) == FILE_HTML)
		file->data[file->bytes] = 0;

	buf = strdup(file->data);
	http_file_free(file);

	return buf;
}

/*
 * image_decode_jpg() - decodes jpeg image stored in buf into image
 */

__inline static void image_decode_jpeg(ImageData *image, void *buf, size_t bytes)
{
	JpegImage *jpeg;

	jpeg = Jpeg_new(image);
	Jpeg_write(jpeg, buf, bytes);
	Jpeg_close(jpeg);
}

/*
 * image_decode_gif() - decodes gif image stored in buf into image
 */

__inline static void image_decode_gif(ImageData *image, void *buf, size_t bytes)
{
	GifImage *gif;

	gif = Gif_new(image);
	Gif_write(gif, buf, bytes);
	Gif_close(gif);
}

/*
 * image_decode_png() - decodes gif image stored in buf into image
 */

__inline static void image_decode_png(ImageData *image, void *buf, size_t bytes)
{	
	PngImage *png;

	png = Png_new(image);
	Png_write(png, buf, bytes);
	Png_close(png);
}


void decode_image(HttpFile *file, ImageData *image)
{
	unsigned int type;

	type = decode_content_type(file->content_type);
	
	switch(type) {

	case IMG_JPEG:
		image_decode_jpeg(image, file->data, file->bytes);
		break;

	case IMG_GIF:
		image_decode_gif(image, file->data, file->bytes);
		break;

	case IMG_PNG:
		image_decode_png(image, file->data, file->bytes);
		break;
		
	default:
		fprintf(stderr, "Unsupported image type encountered.\n");
		break;
	}
}

int http_get_image(uri_t *uri, ImageData *image, GuiMessage *msg)
{
	HttpFile *file;
	int result;
	
	if(!uri || !image || !msg) {
		debug_print("http_get_image: received NULL args");
		return -1;
	}

	result = http_connect(uri->host, uri->port, msg);
	if(result < 0) {
		debug_print("Connection failed");
		http_perror(uri->host, NULL);

		msg->pop = msg->seq + 1;
		msg->seq++;
		return -1;
	}

	file = http_get(uri->abs_path, msg);
	if(!file) {
		debug_print("GET failed");
		http_perror("Unable to retrieve file", NULL);

		msg->pop = msg->seq + 1;
		msg->seq++;
		return -1;
	}
					
	if(!file->content_type) {
		file->content_type = get_type_based_on_ext(uri->abs_path);

		if(!file->content_type) {
			http_close();
			http_file_free(file);
			return -1;
		}
	}	

	debug_print("Content-Length: %d", file->bytes);
	debug_print("Content-Type: %s", file->content_type);

	decode_image(file, image);

	http_close();
	http_file_free(file);

	return 0;
}
