/*******************************************************************************
 *
 * htmltag.c - HTML parsing engine. 
 *                   
 * These routines parse the the individual html tags and store the 
 * attributes into a linked list.
 * 
 * Cheetah Web Browser
 * Copyright (C) 2001 Garett Spencley 
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA. 
 *
 *******************************************************************************/

#include <stdlib.h>
#include <string.h>
#include <ctype.h>

#include "error.h"
#include "htmltag.h"
#include "debug.h"

/*
 * create_tag_args() 
 *
 * creates a linked list of arguments contained
 * within tag
 */

static __inline html_tag_args *add_node(html_tag_args *args, const char *name, const char *value)
{
	html_tag_args *tmp = args;

	if(!args) {
		args = (html_tag_args *)malloc(sizeof(html_tag_args));
		if(!args)
			return NULL;

		args->name  = strdup(name);
		args->value = strdup(value);
		args->next  = NULL;

		return args;
	}

	while(args->next) 
		args = args->next;

	args->next = (html_tag_args *)malloc(sizeof(html_tag_args));
	if(!args->next)
		return NULL;

	args = args->next;

	args->name = strdup(name);
	args->value = strdup(value);

	args->next = NULL;

	return tmp;
}

html_tag_args *create_tag_args(const char *string)
{
	char *p;
	char name[256], value[256];
	html_tag_args *result = NULL;

	p = name;

	while(1) {
		
		/* Skip spaces */

		while(*string && isspace(*string))
			++string;

		if(*string == '=') {

			/* Skip over '=' */
			++string;

			*p = 0;
			p = value;

			if(*string == '"') {

				++string; /* skip open '"' */

				while(*string && *string != '"') 
					*p++ = *string++;
				
				++string; /* skip close '"' */

			} else  {

				while(*string && !isspace(*string)) 
					*p++ = *string++;
			}

			*p = 0;

			result = add_node(result, name, value);
			if(!result) {
				error("Out of memory.");
				return NULL;
			}

			p = name;

		} else if(isspace(*string) || *string == 0) {

			/* Attribute without a value (like 'selected' or 'checked') */

			*p = 0;

			result = add_node(result, name, "");
			if(!result) {
				error("Out of memory.");
				return NULL;
			}

			if(*string == 0) break;

			++string; 

		} else
			*p++ = *string++;
	}

	return result;
}

/*
 * free_tag_args() 
 *
 * frees an html_tag_args list
 */

void free_html_tag_args(html_tag_args *args)
{
	html_tag_args *current;
	html_tag_args *next;

	current = args;

	while(current) {

		free(current->name);
		free(current->value);

		next = current->next;

		free(current);

		current = next;
	}
}
