/*******************************************************************************
 *
 * img_gif.h 
 *                   
 * Original code from dillo http://dillo.sourceforge.net
 *
 * Hacked by Garett Spencley for Cheetah Web Browser
 *
 *  Adapted by Raph Levien from giftopnm.c as found in the
 *  netpbm-1mar1994 release. The copyright notice for giftopnm.c
 *  follows:
 */

/* +-------------------------------------------------------------------+ */
/* | Copyright 1990, 1991, 1993, David Koblas.  (koblas@netcom.com)    | */
/* |   Permission to use, copy, modify, and distribute this software   | */
/* |   and its documentation for any purpose and without fee is hereby | */
/* |   granted, provided that the above copyright notice appear in all | */
/* |   copies and that both that copyright notice and this permission  | */
/* |   notice appear in supporting documentation.  This software is    | */
/* |   provided "as is" without express or implied warranty.           | */
/* +-------------------------------------------------------------------+ */

/*
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA. 
 *
 *******************************************************************************/

#include <stdio.h>
#include <string.h>

#include <gtk/gtk.h>

#include "debug.h"
#include "img_gif.h"
#include "image.h"

static size_t Gif_process_bytes(GifImage * gif, const guchar * buf, size_t bufsize, void *Buf);

/*
 * MIME handler for "image/gif" type
 * (Sets Gif_callback as cache-client)
 */

DwWidget *a_Gif_image(const char *Type, ImageData * Image, void **Data)
{
	if (!Image)
		Image = Image_new(0, 0, NULL, DW_PAINT_DEFAULT_BGND);

	*Data = Image;

	return DW_WIDGET(Image->dw);
}

/*
 * Create a new gif structure for decoding a gif into a RGB buffer
 */

GifImage *Gif_new(ImageData * Image)
{
	GifImage *gif = g_malloc(sizeof(GifImage));

	gif->Image = Image;

	gif->Flags = 0;
	gif->state = 0;
	gif->Start_Ofs = 0;
	gif->linebuf = NULL;
	gif->Background = -1;
	gif->transparent = -1;
	gif->num_spill_lines_max = 0;
	gif->spill_lines = NULL;
	gif->window = 0;
	gif->packet_size = 0;

	return gif;
}

/*
 * Receive and process new chunks of GIF image data
 */

void Gif_write(GifImage *gif, void *Buf, gint BufSize)
{
	guchar *buf;
	size_t bufsize, bytes_consumed;

	/* Sanity checks */
	if (!Buf || !gif->Image || BufSize <= 0)
		return;

	buf = ((guchar *) Buf) + gif->Start_Ofs;
	bufsize = BufSize - gif->Start_Ofs;

	debug_print("Gif_write: %d bytes", BufSize);

	/* Process the bytes in the input buffer. */
	bytes_consumed = Gif_process_bytes(gif, buf, bufsize, Buf);

	if (bytes_consumed < 1)
		return;

	gif->Start_Ofs += bytes_consumed;

	debug_print("exit Gif_write, bufsize=%d", bufsize);
}

/*
 * Finish the decoding process (and free the memory)
 */

void Gif_close(GifImage * gif)
{
	gint i;

	debug_print("destroy gif %x\n", gif);

	if (gif->linebuf != NULL)
		g_free(gif->linebuf);

	if (gif->spill_lines != NULL) {

		for (i = 0; i < gif->num_spill_lines_max; i++)
			g_free(gif->spill_lines[i]);

		g_free(gif->spill_lines);
	}

	g_free(gif);
}


/* --- GIF Extensions --------------------------------------------------------- */

/*
 * This reads a sequence of GIF data blocks.. and ignores them!
 * Buf points to the first data block.
 *
 * Return Value
 * 0 = There wasn't enough bytes read yet to read the whole datablock
 * otherwise the size of the data blocks
 */

static inline size_t Gif_data_blocks(const guchar * Buf, size_t BSize)
{
	size_t Size = 0;

	if (BSize < 1)
		return 0;

	while (Buf[0]) {

		if (BSize <= (size_t) (Buf[0] + 1))
			return 0;

		Size += Buf[0] + 1;
		BSize -= Buf[0] + 1;
		Buf += Buf[0] + 1;
	}

	return Size + 1;
}

/*
 * This is a GIF extension.  We ignore it with this routine.
 * Buffer points to just after the extension label.
 *
 * Return Value
 * 0 -- block not processed
 * otherwise the size of the extension label.
 */

static inline size_t Gif_do_generic_ext(const guchar * Buf, size_t BSize)
{
	size_t Size = Buf[0] + 1, DSize;

	/*The Block size (the first byte) is supposed to be a specific size for each
	 * extension... we don't check.
	 */

	if (Buf[0] > BSize)
		return 0;

	DSize = Gif_data_blocks(Buf + Size, BSize - Size);
	if (!DSize)
		return 0;

	Size += DSize;
	return Size <= BSize ? Size : 0;
}

/*
 * ?
 */

static __inline size_t Gif_do_gc_ext(GifImage * gif, const guchar * Buf, size_t BSize)
{
	/* Graphic Control Extension */
	size_t Size = Buf[0] + 2;
	guint Flags;

	if (Size > BSize)
		return 0;
	Buf++;
	Flags = Buf[0];

	/* The packed fields */
#if 0
	gif->disposal = (Buf[0] >> 2) & 0x7;
	gif->inputFlag = (Buf[0] >> 1) & 0x1;

	/* Delay time */
	gif->delayTime = LM_to_uint(Buf[1], Buf[2]);
#endif

	/* Transparent color index, may not be valid  (unless flag is set) */
	if ((Flags & 0x1)) {
		gif->transparent = Buf[3];
	}
	return Size;
}

#define App_Ext  (0xff)
#define Cmt_Ext  (0xfe)
#define GC_Ext   (0xf9)
#define Txt_Ext  (0x01)

/*
 * ?
 * Return value:
 *    TRUE when the extension is over
 */
static size_t Gif_do_extension(GifImage * gif, guchar Label, const guchar * buf, size_t BSize)
{
	switch (Label) {
	case GC_Ext:		/* Graphics extension */
		return Gif_do_gc_ext(gif, buf, BSize);

	case Cmt_Ext:		/* Comment extension */
		return Gif_data_blocks(buf, BSize);

	case Txt_Ext:		/* Plain text Extension */
		/* This extension allows (rcm thinks) the image to be rendered as text.
		 */
	case App_Ext:		/* Application Extension */
	default:
		return Gif_do_generic_ext(buf, BSize);	/*Ignore Extension */
	}
}

/* --- General Image Decoder ------------------------------------------------------ */
/* Here begins the new push-oriented decoder. It should be quite a bit
 * faster than the old code, which was adapted to be asynchronous from
 * David Koblas's original giftopnm code.
 *
 */

/*
 * ?
 */
static void Gif_lwz_init(GifImage * gif)
{
	gif->num_spill_lines_max = 1;
	gif->spill_lines = g_malloc(sizeof(guchar *) * gif->num_spill_lines_max);

	gif->spill_lines[0] = g_malloc(gif->Width);
	gif->bits_in_window = 0;

	/* First code in table = clear_code +1
	 * Last code in table = first code in table
	 * clear_code = (1<< input code size)
	 */

	gif->last_code = (1 << gif->input_code_size) + 1;
	memset(gif->code_and_byte, 0, (1 + gif->last_code) * sizeof(gif->code_and_byte[0]));
	gif->code_size = gif->input_code_size + 1;
	gif->line_index = 0;
}

/*
 * Send the image line to the dicache, also handling the interlacing.
 */
static void Gif_emit_line(GifImage * gif, const guchar * linebuf)
{
	Image_write(gif->Image, linebuf, gif->y, TRUE);

	if (gif->Flags & INTERLACE) {
		switch (gif->pass) {
		case 0:
		case 1:
			gif->y += 8;
			break;
		case 2:
			gif->y += 4;
			break;
		case 3:
			gif->y += 2;
			break;
		}
		if (gif->y >= gif->Height) {
			gif->pass++;
			switch (gif->pass) {
			case 1:
				gif->y = 4;
				break;
			case 2:
				gif->y = 2;
				break;
			case 3:
				gif->y = 1;
				break;
			default:
				/* arriving here is an error in the input image. */
				gif->y = 0;
				break;
			}
		}
	} else {
		if (gif->y < gif->Height)
			gif->y++;
	}
}

/*
 * I apologize for the large size of this routine and the goto error
 * construct - I almost _never_ do that. I offer the excuse of
 * optimizing for speed.
 *
 * RCM -- busted these down into smaller subroutines... still very hard to
 * read.
 */


/*
 * Decode the packetized lwz bytes
 */
static void Gif_literal(GifImage * gif, guint code)
{
	gif->linebuf[gif->line_index++] = code;

	if (gif->line_index >= gif->Width) {
		Gif_emit_line(gif, gif->linebuf);
		gif->line_index = 0;
	}

	gif->length[gif->last_code + 1] = 2;
	gif->code_and_byte[gif->last_code + 1] = (code << 8);
	gif->code_and_byte[gif->last_code] |= code;
}

/*
 * ?
 */
/* Profiling reveals over half the GIF time is spent here: */

static void Gif_sequence(GifImage * gif, guint code)
{
	guint o_index, o_size, orig_code;
	guchar *obuf = NULL;
	guint sequence_length = gif->length[code];
	guint line_index = gif->line_index;
	gint num_spill_lines;
	gint spill_line_index = gif->spill_line_index;
	guchar *last_byte_ptr;

	gif->length[gif->last_code + 1] = sequence_length + 1;
	gif->code_and_byte[gif->last_code + 1] = (code << 8);

	/* We're going to traverse the sequence backwards. Thus,
	 * we need to allocate spill lines if the sequence won't
	 * fit entirely within the present scan line. */

	if (line_index + sequence_length <= gif->Width) {
		num_spill_lines = 0;
		obuf = gif->linebuf;
		o_index = line_index + sequence_length;
		o_size = sequence_length - 1;
	} else {
		num_spill_lines = (line_index + sequence_length - 1) / gif->Width;
		o_index = (line_index + sequence_length - 1) % gif->Width + 1;
		if (num_spill_lines > gif->num_spill_lines_max) {
			/* Allocate more spill lines. */
			spill_line_index = gif->num_spill_lines_max;
			gif->num_spill_lines_max = num_spill_lines << 1;
			gif->spill_lines = g_realloc(gif->spill_lines, gif->num_spill_lines_max * sizeof(guchar *));

			for (; spill_line_index < gif->num_spill_lines_max; spill_line_index++)
				gif->spill_lines[spill_line_index] = g_malloc(gif->Width);
		}
		spill_line_index = num_spill_lines - 1;
		obuf = gif->spill_lines[spill_line_index];
		o_size = o_index;
	}
	gif->line_index = o_index;	/* for afterwards */

	/* for fixing up later if last_code == code */
	orig_code = code;
	last_byte_ptr = obuf + o_index - 1;

	/* spill lines are allocated, and we are clear to
	 * write. This loop does not write the first byte of
	 * the sequence, however (last byte traversed). */

	while (sequence_length > 1) {

		sequence_length -= o_size;

		/* Write o_size bytes to
		 * obuf[o_index - o_size..o_index). */

		for (; o_size > 0 && o_index > 0; o_size--) {
			guint code_and_byte = gif->code_and_byte[code];

			obuf[--o_index] = code_and_byte & 255;
			code = code_and_byte >> 8;
		}

		/* Prepare for writing to next line. */

		if (o_index == 0) {

			if (spill_line_index > 0) {
				spill_line_index--;
				obuf = gif->spill_lines[spill_line_index];
				o_size = gif->Width;
			} else {

				obuf = gif->linebuf;
				o_size = sequence_length - 1;
			}

			o_index = gif->Width;
		}
	}

	/* Ok, now we write the first byte of the sequence. */
	/* We are sure that the code is literal. */

	obuf[--o_index] = code;
	gif->code_and_byte[gif->last_code] |= code;

	/* Fix up the output if the original code was last_code. */

	if (orig_code == gif->last_code)
		*last_byte_ptr = code;

	/* Output any full lines. */

	o_index = gif->line_index;

	if (gif->line_index >= gif->Width) {
		Gif_emit_line(gif, gif->linebuf);
		gif->line_index = 0;
	}

	if (num_spill_lines) {
		if (gif->line_index)
			Gif_emit_line(gif, gif->linebuf);
		for (spill_line_index = 0; spill_line_index < num_spill_lines - (o_index ? 1 : 0); spill_line_index++)
			Gif_emit_line(gif, gif->spill_lines[spill_line_index]);
	}

	if (num_spill_lines) {
		/* Swap the last spill line with the gif line, using
		 * linebuf as the swap temporary. */
		guchar *linebuf = gif->spill_lines[num_spill_lines - 1];

		gif->spill_lines[num_spill_lines - 1] = gif->linebuf;
		gif->linebuf = linebuf;
	}

	gif->spill_line_index = spill_line_index;
}

/*
 * ?
 *
 * Return Value:
 *   2 -- quit
 *   1 -- new last code needs to be done
 *   0 -- okay, but reset the code table
 *   < 0 on error
 *   -1 if the decompression code was not in the lookup table
 */

static gint Gif_process_code(GifImage * gif, guint code, guint clear_code)
{

	/* A short table describing what to do with the code:
	 * code < clear_code  : This is uncompressed, raw data
	 * code== clear_code  : Reset the decmpression table
	 * code== clear_code+1: End of data stream
	 * code > clear_code+1: Compressed code; look up in table
	 */
	if (code < clear_code) {
		/* a literal code. */
		Gif_literal(gif, code);
		return 1;

	} else if (code >= clear_code + 2) {

		/* a sequence code. */
		if (code > gif->last_code)
			return -1;

		Gif_sequence(gif, code);

		return 1;

	} else if (code == clear_code) {
		/* clear code. Resets the whole table */
		return 0;
	} else {
		/* end code. */
		return 2;
	}
}

/*
 * ?
 */
static gint Gif_decode(GifImage * gif, const guchar * buf, size_t BSize)
{
	/*
	 * Data block processing.  The image stuff is a series of data blocks.  Each
	 * data block is 1 to 256 bytes long.  The first byte is the length of the
	 * data block.  0 == the last data block.
	 */
	size_t bufsize, packet_size;
	guint clear_code;
	guint window;
	gint bits_in_window;
	guint code;
	gint code_size;
	guint code_mask;

	bufsize = BSize;

	/* Want to get all inner loop state into local variables. */
	packet_size = gif->packet_size;
	window = gif->window;
	bits_in_window = gif->bits_in_window;
	code_size = gif->code_size;
	code_mask = (1 << code_size) - 1;
	clear_code = 1 << gif->input_code_size;

	/* If packet size == 0, we are at the start of a data block.
	 * The first byte of the data block indicates how big it is (0 == last
	 * datablock)
	 * packet size is set to this size; it indicates how much of the data block
	 * we have left to process.
	 */
	while (bufsize > 0) {
		/* lwz_bytes is the number of remaining lwz bytes in the packet. */
		gint lwz_bytes = MIN(packet_size, bufsize);

		bufsize -= lwz_bytes;
		packet_size -= lwz_bytes;
		for (; lwz_bytes > 0; lwz_bytes--) {
			/* printf ("%d ", *buf) would print the depacketized lwz stream. */

			/* Push the byte onto the "end" of the window (MSB).  The low order
			 * bits always come first in the LZW stream. */
			window = (window >> 8) | (*buf++ << 24);
			bits_in_window += 8;

			while (bits_in_window >= code_size) {
				/* Extract the code.  The code is code_size (3 to 12) bits long,
				 * at the start of the window */
				code = (window >> (32 - bits_in_window)) & code_mask;

				/* printf ("%d\n", code); would print the code stream. */

				bits_in_window -= code_size;
				switch (Gif_process_code(gif, code, clear_code)) {
				case 1:	/* Increment last code */
					gif->last_code++;
					/*gif->code_and_byte[gif->last_code+1]=0; */

					if ((gif->last_code & code_mask) == 0) {
						if (gif->last_code == (1 << MAX_LWZ_BITS))
							gif->last_code--;
						else {
							code_size++;
							code_mask = (1 << code_size) - 1;
						}
					}
					break;

				case 0:	/* Reset codes size and mask */
					gif->last_code = clear_code + 1;
					code_size = gif->input_code_size + 1;
					code_mask = (1 << code_size) - 1;
					break;

				case 2:	/* End code... consume remaining data chunks..? */
					goto error;	/*Could clean up better? */
				default:
					fprintf(stderr, "dillo_gif_decode: error!\n");
					goto error;
				}
			}
		}

		/* We reach here if
		 * a) We have reached the end of the data block;
		 * b) we ran out of data before reaching the end of the data block
		 */
		if (bufsize <= 0)
			break;	/* We are out of data; */

		/* Start of new data block */
		bufsize--;
		if (!(packet_size = *buf++)) {
			/* This is the "block terminator" -- the last data block */
			gif->state = 999;	/* BUG: should Go back to getting GIF blocks. */
			break;
		}
	}

	gif->packet_size = packet_size;
	gif->window = window;
	gif->bits_in_window = bits_in_window;
	gif->code_size = code_size;

	return BSize - bufsize;

      error:
	gif->state = 999;
	return BSize - bufsize;
}

/*
 * ?
 */
static gint Gif_check_sig(GifImage * gif, const guchar * ibuf, size_t Size)
{
	/* at beginning of file - read magic number */
	if (Size < 6)
		return 0;

	if (strncmp(ibuf, "GIF", 3) != 0) {
		gif->state = 999;
		return 6;
	}

	if (strncmp(ibuf + 3, "87a", 3) != 0 && strncmp(ibuf + 3, "89a", 3) != 0) {
		gif->state = 999;
		return 6;
	}

	gif->state = 1;
	return 6;
}

/* Read the color map
 *
 * Implements, from the spec:
 * Global Color Table
 * Local Color Table
 */

static inline size_t Gif_do_color_table(GifImage * gif, void *Buf, const guchar * buf, size_t BSize, size_t CT_Size)
{
	size_t Size = 3 * (1 << (1 + CT_Size));

	if (Size > BSize)
		return 0;

	gif->ColorMap_ofs = (gulong) buf - (gulong) Buf;
	gif->NumColors = (1 << (1 + CT_Size));
	return Size;
}

/*
 * This implements, from the spec:
 * <Logical Screen> ::= Logical Screen Descriptor [Global Color Table]
 */
static size_t Gif_get_descriptor(GifImage * gif, void *Buf, const guchar * buf, size_t BSize)
{

	/* screen descriptor */
	size_t Size = 7,	/* Size of descriptor */
	 mysize;		/* Size of color table */
	guchar Flags;

	if (BSize < 7)
		return 0;
	Flags = buf[4];

	if (Flags & LOCALCOLORMAP) {
		mysize = Gif_do_color_table(gif, Buf, buf + 7, BSize - 7, Flags & 0x7);
		if (!mysize)
			return 0;
		Size += mysize;	/* Size of the color table that follows */
		gif->Background = buf[5];
	}
	/*   gif->Width = LM_to_uint(buf[0], buf[1]);
	   gif->Height = LM_to_uint(buf[2], buf[3]); */
	gif->ColorResolution = (((buf[4] & 0x70) >> 3) + 1);
	/*   gif->AspectRatio     = buf[6]; */

	return Size;
}

/*
 * This implements, from the spec:
 * <Table-Based Image> ::= Image Descriptor [Local Color Table] Image Data
 *
 * ('Buf' points to just after the Image separator)
 * we should probably just check that the local stuff is consistent
 * with the stuff at the header. For now, we punt...
 */

static size_t Gif_do_img_desc(GifImage * gif, void *Buf, const guchar * buf, size_t bsize)
{
	guchar Flags;
	guchar *buffer;
	size_t Size = 9 + 1;	/* Size of image descriptor + first byte of image data */

	if (bsize < 10)
		return 0;

	gif->Width = LM_to_uint(buf[4], buf[5]);
	gif->Height = LM_to_uint(buf[6], buf[7]);
	gif->linebuf = g_malloc(gif->Width);

	Flags = buf[8];

	gif->Flags |= Flags & INTERLACE;
	gif->pass = 0;
	bsize -= 9;
	buf += 9;

	if (Flags & LOCALCOLORMAP) {

		size_t LSize = Gif_do_color_table(gif, Buf, buf, bsize, Flags & 0x7);

		if (!LSize)
			return 0;
		Size += LSize;
		buf += LSize;
		bsize -= LSize;
	}

	/* Finally, get the first byte of the LZW image data */
	if (bsize < 1)
		return 0;

	gif->input_code_size = *buf++;

	if (gif->input_code_size > 8) {
		gif->state = 999;
		return Size;
	}

	gif->y = 0;
	Gif_lwz_init(gif);
	gif->spill_line_index = 0;
	gif->state = 3;		/*Process the lzw data next */

	if(gif->Image && gif->ColorMap_ofs) 
		Image_set_cmap(gif->Image, (guchar *)Buf + gif->ColorMap_ofs);

	buffer = g_malloc(gif->Width * gif->Height * 3);
	Image_set_parms(gif->Image, buffer, gif->Width, gif->Height, IMG_TYPE_INDEXED);

	return Size;
}

/* --- Top level data block processors -------------------------------------- */
#define Img_Desc (0x2c)
#define Trailer  (0x3B)
#define Ext_Id   (0x21)

/*
 * This identifies which kind of GIF blocks are next, and processes them.
 * It returns if there isn't enough data to process the next blocks, or if
 * the next block is the lzw data (which is streamed differently)
 *
 * This implements, from the spec, <Data>* Trailer
 * <Data> ::= <Graphic Block> | <Special-Purpose Block>
 * <Special-Purpose Block> ::= Application Extension | Comment Extension
 * <Graphic Block> ::= [Graphic Control Extension] <Graphic-Rendering Block>
 * <Graphic-Rendering Block> ::= <Table-Based Image> | Plain Text Extension
 *
 * <Data>* --> GIF_Block
 * <Data>  --> while (...)
 * <Special-Purpose Block> --> Gif_do_extension
 * Graphic Control Extension --> Gif_do_extension
 * Plain Text Extension --> Gif_do_extension
 * <Table-Based Image> --> Gif_do_img_desc
 *
 * Return Value
 * 0 if not enough data is present, otherwise the number of bytes
 * "consumed"
 */
static size_t GIF_Block(GifImage * gif, void *Buf, const guchar * buf, size_t bsize)
{
	size_t Size = 0, mysize;
	guchar C;

	if (bsize < 1)
		return 0;
	while (gif->state == 2) {
		if (bsize < 1)
			return Size;
		bsize--;
		switch (*buf++) {
		case Ext_Id:
			/* get the extension type */
			if (bsize < 2)
				return Size;

			/* Have the extension block intepreted. */
			C = *buf++;
			bsize--;
			mysize = Gif_do_extension(gif, C, buf, bsize);

			if (!mysize)
				/* Not all of the extension is there.. quit until more data
				 * arrives */
				return Size;

			bsize -= mysize;
			buf += mysize;

			/* Increment the amount consumed by the extension introducer
			 * and id, and extension block size */
			Size += mysize + 2;
			/* Do more GIF Blocks */
			continue;

		case Img_Desc:	/* Image descriptor */
			mysize = Gif_do_img_desc(gif, Buf, buf, bsize);
			if (!mysize)
				return Size;

			/* Increment the amount consumed by the Image Separator and the
			 * Resultant blocks */
			Size += 1 + mysize;
			return Size;

		case Trailer:
			gif->state = 999;	/* BUG: should close the rest of the file */
			return Size + 1;
			break;	/* GIF terminator */

		default:	/* Unknown */
			/* BUG: we should gripe and complain
			 * g_print(0x%x\n", *(buf-1)); */
			gif->state = 999;
			return Size + 1;
		}
	}
	return Size;
}


/*
 * Process some bytes from the input gif stream. It's a state machine.
 *
 * From the GIF spec:
 * <GIF Data Stream> ::= Header <Logical Screen> <Data>* Trailer
 *
 * <GIF Data Stream> --> Gif_process_bytes
 * Header            --> State 0
 * <Logical Screen>  --> State 1
 * <Data>*           --> State 2
 * Trailer           --> State > 3
 *
 * State == 3 is special... this is inside of <Data> but all of the stuff in
 * there has been gotten and set up.  So we stream it outside.
 */
static size_t Gif_process_bytes(GifImage * gif, const guchar * ibuf, size_t bufsize, void *Buf)
{
	gint tmp_bufsize = bufsize;
	size_t mysize;

	switch (gif->state) {
	case 0:

		mysize = Gif_check_sig(gif, ibuf, tmp_bufsize);
		if (!mysize)
			break;

		tmp_bufsize -= mysize;
		ibuf += mysize;
		if (gif->state != 1)
			break;

	case 1:
		mysize = Gif_get_descriptor(gif, Buf, ibuf, tmp_bufsize);
		if (!mysize)
			break;

		tmp_bufsize -= mysize;
		ibuf += mysize;
		gif->state = 2;

	case 2:
		/* Ok, this loop construction looks weird.  It implements the <Data>* of
		 * the GIF grammar.  All sorts of stuff is allocated to set up for the
		 * decode part (state ==2) and then there is the actual decode part (3)
		 */
		mysize = GIF_Block(gif, Buf, ibuf, tmp_bufsize);
		if (!mysize)
			break;

		tmp_bufsize -= mysize;
		ibuf += mysize;

		if (gif->state != 3)
			break;

	case 3:
		/* get an image byte */
		/* The users sees all of this stuff */
		mysize = Gif_decode(gif, ibuf, tmp_bufsize);
		if (mysize == 0)
			break;

		ibuf += mysize;
		tmp_bufsize -= mysize;

	default:
		/* error - just consume all input */
		tmp_bufsize = 0;
		break;
	}

	return bufsize - tmp_bufsize;
}
