/*
 *
 * UNICON - The Console Chinese & I18N
 * Copyright (c) 1999-2002
 *
 * This file is part of UNICON, a console Chinese & I18N
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * See the file COPYING directory of this archive
 * Author: see CREDITS
 */

#include 	<stdio.h>
#include 	<unistd.h>
#include 	<stdlib.h>
#include 	<stdarg.h>
#include	<string.h>
#include        <ctype.h>
#include        <time.h>

#include        <xl_defs.h>
#include        <xl_system.h>
#include        <xl_getcap.h>
#include        <xl_errors.h>
#include        <xl_hzfb.h>
#include 	<xl_pinyin.h>
#include 	<xl_hzinput.h>
#include        <xl_key.h>

/***************************************************************************
 *                           variable defines                              *
 ***************************************************************************/
hz_input_table *input_table[NR_INPUTMETHOD];
/* 6 bit a key mask */
unsigned long mask[]=
{
   0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 0x00000000, 
   0x3F000000, 0x3FFC0000, 0x3FFFF000, 0x3FFFFFC0, 0x3FFFFFFF, 0x3FFFFFFF,
   0x3FFFFFFF, 0x3FFFFFFF, 0x3FFFFFFF, 0x3FFFFFFF, 0x3FFFFFFF, 0x3FFFFFFF
};

/***************************************************************************
 *                     Internal Code Input Method                          *
 ***************************************************************************/

hz_input_table *CCE_IntCode_Init(void)
{
  hz_input_table *table;
  int i,index;

  pClient->UseAssociateMode = 0;  /* force to no associate */
  table = malloc(sizeof(hz_input_table));
  if (table == NULL)
    error("Out of memory in loading internal code input method");

   strcpy(table->magic_number,MAGIC_NUMBER);
   strcpy(table->ename, "IntCode");
   strcpy(table->cname, "롿");
   strcpy(table->selkey, "0123456789abcdef");

   table->last_full = 1;
   for(i = 0; i < 128; i++)
   {
      table->KeyMap[i] = 0;
      if ((i >= '0' && i <= '9') || (i >= 'a' && i <= 'f'))
      {
         if (i >= '0' && i <= '9')
            index = i - '0';
         else index = i -'a' + 10;
         table->KeyMap[i] = index;
         table->KeyName[index] = toupper(i);
      }
   }    
   return table;
}

/* pClient->StartKey <= index < pClient->EndKey is valid */
void IntCode_FindMatchKey(void)
{
   unsigned long Key = (pClient->InpKey[0] << 12) | (pClient->InpKey[1] << 8);
   switch(pClient->InputCount)
   {
      case 0:
      case 1:
         pClient->StartKey = pClient->EndKey = 0;    /* not display selection */
         break;
      case 2:
         pClient->StartKey = Key + 0xA1; 
         pClient->EndKey = Key + 0xFF;  /* A1-A9,B0-F7 A1-FE */
         break;
      case 3:
         pClient->StartKey = Key + (pClient->InpKey[2] << 4);
         if (pClient->InpKey[2] == 10) pClient->StartKey++; /* A1 */
         pClient->EndKey = pClient->StartKey + 0x10;  
         if (pClient->InpKey[2] == 15) pClient->EndKey--;  /* FE */ 
         break;
    }
}

/* ABCD  AB then C=a..f, Sel=0-9 begin display selection
        ABC then D=0-f,  Sel=0-f 
*/
void IntCode_FillMatchChars(int index)
{
    int MaxSel,i;
   
    pClient->CurSelNum = 0;
    if (pClient->InputCount < 2) return;
    if (pClient->InputCount == 2) MaxSel = 10; else MaxSel = 16;
    
    if (index % 256 == 0xA1 && pClient->InputCount == 3) 
    {
       pClient->seltab[0][0] = '\0';
       pClient->CurSelNum++;
    }

    while( pClient->CurSelNum < MaxSel && index < pClient->EndKey)
    {
         pClient->seltab[pClient->CurSelNum][0] = index / 256; 
         pClient->seltab[pClient->CurSelNum][1] = index % 256;
         pClient->seltab[pClient->CurSelNum][2] = '\0';
         pClient->CurSelNum++;
         index++;
    }
    
    for(i = pClient->CurSelNum; i < 16; i++)
       pClient->seltab[i][0] = '\0';  /* zero out the unused area */

    pClient->InputMatch = pClient->InputCount;

    /* check if more than one page */
    if ( index < pClient->EndKey && pClient->CurSelNum == MaxSel && MaxSel == 10)
    {
   /* has another matched key, so enter pClient->MultiPageMode, has more pages */
        pClient->NextPageIndex = index;
        pClient->MultiPageMode = 1;
    }  
    else if (pClient->MultiPageMode)
    {
        pClient->NextPageIndex = pClient->StartKey; /* rotate selection */
    }
    else pClient->MultiPageMode = 0;
}

/* return value: characters output to tty_fd */
void Intcode_HZFilter(int tty_fd, unsigned char key)
{
    int inkey = 0;
    switch (key)
    {
     case '\010':  /* BackSpace Ctrl+H */
     case '\177':  /* BackSpace */
        if ( pClient->InputCount > 0 ) 
        {
           pClient->InpKey[--pClient->InputCount]=0;
           if (pClient->InputCount == 0)
               ResetInput();
           else if (pClient->InputCount >= 2)
           {
               IntCode_FindMatchKey();
               pClient->MultiPageMode = 0;
               pClient->CurrentPageIndex = pClient->StartKey;
               IntCode_FillMatchChars(pClient->StartKey); 
           }
        }
        else outchar(tty_fd,key);
        break;

   case '\033':  /* ESCAPE */
        if (pClient->InputCount > 0) 
           ResetInput();
        else outchar(tty_fd,key);
        break;

     case '<':
     case '-':
     case ',':
     case '[':
        if ( pClient->MultiPageMode )
        {
           if ( pClient->CurrentPageIndex > pClient->StartKey) 
                pClient->CurrentPageIndex = pClient->CurrentPageIndex - 10;
           else pClient->CurrentPageIndex = pClient->StartKey;
           IntCode_FillMatchChars(pClient->CurrentPageIndex);
        }
        else outchar(tty_fd, key);
        break;

     case '>':
     case ']':
     case '.':
     case '=':
        if ( pClient->MultiPageMode )
        {
           pClient->CurrentPageIndex = pClient->NextPageIndex;
           IntCode_FillMatchChars(pClient->CurrentPageIndex);
        }
        else outchar(tty_fd, key);
        break;

     case ' ':
        if ( pClient->CurSelNum == 0 )
           outchar(tty_fd, key);
        if ( pClient->seltab[0][0] )
           putstr(tty_fd, pClient->seltab[0]);
        break;

     default:
        inkey   = pClient->cur_table->KeyMap[key];
        switch(pClient->InputCount)
        {
           case 0:  /* A-F */
           case 1:  /* 0-F */
           case 2:  /* A-F */
              if (inkey >=0 && !(pClient->InputCount != 1 && inkey < 10) &&
                   !(pClient->InputCount == 1 && 
                    ((pClient->InpKey[0]==10 && inkey>9) ||       //A1-A9 
                    (pClient->InpKey[0]==10 && inkey==0) ||
                    (pClient->InpKey[0]==15 && inkey>7))))      //B0-F7  
              {
                  pClient->InpKey[pClient->InputCount++] = inkey;
                  if (pClient->InputCount >=2)
                  {
                      IntCode_FindMatchKey();
                      pClient->MultiPageMode = 0;
                      pClient->CurrentPageIndex = pClient->StartKey;
                      IntCode_FillMatchChars(pClient->StartKey);
                  }
              }
              else if (pClient->InputCount == 2 && 
                       inkey >= 0 && inkey < pClient->CurSelNum &&
                       pClient->seltab[inkey][0])
              {
                   putstr(tty_fd, pClient->seltab[inkey]);
              }
              else /* not select key or input key */
              {
                  ResetInput();
                  outchar(tty_fd, key);
                  return;
              }
              break;
           case 3:   /* output char */
                      /* A1A1-FEFE */
              if (inkey >=0 && !(pClient->InpKey[2] == 10 && inkey == 0) &&
                  !(pClient->InpKey[2] == 15 && inkey == 15))
              {
                  pClient->seltab[0][0] = (pClient->InpKey[0] << 4) | 
                                         pClient->InpKey[1];
                  pClient->seltab[0][1] = (pClient->InpKey[2] << 4) | inkey ;
                  pClient->seltab[0][2] = '\0';
                  putstr(tty_fd, pClient->seltab[0]);
              }
              else
              {
                  ResetInput();
                  outchar(tty_fd, key);
                  return;
              }
              break;
        }
  } /* switch */
  return pClient->CurSelNum;
}

