/*****************************************************************************\
 * FILE: guiObject.h
 *
 * PURPOSE: The base object of my simple widgets
 *
 * Created by Eric Akers, 18 Dec 2003
 *
 * ChangeLog:
 *     ELA - <Date> - Initial Working Version
 *
 *
 *
 *   
 *   Copyright (C) 2003 Eric Akers
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
\*****************************************************************************/



#ifndef GUIOBJECT_H
#define GUIOBJECT_H

// Header Files #############################################################
#include <SDL/SDL.h>
#include <GL/gl.h>

#include "glfont2.h"

// Macros ###################################################################


// Structures ###############################################################
struct GUI_COLORS {
  GLdouble fgr, fgg, fgb, fga;
  GLdouble bgr, bgg, bgb, bga;

  GUI_COLORS();
};


// Global Variables #########################################################


// Stores the default colors for all widgets


// Class Definition #########################################################
class guiObject
{
 public:
  guiObject();
  virtual ~guiObject();

  // Setters
  virtual void setPosition( GLdouble x, GLdouble y, GLdouble z = 0.0 );
  virtual void setSize( GLdouble width, GLdouble height );
  virtual void setWidth( GLdouble width );
  virtual void setHeight( GLdouble height );
  virtual void setLabel( string label );
  virtual void setGLFont( GLFont * font );
  virtual void setFGColor( GLdouble r, GLdouble g, GLdouble b, 
			   GLdouble a = 1.0 );
  virtual void setBGColor( GLdouble r, GLdouble g, GLdouble b, 
			   GLdouble a = 1.0 );
  virtual void setBorder( bool useBorder );
  virtual void setVisible( bool isVisible );
  void setIDNumber( int idNumber );
  void setCommandCode( string code );

  // Getters
  void getPosition( GLdouble & x, GLdouble & y, GLdouble & z ) const;
  void getSize( GLdouble & width, GLdouble & height ) const;
  bool getBorder() const;
  string getLabel( string & label ) const;
  string getLabel() const;
  bool isVisible() const;
  int getIDNumber() const;
  string getCommandCode() const;

  // Events
  virtual bool doMouseEvent( Uint8 button, Uint8 state, Uint16 x, Uint16 y );
  virtual bool doKeyboardEvent( Uint8 state, SDL_keysym keysym );
  virtual void draw();

  // Actions
  virtual void actionPerformed( int eventType, guiObject * eventObject );
  void addActionListener( guiObject * listener );
  void addActionCallback( void (*callback)( int, guiObject * ) );


 protected:
  // Text for the widget
  string widgetLabel;

  // Position and bounding box of the widget
  GLdouble xPos, yPos, zPos;
  GLdouble widgetWidth, widgetHeight;

  // Colors for the widget
  GLdouble fgr, fgg, fgb, fga, bgr, bgg, bgb, bga;

  // The font to use
  GLFont * widgetFont;

  // Draw a border
  bool widgetBorder;

  // Flag of visibility
  bool isWidgetVisible;

  // The action listener
  guiObject * actionListener;

  // The callback used for events
  void (*actionCallback)( int, guiObject * );

  // Functions that return the label dimensions
  GLdouble getLabelWidth() const;
  GLdouble getLabelHeight() const;

 private:
  // Dimensions of the label
  GLdouble labelWidth, labelHeight;

  // For events
  string commandCode;
  int widgetID;
};


// Global Functions ###########################################################

// Function to set default colors for all widgets
void GUI_setDefaultColors( GLdouble fgr, GLdouble fgg, GLdouble fgb, 
			   GLdouble fga, GLdouble bgr, GLdouble bgg, 
			   GLdouble bgb, GLdouble bga );


#endif
