/*****************************************************************************\
 * FILE: guiComboBox.cpp
 *
 * PURPOSE: Implementation of the combo box class
 *
 * Created by Eric Akers, 22 Dec 2003
 *
 * ChangeLog:
 *     ELA - <Date> - Initial Working Version
 *
 *
 *
 *   
 *   Copyright (C) 2003 Eric Akers
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
\*****************************************************************************/



// Header Files #############################################################
#include <assert.h>

#include "guiActionListener.h"
#include "guiComboBox.h"

// Macros ###################################################################


// Structures ###############################################################


// Class Function Definitions ###############################################
guiComboBox::guiComboBox() : guiButton()
{
  drawOptions = false;
  numItems = 0;
}



guiComboBox::guiComboBox( GLdouble topLeftX, GLdouble topLeftY, 
			  GLdouble width, GLdouble height )
  : guiButton( topLeftX, topLeftY, width, height )
{
  drawOptions = false;
  numItems = 0;
}


guiComboBox::~guiComboBox()
{}


void guiComboBox::setLabel( string label )
{
  guiObject::setLabel( label );
}


void guiComboBox::setGLFont( GLFont * font )
{
  guiObject::setGLFont( font );
  menu.setGLFont( font );
}


void guiComboBox::setFGColor( GLdouble r, GLdouble g, GLdouble b, 
			      GLdouble a )
{
  // Set the color as normal
  guiObject::setFGColor( r, g, b, a );

  // Set the drop down menu color
  menu.setFGColor( r, g, b, a );
}


void guiComboBox::setBGColor( GLdouble r, GLdouble g, GLdouble b, 
			      GLdouble a )
{
  // Set the color as normal
  guiObject::setBGColor( r, g, b, a );

  // Set the drop down menu color
  menu.setBGColor( r, g, b, a );
}



// Add items to the combo box
void guiComboBox::addItem( guiMenuItem * item )
{
  //  printf( "Adding item: %s\n", item->getLabel().c_str() );
  string tmp = item->getLabel();
  if( numItems == 0 && getLabel().size() <= 0 ) {
    // The label for this combo box needs to be set if one isn't already
    setLabel( tmp );
  }

  // Set the properties of the menu item
  item->addActionListener( this );
  item->setIDNumber( numItems++ );
  item->setCommandCode( tmp );

  // Add the menu item to the drop down menu
  menu.addItem( item );
}



// Handle mouse events
bool guiComboBox::doMouseEvent( Uint8 button, Uint8 state, Uint16 x, Uint16 y )
{
  bool activeMouse = guiButton::doMouseEvent( button, state, x, y );
  //  printf( "***Combo box: doMouseEvent.\n" );

  if( drawOptions ) {
    // The state here is SDL_RELEASED because only a click can open
    // the drop down menu.

    // This unclick will turn off the menu no matter what
    drawOptions = false;
      
    // Now handle the drop down menu
    menu.doMouseEvent( button, state, x, y );

    // Always return true whether the click was on the boundary or not,
    // because the drop down menu was opened, and this releases it
    return true;
  }
  else {
    // Only an initial SDL_PRESSED event can open the drop down menu
    if( activeMouse == false ) {
      //      printf("****comboBos:doMouseEvent: false\n" );
      return false;
    }
    else {
      if( state == SDL_PRESSED ) {
	// Open the menu
	drawOptions = true;
      }

      // The click happened in the boundary, so we always return true
      //      printf("****comboBos:doMouseEvent: true\n" );
      return true;
    }
  }
}




void guiComboBox::actionPerformed( int eventType, guiObject * eventObject )
{
  assert( eventType == GUI_EventMenuItem );
  setLabel( eventObject->getCommandCode() );

  // Call the action listener
  if( actionListener != NULL ) {
    actionListener->actionPerformed( GUI_EventComboBox, this );
  }
  // The callback as well
  if( actionCallback != NULL ) {
    (*actionCallback)( GUI_EventComboBox, this );
  }

}



// Draw the box
void guiComboBox::draw()
{
  // Draw the box but add a triangle at the end
  guiButton::draw();

  GLdouble height = getLabelHeight() / 2;
  GLdouble topLeftX = xPos + widgetWidth - 20;
  GLdouble topRightX = topLeftX + 15;
  GLdouble bottomX = topLeftX + 7.5;
  GLdouble topY = yPos - (widgetHeight / 2) + (height / 2);
  GLdouble bottomY = yPos - (widgetHeight / 2) - (height / 2);

  glColor4d( fgr, fgg, fgb, fga );
  glBegin( GL_TRIANGLES );
  glNormal3d( 0.0, 0.0, 1.0 );
  glVertex3d( topLeftX, topY, zPos );
  glVertex3d( topRightX, topY, zPos );
  glVertex3d( bottomX, bottomY, zPos );
  glEnd();

  // See if the drop down menu needs to be drawn
  if( drawOptions ) {
    menu.setSize( widgetWidth, widgetHeight * numItems );
    menu.setPosition( xPos, yPos - widgetHeight );
    menu.draw();
  }
}

