/*
 * cowbell
 * Copyright (c) 2005 Brad Taylor
 *
 * This file is part of cowbell.
 *
 * cowbell is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * cowbell is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with cowbell; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

using System;
using System.Text;
using System.Text.RegularExpressions;
using Gtk;
using Cowbell.Base;
using System.Collections;

namespace Cowbell.Gui
{
	public class SongDatabaseTreeView : TreeView
	{
		/* public methods */
		public SongDatabaseTreeView ()
		{
			CellRendererText renderer;
				
			count = 0;
			Reorderable = false;
			RulesHint = true;
			HeadersVisible = true;
			
			popup_menu = null;

			// TODO: Rewrite this with TreeCellDataFuncs
			store = new ListStore (typeof (uint),		// Track number
			                       typeof (string),		// Title
			                       typeof (string),		// Artist
			                       typeof (string),		// Disc
			                       typeof (int));		// Location in Database

			store.SetSortColumnId (0, SortType.Ascending);
			Model = store;

			// Track Column
			renderer = new CellRendererText ();
			renderer.Editable = true;
			renderer.Edited += new EditedHandler (OnTrackEdited);
			AppendColumn (Catalog.GetString ("Track"), renderer, "text", 0);

			// Title Column
			renderer = new CellRendererText ();
			renderer.Editable = true;
			renderer.Edited += new EditedHandler (OnTitleEdited);
			AppendColumn (Catalog.GetString ("Title"), renderer, "text", 1);

			// Artist Column
			renderer = new CellRendererText ();
			renderer.Editable = true;
			renderer.Edited += new EditedHandler (OnArtistEdited);
			artist_col = AppendColumn (Catalog.GetString ("Artist"), renderer, "text", 2);
			artist_col.Visible = false;

			// Album Column
			renderer = new CellRendererText ();
			renderer.Editable = true;
			renderer.Edited += new EditedHandler (OnAlbumEdited);
			album_col = AppendColumn (Catalog.GetString ("Album"), renderer, "text", 3);
			album_col.Visible = false;
	
			// Set up right click menu
			AddEvents ((int)Gdk.EventMask.ButtonPressMask);
			ButtonPressEvent += new ButtonPressEventHandler (OnButtonPressed);	

			// Set up DnD
			Drag.DestSet (this, DestDefaults.All, SongDragEntries, Gdk.DragAction.Copy);
			DragDataReceived += new DragDataReceivedHandler (OnDragDataReceived);

			// Set up default column sizes	
			ColumnsAutosize ();
			foreach (TreeViewColumn column in Columns)
			{
				column.Reorderable = true;
				column.Resizable = true;
				column.Clickable = true;
				column.Clicked += new EventHandler (OnTreeViewColumnClicked);
	
				switch (column.Title) {
				case "Track":
					column.MinWidth = 20;
					column.SortIndicator = true;
					break;
				case "Artist":
				case "Album":
					column.MinWidth = 70;
					break;
				}
			}

			// Lets start getting database change notifications
			Runtime.Database.Modified += new DatabaseModifiedHandler (OnDatabaseModified);
			Runtime.Database.MultipleArtistsChanged += new VoidHandler (OnMultipleArtistsChanged);
			Runtime.Database.MultipleDiscsChanged += new VoidHandler (OnMultipleDiscsChanged);
		}

		/* private enumerations */
		enum TargetType {
			UriList
		};

		/* private fields */	
		static TargetEntry[] SongDragEntries = new TargetEntry[] {
			new TargetEntry ("text/uri-list", 0, (uint)TargetType.UriList)
		};	

		ListStore store;
		int count;
		TreeViewColumn artist_col, album_col;
		Menu popup_menu;

		/* for SearchForeach */	
		TreeIter search_iter;
		int search_location;

		/* private methods */
		private TreeIter Search (int location)
		{
			search_location = location;
			store.Foreach (new TreeModelForeachFunc (SearchForeach));

			return search_iter;
		}

		private bool SearchForeach (TreeModel model, TreePath path, TreeIter iter)
		{
			if ((int)model.GetValue (iter, 4) == search_location) {
				search_iter = iter;
				return true;
			}

			return false;
		}

		private void AddRow (Song s, int location)
		{
			store.AppendValues (s.TrackNumber, s.Title, s.Artist, s.Album, location);
			count++;
			ColumnsAutosize ();
		}

		private void RemoveRow (int location)
		{
			TreeIter iter = Search (location);

			if (iter.Equals (TreeIter.Zero)) {
				return;
			}

			store.Remove (ref iter);
			count--;
			ColumnsAutosize ();
		}

		private void ChangeRow (int location, Song s)
		{
			TreeIter iter = Search (location);

			if (iter.Equals (TreeIter.Zero)) {
				return;
			}
			
			store.SetValue (iter, 0, s.TrackNumber);
			store.SetValue (iter, 1, s.Title);
			store.SetValue (iter, 2, s.Artist);
			store.SetValue (iter, 3, s.Album);
			store.EmitRowChanged (store.GetPath (iter), iter);
			ColumnsAutosize ();
		}

		private int GetSelectedLocation ()
		{
			TreeModel model;
			TreeIter iter;

			if (Selection.GetSelected (out model, out iter)) {
				return (int)model.GetValue (iter, 4);
			}

			return -1;
		}

		private Menu CreatePopupMenu ()
		{
			Menu popup = new Menu ();
			ImageMenuItem item;

			item = new ImageMenuItem (Catalog.GetString ("_Remove from List"));
			item.Activated += new EventHandler (OnRemoveFromListMenuItemActivated);
			item.Image = new Image (Gtk.Stock.Delete, IconSize.Menu);
			popup.Add (item); 

			popup.Add (new SeparatorMenuItem ());

			item = new ImageMenuItem (Catalog.GetString ("_Properties"));
			item.Activated += new EventHandler (OnPropertiesMenuItemActivated);
			item.Image = new Image (Gtk.Stock.Properties, IconSize.Menu);
			popup.Add (item);

			return popup;
		}

		private void OnTrackEdited (object o, EditedArgs args)
		{
			int location = GetSelectedLocation ();
			Song s = (Song)Runtime.Database[location];
			uint val;

			if (location < 0) {
				return;
			}

			try {
				val = Convert.ToUInt32 (args.NewText);
			} catch (Exception e) {
				val = s.TrackNumber;
			}

			if (s.TrackNumber != val) {
				s.TrackNumber = val;
				ChangeRow (location, s);
			}

			Runtime.Database[location] = s;
		}

		private void OnTitleEdited (object o, EditedArgs args)
		{
			int location = GetSelectedLocation ();
			Song s = (Song)Runtime.Database[location];

			if (location < 0) {
				return;
			}

			if (s.Title != args.NewText) {
				s.Title = args.NewText;
				ChangeRow (location, s);
			}

			Runtime.Database[location] = s;
		}

		private void OnAlbumEdited (object o, EditedArgs args)
		{
			int location = GetSelectedLocation ();
			if (location < 0) {
				return;
			}

			Song s = (Song)Runtime.Database[location];
			if (s.Album != args.NewText) {
				s.Album = args.NewText;
				ChangeRow (location, s);
			}

			Runtime.Database[location] = s;
		}

		private void OnArtistEdited (object o, EditedArgs args)
		{
			int location = GetSelectedLocation ();
			if (location < 0) {
				return;
			}

			Song s = (Song)Runtime.Database[location];
			if (s.Artist != args.NewText) {
				s.Artist = args.NewText;
				ChangeRow (location, s);
			}

			Runtime.Database[location] = s;
		}
		
		[GLib.ConnectBefore]
		private void OnButtonPressed (object o, ButtonPressEventArgs args)
		{
			if (args.Event.Button == 3) {
				if (Runtime.Database.Count < 1) {
					return;
				}

				if (popup_menu == null) {
					popup_menu = CreatePopupMenu ();
				}

				popup_menu.Popup (null, null, null,
				                  args.Event.Button, args.Event.Time);
				popup_menu.ShowAll ();
			}
		}

		private void OnDragDataReceived (object o, DragDataReceivedArgs args)
		{
                        string data = Encoding.UTF8.GetString (args.SelectionData.Data);

                        if (args.Info == (uint)TargetType.UriList) {
                                string[] tokens = Regex.Split (data, "\r\n");
                                foreach (string token in tokens)
                                {
					// Nautilius always seems to send the first token
					// as empty... must investigate this.
					if (token == String.Empty) {
						continue;
					}

                                        try {
                                                string path = Filesystem.GetLocalPathFromUri (token);
                                                foreach (Song s in Filesystem.GetSongsFromFilesystem (path))
						{
							Runtime.Database.Add (s);
						}
                                        } catch (Exception e) {
						Runtime.Debug ("Drag of \"{0}\" failed with exception:", token);
						Runtime.Debug (e.Message);
						Runtime.Debug (e.StackTrace);
                                        }
                                }

				if (Runtime.Database.Count > 0) {
					Runtime.Database.GlobalData = (Song)Runtime.Database[0];
				}
                        }
                        Gtk.Drag.Finish (args.Context, true, false, args.Time);
		}

		private void OnDatabaseModified (int location, ModificationType type)
		{
			switch (type) {
			case ModificationType.Add:
				AddRow ((Song)Runtime.Database[location], location);
				break;
			
			case ModificationType.Change:
				ChangeRow (location, (Song)Runtime.Database[location]);
				break;
			
			case ModificationType.Delete:
				RemoveRow (location);
				break;
			}
		}

		private void OnMultipleArtistsChanged ()
		{
			artist_col.Visible = Runtime.Database.MultipleArtists;
		}

		private void OnMultipleDiscsChanged ()
		{
			album_col.Visible = Runtime.Database.MultipleDiscs;
		}

		private void OnTreeViewColumnClicked (object o, EventArgs args)
		{
			TreeViewColumn col = (TreeViewColumn)o;

			for (int i = 0; i < Columns.Length; i++)
			{
				if (col == Columns[i]) {
					if (col.SortIndicator && col.SortOrder == SortType.Ascending) {
						col.SortIndicator = true;
						col.SortOrder = SortType.Descending;
						store.SetSortColumnId (i, col.SortOrder);
					} else {
						col.SortIndicator = true;
						col.SortOrder = SortType.Ascending;
						store.SetSortColumnId (i, col.SortOrder);
					}
					continue;
				}
				Columns[i].SortIndicator = false;
			}
		}

		private void OnPropertiesMenuItemActivated (object o, EventArgs args)
		{
			int location = GetSelectedLocation ();
			if (location < 0) {
				return;
			}

			TreeIter iter_first;
			if (!store.GetIterFirst (out iter_first)) {
				return;
			}

			int position = 0;
			ArrayList resultset = new ArrayList ();
			do {
				int pos = (int)store.GetValue (iter_first, 4);
				resultset.Add (Runtime.Database[pos]);

				if (pos == location) {
					position = resultset.Count - 1;
				}
			} while (store.IterNext (ref iter_first));

			PropertiesDialog d = new PropertiesDialog (resultset, position);
			d.Show ();
		}

		private void OnRemoveFromListMenuItemActivated (object o, EventArgs args)
		{
			int location = GetSelectedLocation ();
			if (location < 0) {
				return;
			}

			Runtime.Database.RemoveAt (location);
		}
	}
}
