/* By Laurent Coustet (c) 2005 */

#include <stdlib.h>
#include <stdio.h>
#include <errno.h>
#include <signal.h>

#if WIN32
 #include <winsock2.h>
#else
 #include <netdb.h>
 #include <netinet/in.h>
 #include <sys/socket.h>
 #include <arpa/inet.h>
#endif

#include <strings.h>
#include <string.h>
#include <unistd.h>
#include <time.h>
#include <fcntl.h>
#include <dirent.h>
#include <sys/stat.h>

#ifndef WIN32
 #include <regex.h>
 #include <locale.h>
#endif

#include "cpige.h"
#include "tool.h"
#include "pool.h"
#include "mytime.h"
#include "debug.h"

#ifndef NOCONFIG
 #include "config.h"
 char *configFile = NULL;
#endif

/* Windows needs file to be opened in byte mode */
#if WIN32
#define WAITALL 0
#define WRITE "wb+"
#define sleep(msec) _sleep(msec)
#else
#define WAITALL 0x100
#define WRITE "w+"
#endif

/* Please change this only if really needed... */
#define USER_AGENT "cPige/1.4-1"

#define TENTATIVES -1
#define SOCKET_TIMEOUT 10 /* Socket receives timeout after SOCKET_TIMEOUT seconds */
#define CONNECTION_TIMEOUT 5 /* The connection must be established within 5 seconds */
#define RECONNECT_TIME 5 /* We are waiting 5 seconds before trying to reconnect. */

/* Struct storing command line arguments */
commandLine *cmdLine;

/* Uptime */
time_t start_time;
time_t current_time;
/* for select() */
fd_set rfds;
/* socket timeout */
struct timeval timeout;

/* Servers pool */
urlPool_t *serversPool = NULL;
int poolPosition = 0;

/* Statistics */
stats *cPigeStats;

/* Main Program */
int main (int argc, char **argv)
{
  char *buffer;
  char *titre       = NULL;
  char *oldtitre    = NULL;
  char *nexttitre   = NULL;
  char *filename    = NULL;
  char *meta;
  char *statusline;
  char *extension   = "mp3";
  char *dir;

  long long unsigned int uptime      = 0;
  long long unsigned int metacounter = 0;
 
  /* When operates in quite mode, display stats in logfile
   * evry countdown seconds
   */
  time_t countdown = 3600; /* Evry 1 hour */
  time_t temptime;

  /* Integers ... */
  int size;
  int retval;
  int i;
  int tempsize      = 0;
  int nextsize      = 0;
  int len1   = 0;
  int len2   = 0;
  int maxlen = 0;
  int maxlinelength = 80;
  int songs = 0;
  
  /* Directory & File manipulations */
  DIR *pige_dir;
  FILE *output_file = NULL;
  FILE *logfile     = NULL;
  int fd;

  /* Custom typedef */
  currentSong *curSong;
  icyHeaders *icy_headers = NULL;
  lastCut last_cut;

  last_cut.hour = getHour();
  last_cut.min  = getMinute();
  
  /* Global dynamic configuration */
  cmdLine = parseCommandLine(argc, argv);
  testCommandLine();

  /* Logfile */
  if ((logfile = fopen(cmdLine->logFile, "a+")) == NULL)
  {
    _ERROR("Unable to openlogfile: %s Setting log to stdout\n", cmdLine->logFile);
    logfile = stdout;
  } else {
    VERBOSE("Successfully opened %s\n", cmdLine->logFile);
  }
  
  /* We setup debug to logfile instead of stderr */
  SetDebugFile(logfile);
  
  if (cmdLine->background)
  {
#ifndef WIN32
    int pid;
    FILE *pidfile;

    if (daemon(1, 1) == -1)
    {
      _ERROR("Error daemonizing. %s\n", strerror(errno));
      exit(-1);
    }
    
    pid = (int)getpid();
    
    if (cmdLine->pidFile != NULL)
    {
      pidfile = fopen(cmdLine->pidFile, "w");
      if (pidfile == NULL)
        _ERROR("Unable to open pidfile %s\n", cmdLine->pidFile);
      else
      {
        char *pidchar;
        pidchar = (char *)calloc(64, 1);
        snprintf(pidchar, 63, "%d", pid);
        fwrite(pidchar, 1, strlen(pidchar), pidfile);
        free(pidchar);
        fclose(pidfile);
      }
    }
    
    fprintf(stdout, "cPige launched in background. pid: %d\n", pid);
    fd = open("/dev/null", O_RDWR);
    
    if (fd == -1)
    {
      _ERROR("Error opening /dev/null: %s\n", strerror(errno));
      exit(0);
    }
    
    for (i = 0; i < 3; i++)
      dup2(fd, i);
    
    close(fd);
#else
   printf("Fork not available under WIN32.\n");
#endif
  }

  cPigeStats = (stats *)memory_allocation(sizeof(stats));
  cPigeStats->reconnections = 0;
  cPigeStats->songs = 0;
  
  /* Create output dir if does not exists! */
  if (( pige_dir = opendir(cmdLine->dir)) == NULL)
  {
    _ERROR("Unable to open %s for writing\n", cmdLine->dir);
#ifdef WIN32
    if (mkdir(cmdLine->dir) != 0) {
#else
    if (mkdir(cmdLine->dir, 0755) != 0) {
#endif
      _ERROR("Failed trying to create %s. Verify rights.\n", cmdLine->dir);      
      exit(-1);
    } else {
      VERBOSE("Created %s\n", cmdLine->dir);
      /* pige_dir = opendir(cmdLine->dir); */
    }
  } else {
    VERBOSE("Sucessfully opened %s\n", cmdLine->dir);
    closedir(pige_dir);
  }

  /* If we are keeping One week of stream, check if directories
   * exists, and check disk space
   */
  
  checkWeekBackup();

  /* Start time, for uptime calculation */
  start_time = time((time_t *)NULL);
  temptime = time((time_t *)NULL) + countdown;

  /* We try connecting and receiving good headers infinitely */
  do {
    server_socket = reconnect(RECONNECT_TIME, TENTATIVES, 0);
    if (icy_headers != NULL)
    {
      free_icy_headers(icy_headers);
      free(icy_headers);
      icy_headers = NULL;
    }

    icy_headers = readicyheaders(getHeaders(server_socket));
  } while (icy_headers == NULL);
    
  
  if (icy_headers->metaint != 0)
  {
    VERBOSE("Using metaint: %d\n", icy_headers->metaint);
    buffer = (char *)memory_allocation((int) icy_headers->metaint);
  } else {
    VERBOSE("Using default metaint: %d\n", 32 * 1024);
    icy_headers->metaint = 32*1024;
    buffer = (char *)memory_allocation(32768);
  }

  if (strncmp(icy_headers->content_type, "audio/mpeg", 10) == 0)
    extension = "mp3";
  else if (strncmp(icy_headers->content_type, "audio/aacp", 10) == 0)
    extension = "aac";
  else if (strncmp(icy_headers->content_type, "application/ogg", 15) == 0)
    extension = "ogg";

  VERBOSE("Using extension: %s\n", extension);

  /* Only usefull for debug, not for production ! */
  /* print_icyheaders(icy_headers); */

#ifndef WIN32
  if(cmdLine->useNumbers == 1)
    songs = getSongs(cmdLine->dir);
#endif

  nextsize = 512;
 
  oldtitre = strdup("please.delete");
  
  while (1)
  {
    /* For select() it's a global struct. */
    timeout.tv_sec = SOCKET_TIMEOUT;
    timeout.tv_usec = 0;
    retval = select(server_socket+1, &rfds, NULL, NULL, &timeout);
    if (retval <= 0) 
    {
      _ERROR("Connection Error.\n");
      server_close(server_socket);
      server_socket = reconnect(RECONNECT_TIME, TENTATIVES, 1);
      tempsize = 0;
      nextsize = 512;
      continue;
    }

    size = recv(server_socket, buffer, nextsize, 0);
    if ((size == -1) || ((size == 0) && nextsize != 0))
    {
      _ERROR("Connection error in recv main() size: %d nextsize: %d\n", size, nextsize);
      server_socket = reconnect(RECONNECT_TIME, TENTATIVES, 1);
      tempsize = 0;
      nextsize = 512;
      continue;
    }
    
    if (output_file)
    {
      if (fwrite(buffer, sizeof(char), size, output_file) != size*sizeof(char))
      {
        _ERROR("Error writing output. Disk full ?\n");
        break;
      }
      /* Many thanks to the #hurdfr team! */
      fflush(output_file);
    }

    if ( tempsize == icy_headers->metaint )
    {
      if (cmdLine->pige)
      {
        /* Pige Mode */
        
        if (mustCut(&last_cut) == 1)
        {
          if ((cmdLine->usePigeMeta == 1) && (output_file != NULL) && (strncmp(icy_headers->content_type, "audio/mpeg", 10) == 0))
          {
            char *buffer1, *buffer2, *buffer3;
            
            buffer1 = (char *)memory_allocation(30);
            buffer2 = (char *)memory_allocation(30);

            switch (cmdLine->intervalType)
            {
              case IVAL_HOUR:
                snprintf(buffer1, 30, "pige %.2dh -> %.2dh", last_cut.hour, getCloserInterval(getHour(), cmdLine->interval));
                break;
              case IVAL_MIN:
                snprintf(buffer1, 30, "pige %.2dh%.2d -> %.2dh%.2d", last_cut.hour, last_cut.min, getCloserInterval(getHour(), 1), getCloserInterval(getMinute(), cmdLine->interval));
                break;
            }
            strncpy(buffer2, icy_headers->name, 30);
            
            buffer3 = GetId3v1(buffer2, buffer1, icy_headers->name);
            fwrite(buffer3, sizeof(char), 128, output_file);
            (void)free(buffer3);
            (void)free(buffer2);
            (void)free(buffer1);
          }
          
          if (output_file)
          {
            /* VERBOSE("Closing: %d.mp3\n", lastCut.hour); */
            fclose(output_file);
          }

          if (cmdLine->weekBackup)
          {
            char *dirname = getDayName(time((time_t *)NULL));
            int dirlen = strlen(cmdLine->dir);
            int totlen = dirlen+strlen(dirname)+1+1+1;
            
            dir = (char *)memory_allocation(totlen);
            if (cmdLine->dir[dirlen-1] == '/')
              snprintf(dir, totlen, "%s%s/", cmdLine->dir, dirname);
            else
              snprintf(dir, totlen, "%s/%s/", cmdLine->dir, dirname);

            (void)free(dirname);
          } else {
            dir = strdup(cmdLine->dir);
          }
          
          switch (cmdLine->intervalType)
          {
            case IVAL_HOUR:
              last_cut.hour = getCloserInterval(getHour(), cmdLine->interval);
              last_cut.min  = getCloserInterval(getMinute(), 60);

              len1 = strlen(dir) + 1 + strlen(extension) + 2 + 1;
              filename = (char *)memory_allocation(len1);
              snprintf(filename, len1, "%s%.2d.%s", dir, last_cut.hour, extension);
              break;
              
            case IVAL_MIN:
              last_cut.min  = getCloserInterval(getMinute(), cmdLine->interval);
              last_cut.hour = getCloserInterval(getHour(), 1);
              
              len1 = strlen(dir) + 1 + strlen(extension) + 2 + 2 + 1 + 1;
              filename = memory_allocation(len1);
              snprintf(filename, len1, "%s%.2d-%.2d.%s", dir, getHour(), last_cut.min, extension);
              break;
              
            default:
              fprintf(stderr, "Internal Error: unknown interval type.\n");
              break;
          }

          (void)free(dir);

          if ((output_file = fopen(filename, "r")) != NULL)
          {
            VERBOSE("Erasing %s\n", filename);
            unlink(filename);
            fclose(output_file);
          }
          VERBOSE("Opening: %s\n", filename);
          output_file = fopen(filename, WRITE);
          (void)free(filename);

        }
      } else if (cmdLine->live) {
        printf("Not yet implemented.\n");
        exit(1);
        /* Live Mode */
        /* I must write some stuff here ;) */
      }
      
      metacounter++;
      current_time = time((time_t *)NULL);
      uptime = (long long unsigned int) difftime(current_time, start_time);
      
      /* Classical stdout status line */
      if (!cmdLine->quite)
      {
        fflush(stdout);
        fprintf(stdout, "\r");
      
        /* 100% pur porc */
        for(i = 0; i < maxlinelength; i++)
          fprintf(stdout, " ");
       
        if (strncmp(icy_headers->content_type, "application/ogg", 15) == 0)
          statusline = statusLine(uptime, metacounter, icy_headers->metaint, NULL, NULL);
        else
          statusline = statusLine(uptime, metacounter, icy_headers->metaint, oldtitre, nexttitre);

        fprintf(stdout, "\r%s", statusline);
      
        if ( strlen(statusline) > maxlinelength)
          maxlinelength = strlen(statusline);

        if (statusline != NULL) 
          (void)free(statusline);
      }

      /* Stats evry countdown seconds */
      if ((countdown != 0) && (temptime <= current_time))
      {
        statusline = getStats(uptime, metacounter, icy_headers->metaint);
        if (statusline != NULL)
        {
          if (fwrite(statusline, sizeof(char), strlen(statusline), logfile) != strlen(statusline))
          {
            _ERROR("Fwrite error.\n");
            break; 
          }
          /* Many thanks to the #hurdfr@freenode team! */
          fflush(logfile);
          (void)free(statusline);
        } else {
          VERBOSE("getStats returned NULL values...\n");
        }
        temptime = current_time + countdown;
      }
      
      if ((strncmp(icy_headers->content_type, "audio/mpeg", 10) == 0) ||
          (strncmp(icy_headers->content_type, "audio/aacp", 10) == 0))
      {
        meta = readMeta(server_socket);
      
        if ((meta == NULL) && (server_socket == 0))
        {
          server_socket = reconnect(RECONNECT_TIME, TENTATIVES, 0);
          tempsize = 0;
          nextsize = 512;
          continue;
        }
        
        if (titre != NULL)
          free(titre);

        titre = getTitle(meta);
        
        if (meta != NULL)
          (void)free(meta);
      }

      if (titre != NULL)
      {
        if (strlen(titre) > 0)
        {
          if (strstr(titre, cmdLine->Next) == NULL) /* If title isn't ASUIVRE */
          {
            if (oldtitre != NULL)
            {
              len1 = strlen(oldtitre);
              len2 = strlen(titre);
             
              /* We are determining the maximal comp len */
              if (len1 >= len2); else maxlen = len1;
               
              if (strncmp(titre, oldtitre, maxlen) != 0)
              {
                if (! cmdLine->pige)
                {
                  if (cmdLine->skipSongs > 0)
                  {
                    cmdLine->skipSongs--;
                    VERBOSE("Skipping song `%s' (left %d songs to skip)\n", titre, cmdLine->skipSongs);
                  } else {
                    cPigeStats->songs++;
                    songs++;
                  
                    if (output_file != NULL)
                    {
                      curSong = getCurrentSong(oldtitre);
                      if (curSong != NULL)
                      {
                        if (strncmp(icy_headers->content_type, "audio/mpeg", 10) == 0)
                        {
                          char *buffer1;
                          buffer1 = GetId3v1(curSong->title, curSong->artist, icy_headers->name);
                          fwrite(buffer1, sizeof(char), 128, output_file);
                          (void)free(buffer1);
                        }
                      }
                      fclose(output_file);
                    }
                    
                    if (cmdLine->useNumbers == 0)
                    {
                      len1 = strlen(titre) + strlen(cmdLine->dir)+ 1 + strlen(extension) + 1 + 1;
                      filename = (char *)memory_allocation(len1);
                      snprintf(filename, len1, "%s%s.%s", cmdLine->dir, titre, extension);
                    } else {
                      len1 = 5+strlen(titre) + strlen(cmdLine->dir)+ 1 + strlen(extension) + 1 + 1;
                      filename = (char *)memory_allocation(len1);
                      snprintf(filename, len1, "%s%.4d_%s.%s", cmdLine->dir, songs, titre, extension);
                    }

                    if ((output_file = fopen(filename, "r")) == NULL)
                    { /* Anti doublons */
                      VERBOSE("New file opened: %s\n", filename);
                      output_file = fopen(filename, WRITE);
                    } else {
                      VERBOSE("File already exists %s.\n", filename);
                      fclose(output_file);
                      output_file = NULL;
                    }
                    (void)free(filename);
                  } /* Song skip */
                }
              } /* Title are differents */
            } /* Oldtitre != NULL */
            if (oldtitre != NULL)
              (void)free(oldtitre);

            oldtitre = (char *)memory_allocation(strlen(titre) + 1);
            strncpy(oldtitre, titre, strlen(titre));

          } else { 
            /* Title is "ASUIVRE" */
            if (nexttitre != NULL) 
              (void)free(nexttitre);
            
            nexttitre = (char *)memory_allocation(strlen(titre) + 1);
            strncpy(nexttitre, titre, strlen(titre));
          }
          if (titre != NULL)
          {
            (void)free(titre);
            titre = NULL;
          }
          
        } /* Strlen(titre > 0) */
      }
      tempsize = 0; /* tempsize = 0: chunk received successfully. Awaiting next chunk */
    } else if (tempsize > icy_headers->metaint) {
      _ERROR("Error tempsize > metaint\n");
      break;
    } else
      tempsize = tempsize + size;

    if ((tempsize + 512) >= icy_headers->metaint)
      nextsize = icy_headers->metaint - tempsize;
    else
      nextsize = 512;
  } /* infinite loop */

  server_close(server_socket);

  /* cleanup */
  if (icy_headers != NULL) 
    (void)free(icy_headers);
  if (output_file != NULL) 
    fclose(output_file);
  if (logfile != stdout) 
    fclose(logfile);
  
  (void)free(cmdLine);
  (void)free(cPigeStats);
  
  return 0;
}

void free_icy_headers(icyHeaders *icy)
{
  if (! icy) return;
  if (icy->name)         free(icy->name);
  if (icy->genre)        free(icy->genre);
  if (icy->notice1)      free(icy->notice1);
  if (icy->notice2)      free(icy->notice2);
  if (icy->url)          free(icy->url);
  if (icy->content_type) free(icy->content_type);
}

icyHeaders *readicyheaders(char *headers)
{
  icyHeaders *icy;
  int i, j, k;
  char c;
  char line[50][1024]; /* une ligne ne peut dpasser 1024 octets. */
  int linecount;
  char prefix[512];
  char suffix[512];
  int state = 0;
  int headerslen = 0;
  
  prefix[0] = 0;
  suffix[0] = 0;

  if (headers == NULL) 
    return NULL; 
  else 
    headerslen = strlen(headers);
  
  icy = (icyHeaders *)malloc(sizeof(icyHeaders));
  
  /* Usefull when radio doesn't specify icy name */
  
  icy->name         = NULL;
  icy->content_type = NULL;
  icy->url          = NULL;
  icy->notice1      = NULL;
  icy->notice2      = NULL;
  icy->genre        = NULL;
  
  for(i = 0, j = 0, linecount = 0; i < headerslen; i++)
  {
    if (linecount >= 50) break;
    if (j >= 1023) 
    {
      line[linecount][j] = 0;
      linecount++;
      j = 0;
    }

    c = headers[i];

    if ((c == '\r') && (j == 0))
      break;
    
    if (c == '\n')
    {
      line[linecount][j] = '\n';
      line[linecount][j+1] = '\0';
      linecount++;
      j = 0;
    }
    else if (c != '\r')
      line[linecount][j++] = c;
  }

  for (i = 0; i < linecount; i++)
  {
    prefix[0] = 0;
    suffix[0] = 0;
    k = 0;
    state = 0;
    
    for (j = 0; j < strlen(line[i]); j++)
    {
      c = line[i][j];
      
      if (state == 0)
      {
        if (c == ':')
        {
          prefix[k] = '\0';
          k = 0;
          state = 1;
        } else
          prefix[k++] = c;
      
      } else if (state == 1) { /* On a rencontr les ":" */
        if (c != '\n')
        {
          if ((c == ' ') && (k == 0))
            continue;
          suffix[k++] = c;
        } else
          suffix[k] = '\0';
      }
    } /* for each linechar */
    
    if ((prefix[0] != '\0') && (suffix[0] != '\0'))
    {
      if (strncmp(prefix, "icy-notice1", 11) == 0)
        icy->notice1 = strdup(suffix);

      else if (strncmp(prefix, "icy-notice2", 11) == 0)
      {
        if (strstr("SHOUTcast", suffix) != NULL)
          icy->type = SERVERTYPE_SHOUTCAST;

        icy->notice2 = strdup(suffix);
      }

      else if (strncmp(prefix, "icy-name", 8) == 0)
        icy->name = strdup(suffix);

      else if (strncmp(prefix, "icy-genre", 9) == 0)
        icy->genre = strdup(suffix);

      else if (strncmp(prefix, "icy-url", 7) == 0)
        icy->url = strdup(suffix);

      else if (strncmp(prefix, "icy-pub", 7) == 0)
        icy->pub = (int) atoi((char *) &suffix);

      else if (strncmp(prefix, "icy-metaint", 11) == 0)
        icy->metaint = (int) atoi((char*) &suffix);

      else if (strncmp(prefix, "icy-br", 6) == 0 )
        icy->br = (int) atoi((char *) &suffix[0]);

      else if ((strncmp(prefix, "content-type", 12) == 0 ) ||
        (strncmp(prefix, "Content-Type", 12) == 0))
        icy->content_type = strdup(suffix);

      else if (strncmp(prefix, "Server", 6) == 0)
      {
        if (strstr(suffix, "Icecast") != NULL)
          icy->type = SERVERTYPE_ICECAST;
      }
    }
  } /*  for eachline */
  
  
  if (icy->name == NULL)
    icy->name = strdup("No Name");

  if (icy->content_type == NULL)
    icy->content_type = strdup("audio/mpeg");

  (void)free(headers);
  return icy;
}

cPigeUptime *getUptime(long long unsigned int uptime)
{
  cPigeUptime *cu;
  
  cu = (cPigeUptime *)memory_allocation(sizeof(cPigeUptime));

  /* Calculs bourrins :) */
  cu->day  = (unsigned int) (uptime / (60 * 60 * 24));
  cu->hour = (unsigned int) (uptime / (60 * 60)) - (cu->day * 24);
  cu->min  = (unsigned int) (uptime / (60)) - ((cu->hour * 60) + (cu->day * 24 * 60));
  cu->sec  = (unsigned int) (uptime) - ((cu->min * 60) + (cu->hour * 60 * 60) + (cu->day * 24 * 60 * 60));
  
  return cu; 
}

char *getStats(long long unsigned int uptime, long long unsigned int metacounter, int metaint)
{
  char *line;
  cPigeUptime *cu;
  
  cu = getUptime(uptime);
  line = (char *)memory_allocation(300); /* Exessif. */
  
  snprintf(line, 299, "Uptime: %d days, %d hours, %d min, %d seconds\nDownloaded: %lldKo\nSongs played: %ld\nReconnections: %ld\n", cu->day, cu->hour, cu->min, cu->sec, (long long unsigned int)((metacounter*metaint)/1024), cPigeStats->songs, cPigeStats->reconnections);

  free(cu);
  return line;
}

char *statusLine(long long unsigned int uptime, long long unsigned int metacounter, int metaint, char *titre, char *nexttitre)
{
  char *line;
  int len;
  cPigeUptime *cu;
  
  cu = getUptime(uptime);
  
  /* Pas terrible... */
  line = memory_allocation(300);
  
  len = snprintf(line, 299, "[%dj %dh:%dm:%ds][%lldKo] ", cu->day, cu->hour, cu->min, cu->sec, (long long unsigned int)((metacounter * metaint) / 1024));
  
  if (cmdLine->pige)
    len += snprintf(line + len, 299-len, "%dh -> %dh ", getHour(), getHour()+1);
  
  if (titre != NULL)
    len += snprintf(line+len, 299-len, "%s", titre);
  
  if (nexttitre != NULL) 
    if (strstr(nexttitre, titre) == NULL) 
      snprintf(line+len, 299-len, " -> %s", nexttitre);

  free(cu);
  return line;
}

/* Cette fonction se reconnecte au server servername,
 * port serverport, toute les time secondes, avec un
 * maximum de _nb_tentatives.
 *
 * Si _nb_tentatives est NULL, ou gal a -1, alors il n'y 
 * a aucune limite.
 *
 * Retourne la chausette cre
 */
int reconnect(int time, int _nb_tentatives, int _get_headers)
{
  int tentatives = 0;
  int server_socket;
  serverSettings_t *settings;
  icyHeaders *icy;
  
  if (!time) time = 2;
  
  if (serversPool == NULL)
  {
    _ERROR("Error: not any server defined.\n");
    exit(1);
  }
  
  /* Get an entry in the pool */
  settings = getSettings(serversPool, &poolPosition);
  if (settings == NULL)
  {
    _ERROR("No valid settings in urlPool.\n");
    exit(1);
  }

RECO:
  sleep(time);

  while ( (server_socket = server_connect(settings->serverAddr, settings->port)) < 0)
  {
    /* Get an entry in the pool */
    settings = getSettings(serversPool, &poolPosition);
    if (settings == NULL)
    {
      _ERROR("No valid settings in urlPool.\n");
      exit(1);
    }

    tentatives++;
    if ((tentatives > _nb_tentatives) && (_nb_tentatives != -1))
    {
      MESSAGE("Too many tentatives. Exiting program.\n");
      exit(-2);
    }
    MESSAGE("Reconnecting to http://%s:%d%s [try %d] in %d sec.\n", settings->serverAddr, settings->port, settings->mountpoint, tentatives, time);
    sleep(time);
  }

  VERBOSE("Time spent to reconnect: %d seconds, %d tentatives.\n", (tentatives * time), tentatives);

  if (sendHeaders(server_socket, settings->mountpoint, 1) <= 0)
  {
    _ERROR("Error sending headers: 0 byte sent.\n");
    goto RECO;
  }
  
  if (_get_headers)
  {
    if ( (icy = readicyheaders(getHeaders(server_socket))) == NULL)
      goto RECO;
    else
      free(icy);
  }
  
  cPigeStats->reconnections++;
  
  return server_socket;
}

int sendHeaders(int serversocket, char *mountpoint, int metadata)
{
  int ret = 0;
  
  char *headers;
  headers = (char *)memory_allocation(256);
  
  if (mountpoint == NULL)
    snprintf(headers, 255, "GET / HTTP/1.0\r\n");
  else
    snprintf(headers, 255, "GET %s HTTP/1.0\r\n", mountpoint);

  if (metadata != 0)
    strncat(headers, "Icy-MetaData:1\r\n", 16);
  else
    strncat(headers, "Icy-MetaData:0\r\n", 16);

  strncat(headers, "User-Agent:", 11);
  strncat(headers, USER_AGENT, strlen(USER_AGENT));
  strncat(headers, "\r\n\r\n", 4);
  
  ret = send(serversocket, headers, strlen(headers), 0);
  (void)free(headers);
  return ret;
}


char *readMeta(int serversocket)
{
  char *meta;
  char c;
  int retval;
  int size = 0;
  int readsize = 0;
  char *buffer;
  
  /* For select() it's a global struct. */
  timeout.tv_sec = SOCKET_TIMEOUT;
  timeout.tv_usec = 0;

  retval = select(server_socket+1, &rfds, NULL, NULL, &timeout);
  if (retval <= 0)
  {
    _ERROR("Erreur de connexion!\n");
    server_close(server_socket);
    return NULL;
  } else if (recv(server_socket, &c, 1, 0) != 1) {
    _ERROR("Error reading from server socket: \n", strerror(errno));
    server_close(server_socket);
    return NULL;
  }
  
  if (c > 0)
  {
    meta   = (char *)memory_allocation((c*16)+2);
    buffer = (char *)memory_allocation((c*16)+1);

    /* For select() it's a global struct. */
    timeout.tv_sec = SOCKET_TIMEOUT;
    timeout.tv_usec = 0;
    retval = select(server_socket+1, &rfds, NULL, NULL, &timeout);
    if (retval <= 0)
    {
      _ERROR("Connection error in select. (readmeta)\n");
      (void)free(buffer);
      (void)free(meta);
      server_close(server_socket);
      return NULL;

    } else {
      readsize = 0;
      while(readsize < c*16)
      {
        /* For select() it's a global struct. */
        timeout.tv_sec = SOCKET_TIMEOUT;
        timeout.tv_usec = 0;

        retval = select(server_socket+1, &rfds, NULL, NULL, &timeout);
        if (retval <= 0)
        {
          _ERROR("Erreur de connexion!\n");
          server_close(server_socket);
          return NULL;
        } else {
          size = recv(server_socket, buffer, (c*16 - readsize), 0);
          if (size <= 0)
          {
            VERBOSE("Megaproblem here.\n");
            server_close(server_socket);
          }
          readsize += size;
          strncat(meta, buffer, size);
        }
      }
      (void)free(buffer);
    }
  } else {
    /* Title was not sent. */
    return NULL;
  }
  return meta;
}


/* Lorsque le serveur envoie: "StreamTitle='TITREZIK';"
 * On ne rcupre avec cette fonction que TITREZIK.
 */

char *getTitle(char *titleString)
{
  int i;
  char *result;
  int count = 0;
  int j = 0;
  
  if ((titleString == NULL) || (strlen(titleString) == 0))
    return NULL;
  
  result = (char *)memory_allocation(strlen(titleString) + 1);
  
  for (i=0; i < strlen(titleString); i++)
  {
    if (count == 1) {
      if (titleString[(i+1)] != ';')
        result[j++] = titleString[i];
    }
    
    if (titleString[i] == '\'')
    {
      if (count == 0) { /* Premiere fois qu'on rencontre un ', on count++; */
        count++;
      } else if (titleString[(i+1)] == ';') { /*  Count n'est pas gal a 0, ici */
        result[j++] = '\0';
        break;
      }
    }
  } /* for */
  return result;
}

currentSong *getCurrentSong(char *title) 
{
  currentSong *cursong;
  char c;
  int i;
  int j = 0;
  int flag = 0;

  if (title == NULL) return NULL;

  cursong = (currentSong *)malloc(sizeof(currentSong));
  
  cursong->title  = (char *)memory_allocation(30 + 1); /* id3v1 == 30 chars max */
  cursong->artist = (char *)memory_allocation(30 + 1); /* id3v1 == 30 chars max */
  
  for (i = 0; i < strlen(title); i++)
  {
    c = title[i];
    if (!flag) /* ! Flag */
    {
      if (c == '-') /* Avant le - => artiste */
      {
        if (cursong->artist[(j-1)] == ' ')
          cursong->artist[(j-1)] = '\0';

        flag = 1;
        j = 0; /* Pour l'espace */
      } else {
        if (j < 30) /* "l'espace!" */
          cursong->artist[j++] = c;
      }
    } else { /* Flag */
      if (j < 30)
      {
        if (j == 0 && c == ' ') continue;
        cursong->title[j++] = c;
      }
    }
  } /* boucle :) */

  if (strlen(cursong->title) == 0)
  {
    (void)free(cursong->title);
    cursong->title = NULL;
  }
  
  if (strlen(cursong->artist) == 0)
  {
    (void)free(cursong->artist);
    cursong->artist = NULL;
  }

  /* Last byte of artist & title in the structure
   * are 0 as they have been initialized with calloc.
   */
  return cursong;
}

int getHTTPCode(char *buffer)
{
  char c;
  char array[256]; /* Must check for overflow */
  int i;
  int j = 0;
  int flag = 0;
  int returnCode = 0;

  if ((buffer == NULL) || (strlen(buffer) > 256)) return 0;

  for (i = 0; i < strlen(buffer) && j < 256; i++)
  {
    c = buffer[i];

    if (c == ' ')
    {
      if (flag)
      {
        array[j+1] = 0;
        flag = 0;
      } else
        flag = 1;
    }
    
    if (flag)
      array[j++] = c;

  } /* for */

  array[j] = 0;
  returnCode = atoi((char *) &array);

  if (returnCode == 200)
  {
    return 1;
  } else {
    _ERROR("Error: return code from HTTP server: %d.\n", returnCode);
    return 0;
  }
}

char *getHeaders(int serversocket)
{
  int count = 0;
  int i = 0;
  int j;
  int retval;
  char *ptr;
  char headers[4096]; /* Must check for overflow */
  char *buffer = NULL;
  char c;

  buffer = (char *)memory_allocation(256);
  memset(headers, 0, 4096);

  /* For select() it's a global struct. */
  timeout.tv_sec = SOCKET_TIMEOUT;
  timeout.tv_usec = 0;

  retval = select(server_socket+1, &rfds, NULL, NULL, &timeout);
  if (retval <= 0)
  {
    _ERROR("Erreur de connexion in getHeaders().\n");
    goto error;
  }
  
  i = 0;
  
  while ( (recv(serversocket, &c, 1, 0) > 0) && (i < 256) )
  {
    if (c == '\r')
      continue;
    
    if (c == '\n')
      break;

    buffer[i++] = c;
  }

  if (getHTTPCode(buffer) != 1)
  {
    /* Erreur autre que 200... on attends. */
    goto error;
  }

  (void)free(buffer);
  buffer = NULL;
  
  headers[0] = 0;

  for(j = 0, i = 0; ((j < 4096) && (count != 4)); j++)
  {
    /* For select() it's a global struct. */
    timeout.tv_sec = SOCKET_TIMEOUT;
    timeout.tv_usec = 0;
    retval = select(server_socket+1, &rfds, NULL, NULL, &timeout);
    if (retval <= 0)
    {
      _ERROR("Connection error in select (getHeaders).\n");
      goto error;

    } else if (recv(server_socket, &c, 1, 0) != 1) {
      _ERROR("Error reading data in getHeaders()\n");
      goto error;
    }
    
    if (c == '\n' || c == '\r')
    {
       headers[i++] = c;
       count++;
    } else  {
       headers[i++] = c;
       count = 0;
    }
  }
  headers[i] = 0;
  
  if (!strlen(headers)) 
    return NULL;
  
  ptr = (char *)memory_allocation(strlen(headers)+1);
  strncpy(ptr, headers, strlen(headers));

  return ptr;
  
error:
  server_close(server_socket);
  if (buffer != NULL)
    (void)free(buffer);
  
  return NULL;
}

char *GetId3v1(char *titre, char *artiste, char *radioName)
{
  char *chunk;
  char *relativePointer;
  int padding = 0;
  
  chunk = (char *)memory_allocation(128);
  
  /* id3v1 tag */
  strncat(chunk, "TAG", 3);
  
  /* Title */
  relativePointer = chunk + 3; /* 3 octets */
  if (titre) {
    padding = (30 - strlen(titre));
    if (padding < 0)
    {
      snprintf(relativePointer+3, 31, "%s", titre);
    } else {
      snprintf(relativePointer, 128-3, "%s", titre);
      memset(relativePointer + (30-padding), 0, padding);
    }
  } else {
    memset(relativePointer, 0, 30);
  }
  
  /* Artist */
  relativePointer = relativePointer + 30; /* 33 octets */
  if (artiste) {
    padding = (30 - strlen(artiste));
    if (padding < 0) {
      snprintf(relativePointer, 31, "%s", artiste);
    } else {
      snprintf(relativePointer, 128-33, "%s", artiste);
      memset(relativePointer + (30 - padding), 0, padding);
    }
  } else {
    memset(relativePointer, 0, 30);
  }
  
  /* Album (on met l'url de la radio ...) */
  relativePointer = relativePointer + 30; /* 63 octets */
  if (radioName != NULL) {
    padding = (30 - strlen(radioName));
    if (padding < 0) {
      snprintf(relativePointer, 31, "%s", radioName);
    } else {
      snprintf(relativePointer, 128-63, "%s", radioName);
      memset(relativePointer + (30-padding), 0, padding);
    }
  } else {
    memset(relativePointer, 0, 30);
  }
  
  /* Year */
  relativePointer = relativePointer + 30; /* 93 octets */
  memset(relativePointer, 0, 4);
  
  /* Comment */
  relativePointer = relativePointer + 4; /* 97 octets */
  snprintf(relativePointer, 31, "by cPige http://ed.zehome.com/"); 
  /* Ouah la chance, a rentre! */
  
  /* genre */
  relativePointer = relativePointer + 30; /* 127 octets */
  memset(relativePointer, 1, 1);
  
  /* 128 bytes ! We won :) */
  return chunk;  
}


/* Only for debug, do not use ! */
char *bufToChar(char * string) 
{
  char c;
  char *retour;
  int i;
  retour = malloc(128 + 1);
  for (i=0; i<128; i++)
  {
    c = string[i];
    if (c == '\0')
      retour[i] = ' ';
    else
      retour[i] = c;
  }
  return retour;
}

int server_connect (char *servername, int serverport)
{
  struct sockaddr_in serverSockAddr;
  struct hostent *serverHostEnt;
  long hostAddr;
  
#if WIN32
  int res;
  VERBOSE("Using win32 sockets\n");
  WSADATA WSAData;
  if((res = WSAStartup(MAKEWORD(2,0), &WSAData)) != 0)
    printf("Impossible d'initialiser l'API Winsock 2.0\n");  
#endif
  
  VERBOSE("Entring Server_connect\n");
  
  /* on initialise la socket */
  memset(&serverSockAddr, 0, sizeof(serverSockAddr));
  VERBOSE("Servername: %s\n", servername);
  VERBOSE("Port: %d\n", serverport);
  hostAddr = inet_addr(servername); 
  
  /* If it is an ip address */
  if ((long) hostAddr != (long)-1)
  {
    memcpy(&serverSockAddr.sin_addr, &hostAddr, sizeof(hostAddr));
  } else {
    serverHostEnt = gethostbyname(servername);
    if (serverHostEnt == NULL)
    {
      _ERROR("Error with gethostbyname. exiting.\n");
      return -1;
    }
    memcpy(&serverSockAddr.sin_addr, serverHostEnt->h_addr, serverHostEnt->h_length);
  }
  serverSockAddr.sin_port = htons(serverport);
  serverSockAddr.sin_family = AF_INET; /* FIXEME: IPv6 Support ? */
  
  /* creation de la socket */
  if ( (server_socket = socket(AF_INET, SOCK_STREAM, 0)) < 0)
  {
    _ERROR("Error creating shoutcast socket. Exiting.\n");
    return -2;
  }
  
  VERBOSE("Socket Creation Sucessful.\n");
  VERBOSE("Connection in progress...\n");

  /* requete de connexion */
  if(connect( server_socket, (struct sockaddr *)&serverSockAddr, sizeof(serverSockAddr)) < 0 )
  {
    _ERROR("Remote host connection failed.\n");
    return -3;
  } else {
    VERBOSE("Connected.\n");
  } 
  
  FD_ZERO(&rfds);
  FD_SET(server_socket, &rfds);
  
  return server_socket;
}

int server_close (int serversocket)
{
  VERBOSE("Closing server connection.\n");
  shutdown(server_socket, 2);
  close(server_socket);
  server_socket = 0;
  VERBOSE("Server connection closed.\n");
  return -1;
}

void print_credits()
{
  printf ("cPige %s by Laurent Coustet (c) 2005\n", VERSION);
  return;
}

void print_help()
{
  printf ("cPige help. cPige is a Laurent Coustet product.\n"
          "For more informations about me and my software,\n"
          "please visit http://ed.zehome.com/\n\n"
#ifdef NOCONFIG
          "Usage: ./cpige -h http://stream-hautdebit.frequence3.net:8000/ -d /home/ed/Pige -l logfile.log\n\n"
          "    -h host to connect to.\n"
          "    -V show cpige Version.\n"
          "    -d directory save stream to this directory.\n"
          "    -P Pige mode (takes no argument), save stream by hour.\n"
          "    -M Use pige Meta: will write id3v1 tag (only usefull with pige mode).\n"
          "    -q Quite mode, does not output stream status.\n"
          "    -b Background mode (UNIX only) use cPige as a daemon.\n"
          "    -l Path to logfile.\n"
          "    -I [h|m] pige mode will cut on a hour by hour basis or min by min basis.\n"
          "    -i nb how many \"nb\" hour(s) or minute(s) we should wait before cutting.\n"
#ifndef WIN32
          "    -n cPige will append xxxx to file in 'non pige mode', where xxxx is a number.\n");
#else
          );
#endif
#else
          "Usage: ./cpige -c path/to/cpige.conf\n"
          "    -c path/to/cpige.conf.\n");
#endif
}

void print_icyheaders(icyHeaders *_icy_headers )
{
  if (_icy_headers == NULL)
    return;
  printf("metaint: %d\n", _icy_headers->metaint);
  printf("bitrate: %d\n", _icy_headers->br);
  printf("icy-notice1: %s\n", _icy_headers->notice1);
  printf("icy-notice2: %s\n", _icy_headers->notice2);
  printf("icy-name: %s\n", _icy_headers->name);
  printf("icy-genre: %s\n", _icy_headers->genre);
  printf("icy-url: %s\n", _icy_headers->url);
  printf("content_type: %s\n", _icy_headers->content_type);
}

commandLine *parseCommandLine(int argc, char **argv)
{
  commandLine *cmdLine;
  int i;
  char *c;

#ifndef NOCONFIG
  char *buffer;
  config_t *conf, *workconf;
#endif

  if (argc < 2)
  {
    print_help();
    exit(-1);
  }
  
  cmdLine = (commandLine *)malloc(sizeof(commandLine));
  
  /* default is not pige mode */
  cmdLine->pige  = 0;
  cmdLine->quite = 0;
  cmdLine->live  = 0;
  cmdLine->background   = 0;
  cmdLine->interval     = 1;
  cmdLine->intervalType = IVAL_HOUR;
  cmdLine->usePigeMeta  = 0;
  cmdLine->useNumbers   = 0;
  cmdLine->skipSongs    = 0;

  cmdLine->logFile    = "cpige.log";
  cmdLine->dir        = NULL;
  
  for (i = 1; i < argc; i++)
  {
    c = strchr("c", argv[i][1]);
    if (c != NULL)
    {
      /* from streamripper */
      if ((i == (argc-1)) || (argv[i+1][0] == '-'))
      {
        (void)print_help();
        fprintf(stderr, "option %s requires an argument\n", argv[i]);
        exit(1);
      }
    }
    
    switch (argv[i][1])
    {
      case 'V':
        printf("%s\n", USER_AGENT);
        exit(0);
        break;

#ifndef NOCONFIG
      case 'c':
        i++;
        configFile = strdup(argv[i]);
        break;
#else
      case 'h':
        i++;
        
        if (serversPool == NULL)
          serversPool = addPoolUrl(NULL, argv[i]);
        else
          addPoolUrl(serversPool, argv[i]);

        break;
        
      case 'd':
        i++;
        {
          int len = strlen(argv[i]);
          cmdLine->dir = (char *)memory_allocation(len+2);
          strncpy(cmdLine->dir, argv[i], len);

          /* TODO: Windows compatibility ? */
          if (cmdLine->dir[len-1] != '/')
          {
            cmdLine->dir[len] = '/';
            cmdLine->dir[len+1] = 0;
          }
        }
        break;
#ifndef WIN32      
      case 'n':
        cmdLine->useNumbers = 1;
        break;
#endif        
      case 'l':
        i++;
        cmdLine->logFile = (char *)memory_allocation(strlen(argv[i]) +1);
        strncpy(cmdLine->logFile, argv[i], strlen(argv[i]));
        break;
        
      case 'P':
        if (cmdLine->live == 1)
        {
          printf("You can't use Live Mode and Pige mode simultaneously.\n");
          exit(-1);
        }
        if (cmdLine->pige == 1)
          break;

        cmdLine->pige = 1;
        printf("Pige Mode activated.\n");
        break;
        
      case 'q':
        cmdLine->quite = 1;
        break;
      
      case 'b':
        cmdLine->quite = 1;
        cmdLine->background = 1;
        break;
      
      case 'L':
        if (cmdLine->pige == 1)
        {
          printf("You can't use Live Mode and Pige mode simultaneously.\n");
          exit(-1);
        }
        cmdLine->live = 1;
        printf("Live Mode activated.\n");
        break;
        
      case 'M':
        if (cmdLine->pige == 1)
          cmdLine->usePigeMeta = 1;
        
        break;
        
      case 'I':
        i++;

        if (cmdLine->pige != 1)
        {
          cmdLine->pige = 1;
          printf("Pige Mode activated.\n");
        }

        if ( *argv[i] == 'h' || *argv[i] == 'H' )
        {
          fprintf(stdout, "Setting interval type: HOUR\n");
          cmdLine->intervalType = IVAL_HOUR;
        } 
        else if ( *argv[i] == 'm' || *argv[i] == 'M' )
        {
          fprintf(stdout, "Setting interval type: MINUTE\n");
          cmdLine->intervalType = IVAL_MIN;
          if (cmdLine->interval == 1)
            cmdLine->interval = 30;
        }
        else
        {
          fprintf(stderr, "Unknown interval type.\n");
          exit(1);
        }
        break;
      
      case 'i':
        i++;
       
        if (cmdLine->pige != 1)
        {
          cmdLine->pige = 1;
          printf("Pige Mode activated.\n");
        }
        
        cmdLine->interval = atoi(argv[i]);
        if (cmdLine->interval == 0)
        {
          fprintf(stderr, "Invalid interval 0.\n");
          exit(1);
        }
        
        fprintf(stdout, "Setting interval: `%d'\n", cmdLine->interval);
        break;
        
#endif /* NOCONFIG */

      default:
        fprintf(stderr, "Unknown switch: `%s'\n", argv[i]);
        break;
    } /* switch */
  } /* for */

#ifndef NOCONFIG
  /* Use the config file parser :) */
  
  if (configFile == NULL)
    configFile = strdup("./cpige.conf");
  
  /* Configfile Readin */
  printf("Reading config file: %s\n", configFile);
  if ( (conf = parseConfig( configFile )) == NULL)
  {
    fprintf(stderr, "Could not read config from: %s.\n", configFile);
    exit(0);
  }

  /* Debug */
  /* _conf_printConfig(conf); */

  /* Setting up url pool */
  workconf = conf;
  while (workconf != NULL)
  {
    workconf = _conf_getValue(workconf, "url", &buffer);
    if (buffer == NULL)
      break;
    else
    {
      VERBOSE("Adding %s to the pool.", buffer);
      if (serversPool == NULL)
        serversPool = addPoolUrl(NULL, buffer);
      else
        addPoolUrl(serversPool, buffer);
    }
  }

  /* Setting up cpige common parameters */
  set_str_from_conf(conf, "savedirectory", &buffer, NULL, "savedirectory not found in config file.\n", 1);
  
  {
    int len = strlen(buffer);
    cmdLine->dir = (char *)memory_allocation(len+2);
    strncpy(cmdLine->dir, buffer, len);

    /* TODO: Windows compatibility ? */
    if (cmdLine->dir[len-1] != '/')
    {
      cmdLine->dir[len] = '/';
      cmdLine->dir[len+1] = 0;
    }
  }
  
  /* String values */
#ifndef WIN32
  set_str_from_conf(conf,  "pidfile", &(cmdLine->pidFile), "/var/run/cpige.pid", "Warning: no pid file defined. Using /var/run/cpige.pid\n", 0);
#endif
  set_str_from_conf(conf,  "logfile",   &(cmdLine->logFile), "./cpige.log", NULL, 0);
  set_str_from_conf(conf,  "nexttitle", &(cmdLine->Next),    "A suivre",    NULL, 0);

#ifndef WIN32
  set_str_from_conf(conf,  "locale", &buffer, "C", NULL, 0);

  if (setlocale(LC_TIME, buffer) == NULL)
    _ERROR("Error setting up locale: `%s'.\n", buffer);
#endif

  /* Int values */
  set_int_from_conf(conf, "cutdelay",  &(cmdLine->interval),  30, NULL, 0);
  set_int_from_conf(conf, "skipsongs", &(cmdLine->skipSongs), 0,  NULL, 0);
  set_str_from_conf(conf, "cuttype",   &buffer,              "h", NULL, 0);
  if ( (*buffer == 'h') || (*buffer == 'H') )
  {
    fprintf(stdout, "Setting interval type: HOUR\n");
    cmdLine->intervalType = IVAL_HOUR;
  } 
  else if ( (*buffer == 'm') || (*buffer == 'M') )
  {
    fprintf(stdout, "Setting interval type: MINUTE\n");
    cmdLine->intervalType = IVAL_MIN;
    if (cmdLine->interval == 1)
      cmdLine->interval = 30;
  } else {
    fprintf(stderr, "Unknown interval type: `%s'. Should be 'm' or 'h'\n", buffer);
    exit(1);
  }

  /* Boolean values */
  set_bool_from_conf(conf, "weekbackup", &(cmdLine->weekBackup),  0, NULL, 0);
  set_bool_from_conf(conf, "pigemode",   &(cmdLine->pige),        0, NULL, 0);
  set_bool_from_conf(conf, "pigemeta",   &(cmdLine->usePigeMeta), 1, NULL, 0);
  set_bool_from_conf(conf, "quitemode",  &(cmdLine->quite),       0, NULL, 0);
  set_bool_from_conf(conf, "background", &(cmdLine->background),  0, NULL, 0);
  set_bool_from_conf(conf, "usenumbers", &(cmdLine->useNumbers),  0, NULL, 0);

  /* TODO: I know there's a little memleak there,
   * as we are not freeing the conf chained list,
   * and not freeing var(s) / val(s)
   */

  if (cmdLine->background)
    cmdLine->quite = 1;

#endif /* Using config file */

  if (cmdLine->dir == NULL)
    cmdLine->dir = strdup("./");
 
  /* Don't forget to free it on exit ! */
  return cmdLine;
}

int getCloserInterval(int now, int interval)
{
  int tmp;

  tmp = (now % interval);
  return (now - tmp);
}

int mustCut(lastCut *cut)
{
  int ret = 0;
  int closer;
  
  if (cmdLine->intervalType == IVAL_HOUR)
  {
    closer = getCloserInterval(getHour(), cmdLine->interval);
    if (closer != cut->hour)
      ret = 1;
  } else {
    closer = getCloserInterval(getMinute(), cmdLine->interval);
    if (closer != cut->min)
      ret = 1;
  }
  
  /*
   * if (ret)
   * printf("must cut will return true: itype: %d ival: %d closer: %d hour: %d min: %d\n", cmdLine->intervalType, cmdLine->interval, closer, cut->hour, cut->min);
   */
  return ret;
}

/* Returns what song number we lastly saved
 * the stream to
 */

#ifndef WIN32
int getSongs(char *dir)
{
  DIR *dirp;
  char *filename;
  static regex_t *match = NULL;
  int songs = 0;
  int current = 0;
  struct dirent *cur_dir;
  
  if (match == NULL)
  {
    match = (regex_t *)malloc(sizeof(regex_t));
    if (regcomp(match, "^[0-9]+.*\\.(mp3|aac)$", REG_EXTENDED) != 0)
    {
      fprintf(stderr, "Regex compilation error... Please contact developper!\n");
      return 0;
    }
  }

  if ( (dirp = opendir(dir)) == NULL)
  {
    _ERROR("Unable to open directory: %s\n", strerror(errno));
    return 0;
  }

  while ((cur_dir = readdir(dirp)) != NULL)
  {
    filename = cur_dir->d_name;
    
    if ((*filename == '.') || (strcmp(filename, "..") == 0))
      continue;

    if (regexec(match, filename, 0, 0, 0) == 0)
    {
      /* Match */
      current = atoi(filename); /* A bit uggly.. */
      if (current > songs)
        songs = current;
    }
  }
  
  closedir(dirp);

  return songs;
}
#endif

int checkInterval()
{
  int ret = 0;
  if (cmdLine->intervalType == IVAL_HOUR)
  {
    if ((cmdLine->interval <= 0) || (cmdLine->interval > 12))
      ret = 1;
    
  } else if (cmdLine->intervalType == IVAL_MIN) {
    if ((cmdLine->interval <= 0 || cmdLine->interval > 30))
      ret = 1;
  } else {
    ret = 1;
    fprintf(stderr, "Intenal Error: intervalType unknown!\n");
  }
  
  return ret;
}

void testCommandLine()
{
  if (cmdLine == NULL)
  {
    (void) print_help();
    exit(-1);
  }

  if (serversPool == NULL)
  {
    (void) print_help();
    exit(-1);
  }

  if (checkInterval() != 0)
  {
    fprintf(stderr, "Incorrect interval specified. Exiting...\n");
    exit(1);
  }
}

/* Tries to create directories for week backup */
void checkWeekBackup()
{
  int i;
  char *dayName;
  char *dirName;
  int dirNameLen;
  time_t when;
  DIR *weekdir;
  
  if ((! cmdLine->weekBackup) || (cmdLine->pige == 0))
    return;

  when = time((time_t *) NULL);
  /* One week => 7 days ! */
  for (i = 0; i < 7; i++)
  {
    dayName = getDayName((time_t)(when+(i*86400)));
    
    dirNameLen = strlen(cmdLine->dir) + strlen(dayName) + 2;
    dirName = (char *) memory_allocation(dirNameLen);
    snprintf(dirName, dirNameLen, "%s%s/", cmdLine->dir, dayName);
    (void)free(dayName);

    /* Create output dir if does not exists! */
    if (( weekdir = opendir(dirName)) == NULL)
    {
      _ERROR("Unable to open %s for writing\n", dirName);
  #ifdef WIN32
      if (mkdir(dirName) != 0) {
  #else
      if (mkdir(dirName, 0755) != 0) {
  #endif
        _ERROR("Failed trying to create %s. Verify rights.\n", dirName);
        fprintf(stderr, "Failed trying to create %s. Verify rights.\n", dirName);
        exit(-1);
      } else {
        VERBOSE("Created %s\n", dirName);
        /* pige_dir = opendir(cmdLine->dir); */
      }
    } else {
      VERBOSE("Sucessfully opened %s\n", dirName);
      closedir(weekdir);
    }

    (void)free(dirName);
  }
}
