// This may look like C code, but it is really -*- C++ -*-
/*
 ************************************************************************
 *
 * 			Validating server-side TCP streams
 *
 * This code listens on a specified port. When a new connection
 * comes, the server accepts it, sends a timestamp, takes one (not too-long)
 * CRLF-terminated string, echoes it, and closes the connection.
 * The main server shuts down after accepting ACCEPTED_CONN_MAX connections.
 * This code tries to listen to a port as given by an env variable (VTCP_PORT)
 * (9000 is the default).
 * The server can be either forking or non-forking, as determined by
 * a #define'd parameter DO_FORK. Compile with a flag -DDO_FORK to get
 * a forking server.
 *
 * $Id: vTCPstream_server.cc,v 1.2 2000/11/15 03:35:54 oleg Exp oleg $
 *
 ************************************************************************
 */

#include "TCPstream.h"
#include "myenv.h"
#include <iostream.h>
#include <string.h>
#include <stdlib.h>
#include <time.h>
#if defined(unix)
#include <unistd.h>
#include <sys/wait.h>
#endif
				// Configurational parameters: adjust
				// if necessary
static int Server_port = atoi(xgetenv("VTCP_PORT","9000"));
static int ACCEPTED_CONN_MAX = 2; // Quit after accepting that many connections

				// Handle a client
static void handle_client(iostream& link)
{
  				// Fork from a parent, if desired
#if defined(DO_FORK)
  const pid_t kid_id = fork();
  if( kid_id != 0 )
  {			// We're in the parent process
    if( kid_id < 0 )
      CurrentNetCallback::on_error("Fork error");
    cerr << "\nLaunched a process " << kid_id << " to handle a new client"
	 << endl;
    return;			// The kid will do all the work...
  }
#endif
  time_t timestamp;
  link << "Tstamp: " << (time(&timestamp),ctime(&timestamp)) << endl;
  				// Echo until CRLF...
  char buffer [128];
  link.getline(buffer,sizeof(buffer)-1,'\r');
  assert( link.get() == '\n' );
  link << buffer;
  link.flush();
#if defined(DO_FORK)
  exit(0);		// if we're in a child process, exit
#endif
}

				// Root module
int main(void)
{
  cerr << "\n\nValidating TCPstream module...\n" << endl;
  cerr << "\nStaring a "
#if defined(DO_FORK)
       << "forking "
#else
       << "non-forking "
#endif
       << "server on port " << Server_port << endl;

  SocketAddr server_addr = SocketAddr(IPaddress(),Server_port);
  
  int listening_socket = socket(AF_INET,SOCK_STREAM,0);
  if( listening_socket < 0 )
   CurrentNetCallback::on_error("Socket creation error");
       
  {
    int value = 1;
    if( setsockopt(listening_socket, SOL_SOCKET, SO_REUSEADDR,
		   (char*)&value, sizeof(value)) < 0 )
      CurrentNetCallback::on_error("setsockopt SO_REUSEADDR error");
  }

  if( bind(listening_socket, (sockaddr *)server_addr,sizeof(server_addr)) < 0 )
   CurrentNetCallback::on_error("Socket binding error");

  if( listen(listening_socket, 5) < 0 )
   CurrentNetCallback::on_error("Socket listen error");
  
  cerr << "Entering the server loop..." << endl;
  
  TCPstream link;
  for(int conn_count=0; conn_count<ACCEPTED_CONN_MAX; conn_count++)
  {
    SocketAddr peeraddr = SocketAddr(IPaddress(),1);
    link.rdbuf()->accept(listening_socket,peeraddr);
    cerr << "\nAccepted connection from " << peeraddr << endl;
    handle_client(link);
    link.close();
  }
  cerr << "\nAccepted " << ACCEPTED_CONN_MAX << " connections" << endl;
#if defined(DO_FORK)
  while( ::wait(0) > 0 );	// wait until all forked children finish...
#endif
  close(listening_socket);
  cerr << "\nAll Done\n" << endl;
  return 0;
}
