// This may look like C code, but it is really -*- C++ -*-
//		Verify the vocabulary operations
//
// $Id: vvoc.cc,v 1.4 1999/12/24 02:49:50 oleg Exp oleg $


#include "myenv.h"
#include "std.h"
#include <fstream.h>

#include "voc.h"

				// Testing unloading/loading of a vocabulary
static void test_loading(void)
{
  cout << "\nTest of unloading/loading of a vocabulary" << endl;

  const char * temp_name = tmpnam(0);
  
  cout << "\tCreating a sample envelop vocabulary" << endl;
  Voc& voc_env = new_Voc("Load1Env","Sample Envelop Voc 1");
  {
    VocPathMark mark;
    cout << "\tCreating subvocs in the envelop vocabulary" << endl;
    Voc& voc_sub = new_subVoc("Load2Voc","Voc to unload");
    VocPath::push("Load2Voc");
    assert( voc_sub.q_empty() );
    new_DoubleVocItem("load2item1",3.5);
    new_IntVocItem("load2item2",91);
    new_StrVocItem("load2item3","string\"\"");
    cout << "\tunloading the subvoc to " << temp_name << endl;
    ofstream outs(temp_name);
    outs << voc_sub;
  }
  
  cout << "\tCheck to make sure the subvoc is out of path" << endl;
  assert( VocPath::find_with_default("load2item2",11) == 11 );
  assert( &(VocPath::find_with_default("Load2Env",Dummy_Voc)) == &Dummy_Voc );
  
  cout << "\tCreating a proxy" << endl;
  new_VocProxy("Load2Voc",temp_name);

  cout << "\tReferencing a proxy" << endl;
  VocPath::push("Load2Voc");
  assert( VocPath::find_double("load2item1") == 3.5 );
  assert( VocPath::find_with_default("load2item2",11) == 91 );
  assure( strcmp(VocPath::find_str("load2item3"),"string\"\"") == 0, "find_str failed" );
  
  cout << "\nDone\n";	
}

				// Testing lookups in the several vocabularies
				// (within the path) and operations on VocPath
static void test_vocpath(void)
{
  cout << "\nTest searching/updating the Voc path" << endl;

  {
    cout << "\tCreating a sample vocabulary for reference" << endl;
    Voc& voc_aux = new_Voc("Path1Voc","Sample Voc 1");
    assert( voc_aux.q_empty() );
    new_DoubleVocItem("path1item1",4.0);
    new_IntVocItem("path1item2",7);
    assert( voc_aux.count() == 2 );
  }

  {
    cout << "\tCreating another vocabulary for reference" << endl;
    Voc& voc_aux = new_Voc("Path2Voc","Sample Voc 2");
    assert( voc_aux.q_empty() );
    new_DoubleVocItem("path1item1",5.0);
    new_IntVocItem("path1item2",8);
    assert( voc_aux.count() == 2 );
  }

  {
    cout << "\tCheck retrieving items" << endl;
    assert( VocPath::find_double("path1item1") == 5.0);
    assert( VocPath::find_int("path1item2") == 8 );
    assert( VocPath::find_with_default("path1item1",6.0) == 5.0);
    assert( VocPath::find_with_default("path1item2",7.1) == 7.1);
    VocPath::pop();
    assert( VocPath::find_double("path1item1") == 4.0);
    assert( VocPath::find_int("path1item2") == 7 );
    assert( VocPath::find_with_default("path1item1",6.0) == 4.0);
  }

  VocPathMark mark;
  {
    cout << "\tCreating subvocs in the 1st vocabulary" << endl;
    Voc& voc_sub = new_subVoc("Path3Voc","Sample Voc 3");
    VocPath::push("Path3Voc");
    assert( voc_sub.q_empty() );
    new_IntVocItem("path1item1",15);
    new_IntVocItem("path1item2",9);
    assert( voc_sub.count() == 2 );
    Voc& voc_sub2 = new_subVoc("Path4Voc","Sample Voc 4");
    VocPath::push("Path4Voc");
    VocPath::print();
  }

  {
    cout << "\tCheck retrieving items" << endl;
    assert( VocPath::find_double("path1item1") == 4.0);
    assert( VocPath::find_int("path1item2") == 9 );
    assert( VocPath::find_with_default("path1item1",6.0) == 4.0);
    assert( VocPath::find_with_default("path1item1",7) == 15);
    mark.back_off();
    assert( VocPath::find_int("path1item2") == 7 );
  }

  cout << "\nDone\n";
}

				// Test writing/reading of a vocabulary
static void test_io(void)
{
  cout << "\nTest reading/writing of a vocabulary" << endl;

  {
    cout << "\tCreating a sample vocabulary for reference" << endl;
    Voc& voc_aux = new_Voc("SampleRefVoc","Sample IO Voc");
    assert( voc_aux.q_empty() );
    new_DoubleVocItem("aux voc item 1",4.0);
    new_IntVocItem("aux voc item 2",7);
    assert( voc_aux.count() == 2 );
  }

  cout << "\tCreating a sample vocabulary for io" << endl;
  Voc& voc1 = new_Voc("SampleVoc","Sample IO Voc");
  assert( voc1.q_empty() );
  assert( voc1.count() == 0 );

  cout << "\tAdd some items in the voc1" << endl;
  new_IntVocItem("1st item",4);
  new_DoubleVocItem("2nd item",5.0);
  new_StrVocItem("3d  item","sample string for io!");
  new_StrVocItem("4th  item","sample string\\n with a \" sign");
  Voc& subvoc = new_subVoc("5th item","a subvoc");
  VocPath::push(subvoc.q_name());
  new_IntVocItem("subvoc 1st item",14);
  VocPath::pop();
  assert( subvoc.count() == 1 );
  new_subVoc("7th item","empty subvoc");
  new_VocRefItem("SampleRefVoc");
  assert( voc1.count() == 7 );
  voc1.print("vocabulary before dumping");
  cout << voc1;

  const char * temp_name = tmpnam(0);
  {
    cout << "dumping the vocabulary into the file '" << temp_name << "'" 
         << endl;
    ofstream sout(temp_name);
    sout << voc1;
    assert( sout.good() );
    sout.close();
  }
  {
    cout << "reading the vocabulary from the file '" << temp_name << "'" 
         << endl;
    ifstream stin(temp_name);
    Voc& voc_read = new_Voc(stin);
    voc_read.print("read vocabulary");
    assert( voc_read.find_int("1st item") == 4 );
    assert( voc_read.find_with_default("2nd item",11.0) == 5.0 );
    assert( strcmp(voc_read.find_str("3d  item"),
		   "sample string for io!") == 0 );
    assert( strcmp(voc_read.find_str("4th  item"),
		   "sample string\\n with a \" sign") == 0 );
    assert( voc_read.find_voc("5th item").count() == 1 );
    assert( voc_read.find_voc("5th item").find_int("subvoc 1st item") == 14 );
    assert( voc_read.find_with_default("7th item",voc1).q_empty() );
    assert( voc_read.find_with_default("SampleRefVoc",voc1).count() == 2 );
  }
}
				// Test putting items of various types in a
				// voc and finding them
static void test_search_one_voc(void)
{
  cout << "\nTest putting items of various types in a voc and finding them\n";

  cout << "\tCreating a sample vocabulary\n";
  Voc& voc1 = new_Voc("SampleVoc","Sample Voc");
  assert( voc1.q_empty() );
  assert( voc1.count() == 0 );

  {	
    cout << "\tAdd some items in the voc1\n";
					// Add an int item
    const IntVocItem& item1 = new_IntVocItem("1st item",1);
    assert( !voc1.q_empty() );

					// Add a string item
    const StrVocItem& item2 = new_StrVocItem("1st str item","First str item");
  }
  assert( !voc1.q_empty() );
  assert( voc1.count() == 2 );
  assert( strcmp(voc1.top().q_name(),"1st str item") == 0 );
  assert( strcmp(voc1.bottom().q_name(),"1st item") == 0 );

  cout << "\tCreating a bigger vocabulary, voc2\n";
  Voc& voc2 = new_Voc("BiggerVoc","Bigger Voc");
  assert( voc2.q_empty() );
  assert( voc2.count() == 0 );

  cout << "\tAdding a few items\n";
  new_IntVocItem("voc2_1",10);
  new_DoubleVocItem("voc2_2",1.1);
  new_StrVocItem("voc2_3","str item voc2");
  new_VocRefItem(voc1);

  cout << "\tChecking the top and the bottom of voc2\n";
  assert( strcmp(voc2.top().q_name(),"SampleVoc") == 0 );
  assert( strcmp(voc2.bottom().q_name(),"voc2_1") == 0 );
  assert( voc2.count() == 4 );

  print_all_vocabularies();

  cout << "\tTrying to find all items put into voc2\n";
  assert( voc2.find_double("voc2_2") == 1.1 );
  assert( voc2.find_int("voc2_1") == 10 );
  assert( strcmp(voc2.find_str("voc2_3"),"str item voc2") == 0 );
  assert( voc2.find_voc("SampleVoc").count() == 2 );
  assert( voc2.find_voc("SampleVoc").find_int("1st item") == 1 );

  cout << "\tTrying to find non-existing items (this would fail)\n";
//  voc2.find_int("voc2_3");	// this should fail...

  cout << "\tChecking finding with defaults\n";
  assert( voc2.find_with_default("voc2_2",-1.0) == 1.1 );
  assert( voc2.find_with_default("voc2_3",-1) == -1 );
  assert( voc2.find_with_default("voc2_4",-1) == -1 );
  assert( strcmp(voc2.find_with_default("voc2_4","aa"),"aa") == 0 );
  assert( strcmp(voc2.find_with_default("voc2_3","aa"),"str item voc2") == 0 );
  assert( voc2.find_with_default("SampleVoc",voc2).count() == 2 );
  assert( voc2.find_with_default("Sample Voc",voc2).count() == 4 );
  assert( voc2.find_with_default("voc2_2",voc2).count() == 4 );

  cout << "\nDone\n";
}

				// Test very primitive operations
static void test_very_primitive(void)
{
					// Make a sample vocabulary
  Voc& new_voc = new_Voc("SampleVoc","Sample Voc");
  assert( new_voc.q_empty() );
  assert( new_voc.count() == 0 );

					// Add one item
  const IntVocItem& item1 = new_IntVocItem("1st item",1);
  assert( !new_voc.q_empty() );
  assert( new_voc.count() == 1 );
  item1.print("item1");

					// Add another item
  new_IntVocItem("2nd item",-1);
  assert( !new_voc.q_empty() );
  assert( new_voc.count() == 2 );

  print_all_vocabularies();		// and print everything
  VocPath::print();
}


int main(void)
{
//  test_very_primitive();
  test_search_one_voc();
  test_io();
  test_vocpath();
  test_loading();
  return 0;
}
