// This may look like C code, but it is really -*- C++ -*-
/*
 ************************************************************************
 *
 *		           Arithmetic Coding
 *	      	 Basics for the model of the data source
 *
 * The functions in the present file are responsible for providing the
 * arithmetic coder with the probability distribution for the data
 * to encode/decode. To be precise, the function should supply the
 * coder with the cumulative frequency count for the any symbol of interest.
 *
 * The present file provides the basic functionality that any input data
 * model is going to need. As a matter of fact, the functions described here
 * are all required by the interface with the arithmetic coder. Any particular
 * model of input source is to be derived from the Input_Data_Model class
 * and inherits all the basics, probably extending it to account for
 * the specific data model.
 *
 * The program assumes the total no. of distinct input symbols
 * (integers) is relatively small, so simple linear arrays can be used
 * for storing and looking up the frequency tables. 
 *
 * $Id: arithm_model.cc,v 2.2 1998/03/15 22:50:56 oleg Exp oleg $
 *
 ************************************************************************
 */

#include "arithm.h"


/*
 *------------------------------------------------------------------------
 *			Constructors and Destructors
 */

				// Constructor
Input_Data_Model::Input_Data_Model(void)
	: Top_freq(0), EOF_index(0),
	  no_indices(0),
	  frequencies(0), cum_frequencies(0)
{
}

void Input_Data_Model::allocate_model(const int no_symbols)
{
  assure(no_symbols > 1,
	 "Cannot construct input model for less than 2 symbols");
  no_indices = no_symbols+1;		// Because EOF_index is extra
  EOF_index  = no_indices;
				// Since the index is always >0, the 0th
				// element in these arrays is reserved for
				// the total freq (cum freq)
  frequencies     = new Lword[no_indices+1];
  cum_frequencies = new Lword[no_indices+1];
}

				// Destructor
Input_Data_Model::~Input_Data_Model(void)
{
  assert( frequencies != 0 && cum_frequencies != 0 );
  delete [] frequencies;
  delete [] cum_frequencies;
}
					// Set the Top_freq that is
					// required by coder
void Input_Data_Model::agree_on_top_freq(const int top_freq)
{
  assure(top_freq >= 255, 
	 "The top value for the cumulative frequency must be at least 255");
  Top_freq = top_freq;
  assure(total_cum_freq() <= Top_freq,
	 "Total cum_freq count is larger than the value set up by the codec");
}

/*
 *------------------------------------------------------------------------
 *		      Some index manipulation functions
 */
				// Find an index given its cum freq
				// Note that cum_fr *CAN* be zero!
				// In fact, cum_fr for EOF_index is
				// _exactly_ 0, see, say, arithm_modadapt.cc
Index Input_Data_Model::find_index(const Lword cum_fr) const
{
  if( cum_fr > cum_frequencies[0] )
    _error("find_index: cum frequency %d is out of boundaries [0,%d]",
	   cum_fr, cum_frequencies[0]);
  Index index;
  for(index=1; cum_frequencies[index] > cum_fr && index <= no_indices;
      index++)
    ;
  if( index > no_indices )    
    _error("Failure to find the index for the cumulative frequency %d",
	   cum_fr);
  return index;
}
