// This may look like C code, but it is really -*- C++ -*-
/*
 ************************************************************************
 *
 *		            Arithmetic Coding
 *	      	   Non-Adaptive model for the source of data
 *
 * This is an extension over the basic Input_Data_Model which makes it
 * an exact model of the input source by computing the histogram
 * first. The model thus obtained is exact providing the source of
 * information is stationary, ie doesn't change with the time. As a
 * trade-off, the model requires all the data to code be available
 * for constructing the histogram. That's the model requires one
 * more pass through the data (in addition to the coding pass).
 *
 * The program assumes the total no. of distinct input symbols
 * (integers) is relatively small, so simple linear arrays can be used
 * for storing and looking up the frequency tables. 
 *
 * $Id: arithm_modhist.h,v 1.1 1992/05/17 10:35:45 oleg Exp $
 *
 ************************************************************************
 */

#pragma once
#pragma interface

#include "arithm.h"

class HistModel;
				// Computes the histogram of the sequence
				// of integers with unit bin size
class Histogram
{
  friend HistModel;

  const Symbol symbol_lwb;		// Region potential input symbols
  const Symbol symbol_upb;		// are expected in
  const int no_potential_symbols;
  unsigned int * counts;		// Counts for the symbols

public:
					// Construct a model for symbols
					// in [lwb,upb]
  Histogram(const Symbol lwb, const Symbol upb);
  ~Histogram();

  void put(const Symbol symbol);	// Count the next symbol
  int no_distinct_symbols() const;	// Give the no. of distinct symbols
};

class HistModel : public Input_Data_Model
{
  Symbol symbol_lwb;			// Region potential input symbols
  Symbol symbol_upb;			// are expected in
  int no_potential_symbols;		// which are expected to occur
  int no_distinct_symbols;		// which have occurred

  Index  * symbol_to_index;	// Symbol-to-index conversion
  Symbol * index_to_symbol;	// Conversion from symbol index to symbol value

  void initialize_model(const Histogram& histogram);

public:
					// Construct a model for a given
					// histogram
  HistModel(const Histogram& histogram);
  HistModel();				// With parameters are assumed to
					// be read in
  ~HistModel();

					// Write out/Read in the histogram
					// to/from the coded stream
  void open_for_reading(EndianIO& file);
  void open_for_writing(EndianIO& file);

					// Model is non-adaptive, doesn't
					// have to be updated
  void update_model(const Index index) {}

					// Return the index of a symbol
  Index  get_index(const Symbol symbol) const;
					// and the symbol for an index
  Symbol get_symbol(const Index index) const;
};
