// This may look like C code, but it is really -*- C++ -*-
/*
 ************************************************************************
 *
 * 			Validating TCP streams
 *
 * This code tries to connect to an echo port of a local or remote (ECHO_HOST)
 * host, and send out progressively larger chunks of data. The code reads
 * the response, making sure that it was really an echo of what it was sent.
 *
 * $Id: vTCPstream.cc,v 1.4 1999/12/24 02:50:58 oleg Exp oleg $
 *
 ************************************************************************
 */

#include "TCPstream.h"
#include "myenv.h"
#include <iostream.h>
#include <string.h>


				// Configurational parameters: adjust
				// if necessary
static const char * const Echo_host_name = xgetenv("ECHO_HOST","127.0.0.1");
static const int  Echo_port = 7;

				// The following tells what chunks are being
				// sent
static const int Smallest_chunk = 10;
static const int Largest_chunk = 15000;
static const int Chunk_size_increment = 150;


class Chunk
{
  char * const data_ptr;	// Ptr to the chunk's data
  const size_t size;
  
  void fill_pattern(void);	// Fill the chunk with the pattern data
  void check_pattern(void);	// Check the chunk's pattern
  void destroy_pattern(void);
  
  Chunk(const Chunk&);		// Private and not implemented:
  void operator = (const Chunk&);	// cloning is not allowed

public:
  Chunk(const size_t _size);
  ~Chunk(void);
  friend ostream& operator << (ostream& os, const Chunk& chunk);
  friend istream& operator >> (istream& is, Chunk& chunk);
};

Chunk::Chunk(const size_t _size)
	: data_ptr(new char [_size]),
	  size(_size)
{
  fill_pattern();
}

Chunk::~Chunk(void)
{
  delete data_ptr;
}

			// Fill the chunk with the pattern data
void Chunk::fill_pattern(void)
{
  for(register size_t i=0; i<size; i++)
    data_ptr[i] = i+size;
}

			// Check the chunk's pattern set as above
void Chunk::check_pattern(void)
{
  for(register size_t i=0; i<size; i++)
    if( data_ptr[i] != (char)(i+size) )
      _error("Corrupted chunk's pattern: offset %d, expected %x, found %x\n",
             i,(char)(i+size),data_ptr[i]);
}

void Chunk::destroy_pattern(void)
{
  memset(data_ptr,0,size);
}

			// Write out a chunk
ostream& operator << (ostream& os, const Chunk& chunk)
{
  os.write(chunk.data_ptr,chunk.size);
  assert( os.good() );
  return os;
}

			// Read in a chunk and verify it
istream& operator >> (istream& is, Chunk& chunk)
{
  chunk.destroy_pattern();
  is.read(chunk.data_ptr,chunk.size);
  assert( is.good() );
  chunk.check_pattern();
  return is; 
}

static void run_test(const int smallest_chunk, const int largest_chunk,
		     const int chunk_size_increment)
{
  TCPstream tcp_stream;
  
  const SocketAddr target_addr(Echo_host_name,Echo_port);
  cerr << "Attempting to connect to " << target_addr << endl;
  
  tcp_stream.connect(target_addr);
  if( !tcp_stream.good() )
    cerr << "Connection failed!" << endl, exit(4);
    
  for(register int size=smallest_chunk; size <= largest_chunk;
      size += chunk_size_increment)
  {
    cerr << "sending chunk of size " << size << "...";
    Chunk * const chunk_ptr = new Chunk(size);
    tcp_stream << (*chunk_ptr);
    cerr << " receiving...";
    tcp_stream >> (*chunk_ptr);
    cerr << endl;
    delete chunk_ptr;
  }
  cerr << "\nDone\n" << endl;
}


class mycallbacks : public NetCallback
{
  bool async_io_hint(void)
  { cerr << "this async_io_hint will return true" << endl;
    return true;
  }
}; 
			// Root module
int main(void)
{
  cerr << "\n\nValidating TCPstream module...\n" << endl;
  
  {
    cerr << "...using a non-blocking i/o if implemented" << endl;
    mycallbacks mycallbacks_instance;
    run_test(Smallest_chunk,Largest_chunk,2*Chunk_size_increment);
   }
  
   cerr << "...reverting to default CurrentNetCallback" << endl;
   run_test(Smallest_chunk,Largest_chunk,Chunk_size_increment);

  cerr << "\nAll Done\n" << endl;
  return 0;
}
