"=====================================================================
" imaps.vim -- template expansions
"
" Cream -- An easy-to-use configuration of the famous Vim text editor
" [ http://cream.sourceforge.net ] Copyright (C) 2002-2004  Steve Hall
" 
" License:
" This program is free software; you can redistribute it and/or modify
" it under the terms of the GNU General Public License as published by
" the Free Software Foundation; either version 2 of the License, or
" (at your option) any later version.
" [ http://www.gnu.org/licenses/gpl.html ]
" 
" This program is distributed in the hope that it will be useful, but
" WITHOUT ANY WARRANTY; without even the implied warranty of
" MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
" General Public License for more details.
" 
" You should have received a copy of the GNU General Public License
" along with this program; if not, write to the Free Software
" Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
" 02111-1307, USA.

" Info: {{{1
" --------------------------------------------------------------------
" Source:
"
"        File: imaps.vim
"      Author: Srinath Avadhanula
"              ( srinath@fastmail.fm )
"         WWW: http://robotics.eecs.berkeley.edu/~srinath
" Description: insert mode template expander with cursor placement
"              while preserving filetype indentation.
" Last Change: Sun Mar 10 02:00 AM 2002 PST
"
" Note: Revisions of this script are now packed with vim-latex and are
" not regularly updated at SourceForge. See the CVS location: 
" http://cvs.sourceforge.net/cgi-bin/viewcvs.cgi/vim-latex/vimfiles/plugin/imaps.vim
" ......................................................................
" 
" Cream Adaptations:
" * Adapted the single find character for cursor placement to
"   a variable. This allows multiple characters to be used in addition
"   to abstracting the character itself.
" * Cream changed the mapping to Alt+Space
" * The mappings themselves have been drastically altered and
"   appended.
" * Prepended "imaps_" to all mappings so we can find them afterwards.
" * Commented out Snip() function.
"
" --------------------------------------------------------------------
"
" Description:
" this script provides a way to generate insert mode mappings which do not
" suffer from some of the problem of mappings and abbreviations while allowing
" cursor placement after the expansion. It can alternatively be thought of as
" a template expander. 
"
" Consider an example. If you do
"
" imap lhs something
"
" then a mapping is set up. However, there will be the following problems:
" 1. the 'ttimeout' option will generally limit how easily you can type the
"    lhs. if you type the left hand side too slowly, then the mapping will not
"    be activated.
" 2. if you mistype one of the letters of the lhs, then the mapping is
"    deactivated as soon as you backspace to correct the mistake.
"
" If, in order to take care of the above problems, you do instead
"
" iab lhs something
"
" then the timeout problem is solved and so is the problem of mistyping.
" however, abbreviations are only expanded after typing a non-word character.
" which causes problems of cursor placement after the expansion and invariably
" spurious spaces are inserted.
"
" this script attempts to solve all these problems by providing an emulation
" of imaps wchich does not suffer from its attendant problems. Because maps
" are activated without having to press additional characters, therefore
" cursor placement is possible. furthermore, file-type specific indentation is
" preserved, because the rhs is expanded as if the rhs is typed in literally
" by the user.
"  
" The script already provides some default mappings. each "mapping" is of the
" form:
"
" let s:<filetype>_<lhs> = "rhs"
"
" Consider a working example:
"
" let s:tex_bit  = "\\begin{itemize}\<cr>\\item \<cr>\\end{itemize}"
" 
" This effectively sets up the map
" 
" imap bit<leader>
"
" whenever you edit a latex file. i.e, if you type the leader character ('\'
" by default), after the word 'bit', then its expanded as follows:
"
" \begin{itemize}
" \item *
" \end{itemize}
"
" where * shows the cursor position. The special character  (typed as
" CTRL-K " . s:findchar . " a " . s:findchar . " :) decides the cursor placement after the expansion. If there
" is no , then the cursor is left at the end.
"
" however, unlike in mappings, it is not necessary to enter the keys
" b,i,t,<leader> in quick succession. this works by just mapping the last
" character, which is chosen to be <leader> instead of all the characters. a
" check is then made to see if the characters entered before match the total
" LHS of the required mapping.
" 
" NOTE: There are other plugins out there which do the same thing. For
" example, Dr. Chip's Cabbr script at 
"    http://users.erols.com/astronaut/vim/index.html#CAbbrv
" However, this script is a bit more general in that mappings for various file
" types can be put in the same place and also, instead of having to extend a
" function by putting additional 'if ...' lines, you directly type in
" additional mappings as variables.
" 
"--------------------------------------%<--------------------------------------
" Bonus: This script also provides a command Snip which puts tearoff strings,
" '----%<----' above and below the visually selected range of lines. The
" length of the string is chosen to be equal to the longest line in the range.
"--------------------------------------%<--------------------------------------

" 1}}}

" Abstract find char so we can experiment with conditioning across
" code languages and spoken (multi-byte) languages. Note that the
" width of the character is calculated at the approproate location so
" it can be deleted.
let s:findchar = "{+}"

" Map listing
" Cream_imap_map() {{{1
function! Cream_imap()
	
endfunction

" Cream_imap_map() {{{1
function! Cream_imap_map(ftype, mapping, string)
" set template mappings
" arguments (optional, three required):
"
"   Cream_imap_map({filetype}, {map}, {string})
"
"   {map} -- Word to complete template on. Must be in form applicable
"     to this function, i.e., with preceding fileformat and
"     underscore. (Maps applicable to all fileformats still require
"     leading underscore.)
"   {string} -- String to insert on completion. To pass double-quotes,
"     pass (itself contained in double-quotes) escaped with *three*
"     backslashes. To pass backslashes, use decimal 008 ().

	"+++ Cream: when passed arguments, use them to create
	"           script-scoped variable
	" if two arguments, auto-create variable
	if a:0 == 3
		" remember to quote the second string!
		execute "let g:imaps_" . a:1 . "_" . a:2 . " = \"" . a:3 . "\""

		" don't use empty for general in these arrays, use "general"
		if a:1 == ""
			let ft = ""
		else
			let ft = a:1
		endif
		" register with both filetype*s* array AND particular filetype array
		" * "g:imaps"      arrays all filetypes with templates available
		" * "g:imaps . ft" arrays each filetype's templates
		" initialize filetype's global ("g:imaps_html")
		if !exists("g:imaps_" . ft)
			" verify filetype is registered (so we don't have to search
			" through all variables later)
			" initialize
			if !exists("g:imaps")
				let g:imaps = ft . "\n"
			else
				" only add if not already in array
				if match(g:imaps, ft . "\\n") == -1
					execute "let g:imaps = g:imaps . \"" . ft . "\" . \"\\n\""
				endif
			endif
			" initialize container
			execute "let g:imaps_" . ft . " = ''"
		endif
		" append current to global registration for that filetype
		execute "let g:imaps_" . ft . " = g:imaps_" . ft . " . \"" . a:2 . "\" . \"\\n\""

		return

	endif
	"+++

	" unless we are at the very end of the word, we need to go back in order
	" to find the last word typed.
	"if virtcol('.') != virtcol('$')
		normal h
		let word = expand('<cword>')
		normal l
	"else
	"    let word = expand('<cword>')
	"end

	let rhs = ''
	if exists('g:imaps_' . &filetype . '_' . word)
	" first find out whether this word is a mapping in the current file-type.
		execute 'let rhs = g:imaps_' . &filetype . '_' . word
	elseif exists('g:imaps__' . word)
	" also check for general purpose mappings.
		execute 'let rhs = g:imaps__' . word
	end

	" if this is a mapping
	if rhs != ''
		" erase the map's letters with equal number of backspaces
		let bkspc = "\<BS>" . substitute(word, ".", "\<BS>", 'ge')

		" if the RHS contains an , then go there. this enables easy cursor
		" placement. also put a call to the SAImaps_RemoveLastHistoryItem()
		" functions in order to clean up the search history.
		"+++ Cream: use s:findchar in favor of orginal hard-coded ""
		"           to eliminate problems with multi-byte languages.
		"
		" trailing <Del> count must match length of "find char" - 1
		"let i = 0
		"let mydeletes = ""
		"while i < strlen(s:findchar)
		"    "let mydeletes = mydeletes . "\<DEL>"
		"    let i = i + 1
		"endwhile
		if stridx(rhs, s:findchar) != -1
			let i = strlen(s:findchar)
		else
			let i = 0
		endif
		if rhs =~ s:findchar
			if i > 0
				let movement = "\<C-o>?" . s:findchar . "\<CR>" . i . "s"
			else
				let movement = "\<C-o>?" . s:findchar . "\<CR>"
			endif
		else
			let movement = ""
		end

		return bkspc . rhs . movement

	else

		return ''

	end
endfunction

" 1}}}
" Map listing
" Cream_imaps_listvars() {{{1
function! Cream_imaps_listvars()
" register all imaps

	" compose display string
	let mystr = ""

	" get variable for each filetype registered
	let x = MvNumberOfElements(g:imaps, "\n")
	let i = 0
	while i < x
		" form: "g:imaps_html"
		let myvar = MvElementAt(g:imaps, "\n", i)
		"let myarray = g:imaps_{myvar}
		let y = MvNumberOfElements(g:imaps_{myvar}, "\n")
		let j = 0
		while j < y
			" form: "g:imaps_html_table"
			let myword = MvElementAt(g:imaps_{myvar}, "\n", j)
			let mytemplate = g:imaps_{myvar}_{myword}

			" compose string
			" header
			if j == 0
				let mystr = mystr . "\n"
				let mywidth = winwidth(0) - 1
				let k = 0
				while k < mywidth
					let mystr = mystr . "_"
					let k = k + 1
				endwhile
				let mystr = mystr . "\n"
				" empty is applicable to all filetypes
				if myvar == ""
					let mystr = mystr . "General Templates (available in all files)" . "\n\n"
				else
					let mystr = mystr . toupper(myvar) . " Templates (available in " . myvar . " files only)\n\n"
				endif
				let mystr = mystr . "WORD        TEMPLATE                                     \n"
				let mystr = mystr . "---------   -------------------------------------------- \n"
			endif

			" remove *initial* newline or return
			let mytemplate = substitute(mytemplate, '^[\n\r]', '', 'g')
			" remove dec 001-007 chars
			let mytemplate = substitute(mytemplate, '[-]', '', 'g')
			" remove dec 008 (<BS>)
			let mytemplate = substitute(mytemplate, '[]', '', 'g')
			" remove "<BS>"
			let mytemplate = substitute(mytemplate, '\<BS>', '', 'g')
			"" remove dec 009 (<TAB>)
			"let mytemplate = substitute(mytemplate, '[	]', '', 'g')
			" remove dec 011-012 chars
			let mytemplate = substitute(mytemplate, '[-]', '', 'g')
			" remove dec 014-031 chars
			let mytemplate = substitute(mytemplate, '[-]', '', 'g')
			"" trim off remaining returns and replace with ellipses ("...")
			"let mytemplate = substitute(mytemplate, "\n.*$", "...", "")
			" remove s:findchar
			let mytemplate = substitute(mytemplate, s:findchar, '', 'g')

			" space off templates from left margin
			let myspacer = "            "
			" preceed entries containing return second lines with a spacer
			if match(mytemplate, "\n") != -1
				let mytemplate = substitute(mytemplate, "\n", "\n" . myspacer, "g")
			endif

			" space single/first string
			let myspacerlen = strlen(myspacer)
			let myspacer = ""
			" insert character &textwidth number of times
			let k = 0
			while k < myspacerlen - strlen(myword)
				let myspacer = myspacer . " "
				let k = k + 1
			endwhile

			" cat
			let mystr = mystr . myword . myspacer . mytemplate . "\n"

			let j = j + 1
		endwhile
		let i = i + 1
	endwhile

	" set margin, right
	let k = 0
	let myright = ""
	while k < 1
		let myright = myright . " "
		let k = k + 1
	endwhile
	let mystr = substitute(mystr, "\n", '\n' . myright, 'g')
	" set margin, left
	let mywidth = winwidth(0) - 1
	let mystr = substitute(mystr, "\\v([^\n]{," . mywidth . "})[^\n]*", '\1', 'g')

	" now add pre-header explanation, so it can wrap/return without concerns
	let mystr = 
		\ "\n" .
		\ "    The listing below indicates words that can be expanded\n" . 
		\ "    into the adjacent template. The templates are grouped \n" .
		\ "    by the file type in which each is available.\n" .
		\ "\n" .
		\ "    Simply press Alt+Space with the cursor at the end\n" . 
		\ "    of the word. (Note that the word must be separated\n" .
		\ "    by a space or return from the following word.)\n" .
		\ "\n" .
		\ "    Press the Space Bar to continue... (Esc to quit)\n" .
		\ mystr

	echo mystr

endfunction
" 1}}}
" Mappings
" General {{{1

" date: "20 November 2002" (See F11 for stamps)
call Cream_imap_map("", "date", "\<C-r>=strftime('%d %B %Y')\<CR>")
" datestamp: "2002-12-04T11:16:27EST" (Same as F11x4)
call Cream_imap_map("", "datestamp", "\\<C-r>=strftime('%Y-%m-%dT%H:%M:%S%Z')\<CR>")

" URLs
call Cream_imap_map("", "url", "http://www.")
call Cream_imap_map("", "urlcream", "http://cream.sourceforge.net")
call Cream_imap_map("", "urlvim", "http://www.vim.org")
call Cream_imap_map("", "urlgpl", "http://www.gnu.org/licenses/gpl.html")

" ASCII chars 32-126 (Note "!" and " " are swapped for readibility)
call Cream_imap_map("", "ascii", "! \\\"#$%&'()*+,-./0123456789:;\<=>?@ABCDEFGHIJKLMNOPQRSTUVWXYZ[\\\\]^_`abcdefghijklmnopqrstuvwxyz{\|}~")

" a "ruler" -- nice for counting the length of words
call Cream_imap_map("", "ruler", "1234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890\n" .
	\ "       10^       20^       30^       40^       50^       60^       70^       80^       90^      100^\n")

" some important numbers
call Cream_imap_map("", "e", "2.7182818284590452353602874713526624977573")
call Cream_imap_map("", "pi", "3.1415926535897932384626433832795028841972")
" pi to 1,000 places, <http://arvindn.cjb.net/pi/> 
" each line is 50 chars. (who could resist, eh?)
call Cream_imap_map("", "PI",
	\ "141 592 6535 897 932 3846 264 338 3279 502 884 1971 693 993 7510\n" .
	\ "582 097 4944 592 307 8164 062 862 0899 862 803 4825 342 117 0679\n" .
	\ "821 480 8651 328 230 6647 093 844 6095 505 822 3172 535 940 8128\n" .
	\ "481 117 4502 841 027 0193 852 110 5559 644 622 9489 549 303 8196\n" .
	\ "442 881 0975 665 933 4461 284 756 4823 378 678 3165 271 201 9091\n" .
	\ "456 485 6692 346 034 8610 454 326 6482 133 936 0726 024 914 1273\n" .
	\ "724 587 0066 063 155 8817 488 152 0920 962 829 2540 917 153 6436\n" .
	\ "789 259 0360 011 330 5305 488 204 6652 138 414 6951 941 511 6094\n" .
	\ "330 572 7036 575 959 1953 092 186 1173 819 326 1179 310 511 8548\n" .
	\ "074 462 3799 627 495 6735 188 575 2724 891 227 9381 830 119 4912\n" .
	\ "983 367 3362 440 656 6430 860 213 9494 639 522 4737 190 702 1798\n" .
	\ "609 437 0277 053 921 7176 293 176 7523 846 748 1846 766 940 5132\n" .
	\ "000 568 1271 452 635 6082 778 577 1342 757 789 6091 736 371 7872\n" .
	\ "146 844 0901 224 953 4301 465 495 8537 105 079 2279 689 258 9235\n" .
	\ "420 199 5611 212 902 1960 864 034 4181 598 136 2977 477 130 9960\n" .
	\ "518 707 2113 499 999 9837 297 804 9951 059 731 7328 160 963 1859\n" .
	\ "502 445 9455 346 908 3026 425 223 0825 334 468 5035 261 931 1881\n" .
	\ "710 100 0313 783 875 2886 587 533 2083 814 206 1717 766 914 7303\n" .
	\ "598 253 4904 287 554 6873 115 956 2863 882 353 7875 937 519 5778\n" .
	\ "185 778 0532 171 226 8066 130 019 2787 661 119 5909 216 420 1989\n" .
	\ "  Source:  http://arvindn.cjb.net/pi/\n")

" standard abbreviations
call Cream_imap_map("", "i18n", "internationalization")
call Cream_imap_map("", "l10n", "localization")

" HTML {{{1

call Cream_imap_map("html", "table", "\<table cellpadding=\\\"4\\\" cellspacing=\\\"0\\\" border=\\\"0\\\">\n\<tr>\n\<TAB>\<td>\" . s:findchar . \"\</td>\n\<BS>\</tr>\n\</table>\n")
call Cream_imap_map("html", "tr", "\<tr>\n\<TAB>\<td>\" . s:findchar . \"\</td>\n\<BS>\</tr>\n")
call Cream_imap_map("html", "td", "\<td>\" . s:findchar . \"\</td>\n")
call Cream_imap_map("html", "p",  "\<p>\" . s:findchar . \"\</p>\n")
call Cream_imap_map("html", "a",  "\<a href=\\\"\" . s:findchar . \"\\\">\</a>")
call Cream_imap_map("html", "mail", "\<a href=\\\"mailto:\" . s:findchar . \"\\\">\</a>")
call Cream_imap_map("html", "b",  "\<b>\" . s:findchar . \"\</b>")
call Cream_imap_map("html", "i",  "\<i>\" . s:findchar . \"\</i>")
call Cream_imap_map("html", "u",  "\<u>\" . s:findchar . \"\</u>")
call Cream_imap_map("html", "ol", "\<ol>\n\<li>\" . s:findchar . \"\</li>\n\</ol>\n")
call Cream_imap_map("html", "ul", "\<ul>\n\<li>\" . s:findchar . \"\</li>\n\</ul>\n")
call Cream_imap_map("html", "li", "\<li>\" . s:findchar . \"\</li>\n")
call Cream_imap_map("html", "h1",  "\<h1>\" . s:findchar . \"\</h1>\n")
call Cream_imap_map("html", "h2",  "\<h2>\" . s:findchar . \"\</h2>\n")
call Cream_imap_map("html", "h3",  "\<h3>\" . s:findchar . \"\</h3>\n")
call Cream_imap_map("html", "h4",  "\<h4>\" . s:findchar . \"\</h4>\n")
call Cream_imap_map("html", "h5",  "\<h5>\" . s:findchar . \"\</h5>\n")
call Cream_imap_map("html", "h6",  "\<h6>\" . s:findchar . \"\</h6>\n")

" HTML character equivilences {{{1
"
" Are these really helpful? A list would be better.
"
"call Cream_imap_map("html", " ", "\<Left>\&nbsp;\<DEL>\n\<Left>")
call Cream_imap_map("html", "_", "\&nbsp;\n\<Left>" . s:findchar)

" HTML greek characters {{{1
"
" Cream: single letters are more valuable otherwise
"let s:imaps_html_a = "\&alpha;"
"let s:imaps_html_b = "\&beta;"
"let s:imaps_html_c = "\&chi;"
"let s:imaps_html_d = "\&delta;"
"let s:imaps_html_e = "\&epsilon;"
"let s:imaps_html_f = "\&phi;"
"let s:imaps_html_g = "\&gamma;"
"let s:imaps_html_h = "\&eta;"
"let s:imaps_html_k = "\&kappa;"
"let s:imaps_html_l = "\&lambda;"
"let s:imaps_html_m = "\&mu;"
"let s:imaps_html_n = "\&nu;"
"let s:imaps_html_p = "\&pi;"
"let s:imaps_html_q = "\&theta;"
"let s:imaps_html_r = "\&rho;"
"let s:imaps_html_s = "\&sigma;"
"let s:imaps_html_t = "\&tau;"
"let s:imaps_html_u = "\&upsilon;"
"let s:imaps_html_v = "\&varsigma;"
"let s:imaps_html_w = "\&omega;"
"let s:imaps_html_x = "\&xi;"
"let s:imaps_html_y = "\&psi;"
"let s:imaps_html_z = "\&zeta;"
"let s:imaps_html_A = "\&Alpha;"
"let s:imaps_html_B = "\&Beta;"
"let s:imaps_html_C = "\&Chi;"
"let s:imaps_html_D = "\&Delta;"
"let s:imaps_html_E = "\&Epsilon;"
"let s:imaps_html_F = "\&Phi;"
"let s:imaps_html_G = "\&Gamma;"
"let s:imaps_html_H = "\&Eta;"
"let s:imaps_html_K = "\&Kappa;"
"let s:imaps_html_L = "\&Lambda;"
"let s:imaps_html_M = "\&Mu;"
"let s:imaps_html_N = "\&Nu;"
"let s:imaps_html_P = "\&Pi;"
"let s:imaps_html_Q = "\&Theta;"
"let s:imaps_html_R = "\&Rho;"
"let s:imaps_html_S = "\&Sigma;"
"let s:imaps_html_T = "\&Tau;"
"let s:imaps_html_U = "\&Upsilon;"
"let s:imaps_html_V = "\&Varsigma;"
"let s:imaps_html_W = "\&Omega;"
"let s:imaps_html_X = "\&Xi;"
"let s:imaps_html_Y = "\&Psi;"
"let s:imaps_html_Z = "\&Zeta;"

" Vim {{{1

call Cream_imap_map("vim", "function",
	\ "function! \" . s:findchar . \"()\n" . 
	\ "\n" . 
	\ "endfunction\n")

call Cream_imap_map("vim", "while",
	\ "let i = 0\n" . 
	\ "while i \< \" . s:findchar . \"\n" . 
	\ "\n" . 
	\ "\<TAB>let i = i + 1\n" . 
	\ "\<BS>endwhile\n")

call Cream_imap_map("vim", "debug",
	\ "\\\"*** DEBUG:\n" . 
	\ "let n = confirm(\n" . 
	\ "\<TAB>\\\\ \\\"DEBUG:\\\\n\\\" .\n" . 
	\ "\\\\ \\\"  \" . s:findchar . \"myvar    = \\\\\\\"\\\" . myvar . \\\"\\\\\\\"\\\\n\\\" .\n" . 
	\ "\\\\ \\\"\\\\n\\\", \\\"&Ok\\\\n&Cancel\\\", 1, \\\"Info\\\")\n" . 
	\ "if n != 1\n" . 
	\ "return\n" . 
	\ "\<BS>endif\n" . 
	\ "\\\"***\n\<BS>")

call Cream_imap_map("vim", "confirm",
	\ "call confirm(\\n" . 
	\ "\<TAB>\\\\ \\\"\" . s:findchar . \"\\\\n\\\" .\\n" . 
	\ "\\\\ \\\"\\\\n\\\", \\\"&Ok\\\", 1, \\\"Info\\\")\n\<BS>")

" VB (Basic) {{{1

call Cream_imap_map("basic", "debug", 
	\ "\'*** DEBUG:\\n" . 
	\ "\Msg (\\\"Progress to here!\\\" & Chr(13) & _\\n" . 
	\ "\\<TAB>\\\"  \" . s:findchar . \"myvar  = \\\" & myvar & Chr(13) & _\\n" . 
	\ "Chr(13))\\n" . 
	\ "\'***\\n" . 
	\ "")

" Lisp {{{1

call Cream_imap_map("lisp", "debug", 
	'\<CR>(princ \"DEBUG:  \"' . s:findchar . 'myvar : \") (princ myvar) (princ \"' . 
	'\\n' . '\")\<CR>\<CR>')


"1}}}
" vim:foldmethod=marker
