/***********************************************************************
 *  Copyright (C) 2002   Samuel Benzaquen 
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 ***********************************************************************/
#include <stdlib.h>
#include <stdio.h>
#include <time.h>
#include <vector>
#include <string>
#include <iostream>
#include <algorithm>
#include <map>
#include <utility>

#include "contador.h"

using namespace std;


/*
  Transforma la salida del crossword a formato tex de forma que pueda
  ser compilado con latex
  La salida de `crossword' puede ser utilzada en crucigramas de definiciones
  y en crucigramas de relleno
  Por defecto genera un crucigrama de relleno.
  Utilizando la opcion `-d <definiciones>' genera en crucigrama de definiciones
  utilizando el archivo de definiciones especificado. Este archivo debe
  tener el siguiente formato:
  <pal1>: <def1.1> // <def1.2> // ... // <def1.n>
  <pal2>: <def2.1> // <def2.2> // ... // <def2.m>
  ...
*/

char *puzzle;

map <pair<int, int>, int> numeros;

map <string, vector<string> > definiciones;

void buscar_puntos(char *puzzle, int filas, int columnas){
	int numero = 0;
	for (int i = 1; i < filas-1; i ++){
		for (int j = 1; j < columnas-1; j++){
			if (((puzzle[(i*columnas) + j] != '*') &&
				  (puzzle[(i*columnas) + j + 1] != '*') &&
				  (puzzle[(i*columnas) + j - 1] == '*')) ||
				 ((puzzle[(i*columnas) + j] != '*') &&
				  (puzzle[((i+1)*columnas) + j] != '*') &&
				  (puzzle[((i-1)*columnas) + j] == '*'))){
				numeros[make_pair(i,j)] = ++numero;
			}
		}
	}
}

void leer_defs(FILE *defs){
	char linea[1024] = "0";
	set_punto(0,100);
	while(1){
		linea[0] = '\0';
		int c;
		for(c = fgetc(defs); (c != '\n') && (c != EOF); c = fgetc(defs)){
			sprintf(linea, "%s%c", linea, c);
		}
		if (strlen(linea) > 0){
			agregar_contador(0, stderr);
			string palabra = string(strtok(linea, ":"));
			for (char *def = strtok(NULL, "//"); def != NULL; def = strtok(NULL, "//")){
				definiciones[palabra].push_back(string(def));
			}
		}
		else{
			if (c == EOF)
				break;
		}
	}
	imprimir_contador(0, stderr);
}

void leer_puzzle(FILE *file, int filas, int columnas){
	puzzle = (char *) malloc(filas * columnas * sizeof(char));
	for (int i = 0; i < filas; i++){
		for (int j = 0; j < columnas; j++){
			char c;
			fscanf(file, "%c", &c);
			while((c != ' ') && (c != '*'))
				fscanf(file, "%c", &c);
			puzzle[i*columnas + j] = c;
		}
	}
}

void imprimir_encabezado(){
	cout << "\\documentclass[letterpaper]{article}" << endl;
	cout << "\\usepackage{texdraw}" << endl;
	cout << "\\addtolength{\\hoffset}{-1.0cm}" << endl;
	cout << "\\addtolength{\\textwidth}{2cm}" << endl;
	cout << "\\addtolength{\\voffset}{-1.0cm}" << endl;
	cout << "\\addtolength{\\textheight}{2cm}" << endl;
	cout << "\\begin{document}" << endl;
}

void imprimir_finalizacion(){
	cout << "\\end{document}" << endl;
}

//Imprime la matriz utilizando los numeros que existen en el mapa 'numeros'
void imprimir_matriz(char *puzzle, int filas, int columnas){
	cout << "\\centertexdraw{" << endl;
	float tamano = 0.22;
	//Se genera la matriz del crucigrama
	for (int i = 1; i < filas-1; i++){
		for (int j = 1; j < columnas-1; j++){
			cout << "\\move (" << j*tamano << " " << i*tamano << ")" << endl;
			cout << "\\lvec (" << j*tamano << " " << (i+1)*tamano << ")" << endl;
			cout << "\\lvec (" << (j+1)*tamano << " " << (i+1)*tamano << ")" << endl;
			cout << "\\lvec (" << (j+1)*tamano << " " << i*tamano << ")" << endl;
			cout << "\\lvec (" << j*tamano << " " << i*tamano << ")" << endl;
			if (puzzle[(filas-1-i)*columnas + j] == '*')
				cout << "\\lfill f:0.7" << endl;
			if (numeros[make_pair(i, j)] != 0){
				//Imprimir el numerito para los crucigramas
			}
		}
	}
	cout << "}" << endl;
}

int devolver_sig_palabra(FILE *file, char *palabra, int &i, int &j, int &v){
	if (fscanf(file, "%c", palabra) == EOF)
		return EOF;
	int fileeof;
	while(palabra[0] != '('){
		if ((fileeof = fscanf(file, "%c", palabra)) == EOF)
			return EOF;
	}
	if (fileeof == EOF)
		return EOF;
	fscanf(file, "%d", &i);
	fscanf(file, "%d", &j);
	fscanf(file, "%c", palabra);
	fscanf(file, "%c", palabra);
	if (palabra[0] == 'V')
		v = 1;
	else
		v = 0;
	fscanf(file, "%s",palabra);
	return !EOF;
}

void generar_fill_in(FILE *file, int filas, int columnas){
	vector<string> palabras[30];
//Se leen las palabras y se ignoran la posicion y orientacion de estas
//Se ordenan por tamano y alfabeticamente
	while(!feof(file)){
		char palabra[30];
		int i, j, v = 0;
		if (devolver_sig_palabra(file, palabra, i, j, v) == EOF)
			break;
		palabras[strlen(palabra)].push_back(string(palabra));
	}
	for (int i = 0; i < 30; i++)
		sort(palabras[i].begin(), palabras[i].end());
	//Se colocan en un solo vector y se agregan lineas del formato final
	vector<string> total;
	for (int i = 0; i < 30; i++){
		if (palabras[i].size() > 0){
			char letras[15];
			sprintf(letras, "%d letras", i);
			total.push_back(string(letras));
			for (unsigned int j = 0; j < palabras[i].size(); j++){
				string nuevo = "";
				for (unsigned int k = 0; k < palabras[i][j].size(); k++){
					//Se filtran letras no imprimibles normalmente por latex
					switch(palabras[i][j][k]){
						case '':
							nuevo += "\\~n";
							break;
						default:
							nuevo += palabras[i][j][k];
					}
				}
				total.push_back(nuevo);
			}
			total.push_back("");
		}
	}
	total.pop_back();
	//Se genera el archivo tex

	imprimir_encabezado();

	imprimir_matriz(puzzle, filas, columnas);

	int cols = 7;
	while(total.size() % cols)
		total.push_back("");
	//Se imprimen las palabras
	cout << "\\begin{center}\\begin{tabular}{";
	cout << "p{2.0cm}";
	for (int i = 1; i < cols; i++)
		cout << "@{} p{1.7cm}";
	cout << "}" << endl;
	for (unsigned int i = 0; i < (total.size()/cols); i++){
		cout << total[i];
		for (int j = 1; j < cols; j++){
			cout << " & " << total[i+((total.size()/cols))*j];
			fflush(stdout);
		}
		cout << "\\\\" << endl;
	}
	cout << "\\end{tabular}\\end{center}" << endl;

	imprimir_finalizacion();
}

void generar_crossword(FILE *file, int filas, int columnas){
	vector<pair<pair<int, int>, string> > palabrasV;
	vector<pair<pair<int, int>, string> > palabrasH;
	srand(time(NULL));
//Se ordenan por numero de crucigrama y se separan por orientacion
	while(!feof(file)){
		char palabra[30];
		int i, j, v = 0;
		if (devolver_sig_palabra(file, palabra, i, j, v) == EOF)
			break;
		if (v)
			palabrasV.push_back(make_pair(make_pair(i, j), string(palabra)));
		else
			palabrasH.push_back(make_pair(make_pair(i, j), string(palabra)));
	}
	sort(palabrasV.begin(), palabrasV.end());
	sort(palabrasH.begin(), palabrasH.end());

	//Se colocan en un solo vector y se agregan lineas del formato final
	vector<pair<pair<int, int>, string> > total;
	total.push_back(make_pair(make_pair(-1,-1),string("Horizontales:")));
	for (unsigned int i = 0; i < palabrasH.size(); i++){
		string nuevo = "";
		for (unsigned int k = 0; k < palabrasH[i].second.size(); k++){
			//Se filtran letras no imprimibles normalmente por latex
			switch(palabrasH[i].second[k]){
				case '':
					nuevo += "\\~n";
					break;
				default:
					nuevo += palabrasH[i].second[k];
			}
		}
		total.push_back(make_pair(palabrasH[i].first, nuevo));
	}
	total.push_back(make_pair(make_pair(-1,-1),string("Verticales:")));
	for (unsigned int i = 0; i < palabrasV.size(); i++){
		string nuevo = "";
		for (unsigned int k = 0; k < palabrasV[i].second.size(); k++){
			//Se filtran letras no imprimibles normalmente por latex
			switch(palabrasV[i].second[k]){
				case '':
					nuevo += "\\~n";
					break;
				default:
					nuevo += palabrasV[i].second[k];
			}
		}
		total.push_back(make_pair(palabrasV[i].first, nuevo));
	}

	//Se genera el archivo tex

	imprimir_encabezado();

	imprimir_matriz(puzzle, filas, columnas);

	int cols = 3;
	while(total.size() % cols)
		total.push_back(make_pair(make_pair(-1,-1), string("")));
	//Se imprimen las palabras
	cout << "\\begin{center}\\begin{tabular}{";
	cout << "p{4.3cm}";
	for (int i = 1; i < cols; i++)
		cout << " p{4.3cm}";
	cout << "}" << endl;
	for (unsigned int i = 0; i < (total.size()/cols); i++){
		if (numeros[total[i].first] != 0){
			cout << numeros[total[i].first] << ".- ";
			if (definiciones[total[i].second].size() > 0)
				cout << definiciones[total[i].second][rand() % definiciones[total[i].second].size()]; // agarrar una de las definiciones
			else
				cout << "$<$" << total[i].second << "$>$";
		}
		else
			cout << total[i].second;
		for (int j = 1; j < cols; j++){
			cout << " & ";
			if (numeros[total[i+((total.size()/cols))*j].first] != 0){
				cout << numeros[total[i+((total.size()/cols))*j].first] << ".- ";
				if (definiciones[total[i+((total.size()/cols))*j].second].size() > 0)
					cout << definiciones[total[i+((total.size()/cols))*j].second][rand() % definiciones[total[i+((total.size()/cols))*j].second].size()]; // agarrar una de las definiciones
				else
					cout << "$<$" << total[i+((total.size()/cols))*j].second << "$>$";
			}
			else
				cout << total[i+((total.size()/cols))*j].second;
			fflush(stdout);
		}
		cout << "\\\\" << endl;
	}
	cout << "\\end{tabular}\\end{center}" << endl;

	imprimir_finalizacion();
}

int main(int argc, char **argv){
	FILE *file = stdin, *defs = NULL;
	int def = 0;
	for (int i = 1; i < argc; i++){
		if (def){
			defs = fopen(argv[i], "r");
			if (!defs){
				fprintf(stderr, "Error al abrir el archivo de definiciones %s\n", argv[i]);
				exit(2);
			}
			def = 0;
		}
		else{
			if (!strcmp(argv[i], "-d"))
				def = 1;
			else{
				file = fopen(argv[i], "r");
				if (!file){
					fprintf(stderr, "Error al abrir el archivo %s.\n", argv[1]);
					exit (2);
				}
			}
		}
	}
	int filas, columnas;
	fscanf(file, "%d",&filas);
	fscanf(file, "%d",&columnas);

//Lee el puzzle del archivo
	leer_puzzle(file, filas, columnas);
	if (defs){ //Con definicion (crossword)
//Enumera los puntos (utilzado para el crossword)
		buscar_puntos(puzzle, filas, columnas);
		leer_defs(defs);
		generar_crossword(file, filas, columnas);
	}
	else{ //Sin definicion (fill-in puzzle)
		generar_fill_in(file, filas, columnas);
	}
	
}
