/*  cssed (c) Iago Rubio 2003, 2004 - A tiny CSS editor.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
 
/** @file plugin.h
 *  @brief Header file for cssed plugin interface. 
 *  Those functions are the plugable interface to extend cssed.
 */

/** @struct _CssedPluginPriv.  
 *  @brief This object is opaque and should be ignored.
 */	
typedef struct _CssedPluginPriv CssedPluginPriv;

/** @typedef CssedPlugin  
 *  @brief A data type for ::_CssedPlugin structures.
 */		
typedef struct _CssedPlugin CssedPlugin;

/** @struct _CssedPlugin.  
 *  @brief This object represents a plugin.
 * 	One ::CssedPlugin object's pointer must be returned with the function   
 *  ::CssedPlugin *init_plugin() exported by the module .
 *  <br><br>
 *	You can create this object static or reserve memory for it and then return
 *	it in the <b>init_plugin()</b> function .
 *	<br><br>
 *	@ref hello_world "[view example]"
 */
struct _CssedPlugin
{
	/** The plugin's name. One name to show to users in the plugin's list. */
	gchar *name; 		
	/** Plugin short description. One short description to show to users in the plugin's list.*/	
	gchar *description;						
	/** Plugin load function. It will be called by cssed to initialize the plugin, put all your UI construction and initialization code here. */
	gboolean (*load_plugin) (CssedPlugin*);	
	/** Plugin clean function. It will be called by cssed before unloading the plugin, put all your UI destruction and free all memory here. */
	void (*clean_plugin) (CssedPlugin*); 	
	/** Use this field to store plugin's needed data. You can use this field to pass any data needed by your implementation. */
	gpointer user_data;	
	/** Private data. This field is opaque and should be ignored */	
	CssedPluginPriv* priv;
};


/** To be used as a widget callback.  
 *	This function lets you unload the plugin from a signal, using it as the signal's callback.
 *	@param widget - The widget that caused the signal.
 *	@param plugin - A pointer to the plugin object.
 */
void cssed_plugin_unload_callback( GtkWidget* widget, gpointer plugin );
/** Adds a menu item to the plugin's menu.    
 *	This function lets you add an item to the plugin's menu, you must set the callbacks.
 *	yourself.
 *	@param plugin - A pointer to the plugin object.
 *  @param menu_item - The menu item widget to add.
 */ 
gboolean
cssed_add_menu_item_to_plugins_menu( CssedPlugin* plugin, GtkWidget* menu_item);
/** Adds a menu item to the toolbar.  
 *	This function lets you add an item to the plugin's menu, you must set the callbacks.
 *	yourself.
 *	@param plugin - A pointer to the plugin object.
 *  @param widget - The widget to add.
 *	@param tooltip_text - The text shown to the user in a tool tip box.
 *	@param tooltip_private_text - A biggest description to show to users.
 */
gboolean
cssed_plugin_add_widget_to_toolbar( CssedPlugin* plugin, 
							 GtkWidget* widget,
							 const char *tooltip_text,
							 const char *tooltip_private_text	);
/** Adds a widget to the application's footer panel.  
 *	This function lets you add an item to the footer's panel, you must set the callbacks.
 *	yourself.
 *	@param plugin - A pointer to the plugin object.
 *  @param widget - The menu item widget to add.
 *  @param label_text - A text to be shown in the notebook's tab label. 
 */							 
gboolean
cssed_plugin_add_page_with_widget_to_footer( CssedPlugin* plugin,
											GtkWidget* widget,
											gchar* label_text );
/** Removes a widget from the application's footer panel. 
 *	This function lets you remove a widget from the footer's panel
 *	@param plugin - A pointer to the plugin object.
 *  @param widget - The widget to be removed.
 */
gboolean
cssed_plugin_remove_page_with_widget_in_footer( CssedPlugin* plugin,
											GtkWidget* widget );

/** Adds a widget to the sidebar.  
 *	This function lets you add a widget to the sidebar, you must set the callbacks.
 *	yourself.
 *	@param plugin - A pointer to the plugin object.
 *  @param widget - The widget to add.
 *  @param label_text - A text to be shown in the notebook's tab label. 
 */		
gboolean
cssed_plugin_add_page_with_widget_to_sidebar( CssedPlugin* plugin,
											GtkWidget* widget,
											gchar* label_text );

/** Removes a widget from the application's side bar. 
 *	This function lets you remove a widget from the side bar.
 *	@param plugin - A pointer to the plugin object.
 *  @param widget - The widget to be removed.
 */										
gboolean
cssed_plugin_remove_page_with_widget_in_sidebar( CssedPlugin* plugin,
											GtkWidget* widget );

/** Selects the page containing a widget. 
 *	This function selects the page containing the widget passed in the sidebar notebook.
 *	@param plugin - A pointer to the plugin object.
 *  @param widget - The widget to be selected.
 */	
gboolean
cssed_plugin_select_page_with_widget_in_sidebar( CssedPlugin* plugin,
											GtkWidget* widget );

/** Selects the page containing a widget. 
 *	This function selects the page containing the widget passed in the footer notebook.
 *	@param plugin - A pointer to the plugin object.
 *  @param widget - The widget to be selected.
 */	
gboolean
cssed_plugin_select_page_with_widget_in_footer( CssedPlugin* plugin,
											GtkWidget* widget );

/** Shows an error message dialog.  
 *	It prompts a dialog to show an error to user,
 *	@param message - Explained error message.
 *  @param title - Brief error message. */ 
void
cssed_plugin_error_message( gchar* message, gchar* title );

/** Shows a confirmation dialog.  
 *	It prompts a cnofirmation dialog asking the user to make an action.
 *	@param message - Explained message.
 *  @param title - Brief message.
 *	@return a boolean with the user's delection. TRUE if accepts the option
 *	explained in the message FALSE otherwise.
 */
 
gboolean
cssed_plugin_confirm_dialog ( gchar* message, gchar* title);


/** Opens a file.  
 *	This function opens the file passed as "filename" in cssed.
 *	@param plugin - The plugin object.
 *  @param filename - Full path to the file to be opened.
 */
void
cssed_plugin_open_file( CssedPlugin* plugin, gchar* filename );

/** Shows a file selector dialog.  
 *	It prompts a dialog to let the user choose a file to be opened.
 *	@param plugin - The plugin object.
 */
void
cssed_plugin_open_file_with_fileselector( CssedPlugin* plugin );

/** Adds the text passed to the document.  
 *	It adds the text passed to the document at the anchor position. The whole
 *	operation will be flagged as a single undo action.
 *	@param plugin - The plugin object.
 *  @param text - The text to add to the document.
 */
void 
cssed_plugin_add_text_to_document( CssedPlugin* plugin, gchar* text );

/** Returns the character position at the start of a line.  
 *	It returns the character position as an integer, in the first character of 
 *	the line number passed.
 *	@param plugin - The plugin object.
 *  @param line - The line number to get the character position.
 *	@return an integer with the character position.
 */
gint
cssed_plugin_get_position_at_line ( CssedPlugin* plugin, gint line );

/** Gets a null terminated string with the current document's text.  
 *	You must free the returned valued if not NULL, with g_free.
 *	@param plugin - The plugin object.
 *  @return - A pointer to the text.
 *	@warning You must free the returned value with g_free when no longer needed.
 */
gchar*
cssed_plugin_get_text( CssedPlugin* plugin );

/** Selects a text range.  
 *	The selection is done from character position "start", to character 
 *	position "end".
 *	@param plugin - The plugin object.
 *  @param start - An integer containing the start of selection.
 *	@param end - An integer containing the end of selection.
 */
void
cssed_plugin_select_text_range( CssedPlugin* plugin, gint start, gint end );

/** Gets a null terminated string with selected text.  
 *	You must free the returned valued if not NULL, with g_free.
 *	@param plugin - The plugin object.
 *  @return - A pointer to the text or NULL if nothing is selected.
 *	@warning You must free the returned value with g_free when no longer needed.
 */
gchar*
cssed_plugin_get_selected_text(  CssedPlugin* plugin  );

/** Deletes the text selected.  
 *	If there's no selection this function has no effect.
 *	@param plugin - The plugin object.
 */
void
cssed_plugin_clear_selected_text( CssedPlugin* plugin );

/** Undoes one previous action.  
 *	It undoes the last text or character insertion in the document.
 *	@param plugin - The plugin object.
 */
void
cssed_plugin_undo_document_action ( CssedPlugin* plugin );

/** Redoes one previuos action.  
 *	It redoes the last action.
 *	@param plugin - The plugin object.
 */
void
cssed_plugin_redo_document_action ( CssedPlugin* plugin );

/** Backwards searches and selects some text..  
 *	It searches and selects the text passed second parameter going from caret position 
 *	to the document's start.
 *	@param plugin - The plugin object. 
 *  @param text_to_search - The text to search for.
 */
void
cssed_plugin_search_prev( CssedPlugin* plugin, gchar* text_to_search );

/** Searches and selects some text..  
 *	It searchs and selects the text passed as second parameter, searching from caret position 
 *	to the document's end.
 *	@param plugin - The plugin object. 
 *  @param text_to_search - The text to search for.
 */
void
cssed_plugin_search_next( CssedPlugin* plugin, gchar* text_to_search );

/** Inserts a css dialog.  
 *	It inserts a css dialog defining a CSS property's value to be used by auto completion
 *	and quick insertion.
 *	@param plugin - The plugin object. 
 *  @param key - A key defining this value, it must be surrounded by parens - as example (border).
 *	@param class - A ::CssedDialogClass defining the dialog.
 */
void 
cssed_plugin_css_dialog_insert( CssedPlugin* plugin, gchar* key, gpointer class );
	
/** Removes a css dialog.  
 *	It removes a css dialog defining a CSS property's value to be used by auto completion
 *	and quick insertion.
 *	@param plugin - The plugin object. 
 *  @param key - The key used to define this value.
 */	
void 
cssed_plugin_css_dialog_remove_by_keyword( CssedPlugin* plugin, gchar* key );

/** Returns a ::CssDialogClass defining a css dialog.  
 *	It returns a ::CssDialogClass defining a CSS proerty's value to be used by
 *	auto completion.
 *	@param plugin - The plugin object. 
 *  @param key - The key used to define this value.
 *	@return A pointer to the ::CssDialogClass you must cast it with (CssedDialogClass*)
 * 	@code
	#include "plugin.h"
	#include "cssdialogs-class.h"
	#include "cssedwindow.h"

	CssedPlugin* plugin;
	CssedWindow* window;
	CssDialogClass* dialog_class;
	gchar* keyword = "(border)";
	GtkWidget* dialog;
	gchar* output_string;
	// ....
	
	dialog_class = (CssDialogClass*) cssed_plugin_css_dialog_lookup_by_keyword( plugin, key );
	if( dialog_class != NULL ){
		gint response;
		window = (CssedWindow*) cssed_plugin_get_window( plugin );
		dialog = dialog_class->create_function ( window, NULL, FALSE, FALSE);
		response = gtk_dialog_run( dialog );
		if( response = GTK_RESPONSE_OK ){
			output_string = dialog_class->get_string_function( window, NULL );
		}
	}  
	@endcode
 */	
gpointer 
cssed_plugin_css_dialog_lookup_by_keyword( CssedPlugin* plugin, gchar* key );

/** Returns a boolean indicating if the given keyword is being used.  
 *	The keyword is the trigger for most auto completion features.
 *	@param plugin - The plugin object. 
 *  @param key - The keyword.
 */	
gboolean 
cssed_plugin_css_dialog_keyword_in_use( CssedPlugin* plugin, gchar* key );

/** Returns a pointer to the main ::CssedWindow.  
 *	You must not free or modify this pointer. You must cast it to ::CssedWindow pointer - use (::CssedWindow*) or CSSED_WINDOW() at your convenience.
 *	@param plugin - The plugin object. 
 */	
gpointer
cssed_plugin_get_window( CssedPlugin* plugin );





