/* cStringBuffer: a file-like string buffer.
   Copyright (C) 2004 Mooneer Salem <mooneer@translator.cx>

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public
   License as published by the Free Software Foundation; either
   version 2.1 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public
   License along with this library; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

/* Include necessary includes. */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdarg.h>
#include <assert.h>
#include <unistd.h>
#include "config.h"

#ifdef HAVE_PTHREAD_H
#include <pthread.h>
#endif

#include "csb_private.h"
#include "csb.h"

/* csb_seek(): Seek to position in buffer.
   Parameters:
     buf: The buffer to operate on.
     pointer: the position to move to.
   Returns: nothing. If pointer is greater than the length of the string, it will
     seek to the end.
*/
void csb_seek(csb_buf *buf, size_t pointer) /*@modifies buf->fileptr, buf->ungetc_buffer, buf->ungetc_length@*/ {
	/* Perform sanity checks to make sure we have a valid buffer. */
	assert(buf != NULL && buf->string != NULL);

	/* If the position requested is past the end of the buffer, just seek
	   to the end of the buffer. */
	if (pointer > buf->stringsize) {
		pointer = buf->stringsize;
	}

	/* Update file pointer to point to the new position. */
	buf->fileptr = buf->string + pointer;

	/* Flush characters in ungetc buffer. */
	if (buf->ungetc_buffer != NULL) {
		free(buf->ungetc_buffer);
		buf->ungetc_buffer = NULL;
		buf->ungetc_length = 0;
	}
}

/* csb_rewind(): Rewind file pointer to the beginning.
   Parameters:
     buf: The buffer to operate on.
   Returns: nothing.
*/
void csb_rewind(csb_buf *buf) /*@modifies buf->fileptr, buf->ungetc_buffer, buf->ungetc_length@*/ {
	/* Call csb_seek() to seek to the beginning (common code). */
	csb_seek(buf, 0);
}

/* csb_tellpos(): Retrieve position of file pointer.
   Parameters:
     buf: The buffer to operate on.
   Returns: the position of the file pointer in the buffer.
*/
size_t csb_tellpos(csb_buf *buf) {
	/* Declare variables. */
	size_t retval;

	/* Perform sanity checks to make sure we have a valid buffer. */
	assert(buf != NULL && buf->string != NULL);

	/* Return the position of the file pointer. */
	retval = (size_t)((unsigned long)(buf->fileptr) - (unsigned long)(buf->string));

	return (retval);
}
