/* cStringBuffer: a file-like string buffer.
   Copyright (C) 2004 Mooneer Salem <mooneer@translator.cx>

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public
   License as published by the Free Software Foundation; either
   version 2.1 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public
   License along with this library; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

/* Include necessary includes. */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdarg.h>
#include <assert.h>
#include <unistd.h>
#include "config.h"

#ifdef HAVE_PTHREAD_H
#include <pthread.h>
#endif

#include "csb_private.h"
#include "csb.h"

/* csb_trunc(): Truncate string at current position.
   Parameters:
     buf: The buffer to operate on.
   Returns: true on success, false otherwise.
*/
int csb_trunc(csb_buf *buf) /*modifies buf->fileptr, buf->stringsize@*/ {
	/* Perform sanity checks to make sure we have a valid buffer. */
	assert(buf != NULL && buf->string != NULL);

	if (buf->stringsize == buf->length) {
		/* No need to truncate; return true. */
		return (TRUE);
	}

	/* Add '\0' at position after the current one. */
	*(buf->fileptr + 1) = '\0';
	buf->stringsize = (size_t)((unsigned long)(buf->fileptr) - (unsigned long)(buf->string)) + 1;

	return (TRUE);
}

/* csb_movebegin(): move the beginning of the string.
   Parameters:
     buf: The buffer to operate on.
   Returns: true on success, false otherwise.
*/
int csb_movebegin(csb_buf *buf) /*@modifies buf->fileptr, buf->string, buf->stringsize, buf->ungetc_buffer, buf->ungetc_length@*/ {
	/* Perform sanity checks to make sure we have a valid buffer. */
	assert(buf != NULL && buf->string != NULL);

	/* Move beginning of string. */
	memmove(buf->string, buf->fileptr, (buf->stringsize - ((unsigned long)(buf->fileptr) - (unsigned long)(buf->string))));
	buf->string[buf->stringsize - ((unsigned long)(buf->fileptr) - (unsigned long)(buf->string))] = '\0';
	buf->stringsize = (buf->stringsize - ((unsigned long)(buf->fileptr) - (unsigned long)(buf->string)));
	csb_rewind(buf);

	/* Return true for success. */
	return (TRUE);
}

/* csb_strcat(): Concatenate a string at the end without regards to the file pointer.
   Parameters:
     buf: The buffer to operate on.
     string: The string to append
   Returns: true on success, false otherwise.
*/
int csb_strcat(csb_buf *buf, /*@unique@*/ const char *str) /*@modifies buf->string, buf->stringsize, buf->length, buf->fileptr, buf->ungetc_buffer@*/ {
	/* Perform sanity checks to make sure we have a valid buffer. */
	assert(buf != NULL && buf->string != NULL);

	/* Dynamically make the buffer bigger until we're sure the string fits. */
	CSB_RESIZE_TO_FIT(buf, str);

	/* Call strcat() to concatenate the string. */
	strcat(buf->string, str);
	buf->stringsize += strlen (str);

	/* Return true; it was successful. */
	return (TRUE);
}

/* csb_cstring(): Returns pointer to string being stored.
   Parameters:
     buf: The buffer to operate on.
   Returns: pointer to string.
     (NOTE: the string is not intended to be modified.)
*/
/*@exposed@*/ char *csb_cstring(csb_buf *buf) {
	/* Perform sanity checks to make sure we have a valid buffer. */
	assert(buf != NULL && buf->string != NULL);

	/* Return pointer to user. */
	return (buf->string);
}

/* csb_length(): Returns the length of the string. 
   Parameters:
     buf: The buffer to operate on.
   Returns: the number of bytes in the string, beginning from the beginning.
*/
size_t csb_length(csb_buf *buf) {
	/* Perform sanity checks to make sure we have a valid buffer. */
	assert(buf != NULL && buf->string != NULL);

	/* Return buf->stringsize to user. */
	return (buf->stringsize);
}

/* csb_memorysize(): Returns the size of the buffer itself.
   Parameters:
     buf: The buffer to operate on.
   Returns: the number of bytes allocated.
*/
size_t csb_memorysize(csb_buf *buf) {
	/* Perform sanity checks to make sure we have a valid buffer. */
	assert(buf != NULL && buf->string != NULL);

	/* Return buf->length to user. */
	return (buf->length);
}
