/***************************************************************************
                           cfilemanager.cpp  -  description
                             -------------------
    begin                : Sat Aug 03 2002
    copyright            : (C) 2002-2004 by Mathias Kster
    email                : mathen@users.berlios.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#ifndef WIN32
#include <unistd.h>
#include <stdlib.h>
#endif

#include <string.h>

#include <dclib/dcos.h>
#include <dclib/core/cbytearray.h>
#include <dclib/cconfig.h>
#include <dclib/core/cstring.h>
#include <dclib/core/cstringlist.h>
#include <dclib/core/cdir.h>
#include <dclib/core/cfile.h>
#include <dclib/cdownloadmanager.h>
#include <dclib/dclib.h>
#include <dclib/core/cmanager.h>
#include <dclib/core/cbz.h>
#include <dclib/core/che3.h>
#include <dclib/csearchindex.h>
#include <dclib/hash/tigertree.h>
#include <dclib/core/cbase32.h>

#include "cfilemanager.h"

class CFileType : public CObject {
public:
	/** */
	CFileType() {};
	/** */
	virtual ~CFileType() {};

	/** */
	eFileTypes m_eFileType;
};

/** filetype list */
typedef struct filetype {
	const char* sExt;
	eFileTypes eFileType;
} filetype;

struct filetype FileType[] = {
	// MP3 2
	{ "AIFF", eftMP3 },
	{ "AU",   eftMP3 },
	{ "MID",  eftMP3 },
	{ "MP2",  eftMP3 },
	{ "MP3",  eftMP3 },
	{ "OGG",  eftMP3 },
	{ "RA",   eftMP3 },
	{ "WAV",  eftMP3 },

	// ARCHIVE 3
	{ "ACE", eftARCHIVE },
	{ "BZ2", eftARCHIVE },
	{ "CAB", eftARCHIVE },
	{ "EX_", eftARCHIVE },
	{ "GZ",  eftARCHIVE },
	{ "LZH", eftARCHIVE },
	{ "RAR", eftARCHIVE },
	{ "RPM", eftARCHIVE },
	{ "TAR", eftARCHIVE },
	{ "TGZ", eftARCHIVE },
	{ "ZIP", eftARCHIVE },
	{ "ZOO", eftARCHIVE },
	{ "Z",   eftARCHIVE },

	// DOCUMENT 4
	{ "CFG",   eftDOCUMENT },
	{ "CONF",  eftDOCUMENT },
	{ "CPP",   eftDOCUMENT },
	{ "CSS",   eftDOCUMENT },
	{ "C",     eftDOCUMENT },
	{ "DIZ",   eftDOCUMENT },
	{ "DOC",   eftDOCUMENT },
	{ "H",     eftDOCUMENT },
	{ "HLP",   eftDOCUMENT },
	{ "HTM",   eftDOCUMENT },
	{ "HTML",  eftDOCUMENT },
	{ "INI",   eftDOCUMENT },
	{ "INF",   eftDOCUMENT },
	{ "LOG",   eftDOCUMENT },
	{ "PHP",   eftDOCUMENT },
	{ "PS",    eftDOCUMENT },
	{ "PDF",   eftDOCUMENT },
	{ "SHTML", eftDOCUMENT },
	{ "SXW",   eftDOCUMENT },
	{ "TXT",   eftDOCUMENT },
	{ "RFT",   eftDOCUMENT },
	{ "RDF",   eftDOCUMENT },
	{ "XML",   eftDOCUMENT },

	// APPL 5
	{ "BAT",  eftAPPLICATION },
	{ "CGI",  eftAPPLICATION },
	{ "COM",  eftAPPLICATION },
	{ "DLL",  eftAPPLICATION },
	{ "EXE",  eftAPPLICATION },
	{ "HQX",  eftAPPLICATION },
	{ "JS",   eftAPPLICATION },
	{ "SH",   eftAPPLICATION },
	{ "SO",   eftAPPLICATION },
	{ "SYS",  eftAPPLICATION },
	{ "VXD",  eftAPPLICATION },

	// PICTURE 6
	{ "3DS",  eftPICTURE },
	{ "A11",  eftPICTURE },
	{ "ACB",  eftPICTURE },
	{ "ADC",  eftPICTURE },
	{ "ADI",  eftPICTURE },
	{ "AFI",  eftPICTURE },
	{ "AI",   eftPICTURE },
	{ "AIS",  eftPICTURE },
	{ "ANS",  eftPICTURE },
	{ "ART",  eftPICTURE },
	{ "B8",   eftPICTURE },
	{ "BMP",  eftPICTURE },
	{ "CBM",  eftPICTURE },
	{ "GIF",  eftPICTURE },
	{ "ICO",  eftPICTURE },
	{ "JPEG", eftPICTURE },
	{ "JPG",  eftPICTURE },
	{ "PIC",  eftPICTURE },
	{ "PNG",  eftPICTURE },
	{ "PCX",  eftPICTURE },
	{ "TGA",  eftPICTURE },
	{ "TIF",  eftPICTURE },
	{ "XPM",  eftPICTURE },

	// VIDEO 7
	{ "AVI",   eftVIDEO },
	{ "ASF",   eftVIDEO },
	{ "ASX",   eftVIDEO },
	{ "DAT",   eftVIDEO },
	{ "MOV",   eftVIDEO },
	{ "MOVIE", eftVIDEO },
	{ "MPG",   eftVIDEO },
	{ "MP4",   eftVIDEO },
	{ "MPEG",  eftVIDEO },
	{ "OGM",   eftVIDEO },
	{ "QT",    eftVIDEO },
	{ "RM",    eftVIDEO },
	{ "VIV",   eftVIDEO },
	{ "WMV",   eftVIDEO },
	{ "DIVX",  eftVIDEO },

	// END
	{ 0, eftALL }
};

/** */
CFileManager::CFileManager()
{
	m_pFileTypeList   = new CStringList();
	m_pFileNameList   = 0;
	m_nFileBaseIndex  = 0;
	m_pSearchIndex    = new CSearchIndex();
	m_pShareList      = new CShareList();

	m_pFileManagerInfo = new CFileManagerInfo();
	m_pFileManagerInfo->m_nProgress = 100;

	InitFileTypeList();

	if ( m_pShareList->Load() == FALSE )
	{
		CreateShareList();
	}
	else
	{
		if ( m_pSearchIndex->LoadIndex() == FALSE )
		{
			CreateShareList();
		}
		else if ( CConfig::Instance()->GetAutoRecreateShareList() )
		{
			if ( CalcShareSize() != m_pShareList->GetShareSize() )
			{
				CreateShareList();
			}
			else
			{
				m_pShareList->CreateList(m_pSearchIndex);
			}
		}
		else
		{
			m_pShareList->CreateList(m_pSearchIndex);
		}
	}

	if ( CConfig::Instance()->GetRecreateShareListTime() != 0 )
	{
		m_tCreateShareListTimeout = time(0)+CConfig::Instance()->GetRecreateShareListTime()*60*60;
	}
	else
	{
		m_tCreateShareListTimeout = 0;
	}

	// init timer
	m_pFileManagerCallback = new CCallback<CFileManager>( this, &CFileManager::FileManagerCallback );
	CManager::Instance()->Add( m_pFileManagerCallback );

	SetInstance(this);
}

/** */
CFileManager::~CFileManager()
{
	Stop();

	SetInstance(0);

	Lock();

	if ( m_pFileManagerCallback )
	{
		CManager::Instance()->Remove(m_pFileManagerCallback);
		delete m_pFileManagerCallback;
	}

	if ( m_pFileNameList )
	{
		delete m_pFileNameList;
		m_pFileNameList = 0;
	}

	if ( m_pFileManagerInfo )
	{
		delete m_pFileManagerInfo;
		m_pFileManagerInfo = 0;
	}

	if ( m_pFileTypeList )
	{
		delete m_pFileTypeList;
		m_pFileTypeList = 0;
	}

	if ( m_pShareList )
	{
		delete m_pShareList;
		m_pShareList = 0;
	}

	if ( m_pSearchIndex )
	{
		delete m_pSearchIndex;
		m_pSearchIndex = 0;
	}

	UnLock();
}

/** */
int CFileManager::GetShareBuffer( eShareBufferType type, CByteArray * sharebuffer, bool decompress )
{
	return m_pShareList->GetShareBuffer( type, sharebuffer, decompress );
}

/** */
unsigned long CFileManager::GetShareBufferSize( eShareBufferType type )
{
	return m_pShareList->GetShareBufferSize(type);
}

/** */
ulonglong CFileManager::GetShareSize()
{
	if ( m_pShareList )
		return m_pShareList->GetShareSize();
	else
		return 0;
}

/** */
void CFileManager::InitFileTypeList()
{
	int i;
	CFileType * filetype;

	for(i=0;FileType[i].sExt != 0;i++)
	{
		filetype = new CFileType();
		filetype->m_eFileType = FileType[i].eFileType;

		m_pFileTypeList->Add( FileType[i].sExt, filetype );
	}
}

/** return the filetype */
eFileTypes CFileManager::GetFileType( CString file )
{
	CString ext;
	eFileTypes res=eftUNKNOWN;
	CFileType * filetype;

	// set the filetype
	ext = CDir::Extension(file);

	if ( ext != "" )
	{
		if ( m_pFileTypeList->Get( ext.ToUpper(), (CObject*&)filetype ) == 0 )
		{
			res = filetype->m_eFileType;
		}
#if 0
		if ( res == eftUNKNOWN )
			printf("unknown extension: %s\n",ext.Data());
#endif
	}

	return res;
}

/** */
CStringList * CFileManager::Search( CString s )
{
	if ( !m_pFileManagerInfo || !m_pSearchIndex )
		return 0;

	if ( m_pFileManagerInfo->m_eFileManagerStatus != efmsIDLE )
		return 0;

	return m_pSearchIndex->Search(s);
}

/** */
CStringList * CFileManager::SearchHash( CString s )
{
	CString ts;
	CByteArray dst;
	CBase32 base32;
	
	if ( !m_pFileManagerInfo || !m_pSearchIndex )
		return 0;

	if ( m_pFileManagerInfo->m_eFileManagerStatus != efmsIDLE )
		return 0;

	if ( s.Left(4) != "TTH:" )
		return 0;

	ts = s.Mid(4,s.Length()-4);
	
	if ( base32.Decode(&dst,&ts) != TIGERSIZE )
		return 0;

	return m_pSearchIndex->SearchHash(dst.Data());
}

/** */
CString CFileManager::GetHash( ulonglong hbi )
{
	if ( !m_pFileManagerInfo || !m_pSearchIndex )
		return "";

	if ( m_pFileManagerInfo->m_eFileManagerStatus != efmsIDLE )
		return "";
	
	return m_pSearchIndex->GetHash(hbi);
}

/** */
bool CFileManager::GetFileBaseObject( CString id, struct filebaseobject * fbo, CString & filename )
{
	if ( !m_pFileManagerInfo || !m_pSearchIndex )
		return FALSE;

	if ( m_pFileManagerInfo->m_eFileManagerStatus != efmsIDLE )
		return FALSE;

	return m_pSearchIndex->GetFileBaseObject(id,fbo,filename);
}

/** */
bool CFileManager::CreateShareList()
{
	bool err = FALSE;

	if ( m_pFileManagerInfo->m_eFileManagerStatus != efmsIDLE )
		return err;
	
	Lock();
	
	m_SharedFolders.Clear();

	m_pShareFolder = 0;
	m_nShareSize   = 0;
	m_sShareIndexBuffer = "";

	m_pSearchIndex->Reset();

	if ( CConfig::Instance()->GetSharedFolders(&m_SharedFolders) != 0 )
	{
		if ( m_pFileNameList )
			delete m_pFileNameList;
		m_pFileNameList  = new CStringList(25);

		m_pFileManagerInfo->m_nProgress = 0;
		m_pFileManagerInfo->m_eFileManagerStatus = efmsCREATESHARELIST;

		if ( CDownloadManager::Instance() )
			CDownloadManager::Instance()->SendFileManagerInfo(m_pFileManagerInfo);

		err = TRUE;
	}
	else
	{
		m_pShareList->SetIndexBuffer( "" );
		m_pShareList->CreateList(m_pSearchIndex);

		m_pSearchIndex->SaveIndex();
	}

	if ( err == TRUE )
	{	
		Start();
	}

	UnLock();

	return err;
}

/** */
void CFileManager::CreateSearchIndex()
{
	if ( m_pFileManagerInfo->m_eFileManagerStatus == efmsIDLE )
	{
		m_nFileBaseIndex = 0;

		m_pSearchIndex->ResetIndex();

		m_pFileManagerInfo->m_nProgress = 0;
		m_pFileManagerInfo->m_eFileManagerStatus = efmsCREATESEARCHINDEX;

		Start();
	}
}

/** */
bool CFileManager::CreateHashList()
{
	bool res = FALSE;

	if ( m_pFileManagerInfo->m_eFileManagerStatus == efmsIDLE )
	{
		m_nFileBaseIndex = 0;

		m_pFileManagerInfo->m_nProgress = 0;
		m_pFileManagerInfo->m_eFileManagerStatus = efmsCREATEHASHLIST;

		Start();
		
		res = TRUE;
	}

	return res;
}

/** */
void CFileManager::ThreadCreateSearchIndex()
{
	int i,k;
	CString s,s1,id;
	double percent;

	for(k=0;k<100;k++)
	{
		if ( (s = m_pSearchIndex->GetFileName(m_nFileBaseIndex)) != "" )
		{
			s = s.ToUpper();

			m_pSearchIndex->AddIndex(s,m_nFileBaseIndex);

			// convert special chars
			s = s.Replace(' ',"/");
			s = s.Replace('.',"/");
			s = s.Replace('-',"/");
			s = s.Replace('_',"/");
			s = s.Replace('(',"/");
			s = s.Replace(')',"/");
			s = s.Replace('!',"/");

			s = CDir().ConvertSeparators(s);

			while( (i=s.Find(DIRSEPARATOR)) != -1 )
			{
				s1 = s.Left(i);
				s  = s.Mid(i+1,s.Length()-i-1);

				i++;

				// max length is 3
				if ( s1.Length() < 3 )
					continue;

				m_pSearchIndex->AddIndex(s1,m_nFileBaseIndex);
			}

			m_nFileBaseIndex++;
		}
		else
		{
			m_pSearchIndex->InitIndex();

			//delete m_pSearchArray;
			//m_pSearchArray = 0;

			printf("ready create index\n");

			m_pSearchIndex->SaveIndex();

			m_pFileManagerInfo->m_nProgress = 100;
			m_pFileManagerInfo->m_eFileManagerStatus = efmsIDLE;

			if ( CDownloadManager::Instance() )
				CDownloadManager::Instance()->SendFileManagerInfo(m_pFileManagerInfo);

			CreateHashList();

			return;
		}
	}

	if ( m_pSearchIndex->IndexCount() > 0 )
	{
		percent = (m_nFileBaseIndex*100.0)/m_pSearchIndex->IndexCount();

		if ( m_pFileManagerInfo->m_nProgress != percent )
		{
			m_pFileManagerInfo->m_nProgress = percent;

			if ( CDownloadManager::Instance() )
				CDownloadManager::Instance()->SendFileManagerInfo(m_pFileManagerInfo);
		}
	}
}

/** */
void CFileManager::ThreadCreateShareList()
{
	CString s,s1;
	CDir dir;

	if ( (m_pShareFolder=m_SharedFolders.Next(m_pShareFolder)) != 0 )
	{
		// directory
		s = m_pShareFolder->m_sPath;

		// is valid ?
		if ( dir.cd(s.Data()) == TRUE )
		{
			s  = dir.Path();
			s1 = dir.DirName();

			if ( s1 != "" )
			{
				s = s.Left( s.Length()-s1.Length() );
			}

			ThreadCreateShareList( 0, "", "", 100/m_SharedFolders.Count() );
		}
		else
		{
			printf("Can't change dir: '%s'\n",s.Data());
		}
	}
	else
	{
		m_pFileManagerInfo->m_nProgress = 100;
		printf("ready create sharelist\n");

		m_pShareList->SetIndexBuffer( m_sShareIndexBuffer );
		m_pShareList->CreateList(m_pSearchIndex);

		if ( m_pFileNameList )
		{
			delete m_pFileNameList;
			m_pFileNameList = 0;
		}

		m_pFileManagerInfo->m_eFileManagerStatus = efmsIDLE;

		if ( CDownloadManager::Instance() )
			CDownloadManager::Instance()->SendFileManagerInfo(m_pFileManagerInfo);

		CreateSearchIndex();

		if ( CDownloadManager::Instance() )
			CDownloadManager::Instance()->SendFileManagerInfo(m_pFileManagerInfo);
	}
}

/** */
void CFileManager::ThreadCreateShareList( int depth, CString curr, CString rel, double percent )
{
	CDir dir;
	CList<CFileInfo> list;
	CString d,s,*ts=0;
	CString path;
	CFileInfo *fileinfo;
	int x,z;
	unsigned long index;
	
	if ( Stopped() )
		return;

	path = dir.SimplePath(m_pShareFolder->m_sPath+DIRSEPARATOR+curr+DIRSEPARATOR+rel);

	dir.SetPath(path);

	d = "";
	z = depth;

//	printf("%s\n",curr.Data());

	// create depth string
	for(x=depth;x>0;x--) d+="\x9";
	z++;
	// use alias
	if ( depth == 0 )
		m_sShareIndexBuffer += m_pShareFolder->m_sAlias + "\xd\xa";
	else
		m_sShareIndexBuffer += d + rel + "\xd\xa";
	d += "\x9";

	// get all files
	if ( dir.ReadEntrys( CDir::Files, &list ) )
	{
		// add files
		fileinfo = 0;
		while( (fileinfo=list.Next(fileinfo)) != 0 )
		{
			if ( Stopped() )
				return;

			if ( fileinfo->name != "" )
			{
				if ( fileinfo->m_bSymlink )
					s = dir.ReadLink(path+DIRSEPARATOR+fileinfo->name);
				else
					s = dir.SimplePath(path+DIRSEPARATOR+fileinfo->name);

				if ( s != "" )
				{
					if ( m_pFileNameList->Get(s,(CObject*&)ts) != 0 )
					{
						if ( dclibVerbose() > 1 )
							printf("CreateShareList: Add file: '%s/%s'\n",path.Data(),fileinfo->name.Data());fflush(stdout);

						m_pFileNameList->Add(s,new CString());
						m_nShareSize   += fileinfo->size;
						
						// add file to the file manager ...
						index = m_pSearchIndex->AddIndex( fileinfo, dir.SimplePath(m_pShareFolder->m_sAlias+DIRSEPARATOR+curr+DIRSEPARATOR+rel), GetFileType(fileinfo->name) );
						
						m_sShareIndexBuffer += d + fileinfo->name + "|" + CString().setNum(index) + "\xd\xa";
					}
				}
			}
		}
	}

	// get all dirs
	if ( dir.ReadEntrys( CDir::Dirs, &list ) )
	{
		if ( list.Count() > 0 )
			percent /= list.Count();
		else
			m_pFileManagerInfo->m_nProgress += percent;

		// add dir
		fileinfo = 0;
		while( (fileinfo=list.Next(fileinfo)) != 0 )
		{
			if ( Stopped() )
				return;

			if ( (fileinfo->name != ".") && (fileinfo->name != "..") )
			{
				if ( fileinfo->m_bSymlink )
					s = dir.ReadLink(path+DIRSEPARATOR+fileinfo->name);
				else
					s = dir.SimplePath(path+DIRSEPARATOR+fileinfo->name);

				if ( s != "" )
				{
					if ( m_pFileNameList->Get(s,(CObject*&)ts) != 0 )
					{
						if ( dclibVerbose() > 0 )
							printf("CreateShareList: Add path: '%s/%s'\n",path.Data(),fileinfo->name.Data());fflush(stdout);

						m_pFileNameList->Add(s,new CString());
						ThreadCreateShareList(z,curr+DIRSEPARATOR+rel,fileinfo->name,percent);
					}
					else
						m_pFileManagerInfo->m_nProgress += percent;

				}
			}
			else
				m_pFileManagerInfo->m_nProgress += percent;

			if ( CDownloadManager::Instance() )
				CDownloadManager::Instance()->SendFileManagerInfo(m_pFileManagerInfo);
		}
	}
}

/** */
ulonglong CFileManager::CalcShareSize()
{
	CString s,s1;
	CDir dir;
	ulonglong iSSize;

	iSSize = 0;
	m_pShareFolder=0;

	if ( CConfig::Instance()->GetSharedFolders(&m_SharedFolders) == 0 )
	{
		printf("No share folderfound !");
		return 0;
	}

	while ( (m_pShareFolder=m_SharedFolders.Next(m_pShareFolder)) != 0 )
	{
		// directory
		s = m_pShareFolder->m_sPath;

		// is valid ?
		if ( dir.cd(s.Data()) == TRUE )
		{
			s  = dir.Path();
			s1 = dir.DirName();

			if ( s1 != "" )
			{
				s = s.Left( s.Length()-s1.Length() );
			}

			iSSize += CalcShareSize( 0, s, s1, "" );
		}
		else
		{
			printf("Can't change to dir: '%s'\n",s.Data());
		}
	}

	return iSSize;
}


/** */
ulonglong CFileManager::CalcShareSize( int depth, CString base, CString curr, CString relpath )
{
	ulonglong lShareSize;
	CDir dir;
	CList<CFileInfo> list;
	CString d,s;
	CString path;
	CString rpath;
	CFileInfo *fileinfo;
	int x,z;

	lShareSize = 0;

	if ( relpath == "" )
		rpath = curr;
	else
		rpath = relpath+DIRSEPARATOR+curr;

	if (base=="")
		path = curr;
	else
		path = base+DIRSEPARATOR+curr;

	dir.SetPath(path);

	d = "";
	z = depth;

	// create depth string
	if ( curr != "" )
	{
		for(x=depth;x>0;x--) d+="\x9";
		z++;
		d += "\x9";
	}

	// get all files
	if ( dir.ReadEntrys( CDir::Files, &list ) )
	{
		fileinfo = 0;
		while( (fileinfo=list.Next(fileinfo)) != 0 )
		{
			if ( fileinfo->name != "" )
			{
				if ( fileinfo->m_bSymlink )
					s = dir.ReadLink(path+DIRSEPARATOR+fileinfo->name);
				else
					s = dir.SimplePath(path+DIRSEPARATOR+fileinfo->name);

				if ( s != "" )
				{
					lShareSize += fileinfo->size;
				}
			}
		}
	}

	// get all dirs
	if ( dir.ReadEntrys( CDir::Dirs, &list ) )
	{
		fileinfo = 0;
		while( (fileinfo=list.Next(fileinfo)) != 0 )
		{
			if ( (fileinfo->name != ".") && (fileinfo->name != "..") )
			{
				if ( fileinfo->m_bSymlink )
					s = dir.ReadLink(path+DIRSEPARATOR+fileinfo->name);
				else
					s = dir.SimplePath(path+DIRSEPARATOR+fileinfo->name);

				if ( s != "" )
				{
					lShareSize += CalcShareSize(z,path,fileinfo->name,rpath);
				}
			}
		}
	}
	return lShareSize;
}

/** */
void CFileManager::ThreadCreateHashList()
{
	CString s,pa;
	struct filebaseobject fbo;
	ulonglong hbi;
	struct tt_context tth;
	unsigned char hash[TIGERSIZE];
	char *buffer;
	int len;
	CFile f;
	CBase32 base32;
	double percent;

	if ( m_pSearchIndex->GetFileBaseObject(CString().setNum(m_nFileBaseIndex),&fbo,s) == TRUE )
	{
		pa = CConfig::Instance()->AliasToPath(s);
		
		if ( pa == "" )
		{
			m_nFileBaseIndex++;
			return;
		}
		
		// check if we have the hash
		if ( m_pSearchIndex->FindHashBaseIndex(&fbo,&hbi) )
		{
			if ( dclibVerbose() > 0 )
				printf("hash found, no changes in file\n");
			fbo.m_nHashIndex = hbi;
			m_pSearchIndex->UpdateIndex(m_nFileBaseIndex,&fbo);
		}
		else
		{
			if ( CConfig::Instance()->GetDisableHashList() == FALSE )
			{
				if ( f.Open( pa, IO_RAW | IO_READONLY ) == TRUE )
				{
					buffer = (char*)malloc(1024*1024);
					
					if ( buffer )
					{
						tt_init(&tth);

						while ( (len = f.Read( buffer, 1024*1024)) > 0 )
						{
							tt_update(&tth,(unsigned char*)buffer,len);
				
							if ( Stopped() )
								break;
						}
					
						free(buffer);
					}
					
					f.Close();
			
					if ( !Stopped() )
					{
						tt_digest(&tth,(unsigned char*)hash);
						m_pSearchIndex->AddHashIndex(m_nFileBaseIndex,hash);
					}
					else
					{
						printf("create hash stop\n");
						m_pFileManagerInfo->m_eFileManagerStatus = efmsIDLE;
						m_pSearchIndex->SaveIndex();
					}
				}
			}
		}
		
		m_nFileBaseIndex++;
	
		if ( m_pSearchIndex->IndexCount() > 0 )
		{
			percent = (m_nFileBaseIndex*100.0)/m_pSearchIndex->IndexCount();

			if ( m_pFileManagerInfo->m_nProgress != percent )
			{
				m_pFileManagerInfo->m_nProgress = percent;

				if ( CDownloadManager::Instance() )
					CDownloadManager::Instance()->SendFileManagerInfo(m_pFileManagerInfo);
			}
		}
	}
	else
	{
		printf("create hash end\n");
		m_pFileManagerInfo->m_eFileManagerStatus = efmsIDLE;
		m_pFileManagerInfo->m_nProgress = 100;

		if ( CDownloadManager::Instance() )
			CDownloadManager::Instance()->SendFileManagerInfo(m_pFileManagerInfo);

		m_pSearchIndex->SaveIndex();
		m_pShareList->CreateList(m_pSearchIndex);
		Stop();
	}
}

/** */
void CFileManager::Thread( CObject * )
{
	int wait = 50;

	Lock();

	switch(m_pFileManagerInfo->m_eFileManagerStatus)
	{
		case efmsCREATESHARELIST:
			ThreadCreateShareList();
			break;
		case efmsCREATESEARCHINDEX:
			ThreadCreateSearchIndex();
			wait = 0;
			break;
		case efmsCREATEHASHLIST:
			ThreadCreateHashList();
			wait = 0;
			break;
		default:
			break;
	}

	UnLock();

	NanoSleep(wait);
}

/** */
int CFileManager::FileManagerCallback( CObject *, CObject * )
{
	if ( CConfig::Instance()->GetRecreateShareListTime() != 0 )
	{
		if ( m_tCreateShareListTimeout != 0 )
		{
			if ( time(0) > m_tCreateShareListTimeout )
			{
				CreateShareList();

				m_tCreateShareListTimeout = time(0)+CConfig::Instance()->GetRecreateShareListTime()*60*60;
			}
		}
		else
		{
			m_tCreateShareListTimeout = time(0)+CConfig::Instance()->GetRecreateShareListTime()*60*60;
		}
	}
	else
	{
		m_tCreateShareListTimeout = 0;
	}

	return 0;
}
