/***************************************************************************
                          ctransfer.cpp  -  description
                             -------------------
    begin                : Fri Feb 22 2002
    copyright            : (C) 2002-2003 by Mathias Kster
    email                : mathen@users.berlios.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <fcntl.h>
#include <stdlib.h>
#include <errno.h>
#include <string.h>

#ifndef WIN32
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <unistd.h>
#else
#include <io.h>
#endif

#include <dclib/dcos.h>
#include <dclib/dclib.h>
#include <dclib/dcobject.h>
#include <dclib/cmessagehandler.h>
#include <dclib/cconnectionmanager.h>
#include <dclib/cencrypt.h>
#include <dclib/core/che3.h>
#include <dclib/core/cbz.h>
#include <dclib/core/cdir.h>
#include <dclib/core/cbytearray.h>
#include <dclib/cconfig.h>
#include <dclib/core/platform.h>
#include <dclib/cfilemanager.h>

#include "ctransfer.h"

#define READ_FILE_BUFFER_SIZE	(40*1024)

CTransfer::CTransfer( bool listener )
{
	pCallback = 0;

	pByteArray = new CByteArray();

	m_bListener   = listener;
	sNick         = "";
	sSrcFilename  = "";
	sDstFilename  = "";
	sBuffer       = "";
	eMedium       = eltNONE;
	eSrcDirection = edNONE;
	eDstDirection = edNONE;
	eMode         = estNONE;

	m_nStartPosition = 0;
	m_nEndPosition   = 0;
	m_nSrcLevel      = 0;
	m_nDstLevel      = 0;
	starttime.tv_sec = 0;
	starttime.tv_usec= 0;
	m_nFileSize      = 0;
	m_nTransfered    = 0;
	m_nTransferRate  = 0;
	m_nChunkSize     = 0;

	sDstNick = "";

	sHubName = "";
	sHubHost = "";

	m_nTransferID = 0;

	m_bIdle = TRUE;
	m_eTransferState = etsNONE;

	m_eTransferType = ettNONE;

	// init transferrate
	memset(&m_AverageTransferRate,0,sizeof(m_AverageTransferRate));
	m_AverageTransferRate.index=0;
	gettimeofday(&m_AverageTransferRate.tv[0],NULL);
}

CTransfer::~CTransfer()
{
	SetCallBackFunction(0);

	TransferThread.Lock();

	if (pByteArray)
	{
		delete pByteArray;
		pByteArray = 0;
	}

	m_File.Close();

	TransferThread.UnLock();
}

/** */
int CTransfer::CallBack_SendObject( CObject * Object )
{
	int err;

	if ( pCallback != 0 )
	{
		err = pCallback->notify( this, Object );
	}
	else
	{
		err = DC_CallBack( Object );
	}

	if ( err == -1 )
	{
		DPRINTF("CallBack failed (state)...\n");
		delete Object;
	}

	return err;
}

/** */
int CTransfer::CallBack_SendError( CString msg )
{
	CMessageError * Object = new CMessageError();

	Object->m_sError = msg;

	return CallBack_SendObject(Object);
}

/** */
void CTransfer::SetBuffer( CByteArray *ba )
{
	TransferThread.Lock();
	pByteArray->Append(ba->Data(),ba->Size());
	TransferThread.UnLock();
}

/** */
long CTransfer::GetBuffer( CByteArray *ba )
{
	TransferThread.Lock();
	long i;
	ba->Append(pByteArray->Data(),pByteArray->Size());
	i = pByteArray->Size();
	TransferThread.UnLock();
	return i;
}

/** */
void CTransfer::InitTime()
{
	TransferThread.Lock();
	gettimeofday(&starttime,NULL);
	TransferThread.UnLock();
}

/** */
ulonglong CTransfer::GetTransferrate()
{
	TransferThread.Lock();

	ulonglong r;

	if ( (starttime.tv_sec == 0) || (m_bIdle == TRUE) )
	{
		TransferThread.UnLock();
		return 0;
	}

	r = GetTraffic();

	TransferThread.UnLock();

	return r;
}

/** */
ulonglong CTransfer::GetBytesForTransferrate( ulonglong rate )
{
	TransferThread.Lock();

	ulonglong r = 0;

	if ( starttime.tv_sec == 0 )
	{
		TransferThread.UnLock();
		return 0;
	}

	r = GetTraffic();

	if ( r < rate )
	{
		r = rate;
	}
	else if ( r > rate )
	{
		r = 0;
	}

	TransferThread.UnLock();

	return r;
}

/** */
void CTransfer::AddTraffic( long n )
{
	int i;
	ulonglong t1,t2;
	struct timeval tp;

	m_nTransfered += n;

	gettimeofday(&tp,NULL);

	t1 = ((tp.tv_sec*1000)+(tp.tv_usec/1000));

	i = m_AverageTransferRate.index;

	t2 = ((m_AverageTransferRate.tv[i].tv_sec*1000)+(m_AverageTransferRate.tv[i].tv_usec/1000));

	if ( (t1-t2) > 1000 )
	{
		i++;

		if ( i >= 10 )
		{
			i = 0;
		}

		m_AverageTransferRate.index = i;
		m_AverageTransferRate.tv[i] = tp;
		m_AverageTransferRate.transfered[i] = 0;
	}

	m_AverageTransferRate.transfered[i] += n;
}

/** */
ulonglong CTransfer::GetTraffic()
{
	int i;
	struct timeval tp;
	ulonglong t1,t2;
	ulonglong n = 0;

	gettimeofday(&tp,NULL);

	t1 = ((tp.tv_sec*1000)+(tp.tv_usec/1000));

	i = m_AverageTransferRate.index;

	if ( i == 9 )
		i = 0;
	else
	{
		i++;
	}

	t2 = ((m_AverageTransferRate.tv[i].tv_sec*1000)+(m_AverageTransferRate.tv[i].tv_usec/1000));

	if ( (t2 == 0) && (i > 0) )
	{
		i = 0;
	}

	t2 = ((m_AverageTransferRate.tv[i].tv_sec*1000)+(m_AverageTransferRate.tv[i].tv_usec/1000));

	if ( t2 != 0 )
	{
		t1 = t1-t2;

		if ( t1 > 0 )
		{
		for(i=0,t2=0;i<10;i++)
		{
			n += m_AverageTransferRate.transfered[i];
		}

		n = (n*1000)/t1;
		}
	}
	else
	{
		DPRINTF("time is null\n"); fflush(stdout);
	}

	return n;
}

/** */
void CTransfer::ConnectionState( eConnectionState state )
{
	CMessageConnectionState *Object;

	Object = new CMessageConnectionState();

	Object->m_eState   = state;
	Object->m_sMessage = GetSocketError();

	if ( state == estCONNECTED )
	{
		eMode = estTRANSFERHANDSHAKE;

		if ( sNick != "" )
		{
			SendMyNick( sNick );
		}
	}
	else if ( state == estSSLCONNECTED )
	{
		DPRINTF("change to ssl mode success\n");

		CallBack_SendObject(new CMessageKey());
	}
	else if ( state == estDISCONNECTED )
	{
		eMode = estNONE;
	}

	CallBack_SendObject(Object);
}

/** */
int CTransfer::StartDownload( CString dstfile, ulonglong startposition, ulonglong endposition, ulonglong size, ulonglong chunksize, CString srcfile )
{
	if ( (srcfile == "" ) && (eMedium == eltFILE) )
	{
		DPRINTF("ctransfer: wrong mode (empty file) %d\n",eMedium);
		return -1;
	}

	if ( m_bIdle == FALSE )
	{
		DPRINTF("ctransfer: other transfer is running\n");
		return -1;
	}

	if ( eMode == estTRANSFERUPLOAD )
	{
		DPRINTF("ctransfer: wrong transfer mode\n");
		return -1;
	}

	SetMode(estTRANSFERDOWNLOAD);
	SetStartPosition(startposition);
	SetEndPosition(endposition);
	SetLength(size);
	SetDstFilename(dstfile);
	SetSrcFilename(srcfile);
	m_nTransfered = 0;
	m_nChunkSize = chunksize;

	/** set starttime */
	InitTime();

	if ( eMedium == eltCLIENTVERSION )
	{
		m_bIdle = TRUE;
	}
	else
	{
#ifdef HAVE_LIBBZ2
		if ( (GetDstFilename() == DC_USER_FILELIST_HE3) && (m_MessageSupports.m_bXMLBZList == TRUE) )
		{
			SendGet( DC_USER_FILELIST_XMLBZ, startposition+1 );
		}
		else if ( (GetDstFilename() == DC_USER_FILELIST_HE3) && (m_MessageSupports.m_bBZList == TRUE) )
		{
			SendGet( DC_USER_FILELIST_BZ, startposition+1 );
		}
		else
#endif
		{
			if ( (m_MessageSupports.m_bChunk == FALSE) || (chunksize == 0) )
			{
				SendGet( GetDstFilename(), startposition+1 );
			}
			else
			{
				SendGet( GetDstFilename(), startposition+1, chunksize );
			}
		}
	}

	return 0;
}

/** */
int CTransfer::StartUpload( CString dstfile, ulonglong length, ulonglong pos, ulonglong chunksize, CString srcfile )
{
	if ( (srcfile == "" ) && (eMedium != eltBUFFER) )
	{
		DPRINTF("ctransfer: wrong mode %d\n",eMedium);
		return -1;
	}

	if ( m_bIdle == FALSE )
	{
		DPRINTF("ctransfer: other transfer is running\n");
		return -1;
	}

	if ( eMode == estTRANSFERDOWNLOAD )
	{
		DPRINTF("ctransfer: wrong transfer mode\n");
		return -1;
	}

	SetMode(estTRANSFERUPLOAD);

	SetStartPosition(pos);
	SetEndPosition(length);
	SetLength(length);
	SetDstFilename(dstfile);
	SetSrcFilename(srcfile);
	m_nTransfered = 0;
	// if chunksize 0 we use the len
	if ( chunksize == 0 )
		m_nChunkSize = length-pos;
	else
		m_nChunkSize = chunksize;
	m_nPendingSendBytes = 0;

	/** set starttime */
	InitTime();

	SendFileLength(length);

	return 0;
}

/** */
int CTransfer::HandleMessage( char * c, int )
{
	int p=0;
	eDCMessage type;
	CObject * Object;
	CDir dir;

	CMessageHandler MessageHandler;

	CString s;

	s = c;

	while( (m_bIdle || (eMode == estTRANSFERUPLOAD)) &&
	       ((type=MessageHandler.Parse(&s,p,Object=0)) != DC_MESSAGE_PARSE_ERROR) )
	{
//		DPRINTF("transfer: handle message: %d\n",type);

		if ( !Object )
		{
			continue;
		}

		switch (type)
		{
			case DC_MESSAGE_FILELENGTH:
			case DC_MESSAGE_LISTLEN:
			{
				bool b = FALSE;
				CMessageFileLength * msg = (CMessageFileLength*)Object;

				if ( eMode != estTRANSFERDOWNLOAD || (m_bIdle == FALSE) )
				{
					DPRINTF("Warning: not in downloadmode/idle FILELENGTH/LISTLEN\n");

					delete Object;
					Object = 0;
				}
				else
				{
					m_nTransfered = 0;

					// update endposition
					if ( (m_nEndPosition == m_nFileSize) || (m_nEndPosition == 0) )
					{
						m_nEndPosition = msg->m_nFileLength;
					}

					// fix chunksize
					if ( (m_nChunkSize == m_nFileSize) || (m_nChunkSize == 0) )
					{
						m_nChunkSize = msg->m_nFileLength;
					}

					m_nFileSize = msg->m_nFileLength;

					if ( eMedium == eltFILE )
					{
						int mode = 0;

						m_File.Close();

						dir.SetPath("");

						if ( dir.IsFile(sSrcFilename) == FALSE )
						{
							// create file
							mode |= IO_CREAT;
						}

						mode |= IO_RAW | IO_WRITEONLY;

						if ( m_File.Open( sSrcFilename, mode, MO_IRUSR|MO_IWUSR|MO_IRGRP|MO_IROTH ) == FALSE )
						{
							CallBack_SendError(strerror(errno));
							perror("File open");
						}
						else
						{
							if ( CConfig::Instance()->GetCreateFile() == TRUE )
							{
								if ( ((mode & IO_CREAT) == IO_CREAT) &&
								     (m_nFileSize > 0) )
								{
									if ( m_File.Seek(m_nFileSize-1,SEEK_SET) == TRUE )
									{
										if ( m_File.Write("\0",1) == 1 )
										{
											b = TRUE;
										}
									}
								}
							}
							else
							{
								b = TRUE;
							}

							if ( b == TRUE )
							{
								// seek to the startposition
								if ( m_File.Seek(m_nStartPosition,SEEK_SET) == FALSE )
								{
									b = FALSE;
								}
								else
								{
									b = TRUE;
								}
							}

							if ( b == FALSE )
							{
								CallBack_SendError(strerror(errno));
								perror("File seek");
								m_File.Close();
							}
						}
					}
					else
					{
						pByteArray->SetSize(0);
						b = TRUE;
					}

					// start transfer
					if ( b == TRUE )
					{
						m_bIdle = FALSE;

						SendSend();
					}
				}

				break;
			}

			case DC_MESSAGE_SEND:
			{
				if ( (eMode != estTRANSFERUPLOAD) || (m_bIdle == FALSE) )
				{
					DPRINTF("Warning: not in uploadmode/idle SEND [%d/%d]\n",eMode,m_bIdle);

					delete Object;
					Object = 0;
				}
				else
				{
					m_nTransfered = 0;

					// init filehandle
					if ( eMedium == eltFILE )
					{
						m_nFileBufferPos    = READ_FILE_BUFFER_SIZE;
						m_nFileBufferSize   = READ_FILE_BUFFER_SIZE;
						m_nPendingSendBytes = 0;

						pByteArray->SetSize(READ_FILE_BUFFER_SIZE);

						m_File.Close();

						if ( m_File.Open( sSrcFilename, IO_RAW | IO_READONLY ) == FALSE )
						{
							CallBack_SendError( CString("(File open) ") + strerror(errno) );
							perror("File open");
						}
						else
						{
							// seek to the startposition
							if ( m_File.Seek(m_nStartPosition,SEEK_SET) == FALSE )
							{
								CallBack_SendError( CString("(File seek) ") + strerror(errno) );
								perror("File seek");

								m_File.Close();
							}
						}

						// check for error
						if ( !m_File.IsOpen() )
						{
							Disconnect(TRUE);
						}
						else
						{
							// start transfer
							m_bIdle = FALSE;
						}
					}
					else
					{
						// start transfer
						m_bIdle = FALSE;
					}

					DPRINTF("start upload ...'%s'\n",sSrcFilename.Data());
				}

				break;
			}

			case DC_MESSAGE_ERROR:
			{
				break;
			}

			case DC_MESSAGE_DIRECTION:
			{
				CMessageDirection * msg = (CMessageDirection*)Object;

				if ( eDstDirection != edNONE )
				{
					// remote will change the direction ... hoho ;-)
					if ( eDstDirection != msg->m_eDirection )
					{
						Disconnect(TRUE);
						DPRINTF("Warning: remote will change his direction (hack) ...\n");
						delete Object;
						Object = 0;
					}
				}
				else
				{
					eDstDirection = msg->m_eDirection;
					m_nDstLevel   = msg->m_nLevel;
				}

				break;
			}

			case DC_MESSAGE_MYNICK:
			{
				CMessageMyNick * msg = (CMessageMyNick*)Object;

				if ( eMode == estTRANSFERHANDSHAKE )
				{
					sDstNick = msg->m_sNick;
										
					m_nSrcLevel = rand()%0x7FFF;
				}
				else
				{
					DPRINTF("Warning: Nick msg from '%s' only allowed in handshake",msg->m_sNick.Data());

					delete Object;
					Object = 0;
				}

				break;
			}

			case DC_MESSAGE_LOCK:
			{
				CString s;
				CEncrypt Encrypt;
				CMessageLock * msg = (CMessageLock*)Object;

				// save lock message
				m_MessageLock = *msg;

				if ( eSrcDirection != edNONE )
				{
					// send supports command
					if ( msg->m_bExtProtocol == TRUE )
					{
						s = "";
#ifdef HAVE_LIBBZ2
						if ( CFileManager::Instance()->GetShareBufferSize(esbtBZ) > 0 )
							s += "BZList ";
						if ( CFileManager::Instance()->GetShareBufferSize(esbtXMLBZ) > 0 )
							s += "XmlBZList ";
#endif
#ifdef HAVE_SSL
						CDir dir;
						if ( dir.IsFile(CConfig::Instance()->GetTransferCert(),FALSE) &&
						     dir.IsFile(CConfig::Instance()->GetTransferKey(),FALSE) )
						{
							s += "SSL ";
						}
#endif
						// add chunk support
						s += "CHUNK ";
						// minislots (files < 16k)
						s += "MiniSlots ";

						if ( s != "" )
						{
							SendSupports(s);
						}
					}

					SendDirection( eSrcDirection, m_nSrcLevel );

					s = "";
					Encrypt.Encrypt(msg->m_sData,s);

					SendKey(s);
				}
				else
				{
					Disconnect(TRUE);
				}

				break;
			}

			case DC_MESSAGE_SUPPORTS:
			{
				CMessageSupports * msg = (CMessageSupports*)Object;
								
				DPRINTF("LOCK %d %d %d\n",
				m_MessageLock.m_eClientVersion,
				m_MessageLock.m_nVersionMajor,
				m_MessageLock.m_nVersionMinor);
				
				// now we need to check some broken clients and limit support
				if ( (m_MessageLock.m_eClientVersion == eucvDCPP) &&
				     (m_MessageLock.m_nVersionMajor == 0) &&
				     (m_MessageLock.m_nVersionMinor <= 403) )
				{
					// disable xml list for dc++ 0.401
					msg->m_bXMLBZList = FALSE;
				}
				
				m_MessageSupports = *msg;

				break;
			}

			case DC_MESSAGE_KEY:
			{
#ifdef HAVE_SSL
				// start ssl socket
				if ( (m_MessageSupports.m_bSSL == TRUE) &&
				     (CConfig::Instance()->GetTransferCert() != "") &&
				     (CConfig::Instance()->GetTransferKey() != "") )
				{
					bool b;
					if ( m_bListener == TRUE )
						b = ChangeSocketMode(esmSSLSERVER, CConfig::Instance()->GetTransferCert(), CConfig::Instance()->GetTransferKey() );
					else
						b = ChangeSocketMode(esmSSLCLIENT);
					DPRINTF("change to ssl mode %d\n",b);
					delete Object;
					Object = 0;
				}
#endif

				break;
			}

			case DC_MESSAGE_MAXEDOUT:
			{
				break;
			}

			case DC_MESSAGE_CANCEL:
			{
				if ( (eMode == estTRANSFERUPLOAD) && (m_bIdle == FALSE) )
				{
					if ( eMedium == eltFILE )
					{
						m_File.Close();
					}

					// free buffer
					pByteArray->SetSize(0);

					// stop transfer
					m_bIdle = TRUE;
				}
				else
				{
					DPRINTF("Warning: Cancel msg not in uploadmode/idle SEND [%d/%d]\n",eMode,m_bIdle);

					delete Object;
					Object = 0;
				}

				break;
			}

			default:
			{
				//DPRINTF("ctransfer: unknown message %d\n",type);
			}
		}

		if (Object)
		{
			((CDCMessage*)Object)->m_eType = type;

			CallBack_SendObject(Object);
		}
	}

	return p;
}

/** */
bool CTransfer::GetEncrypted()
{
	bool res = FALSE;

#ifdef HAVE_SSL
	if ( (GetSocketMode() == esmSSLCLIENT) ||
	     (GetSocketMode() == esmSSLSERVER) )
	{
		res = TRUE;
	}
#endif

	return res;
}

/** */
void CTransfer::DataSend()
{
	int l = 0;
	ulonglong len=0;
	ulonglong sendbyte;
	ulonglong freebuf;
	CMessageTransfer *Object;
	int lo;

	if ( (eMode == estTRANSFERUPLOAD) && (m_bIdle == FALSE) )
	{
		for(lo=0;(lo<5) && (!m_bIdle);lo++)
		{

		sendbyte = 0;

		// check send rate
		if ( m_nTransferRate != 0 )
		{
			sendbyte = GetBytesForTransferrate(m_nTransferRate);

			if ( sendbyte == 0 )
			{
				break;
			}
		}

		freebuf = READ_FILE_BUFFER_SIZE; //GetFreeSendBufferSize();

		if ( (m_MessageSupports.m_bChunk == FALSE) || (m_nChunkSize == 0) )
		{
			len = m_nFileSize - m_nStartPosition - m_nTransfered;
		}
		else
		{
			len = m_nChunkSize - m_nTransfered;
		}

		if ( eMedium == eltFILE )
		{
			//DPRINTF("%llu %llu %llu %llu\n",len,lStartPosition,lChunkSize,lCurrentPosition);

			// load data from filehandle and send ...
			if ( m_nFileBufferPos == m_nFileBufferSize )
			{
				if ( len > READ_FILE_BUFFER_SIZE )
				{
					len = READ_FILE_BUFFER_SIZE;
				}

				m_nFileBufferSize = m_File.Read( (char *)pByteArray->Data(), (long)len );

				if ( m_nFileBufferSize == -1 )
				{
					CallBack_SendError(strerror(errno));
					perror("CTransfer::DataSend() read error ! ");
					Disconnect(TRUE);
					return;
				}
				else if ( m_nFileBufferSize == 0 )
				{
					perror("CTransfer::DataSend() no data read ! ");
					Disconnect(TRUE);
					return;
				}
				else if ( len > (ulonglong)m_nFileBufferSize )
				{
					perror("CTransfer::DataSend() wrong length calculation ! ");
					len = m_nFileBufferSize;
				}

				m_nFileBufferPos = 0;
			}

			// limit to sendbuffer
			if ( len > (ulonglong)(m_nFileBufferSize-m_nFileBufferPos) )
			{
				len = m_nFileBufferSize-m_nFileBufferPos;
			}
		}
		else if ( eMedium == eltBUFFER )
		{
			m_nFileBufferPos = m_nTransfered;
		}

		// limit to transferrate
		if ( sendbyte != 0  )
		{
			if ( len > sendbyte )
			{
				len = sendbyte;
			}
		}

		// limit to free socket sendbuffer
		if ( len > freebuf )
		{
			len = freebuf;
		}

		// fix to pending send bytes
		if ( m_nPendingSendBytes != 0 )
		{
			len = m_nPendingSendBytes;
		}

//		DPRINTF("LEN: %d\n",len);

		// write data to socket
		l = Write( pByteArray->Data()+m_nFileBufferPos, (long)len, TRUE );

		if ( l > 0 )
		{
			AddTraffic(l);

			if ( m_nPendingSendBytes != 0 )
			{
				m_nPendingSendBytes -= l;
			}
			else
			{
				m_nPendingSendBytes = len-l;
			}

			if ( eMedium == eltFILE )
			{
				m_nFileBufferPos += l;
			}

			// add traffic control
			CSocket::m_Traffic.AddTraffic(ettDATATX,len);
		}
		else if ( l == -1 )
		{
			perror("CTransfer::DataSend() write error !\n");
			Disconnect(TRUE);
			return;
		}
		else if ( l == 0 )
		{
			if ( m_nPendingSendBytes == 0 )
			{
				m_nPendingSendBytes = len;
			}

			break;
		}

		if ( ((m_nStartPosition+m_nTransfered) == m_nFileSize) ||
		     ( (m_MessageSupports.m_bChunk == TRUE) &&
		       (m_nChunkSize == m_nTransfered) ) )
		{
			DPRINTF("end found\n");
			if ( eMedium == eltFILE )
			{
				m_File.Close();
			}

			// free buffer
			pByteArray->SetSize(0);


			// stop transfer
			m_bIdle = TRUE;

			// stop loop
			lo = 5;
		}

		Object = new CMessageTransfer();

		Object->m_eDirection  = edUPLOAD;
		Object->m_nTransfered = m_nStartPosition+m_nTransfered;
		Object->m_nLength     = m_nFileSize;

		CallBack_SendObject(Object);

		} // for
	}
}

/** */
void CTransfer::DataTimeout()
{
	if ( (eMode == estTRANSFERDOWNLOAD) && (m_bIdle == FALSE) )
	{
		DPRINTF("CTransfer: download timeout ...\n");

		// send string ...
		SendString("|");
	}
	else if ( eMode == estTRANSFERHANDSHAKE )
	{
		DPRINTF("CTransfer: handshake timeout ...\n");

		// handshake timeout
		Disconnect(TRUE);
	}
	else if ( m_bIdle == TRUE )
	{
		// send string ...
		SendString("|");
	}
}

/** */
void CTransfer::Notify()
{
}

/** */
void CTransfer::DataAvailable( const char * buffer, int len )
{
	int i,r,p;
	CMessageTransfer *Object;

	i = 0;
	p = 0;

	while ( i != len )
	{
		if ( (eMode == estTRANSFERDOWNLOAD) && (m_bIdle == FALSE) )
		{
			while ( (i != len) && (!m_bIdle) )
			{
				if ( eMedium == eltFILE )
				{
					r = HandleFileTransfer(buffer+i,len-i);
				}
				else if ( eMedium == eltBUFFER )
				{
					r = HandleBufferTransfer(buffer+i,len-i);
				}
				else
				{
					r = -1;
				}

				if ( r >= 0 )
				{
					i += r;
				}
				else
				{
					break;
				}

				// add traffic control
				CSocket::m_Traffic.AddTraffic(ettDATARX,r);

				// set transfer to idle on finished chunk if remote
				// support chunks or if fileend reached
				if ( (m_nChunkSize == m_nTransfered) &&
				    ((m_MessageSupports.m_bChunk == TRUE) ||
				     ((m_nStartPosition+m_nChunkSize) == m_nFileSize) ) )
				{
					m_bIdle = TRUE;
				}

				// send transfer message to handle new chunk-end
				Object = new CMessageTransfer();
				Object->m_eDirection  = edDOWNLOAD;
				Object->m_nTransfered = m_nTransfered;
				Object->m_nLength     = m_nFileSize;
				CallBack_SendObject(Object);

				// check end
				if ( m_nChunkSize == m_nTransfered )
				{
					// if endposition not filesize and remote don't support chunks we must disconnect
					if ( ((m_nStartPosition+m_nChunkSize) != m_nFileSize) && (m_MessageSupports.m_bChunk == FALSE) )
					{
						Disconnect();
					}

					m_bIdle = TRUE;

					m_File.Close();
				}
			}
		}
		else
		{
			r = HandleControlTransfer(buffer+i,len-i);

			// add traffic control
			CSocket::m_Traffic.AddTraffic(ettCONTROLRX,r);

			i += r;
		}

		if ( i == p )
		{
			DPRINTF("WARNING: unknown data ! [%d %d]\n",i,len);
			break;
		}
		else
		{
			p = i;
		}
	}
}

/** */
int CTransfer::HandleFileTransfer( const char * buffer, int len )
{
	int i,i1;
	ulonglong l;
	CString serr="";
	bool berr=FALSE;

	// check endposition
	if ( (len+m_nTransfered) > m_nChunkSize )
	{
		l = m_nChunkSize-m_nTransfered;
	}
	else
	{
		l = len;
	}

	i1 = 0;

	// write data
	if ( (m_File.IsOpen()) && (l > 0) )
	{
		while( i1 != l )
		{
			i = m_File.Write( buffer, l );

			if ( i != -1 )
			{
				i1 += i;
			}
			else
			{
				i1 -= m_File.GetBufferPos();

				DPRINTF("CTransfer::HandleFileTransfer: write failed with %d pending bytes\n",m_File.GetBufferPos());
				berr = TRUE;
				serr = strerror(errno);
				break;
			}
		}
	}
	else
	{
		// TODO: error message
	}

	AddTraffic(i1);

	// handle error
	if ( berr == TRUE )
	{
		i1 = -1;

		// close file
		m_File.Close();

		CallBack_SendError(serr);

		// disconnect on error
		Disconnect();
	}

	return i1;
}

/** */
int CTransfer::HandleBufferTransfer( const char * buffer, int len )
{
	int i;
	CString serr="";
	bool berr=FALSE;

	// check endposition
	if ( (len+m_nTransfered) > m_nChunkSize )
	{
		i = m_nChunkSize-m_nTransfered;
	}
	else
	{
		i = len;
	}

	pByteArray->Append((const unsigned char*)buffer,(long)i);

	AddTraffic(i);

	// end found
	if ( m_nTransfered == m_nChunkSize )
	{
		if ( sDstFilename == DC_USER_FILELIST_HE3 )
		{
#ifdef HAVE_LIBBZ2
			// decompress filelist
			if ( (m_MessageSupports.m_bBZList == FALSE) && (m_MessageSupports.m_bXMLBZList == FALSE) )
#endif
			{
				CHE3 *he3=0;
				CString * s=0;

				he3 = new CHE3();
				s = he3->decode_he3_data(pByteArray);
				delete he3;
				pByteArray->SetSize(0);

				if ( s != 0 )
				{
					pByteArray->Append( (unsigned char*)s->Data(), s->Length() );
					delete s;
				}
				else
				{
					berr = TRUE;
					serr = "he3 decompress failed";
				}
			}
#ifdef HAVE_LIBBZ2
			else
			{
				CBZ * bz = 0;
				CByteArray out;

				bz = new CBZ();

				if ( bz->Decompress( pByteArray, &out ) == TRUE )
				{
					pByteArray->SetSize(0);
					pByteArray->Append( out.Data(), out.Size() );
				}
				else
				{
					berr = TRUE;
					serr = "bz2 decompress failed";
					pByteArray->SetSize(0);
				}

				delete bz;
			}
#endif
		}

	}

	// handle error
	if ( berr == TRUE )
	{
		i = -1;

		CallBack_SendError(serr);

		// disconnect on error
		Disconnect();
	}

	return i;
}

/** */
int CTransfer::HandleControlTransfer( const char * buffer, int len )
{
	CString s;
	int i,p;

	i = p = 0;

	s = sBuffer + CString().Set(buffer,len);

	// search last '|'
	i = s.FindRev('|');
	i++;

	if ( i > 0 )
	{
		p = HandleMessage( (char*)s.Data(), i );

		// some clients send data after filelength (e.g. nmdc 2.02)
		if ( (p < s.Length()) && (!m_bIdle) && (eMode == estTRANSFERDOWNLOAD) )
		{
			DPRINTF("WARNING: transfer->client send unwanted data ! [%d %d %d]\n",i,p,len);
		}
	}

	if ( (i == s.Length()) ||
	     ((!m_bIdle) && (eMode == estTRANSFERDOWNLOAD)) )
	{
		sBuffer = "";
	}
	else
	{
		sBuffer = s.Mid( i, s.Length()-i );
		p = len;
	}

	return p;
}
