/*
 * deal -- compute combination probabilities for card draws
 *
 * ($Id: deal.c,v 1.22 1996/09/26 22:31:22 esr Exp $)
 *
 * Based on the `cardprobs' program written by Jeremy C. York
 * <jeremy@stat.cmu.edu> in September 1993; cleanup, comments, 
 * generalization, and interface improvements by Eric Raymond
 * <esr@snark.thyrsus.com> in April 1995.  The -g and -w options
 * were added by esr September 1996.
 *
 * If I have M special cards in my deck, and N total cards in my deck,
 * and I draw k cards from that deck (*without replacement* -- the
 * fact that you don't put them back into the deck can be important
 * for some values of M and N), and if X counts the number of special
 * cards contained in those k drawn cards...
 * 
 * Then X has a hypergeometric distribution, with the following formula
 * for probabilities:
 * 
 *                 B(M, i) * B(N-M, k-i)
 *      P(X = i) = ---------------------
 *                       B(N, k)
 * 
 * where B(a,b) is read "a choose b", and is the number of ways
 * to choose b objects out of a possible objects.  The formula for it is
 * 
 *     B(a,b)  =   a!/(b! (a-b)!)
 *
 * Compile with:
 *
 *	cc deal.c -lm -o deal
 *
 * In the table output, a dot designates a probability less than .5% that
 * rounds to zero.
 */
#include <stdio.h>
#include <math.h>
#include <signal.h>

#define BASE_DEFAULT	7	/* # cards in initial deal */
#define TURNS_DEFAULT	20	/* show this many turns if total is greater */

#define TRUE	1
#define FALSE	0

static int si, sk, sM, sN;
static int sa, sb;
static int verbose = FALSE;

static double log_a_choose_b(int a, int b);

static void hypercatch(int sig)
{
    putchar('\n');
    if (sig == SIGFPE)
	(void) printf("Died while computing hyper(i=%d, k=%d, M=%d, N=%d)\n", si,sk,sM,sN);
    else
	(void) printf("Caught signal %d during hyper computation\n", sig);
}

static void choosecatch(int sig)
{
    putchar('\n');
    if (sig == SIGFPE)
	(void) printf("Died while computing log_a_choose_b(%d, %d)\n", sa, sb);
    else
	(void) printf("Caught signal %d during choose computation\n", sig);
}

static void usage(void)
{
    (void) printf("usage: deal [-h] [-b initial] [-t turns] [-c columns] [-g] specials total\n");
    exit(0);
}

main(argc, argv)
int argc;
char **argv;
{
    extern char	*optarg;	/* set by getopt */
    extern int	optind;		/* set by getopt */
    static double hyper_prob();

    int status, special, total, columns, i, j;
    int base = BASE_DEFAULT;
    int turns = TURNS_DEFAULT;
    int gammatest = FALSE;
    int ormore = FALSE;
    int hypertest = FALSE;
    int choosetest = FALSE;
    int headers = TRUE;
    int wide = FALSE;

    while ((status = getopt(argc, argv, "b:c:ght:vwBGH")) != EOF)
    {
	switch (status)
	{
	case 'b':
	    base = atoi(optarg);
	    break;

	case 'c':
	    columns = atoi(optarg);
	    break;

	case 'G':
	    gammatest = TRUE;
	    break;

	case 'H':
	    hypertest = TRUE;
	    break;

	case 'g':
	    wide = ormore = TRUE;
	    break;

	case 'h':
	    headers = FALSE;
	    break;

	case 'B':
	    choosetest = TRUE;
	    break;

	case 't':
	    turns = atoi(optarg);
	    break;

	case 'v':
	    verbose = TRUE;
	    break;

	case 'w':
	    wide = TRUE;
	    break;

	default:
	    usage();
	}
    }	

    if (gammatest) {
	double p = lgamma(atoi(argv[optind]));

	(void) printf("lgamma(%d) = %f\n", atoi(argv[optind]), p);
	exit(0);
    }

    if (hypertest) {
	double p;

	si = atoi(argv[optind]);
	sk = atoi(argv[optind + 1]);
	sM = atoi(argv[optind + 2]);
	sN = atoi(argv[optind + 3]);

	p = hyper_prob(si, sk, sM, sN);

	(void) printf("hyper(%d, %d, %d, %d) = %f\n", si, sk, sM, sN, p);
	exit(0);
    }

    if (optind + 2 > argc)
	usage();

    special = atoi(argv[optind]);
    total = atoi(argv[optind + 1]);

    if (choosetest) {
	double p = log_a_choose_b(special, total);

	(void) printf("log_a_choose_b(%d, %d) = %f\n", special, total, p);
	exit(0);
    }

    if (total < special) {
	(void) printf("Can't have more specials than total cards!\n");
	exit(-1);
    } else if (total <= base) {
	(void) printf("All specials would be in the initial hand!\n");
	exit(-1);
    }

    if (total - base < TURNS_DEFAULT)
	turns = total - base;

    if (turns > total - base) {
	(void) printf("The deck can't last that long!\n");
	exit(-1);
    }

    columns = special + 1;
    if (argc >= 5)
	columns = atoi(argv[4]);
    if (columns > special + 1) {
	(void) printf("using %d columns because can only have ",special+1);
	(void) printf("0 to %d specials\n",special);
	columns = special + 1;
    }

    if (headers)
    {
	(void) printf("%d in initial hand, %d specials out of %d total\n",
		      base, special, total);

	(void) printf("     |  %% chance of given number %sof specials (. means < 0.5%%)\n", ormore ? "or more " : "");
	(void) printf("Turn | ");
	for (j = 0; j < columns; j++)
	    (void) printf(wide ? "%3d " : "%2d ", j);
	(void) printf("\n");
	(void) printf("-----+");
	for (j = 0; j < columns; j++)
	    (void) printf(wide ? "----" : "---", j);
	(void) printf("\n");
    }

    (void) signal(SIGFPE, hypercatch);

    for (i = 1; i <= turns; i++) {
	if (headers)
	    (void) printf("%2d   | ",i);
	for (j = 0; j < columns && j <= i + base && j <= special; j++)
	{
	    double prob = hyper_prob(j, i + base, special, total);
	    int percent, k;

	    if (ormore)
		for (k = j+1; k < columns && k <= i+base && k <= special; k++)
		    prob += hyper_prob(k, i + base, special, total);

	    percent = (int)rint(prob * 100);
	    if (percent == 0)
		(void) fputs(wide ? "  . " : " . ", stdout);
	    else
		(void) printf(wide ? "%3d " : "%2d ", percent);
	}

	(void) printf("\n");
    }
}

/* 
 * We avoid nasty numerical problems by using log of gamma function to
 * calculate factorials, and doing everything on the log scale before
 * giving the final answer.
 */

static double hyper_prob(i, k, M, N)
    int i, k, M, N;
{
    double p, exp();

    /* for the signal handler */
    si = i;
    sk = k;
    sM = M;
    sN = N;

    p = exp(log_a_choose_b(M,i)
	    + log_a_choose_b(N-M,k-i)
	    - log_a_choose_b(N, k));

    if (p > 1)
	p = 1;

    return(p);
}

static double log_a_choose_b(a,b)
    int a, b;
{
    double p, lgamma();

    sa = a;
    sb = b;

    signal(SIGFPE, choosecatch);

    if (b >= a)
	p = 0;
    else
	p = lgamma((double) (a+1)) - lgamma((double) (b+1)) -
                lgamma((double) (a-b+1));

    if (verbose)
	(void) fprintf(stderr, "log_a_choose_b(%d, %d) = %f\n", a, b, p);

    signal(SIGFPE, hypercatch);

    return(p);
}

/* deal.c ends here */
