/*
 * despoof.c - determine if packets are spoofed
 * 
 * Requirements: Libnet 1.0 or higher (http://www.packetfactory.net/libnet/)
 *               libpcap 0.4 (ftp://ftp.ee.lbl.gov/libpcap.tar.Z)
 *
 * Compilation instructions:
 *    gcc `libnet-config --defines` -o despoof despoof.c -lnet -lpcap
 *
 * Tested on RedHat 6.1, but should work most places that libnet and libpcap
 * compile and install properly.
 *
 * 01Jul2000 Initial revision.
 * 
 * 03Jul2000 Bug fixes, added more protocol support.
 * 05Jul2000 More cleanup. Gave up trying to code UDP support (see README).
 * 06Jul2000 Fixed loop problems in grab_packet routine. Finished source address
 *           spoofing support.
 * 07Jul2000 Fixed timing problem that missed packets. Minor bug fixes.
 * 19Jul2000 Added ACK flag support for TCP.
 * 25Jul2000 Made TCP the default protocol.
 */

#include <libnet.h>
#include <pcap.h> 
#include <net/bpf.h>

/*
 * Globals
 */
char *targetname;
u_long targetaddr;
struct bpf_program pcapfilter;
struct in_addr net, mask;
pcap_t *pd;
char pcap_err[PCAP_ERRBUF_SIZE];
u_char *ebuf;
char *dev;
struct libnet_link_int *l;
int num_of_questions, link_offset, verbose, target_answers, unexpected, inquery;
u_short ip_header;
int icmp_target, icmp_packet, tcp_packet, udp_packet, ack;
u_char ttlval; 
u_short sport, dport;
u_long src_ip;
u_char proto;

void total(void)
{
  fprintf(stdout,"Total expected packets received   : %d\n",target_answers);
  fprintf(stdout,"Total unexpected packets received : %d\n",unexpected);
}

/*
 * build and send the packets
 */
void send_packets(void)
{
  int packet_size, network, i;
  u_char tcpflags;
  u_long seq_num;
  u_long ack_num;
  u_char *packet;

  /* compute packet size */
  packet_size = LIBNET_IP_H + ip_header;

  /* get mem for packet */
  libnet_init_packet(packet_size, &packet);
  if (packet == NULL)
  {
    libnet_error(LIBNET_ERR_FATAL, "unable to init packet mem\n");
  }

  /* get network ready */
  network = libnet_open_raw_sock(IPPROTO_RAW);
  if (network == -1)
  {
    libnet_error(LIBNET_ERR_FATAL, "unable to open network for sending\n");
  }

  /* build IP section */
  libnet_build_ip(ip_header,0,666,0,255,proto,src_ip,targetaddr,NULL,0,packet);
  /* build ICMP section */
  switch (inquery)
  {
    case 1:
      libnet_seed_prand();
      seq_num = libnet_get_prand(LIBNET_PRu32);
      if (ack)
      {
        tcpflags = TH_ACK; 
        ack_num = libnet_get_prand(LIBNET_PRu32);
      }
      else 
      {
        tcpflags = TH_SYN;
        ack_num = 0;
      }
      libnet_build_tcp(sport,dport,seq_num,ack_num,tcpflags,1024,0,NULL,0,packet + LIBNET_IP_H);
      break;
    case 2:
      libnet_build_icmp_echo(ICMP_ECHO,0,666,0,NULL,0,packet + LIBNET_IP_H);
      break;
    case 3:
      libnet_build_icmp_timestamp(ICMP_TSTAMP,0,666,0,0x00000000,0x00000000,0x00000000,NULL,0,packet + LIBNET_IP_H);
      break;
  }
  /* do the checksum */
  if (libnet_do_checksum(packet, proto, ip_header) == -1) 
  { 
    libnet_error(LIBNET_ERR_FATAL, "checksum failed on %s\n", targetname); 
  }

  /* send the packet */
  i = libnet_write_ip(network, packet, packet_size);

  if (i == -1)
  {
    libnet_error(LIBNET_ERR_FATAL, "failed to write to network\n");
  }

  if (i < packet_size) 
  { 
    libnet_error(LIBNET_ERR_FATAL, "only wrote %d bytes\n", i); 
  }

  if (verbose)
  { 
    fprintf(stdout,"Sent %d bytes for %s\n", i, targetname); 
  }

  /* clean things up */
  if (libnet_close_raw_sock(network) == -1) 
  { 
    libnet_error(LIBNET_ERR_WARNING, "couldn't close the interface after sending"); 
  } 
  libnet_destroy_packet(&packet); 
}

/*
 * handles packets as they are sniffed and processes them
 */
void grab_packet(u_char *data1, struct pcap_pkthdr* h, u_char *p)
{
  struct ip* ip = (struct ip *)(p + link_offset);
  struct icmp *icmp = (struct icmp *)(ip + 1);
  struct tcphdr *tcphdr = (struct tcphdr *)(ip+1);
  u_long tmp_ip;

  if (targetaddr == ip->ip_src.s_addr) 
  {
    if (((inquery == 1) && (icmp->icmp_type == ICMP_ECHOREPLY)) || ((inquery == 2) && (icmp->icmp_type == ICMP_TIMESTAMPREPLY)) || ((inquery == 3) && (sport == ntohs(tcphdr->th_dport))))
    {
      if (ttlval == ip->ip_ttl)
      {
        fprintf(stdout,"NORMAL: %s TTL appears normal\n", inet_ntoa(ip->ip_src));
      }
      else
      {
        fprintf(stdout,"WARNING: %s TTL mismatch, expected %d, received %d, possible spoofed packet\n",inet_ntoa(ip->ip_src),ttlval,ip->ip_ttl);
      }
      target_answers++;
      alarm(1);
    }
    else
    {
      /* we do not wish to see any odd packets from our own address */
      tmp_ip = libnet_get_ipaddr(l, dev, ebuf);
      tmp_ip = htonl(src_ip);
      if (!(tmp_ip == ip->ip_src.s_addr))
      {  
        if (verbose) fprintf(stdout,"Received unexpected packet from %s\n",inet_ntoa(ip->ip_src));
        unexpected++;
      }
    }
  }
}

/*
 * set up to receive packets 
 */
void receive_packets(void)
{
  int snaplen = 65535, promisc = 1, to_ms = 1000;

  if (pcap_lookupnet(dev,&net.s_addr,&mask.s_addr, pcap_err) == -1) 
  {
    perror(pcap_err);
    exit(-1);
  } 

  if (verbose) fprintf(stdout,"Listening for replies on %s: %s\n", dev, inet_ntoa(net));

  if ((pd = pcap_open_live(dev, snaplen, promisc, to_ms, pcap_err)) == NULL)
  {
    perror(pcap_err);
    exit(-1);
  }

  switch(pcap_datalink(pd)) 
  {
    case DLT_EN10MB:
    case DLT_IEEE802:
      link_offset = 14;
      break;
    case DLT_SLIP: 
      link_offset = 16;
      break;
    case DLT_PPP:
    case DLT_NULL:
      link_offset = 4;
      break;
    case DLT_RAW: 
      link_offset = 0;
      break;
    default:
      fprintf(stderr,"unsupported interface type\n");
      exit(-1);
  }

  while (pcap_loop(pd,0,(pcap_handler)grab_packet,0));
}

/*
 * Called by caughtsig, if we're sniffing to turn off the promisc bit
 */
void stop_sniffing(void)
{
  int fd, i;
  struct ifreq buf[16];
  struct ifconf ifc;

  if ((fd = socket (AF_INET, SOCK_DGRAM, 0)) >= 0)
  {
    ifc.ifc_len = sizeof buf;
    ifc.ifc_buf = (caddr_t) buf;
    if (!(ioctl (fd, SIOCGIFCONF, (char *) &ifc))) 
    {
      i = ifc.ifc_len / sizeof (struct ifreq);
      if (!i)
      {
        fprintf(stderr, "Can't find an interface.\n");
        exit(-1);
      }
      while (i-- > 0)
      {
        /* cmp only length of dev since ifr_name may have extra chars */
        if (!(strncmp(buf[i].ifr_name,dev,strlen(dev))))
        {
          if(!(ioctl(fd, SIOCGIFFLAGS, (char *) &buf[i])))
          {
            if (buf[i].ifr_flags & IFF_PROMISC) 
            {
              buf[i].ifr_flags = buf[i].ifr_flags ^ IFF_PROMISC;
              if(ioctl(fd, SIOCSIFFLAGS, &buf[i])) perror("ioctl");
            }
          }
          else perror("ioctl");
        } 
      } /* end while loop */
    } 
    else perror ("ioctl");
  } 
  else perror ("socket");
  close (fd);
}

void caughtsig(int sig)
{
  if (pd) stop_sniffing();
  if (target_answers == 0) fprintf(stdout,"ERROR: No response from %s\n",targetname);
  if (verbose) total();
  exit(sig);
}

/*
 * usage
 */
void usage(char *prog)
{
  fprintf(stderr,"USAGE:\n");
  fprintf(stderr,"%s [opts] [-d dev] [-i 0-3] [-l|p num ] [-s src] [-t sec] [-T TTL] target\n\n",prog);
  fprintf(stderr,"  opts are a h v\n");
  fprintf(stderr,"    -a set ACK flag on TCP packets\n");
  fprintf(stderr,"    -h this help screen\n");
  fprintf(stderr,"    -v verbose\n");
  fprintf(stderr,"  -d device to grab local IP or sniff from, default is eth0\n");
  fprintf(stderr,"  -l local port to bind to, default is 80\n");
  fprintf(stderr,"  -p target port to send to, default is 80\n");
  fprintf(stderr,"  -i inquery packet type to send/receive, types include the following:\n");
  fprintf(stderr,"       1  tcp (default)\n");
  fprintf(stderr,"       2  icmp echo\n");
  fprintf(stderr,"       3  icmp timestamp\n");
  fprintf(stderr,"  -s spoofed source address\n");
  fprintf(stderr,"  -t time in seconds to wait for all replies (default 10)\n");
  fprintf(stderr,"  -T TTL to test (required)\n");
  fprintf(stderr,"  target (IP address or hostname)\n");
  fprintf(stderr,"\n");
}

int main(int argc, char **argv)
{
  char *prog;
  extern char *optarg;
  extern int optind;
  extern int optopt;
  extern int opterr;
  char ch;
  int spoof = 0, setdev = 0, timeout = 10, ttl = 0;
  u_long dst_ip = 0;
  pid_t pid;
  
  dev = "eth0"; /* default */
  verbose = 0;
  prog = argv[0];
  sport = 80;
  dport = 80;
  icmp_packet = 0;
  tcp_packet = 0;
  udp_packet = 0;
  ack = 0;
  inquery = 1; /* default */

  while ((ch = getopt(argc, argv, "ah?vd:l:p:i:s:t:T:")) != EOF) 
    switch(ch)
    {
      case 'a':
        ack = 1;
        break;
      case 'h':
      case '?':
        usage(prog);
        exit(0);
      case 'v':
        verbose = 1;
        break;
      case 'd':
        dev = optarg;
        setdev = 1;
        break;
      case 'l':
        sport = (u_short) strtol(optarg, NULL, 10);
        break;
      case 'p':
        dport = (u_short) strtol(optarg, NULL, 10);
        break;
      case 'i':
        inquery = (int) strtol(optarg, NULL, 10);
        break;
      case 's':
        if (!(src_ip = libnet_name_resolve(optarg, LIBNET_RESOLVE)))
        {
          fprintf(stderr,"=== Unable to resolve source host,\n");
          fprintf(stderr,"=== try spoofing with an IP address\n");
          usage(prog);
          exit(-1);
        }
        spoof = 1;
        break;
      case 't':
        timeout = (int) strtol(optarg, NULL, 10);
        break;
      case 'T':
        ttl = 1;
        ttlval = (u_char) strtol(optarg, NULL, 10);
        break;
      default:
        usage(prog);
        exit(-1);
    }
  argc -= optind;
  argv += optind;

  /* post arg processing */
  if (verbose) fprintf(stdout,"DESPOOF - determine spoofed packets v0.9\nSimple Nomad - thegnome@razor.bindview.com\nCopyright (c) 2000 BindView Corporation\n");

  if(getuid()!=0) 
  {
        fprintf(stderr, "=== You must be root to run %s!\n\n", prog);
        exit(-1);
  } 

  if (!ttl)
  {
    fprintf(stderr, "=== You must specify a TTL!\n");
    usage(prog);
    exit(-1);
  }

  if (!argv[0] || !strlen(argv[0]))
  {
    fprintf(stderr,"=== You must specify a target!\n");
    usage(prog);
    exit(-1);
  }

  if ((inquery > 3) || (inquery < 1))
  {
    fprintf(stderr,"=== -i value is invalid\n");
    usage(prog);
    exit(-1);
  }

  if ((setdev) && (spoof))
  {
    fprintf(stderr, "=== You cannot specify a device for a source IP address\n");
    fprintf(stderr, "=== and spoof your source IP address at the same time.\n");
    usage(prog);
    exit(-1);
  }

  if(!spoof)
  {
    src_ip = libnet_get_ipaddr(l, dev, ebuf);
    if ((src_ip == -1) || (src_ip == 0))
    {
      fprintf(stderr, "=== Grabbing address from %s failed,\n",dev);
      fprintf(stderr, "=== try a different device.\n");
      usage(prog);
      exit(-1);
    }
    src_ip = htonl(src_ip);
  }

  switch (inquery)
  {
    case 1:
      proto = IPPROTO_TCP;
      ip_header = LIBNET_TCP_H;
      break;
    case 2:
      proto = IPPROTO_ICMP;
      ip_header = LIBNET_ICMP_ECHO_H;
      break;
    case 3:
      proto = IPPROTO_ICMP;
      ip_header = LIBNET_ICMP_TS_H;
      break;
  }

  /* end post arg processing */

  targetname = *argv;
  targetaddr = libnet_name_resolve(targetname, LIBNET_RESOLVE);
  if ((targetaddr == 0xffffffff) || (targetaddr == -1))
  {
    fprintf(stdout, "ERROR: Unable to resolve %s\n",targetname);
    exit(1);
  };

  /* if sending packets, fork off the process so we can start listening
     to replies immediately */
  pid = fork();
  switch(pid)
  {
    case -1:
      fprintf(stderr, "=== Unable to fork off for sending packets\n");
      exit(-1);
    case 0:
      send_packets();
      break;
    default:
      /* set up timeout*/
      signal(SIGALRM, caughtsig);
      alarm(timeout);
      receive_packets();
      break;
  }
  exit(0);
}
