/* DiaCanvas -- A technical drawing canvas.
 * Copyright (C) 1999, Arjan Molenaar
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include "diabaseelement.h"

static void handle_move_x (DiaObject *obj, gfloat x);
static void handle_move_y (DiaObject *obj, gfloat y);

static void destroy (DiaObject *object);
static void draw (DiaObject* obj, DiaRenderer* renderer);
static void draw_handles (DiaObject* obj, DiaRenderer* renderer);
static void draw_cps (DiaObject* obj, DiaRenderer* renderer);
static gfloat distance (DiaObject* obj, Point* point);
static void move (DiaObject *obj, gfloat dx, gfloat dy);
static void move_handle (DiaObject *obj, DiaHandle *handle,
			 gfloat dx, gfloat dy);
static void copy (DiaBaseElement *from, DiaBaseElement *to);
static gint is_empty (DiaObject *obj);
static void calc_bounding_box (DiaBaseElement *elem);
static gint event (DiaObject *obj, DiaEvent *event, DiaLayer *layer);
static gfloat cp_distance (DiaObject *obj, Point *pos, Point *con_pos,
			   DiaConnectionPoint **cp);
static DiaConnectionPoint* cp_connect (DiaObject *obj, DiaHandle *h,
				       Point *pos);
static void cp_disconnect (DiaObject *obj, DiaConnectionPoint *p,
			   DiaHandle *h);

static DiaObjectOps element_ops =
{
  destroy,
  draw,
  draw_handles,
  draw_cps,
  distance,
  move,
  move_handle,
  (DiaCopyFunc) copy,
  event,
  is_empty,
  (DiaCalcBoundingBoxFunc) calc_bounding_box,
  cp_distance,
  cp_connect,
  cp_disconnect
};

//static DiaObjectOps *parent_ops = NULL;
#define parent_ops (dia_base_element_get_type ()->parent_type ()->ops)

DiaObjectType*
dia_base_element_get_type ()
{
  static DiaObjectType element_type =
  {
    "DiaBaseElement",
    sizeof (DiaBaseElement),
    (DiaInitFunc) dia_base_element_init,
    &element_ops,
    dia_object_get_type
  };
  return &element_type;
}

void
dia_base_element_init (DiaBaseElement *element)
{
  gint i;
  DiaObject *obj;
  DiaHandle *h;
  
  obj = DIA_OBJECT (element);
  
  /* call parent */
  //dia_object_init (obj);
  /* save parent options */
  //if (!parent_ops)
    //parent_ops = obj->ops;
  
  /* overwrite old function declarations */
  //obj->object_type = &element_type;
  //obj->ops = &element_ops;
  
  for (i = 0; i < 8; i++)
    {
      h = dia_object_add_handle (obj, 0.0, 0.0);
      h->is_connectable = FALSE;
    }
    
  element->direction = DIA_BASE_ELEMENT_T;

  element->size.left = 0.0;
  element->size.top = 0.0;
  element->size.right = 1.0;
  element->size.bottom = 1.0;
}

void
calc_bounding_box (DiaBaseElement *element)
{
  dia_base_element_calc_size (element);
    
  DIA_OBJECT (element)->bounding_box = element->size;
}

void
dia_base_element_calc_size (DiaBaseElement *element)
{
  DiaObject *obj;
  
  g_return_if_fail (element != NULL);
  
  obj = DIA_OBJECT (element);
  
  element->size.top = ((DiaHandle*) g_ptr_array_index (obj->handles,
							   DIA_BASE_ELEMENT_T))->pos.y;
  element->size.bottom = ((DiaHandle*) g_ptr_array_index (obj->handles,
							   DIA_BASE_ELEMENT_B))->pos.y;
  element->size.left = ((DiaHandle*) g_ptr_array_index (obj->handles,
							   DIA_BASE_ELEMENT_L))->pos.x;
  element->size.right = ((DiaHandle*) g_ptr_array_index (obj->handles,
							   DIA_BASE_ELEMENT_R))->pos.x;
  
}

void
dia_base_element_resize (DiaBaseElement *element, gfloat width, gfloat height)
{
  DiaHandle *handle;
  DiaBaseElementDirection old_dir;
  
  g_return_if_fail (element != NULL);
  
  old_dir = element->direction;
  element->direction = DIA_BASE_ELEMENT_R;
  handle = (DiaHandle*)g_ptr_array_index (DIA_OBJECT (element)->handles, DIA_BASE_ELEMENT_L);
  handle_move_x (DIA_OBJECT (element), handle->pos.x + width);
  
  element->direction = DIA_BASE_ELEMENT_B;
  handle = (DiaHandle*)g_ptr_array_index (DIA_OBJECT (element)->handles, DIA_BASE_ELEMENT_T);
  handle_move_y (DIA_OBJECT (element), handle->pos.y + height);

  element->direction = old_dir;
}

/* "events" */
static void
destroy (DiaObject *object)
{
  parent_ops->destroy (object);
}

static void
draw (DiaObject* obj, DiaRenderer* renderer)
{
  parent_ops->draw (obj, renderer);
}

static void
draw_handles (DiaObject* obj, DiaRenderer* renderer)
{
  parent_ops->draw_handles (obj, renderer);
}

static void
draw_cps (DiaObject* obj, DiaRenderer* renderer)
{
  parent_ops->draw_cps (obj, renderer);
}

static gfloat
distance (DiaObject* obj, Point* point)
{
  return distance_rectangle_point (&DIA_BASE_ELEMENT (obj)->size, point);
}

static void
move (DiaObject *obj, gfloat dx, gfloat dy)
{
  DiaBaseElement *elem = DIA_BASE_ELEMENT (obj);
  
  parent_ops->move (obj, dx, dy);
  
  elem->size.left += dx;
  elem->size.top += dy;
  elem->size.right += dx;
  elem->size.bottom += dy;
}

static void
move_handle (DiaObject *obj, DiaHandle *handle, gfloat dx, gfloat dy)
{
  parent_ops->move_handle (obj, handle, dx, dy);
}

static void
copy (DiaBaseElement *from, DiaBaseElement *to)
{
  parent_ops->copy (DIA_OBJECT (from), DIA_OBJECT (to));
}

static gint
is_empty (DiaObject* obj)
{
  return TRUE;
}

static void
cp_update_x (DiaObject *obj)
{
  gint i;
  DiaConnectionPoint *cp;

  /* update connection points: */
  for (i = 0; i < obj->connections->len; i++)
    {
      cp = (DiaConnectionPoint*)g_ptr_array_index (obj->connections, i);

      /* If CP on right side, just adjust the position: */
      switch (((DiaBaseElementCP*)cp->data)->direction)
	{
	case DIA_BASE_ELEMENT_TL:
	  cp->pos.x = DIA_OBJECT_GET_HANDLE (obj, DIA_BASE_ELEMENT_TL)->pos.x +
	    ((DIA_OBJECT_GET_HANDLE (obj, DIA_BASE_ELEMENT_TR)->pos.x -
	      DIA_OBJECT_GET_HANDLE (obj, DIA_BASE_ELEMENT_TL)->pos.x) *
	     ((DiaBaseElementCP*) cp->data)->factor);
	  break;
	case DIA_BASE_ELEMENT_BR:
	  cp->pos.x = DIA_OBJECT_GET_HANDLE (obj, DIA_BASE_ELEMENT_BR)->pos.x +
	    ((DIA_OBJECT_GET_HANDLE (obj, DIA_BASE_ELEMENT_BL)->pos.x -
	      DIA_OBJECT_GET_HANDLE (obj, DIA_BASE_ELEMENT_BR)->pos.x) *
	     ((DiaBaseElementCP*) cp->data)->factor);
	  break;
	case DIA_BASE_ELEMENT_TR:
	  cp->pos.x = DIA_OBJECT_GET_HANDLE (obj, DIA_BASE_ELEMENT_R)->pos.x;
	  break;
	case DIA_BASE_ELEMENT_BL:
	  cp->pos.x = DIA_OBJECT_GET_HANDLE (obj, DIA_BASE_ELEMENT_L)->pos.x;
	  break;
	default:
	  break;
	}
    } /* for */	    
}

/* See handle_move_y below for an explanation */
static void
handle_move_x (DiaObject *obj, gfloat x)
{
  DiaBaseElementDirection dir;
  
  dir = DIA_BASE_ELEMENT (obj)->direction;
  
  /* adjusting horizontal coordinates */
  if ((dir == DIA_BASE_ELEMENT_R) ||
      (dir == DIA_BASE_ELEMENT_TR) ||
      (dir == DIA_BASE_ELEMENT_BR))
    {
      /* if the object is about to invert itself, we will  */
      if (((DiaHandle*)g_ptr_array_index (obj->handles,
					  DIA_BASE_ELEMENT_L))->pos.x > x)
	{
	  /* set the bottom coordinates equal to the upper coordinates: */
	  DIA_OBJECT_GET_HANDLE (obj, DIA_BASE_ELEMENT_R)->pos.x =
	    DIA_OBJECT_GET_HANDLE (obj, DIA_BASE_ELEMENT_TR)->pos.x =
	    DIA_OBJECT_GET_HANDLE (obj, DIA_BASE_ELEMENT_BR)->pos.x =
	    DIA_OBJECT_GET_HANDLE (obj, DIA_BASE_ELEMENT_L)->pos.x;
	  switch (dir)
	    {
	    case DIA_BASE_ELEMENT_R:
	      DIA_BASE_ELEMENT (obj)->direction = DIA_BASE_ELEMENT_L;
	      break;
	    case DIA_BASE_ELEMENT_TR:
	      DIA_BASE_ELEMENT (obj)->direction = DIA_BASE_ELEMENT_TL;
	      break;
	    case DIA_BASE_ELEMENT_BR:
	      DIA_BASE_ELEMENT (obj)->direction = DIA_BASE_ELEMENT_BL;
	      break;
	    default:
	      break;
	    }
	  /* reenter the function as if we are adjusting the upper row: */
	  handle_move_x (obj, x);
	}
      else
	{
	  /* Now we adjust the box' handles: */
	  DIA_OBJECT_GET_HANDLE (obj, DIA_BASE_ELEMENT_R)->pos.x =
	    DIA_OBJECT_GET_HANDLE (obj, DIA_BASE_ELEMENT_TR)->pos.x =
	    DIA_OBJECT_GET_HANDLE (obj, DIA_BASE_ELEMENT_BR)->pos.x = x;

	  DIA_OBJECT_GET_HANDLE (obj, DIA_BASE_ELEMENT_B)->pos.x =
	    DIA_OBJECT_GET_HANDLE (obj, DIA_BASE_ELEMENT_T)->pos.x =
	    (DIA_OBJECT_GET_HANDLE (obj, DIA_BASE_ELEMENT_L)->pos.x +
	     DIA_OBJECT_GET_HANDLE (obj, DIA_BASE_ELEMENT_R)->pos.x) / 2.0;

	  cp_update_x (obj);
	}
    }
  else
    if ((dir == DIA_BASE_ELEMENT_L) ||
	(dir == DIA_BASE_ELEMENT_TL) ||
	(dir == DIA_BASE_ELEMENT_BL))
      {
	/* if the object is about to invert itself, we will  */
	if (((DiaHandle*)g_ptr_array_index (obj->handles,
					    DIA_BASE_ELEMENT_R))->pos.x < x)
	  {
	  /* recalculate x: */
	    //x -= ((DiaHandle*)g_ptr_array_index (obj->handles, DIA_BASE_ELEMENT_R))->pos.x;
	    /* set the bottom coordinates equal to the upper coordinates: */
	    DIA_OBJECT_GET_HANDLE (obj, DIA_BASE_ELEMENT_L)->pos.x =
	      DIA_OBJECT_GET_HANDLE (obj, DIA_BASE_ELEMENT_TL)->pos.x =
	      DIA_OBJECT_GET_HANDLE (obj, DIA_BASE_ELEMENT_BL)->pos.x =
	      DIA_OBJECT_GET_HANDLE (obj, DIA_BASE_ELEMENT_R)->pos.x;
	    switch (dir)
	      {
	      case DIA_BASE_ELEMENT_L:
		DIA_BASE_ELEMENT (obj)->direction = DIA_BASE_ELEMENT_R;
		break;
	      case DIA_BASE_ELEMENT_TL:
		DIA_BASE_ELEMENT (obj)->direction = DIA_BASE_ELEMENT_TR;
		break;
	      case DIA_BASE_ELEMENT_BL:
		DIA_BASE_ELEMENT (obj)->direction = DIA_BASE_ELEMENT_BR;
		break;
	      default:
		break;
	      }
	    /* reenter the function as if we are adjusting the upper row: */
	    handle_move_x (obj, x);
	  }
	else
	  {
	    /* Now we adjust the box' handles: */
	    DIA_OBJECT_GET_HANDLE (obj, DIA_BASE_ELEMENT_L)->pos.x =
	      DIA_OBJECT_GET_HANDLE (obj, DIA_BASE_ELEMENT_TL)->pos.x =
	      DIA_OBJECT_GET_HANDLE (obj, DIA_BASE_ELEMENT_BL)->pos.x = x;

	    DIA_OBJECT_GET_HANDLE (obj, DIA_BASE_ELEMENT_B)->pos.x =
	      DIA_OBJECT_GET_HANDLE (obj, DIA_BASE_ELEMENT_T)->pos.x =
	      (DIA_OBJECT_GET_HANDLE (obj, DIA_BASE_ELEMENT_L)->pos.x +
	       DIA_OBJECT_GET_HANDLE (obj, DIA_BASE_ELEMENT_R)->pos.x) / 2.0;

	    cp_update_x (obj);
	  }
      }
}

/* Update the CP's if the top or bottom of the element is moved */
static void
cp_update_y (DiaObject *obj)
{
  gint i;
  DiaConnectionPoint *cp;

  /* update connection points: */
  for (i = 0; i < obj->connections->len; i++)
    {      
      cp = (DiaConnectionPoint*)g_ptr_array_index (obj->connections, i);

      /* If CP on right side, just adjust the position: */
      switch (((DiaBaseElementCP*)cp->data)->direction)
	{
	case DIA_BASE_ELEMENT_TR:
	  cp->pos.y = DIA_OBJECT_GET_HANDLE (obj, DIA_BASE_ELEMENT_TR)->pos.y +
	    ((DIA_OBJECT_GET_HANDLE (obj, DIA_BASE_ELEMENT_BR)->pos.y -
	      DIA_OBJECT_GET_HANDLE (obj, DIA_BASE_ELEMENT_TR)->pos.y) *
	     ((DiaBaseElementCP*) cp->data)->factor);
	  break;
	case DIA_BASE_ELEMENT_BL:
	  cp->pos.y = DIA_OBJECT_GET_HANDLE (obj, DIA_BASE_ELEMENT_BL)->pos.y +
	    ((DIA_OBJECT_GET_HANDLE (obj, DIA_BASE_ELEMENT_TL)->pos.y -
	      DIA_OBJECT_GET_HANDLE (obj, DIA_BASE_ELEMENT_BL)->pos.y) *
	     ((DiaBaseElementCP*) cp->data)->factor);
	  break;
	case DIA_BASE_ELEMENT_TL:
	  cp->pos.y = DIA_OBJECT_GET_HANDLE (obj, DIA_BASE_ELEMENT_T)->pos.y;
	  break;
	case DIA_BASE_ELEMENT_BR:
	  cp->pos.y = DIA_OBJECT_GET_HANDLE (obj, DIA_BASE_ELEMENT_B)->pos.y;
	  break;
	default:
	  break;
	}
    } /* for */	    
}

/* This huge function handles the move in the y direction. If an object is
 * about to invert (bottom < top or visa versa) the pointer will automatically
 * switch to the right handle in order to keer the box conststent.
 */
static void
handle_move_y (DiaObject *obj, gfloat y)
{
  DiaBaseElementDirection dir;
 
  dir = DIA_BASE_ELEMENT (obj)->direction;
  
  /* adjusting vertical coordinates */
  if ((dir == DIA_BASE_ELEMENT_B) ||
      (dir == DIA_BASE_ELEMENT_BL) ||
      (dir == DIA_BASE_ELEMENT_BR))
    {
      /* if the object is about to invert itself, we will  */
      if (((DiaHandle*)g_ptr_array_index (obj->handles,
					  DIA_BASE_ELEMENT_T))->pos.y > y)
	{
	  /* set the bottom coordinates equal to the upper coordinates: */
	  DIA_OBJECT_GET_HANDLE (obj, DIA_BASE_ELEMENT_B)->pos.y =
	    DIA_OBJECT_GET_HANDLE (obj, DIA_BASE_ELEMENT_BL)->pos.y =
	    DIA_OBJECT_GET_HANDLE (obj, DIA_BASE_ELEMENT_BR)->pos.y =
	    DIA_OBJECT_GET_HANDLE (obj, DIA_BASE_ELEMENT_T)->pos.y;
	  switch (dir)
	    {
	    case DIA_BASE_ELEMENT_B:
	      DIA_BASE_ELEMENT (obj)->direction = DIA_BASE_ELEMENT_T;
	      break;
	    case DIA_BASE_ELEMENT_BL:
	      DIA_BASE_ELEMENT (obj)->direction = DIA_BASE_ELEMENT_TL;
	      break;
	    case DIA_BASE_ELEMENT_BR:
	      DIA_BASE_ELEMENT (obj)->direction = DIA_BASE_ELEMENT_TR;
	      break;
	    default:
	      break;
	    }
	  /* reenter the function as if we are adjusting the upper row: */
	  handle_move_y (obj, y);
	}
      else
	{
	  /* Now we adjust the box' handles: */
	  DIA_OBJECT_GET_HANDLE (obj, DIA_BASE_ELEMENT_B)->pos.y =
	    DIA_OBJECT_GET_HANDLE (obj, DIA_BASE_ELEMENT_BL)->pos.y =
	    DIA_OBJECT_GET_HANDLE (obj, DIA_BASE_ELEMENT_BR)->pos.y = y;

	  DIA_OBJECT_GET_HANDLE (obj, DIA_BASE_ELEMENT_L)->pos.y =
	    DIA_OBJECT_GET_HANDLE (obj, DIA_BASE_ELEMENT_R)->pos.y =
	    (DIA_OBJECT_GET_HANDLE (obj, DIA_BASE_ELEMENT_T)->pos.y +
	     DIA_OBJECT_GET_HANDLE (obj, DIA_BASE_ELEMENT_B)->pos.y) / 2.0;

	  cp_update_y (obj);
	}
    }
  else
    if ((dir == DIA_BASE_ELEMENT_T) ||
	(dir == DIA_BASE_ELEMENT_TL) ||
	(dir == DIA_BASE_ELEMENT_TR))
      {
	/* if the object is about to invert itself, we will  */
	if (((DiaHandle*)g_ptr_array_index (obj->handles,
					    DIA_BASE_ELEMENT_B))->pos.y < y)
	  {
	    /* set the bottom coordinates equal to the upper coordinates: */
	    DIA_OBJECT_GET_HANDLE (obj, DIA_BASE_ELEMENT_T)->pos.y =
	      DIA_OBJECT_GET_HANDLE (obj, DIA_BASE_ELEMENT_TL)->pos.y =
	      DIA_OBJECT_GET_HANDLE (obj, DIA_BASE_ELEMENT_TR)->pos.y =
	      DIA_OBJECT_GET_HANDLE (obj, DIA_BASE_ELEMENT_B)->pos.y;
	    switch (dir)
	      {
	      case DIA_BASE_ELEMENT_T:
		DIA_BASE_ELEMENT (obj)->direction = DIA_BASE_ELEMENT_B;
		break;
	      case DIA_BASE_ELEMENT_TL:
		DIA_BASE_ELEMENT (obj)->direction = DIA_BASE_ELEMENT_BL;
		break;
	      case DIA_BASE_ELEMENT_TR:
		DIA_BASE_ELEMENT (obj)->direction = DIA_BASE_ELEMENT_BR;
		break;
	      default:
		break;
	      }
	    /* reenter the function as if we are adjusting the upper row: */
	    handle_move_y (obj, y);
	  }
	else
	  {
	    /* Now we adjust the box' handles: */
	    DIA_OBJECT_GET_HANDLE (obj, DIA_BASE_ELEMENT_T)->pos.y =
	      DIA_OBJECT_GET_HANDLE (obj, DIA_BASE_ELEMENT_TL)->pos.y =
	      DIA_OBJECT_GET_HANDLE (obj, DIA_BASE_ELEMENT_TR)->pos.y = y;

	    DIA_OBJECT_GET_HANDLE (obj, DIA_BASE_ELEMENT_L)->pos.y =
	      DIA_OBJECT_GET_HANDLE (obj, DIA_BASE_ELEMENT_R)->pos.y =
	      (DIA_OBJECT_GET_HANDLE (obj, DIA_BASE_ELEMENT_T)->pos.y +
	       DIA_OBJECT_GET_HANDLE (obj, DIA_BASE_ELEMENT_B)->pos.y) / 2.0;
	    
	    cp_update_y (obj);
	  }
      }
}

static gint
event (DiaObject *obj, DiaEvent *event, DiaLayer *layer)
{
  gboolean result = FALSE;
  gint i;

  switch (event->type)
    {
    case DIA_PLACE:
      DIA_OBJECT_SET_FLAGS (obj, DIA_BASE_ELEMENT_NEW);
      if (DIA_OBJECT_GET_HANDLE (obj, DIA_BASE_ELEMENT_BR)->is_movable)
	{
	  DIA_OBJECT_SET_FLAGS (obj, DIA_BASE_ELEMENT_MOVE_HANDLE);
	  DIA_BASE_ELEMENT (obj)->direction = DIA_BASE_ELEMENT_BR;
	  dia_base_element_resize (DIA_BASE_ELEMENT (obj), 0.0, 0.0);
	}
      else
	{
	  DIA_OBJECT_SET_FLAGS (obj, DIA_BASE_ELEMENT_MOVE);
	}
      
      DIA_OBJECT_SET_REQUEST (obj, DIA_REQUEST_FOCUS | DIA_REQUEST_SELECT
			      | DIA_REQUEST_REDRAW | DIA_REQUEST_GRAB);
      result = TRUE;
      break;
    case DIA_BUTTON_PRESS:
      switch (event->button.button)
	{
	case 1:
	  DIA_OBJECT_SET_REQUEST (obj, DIA_REQUEST_FOCUS | DIA_REQUEST_SELECT
				  | DIA_REQUEST_REDRAW | DIA_REQUEST_GRAB);
	  obj->update_box = obj->bounding_box;
	  if (event->button.handle && ((DiaHandle*)event->button.handle)->is_movable)
	    {
	      for (i = 0; i < 8; i++)
		{
		  if (g_ptr_array_index (obj->handles, i)
		          == event->button.handle)
		    {
		      DIA_BASE_ELEMENT (obj)->direction =
			(DiaBaseElementDirection) i;
		      DIA_OBJECT_SET_FLAGS (obj, DIA_BASE_ELEMENT_MOVE_HANDLE);
		      DIA_OBJECT_SET_REQUEST (obj, DIA_REQUEST_STOP_EMIT);
		      result = TRUE;
		      break;
		    }
		}
	    }
	  else
	    {
	      obj->snap_pos = event->button.snap;
	      //point_sub (&obj->snap_pos, &obj->position);
	      DIA_OBJECT_SET_FLAGS (obj, DIA_BASE_ELEMENT_MOVE);
	      result = TRUE;
	    }
	  break; 
	default:
	  break;
	} /* inner switch*/
      break;
    case DIA_MOTION:
      if (obj->flags & DIA_BASE_ELEMENT_MOVE)
	{
	  Point delta;
	  	  
	  delta = event->motion.snap;
	  //point_sub (&delta, &obj->position);
	  /* delta is now relative */
	  point_sub (&delta, &obj->snap_pos);
	  
	  dia_object_move (obj, delta.x, delta.y);
	  
	  DIA_OBJECT_SET_REQUEST (obj, DIA_REQUEST_REDRAW |
				  DIA_REQUEST_EXTENTS |
				  DIA_REQUEST_UPDATE_CONNECTIONS);
	  result = TRUE;
	}
      else if (obj->flags & DIA_BASE_ELEMENT_MOVE_HANDLE)
	{
	  obj->update_box = obj->bounding_box;
	  /* define new position for the handle */

	  handle_move_x (obj, event->motion.snap.x);
	  handle_move_y (obj, event->motion.snap.y);
	    
	  dia_object_calc_bounding_box (obj);
	  /* update the objects connection points */
	  //dia_base_element_update_connection_points (DIA_BASE_ELEMENT (obj));
	  if (DIA_OBJECT_IS_SET (obj, DIA_BASE_ELEMENT_NEW))
	    DIA_OBJECT_UNSET_FLAGS (obj, DIA_BASE_ELEMENT_NEW);
	  
	    	    
	  rectangle_union (&obj->update_box, &obj->bounding_box);
	    
	  DIA_OBJECT_SET_REQUEST (obj, DIA_REQUEST_EXTENTS
				  | DIA_REQUEST_REDRAW
				  | DIA_REQUEST_UPDATE_CONNECTIONS
				  | DIA_REQUEST_STOP_EMIT);
	  result = TRUE;
	}
      break;
    case DIA_BUTTON_RELEASE:
      switch (event->button.button)
	{
	case 1:
	  DIA_OBJECT_UNSET_FLAGS (obj,
				  DIA_BASE_ELEMENT_MOVE_HANDLE
				  | DIA_BASE_ELEMENT_MOVE);
	  DIA_OBJECT_SET_REQUEST (obj, DIA_REQUEST_UNGRAB);
	  if (DIA_OBJECT_IS_SET (obj, DIA_BASE_ELEMENT_NEW)
	      && (DIA_OBJECT_GET_HANDLE (obj, DIA_BASE_ELEMENT (obj)->direction)->is_movable))
	    {
	      dia_base_element_resize (DIA_BASE_ELEMENT (obj), 2.0, 2.0);
	      dia_object_calc_bounding_box (obj);
	      obj->update_box = obj->bounding_box;
	      DIA_OBJECT_SET_REQUEST (obj, DIA_REQUEST_REDRAW
				      | DIA_REQUEST_EXTENTS);	      
	      DIA_OBJECT_UNSET_FLAGS (obj,  DIA_BASE_ELEMENT_NEW);
	    }
	  
	  result = TRUE;
	  break;
	default:
	  break;
	} /* inner switch */
      break;
    default:
      result = parent_ops->event (obj, event, layer);
      break;
    }
  return result;
}

static gfloat
cp_distance (DiaObject *obj, Point *pos, Point *con_pos,
	     DiaConnectionPoint **cp)
{
  return parent_ops->cp_distance (obj, pos, con_pos, cp);
}

static DiaConnectionPoint*
cp_connect (DiaObject *obj, DiaHandle *h, Point *pos)
{
  return parent_ops->cp_connect (obj, h, pos);
}

static void
cp_disconnect (DiaObject *obj, DiaConnectionPoint *cp, DiaHandle *h)
{
  parent_ops->cp_disconnect (obj, cp, h);
}






