/* DiaCanvas -- A technical drawing canvas.
 * Copyright (C) 1999, Arjan Molenaar
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include "diabaseline.h"
//#include "math.h"

static void destroy (DiaObject *object);
static void draw (DiaObject* obj, DiaRenderer* renderer);
static void draw_handles (DiaObject* obj, DiaRenderer* renderer);
static void draw_cps (DiaObject* obj, DiaRenderer* renderer);
//static gfloat distance_real (DiaObject* obj, Point* point,
//			     Point *point_on_line);
static gfloat distance (DiaObject* obj, Point* point);
static void move (DiaObject *obj, gfloat dx, gfloat dy);
static void move_handle (DiaObject *obj, DiaHandle *handle,
			 gfloat dx, gfloat dy);
static void copy (DiaBaseLine* from, DiaBaseLine *to);
static gint is_empty (DiaObject *obj);
static void calc_bounding_box (DiaBaseLine *line);
static gint event (DiaObject *obj, DiaEvent *event, DiaLayer *layer);
static gfloat cp_distance (DiaObject *obj, Point *pos, Point *con_pos,
			   DiaConnectionPoint **cp);
static DiaConnectionPoint* cp_connect (DiaObject *obj, DiaHandle *h,
				       Point *pos);
static void cp_disconnect (DiaObject *obj, DiaConnectionPoint *cp,
			   DiaHandle *h);

static DiaObjectOps line_ops =
{
  destroy,
  draw,
  draw_handles,
  draw_cps,
  distance,
  move,
  move_handle,
  (DiaCopyFunc) copy,
  event,
  is_empty,
  (DiaCalcBoundingBoxFunc) calc_bounding_box,
  cp_distance,
  cp_connect,
  cp_disconnect
};

//static DiaObjectOps *parent_ops = NULL;
#define parent_ops (dia_base_line_get_type ()->parent_type ()->ops)

DiaObjectType*
dia_base_line_get_type ()
{
  static DiaObjectType line_type =
  {
    "DiaBaseLine",
    sizeof (DiaBaseLine),
    (DiaInitFunc) dia_base_line_init,
    &line_ops,
    dia_object_get_type
  };
  return &line_type;
}

void
dia_base_line_init (DiaBaseLine *base_line)
{
  gint i;
  
  base_line->line_caps = DIA_LINE_CAPS_ROUND;
  base_line->line_join = DIA_LINE_JOIN_ROUND;
  base_line->line_style = DIA_LINE_STYLE_SOLID;
  base_line->line_width = 0.1;
  base_line->color = dia_color_black;

  base_line->handle_moving = NULL;
  
  /* add two points (a begin and an end point) for the line */
  for (i = 0; i < 2; i++)
    {
      dia_object_add_handle (DIA_OBJECT (base_line), (gfloat) i, (gfloat) i);
    }
}

DiaObject*
dia_base_line_new (Point *pos)
{
  DiaObject *new_line;
  
  g_return_val_if_fail (pos != NULL, NULL);
  
  new_line = dia_object_create (dia_base_line_get_type ());

  /* move the line to the position where the mouse is: */
  dia_object_move (new_line, pos->x, pos->y);
  
  return new_line;
}

gfloat
dia_base_line_cp_calc_factor (Point *p1, Point *p2, Point *cp_pos)
{
  gfloat factor;

  if (p2->x != p1->x)
    factor = (cp_pos->x - p1->x) / (p2->x - p1->x);
  else
    if (p2->y != p1->y)
      factor = (cp_pos->y - p1->y) / (p2->y - p1->y);
    else
      factor = 0.0;
  
  //factor = (cp_pos->x - p1->x) / (p2->x - p1->x);
  //if (factor == 0.0)
  //  factor = (cp_pos->y - p1->y) / (p2->y - p1->y);
  
  return ABS (factor);
}

void
dia_base_line_remove_point (DiaBaseLine *line, gint index)
{
  gint i;
  DiaObject *obj;
  DiaHandle *prev_handle = NULL;
  DiaHandle *next_handle = NULL;
  DiaBaseLineCP *cp_data;
  DiaHandle *handle;
  
  g_return_if_fail (line != NULL);
  g_return_if_fail (index > 0);
  g_return_if_fail (index < DIA_OBJECT (line)->handles->len - 1);
  
  obj = DIA_OBJECT (line);
  handle = DIA_OBJECT_GET_HANDLE (obj, index);
  prev_handle = DIA_OBJECT_GET_HANDLE (obj, index - 1);
  next_handle = DIA_OBJECT_GET_HANDLE (obj, index + 1);

  /* Update connection points */
  for (i = 0; i < obj->connections->len; i++)
    {
      cp_data = (DiaBaseLineCP*) DIA_OBJECT_GET_CP (obj, i)->data;
      
      if (cp_data->handle == handle)
	{
	  cp_data->handle = prev_handle;
	  cp_data->factor = dia_base_line_cp_calc_factor (&prev_handle->pos,
							  &next_handle->pos,
							  &DIA_OBJECT_GET_CP (obj, i)->pos);
	}
      if (cp_data->handle == prev_handle)
	{
	  cp_data->factor = dia_base_line_cp_calc_factor (&prev_handle->pos,
							  &next_handle->pos,
							  &DIA_OBJECT_GET_CP (obj, i)->pos);
	}
      
    }
  
  dia_object_remove_handle (obj, handle);
  
  dia_base_line_update_connection_points (line, prev_handle);
}

void
dia_base_line_update_connection_points (DiaBaseLine *line, DiaHandle *handle)
{
  DiaObject *obj;
  DiaConnectionPoint *cp;
  DiaHandle *handle_prev = NULL;
  DiaHandle *handle_next = NULL;
  gint i;
  
  obj = DIA_OBJECT (line);

  if (obj->connections->len == 0)
    {
      return;
    }
  
  for (i = 0; i < obj->handles->len; i++)
    {
      if (DIA_OBJECT_GET_HANDLE (obj, i) == handle)
	{
	  if (i > 0)
	    handle_prev = DIA_OBJECT_GET_HANDLE (obj, i - 1);
	  if ( i < obj->handles->len - 1)
	    handle_next = DIA_OBJECT_GET_HANDLE (obj, i + 1);
	  
	  break;
	}
    }

  if (!handle_next && !handle_prev)
    {
      g_warning ("Handle is alone...");
      return;
    }
  
  for (i = 0; i < obj->connections->len; i++)
    {
      cp = g_ptr_array_index (obj->connections, i);
     
      if ((((DiaBaseLineCP*) cp->data)->handle == handle) && handle_next)
	{
	  cp->pos.x = handle->pos.x + ((handle_next->pos.x - handle->pos.x) *
	     ((DiaBaseLineCP*) cp->data)->factor);
	  cp->pos.y = handle->pos.y + ((handle_next->pos.y - handle->pos.y) *
	     ((DiaBaseLineCP*) cp->data)->factor);
	}
      if ((((DiaBaseLineCP*) cp->data)->handle == handle_prev) && handle_prev)
	{
	  cp->pos.x = handle_prev->pos.x - ((handle_prev->pos.x - handle->pos.x) *
	     ((DiaBaseLineCP*) cp->data)->factor);
	  cp->pos.y = handle_prev->pos.y - ((handle_prev->pos.y - handle->pos.y) *
	     ((DiaBaseLineCP*) cp->data)->factor);
	}
    }
}

/* "events" */
static void
destroy (DiaObject *object)
{
  //DiaBaseLine *line = DIA_BASE_LINE (object);
  
  parent_ops->destroy (object);
}

static void
draw (DiaObject* obj, DiaRenderer* renderer)
{
  DiaBaseLine *line;
  gint i;
  Point *points;
  
  line = DIA_BASE_LINE (obj);
  
  renderer->ops->set_linewidth (renderer, line->line_width);
  renderer->ops->set_linestyle (renderer, line->line_style);
  renderer->ops->set_linecaps (renderer, line->line_caps);
  renderer->ops->set_linejoin (renderer, line->line_join);
  //renderer->ops->set_origin (renderer, obj->position.x, obj->position.y);

  points = g_new (Point, obj->handles->len);
  for (i = 0; i < obj->handles->len; i++)
    {
      points[i] = DIA_OBJECT_GET_HANDLE (obj, i)->pos;
    }
  renderer->ops->draw_polyline (renderer, points, obj->handles->len,
				 &line->color);
  
  g_free (points);
}

static void
draw_handles (DiaObject* obj, DiaRenderer* renderer)
{
  parent_ops->draw_handles (obj, renderer);
}

static void
draw_cps (DiaObject* obj, DiaRenderer* renderer)
{
  parent_ops->draw_cps (obj, renderer);
}

static gfloat
distance (DiaObject* obj, Point* point)
{
  return dia_base_line_distance_real (obj, point, NULL);
}

void
calc_bounding_box (DiaBaseLine* line)
{
  Rectangle *bb;
  DiaHandle *h;
  gint i;
  
  bb = &DIA_OBJECT (line)->bounding_box;
  
  h = DIA_OBJECT_GET_HANDLE (line, 0);
  
  bb->top = bb->bottom = h->pos.y;
  bb->left = bb->right = h->pos.x;
  
  for (i=1; i < DIA_OBJECT (line)->handles->len; i++)
    {
      h = DIA_OBJECT_GET_HANDLE (line, i);
      bb->top = MIN (bb->top, h->pos.y);
      bb->bottom = MAX (bb->bottom, h->pos.y);
      bb->left = MIN (bb->left, h->pos.x);
      bb->right = MAX (bb->right, h->pos.x);
    }

  bb->top -= line->line_width / 2.0;
  bb->bottom += line->line_width / 2.0;
  bb->left -= line->line_width / 2.0;
  bb->right += line->line_width / 2.0;
}

gfloat
dia_base_line_distance_real (DiaObject* obj, Point* point,
			     Point *point_on_line)
{
  gfloat min_dist = G_MAXFLOAT;
  gfloat dist;
  gint i;
  DiaBaseLine *line;
  Point p_on_line;
  
  g_return_val_if_fail (obj != NULL, min_dist);
  g_return_val_if_fail (point != NULL, min_dist);

  line = DIA_BASE_LINE (obj);
 
  for (i=0; i < obj->handles->len - 1; i++)
    {
      dist = distance_line_point (&DIA_OBJECT_GET_HANDLE (obj, i)->pos,
				  &DIA_OBJECT_GET_HANDLE (obj, i+1)->pos,
				  line->line_width, point, &p_on_line);
      if (dist < min_dist)
	{
	  min_dist = dist;
	  if (point_on_line)
	    *point_on_line = p_on_line;
	}
    }
  
  return min_dist;
}

static void
move (DiaObject *obj, gfloat dx, gfloat dy)
{
  parent_ops->move (obj, dx, dy);
}

static void
move_handle (DiaObject *obj, DiaHandle *handle, gfloat dx, gfloat dy)
{
  parent_ops->move_handle (obj, handle, dx, dy);
}

static void
copy (DiaBaseLine *from, DiaBaseLine *to)
{  
  DiaBaseLineCP *cp_data;
  DiaBaseLineCP *orig_cp_data;  
  gint i;
  gint n;
  
  parent_ops->copy (DIA_OBJECT (from), DIA_OBJECT (to));
  
  to->line_caps = from->line_caps;
  to->line_join = from->line_join;
  to->line_style = from->line_style;
  to->line_width = from->line_width;
  to->color = from->color;

  /* copy CP data: */
  for (i = 0; i < DIA_OBJECT (from)->connections->len; i++)
    {
      orig_cp_data = ((DiaBaseLineCP*) DIA_OBJECT_GET_CP (from, i)->data);
      cp_data = g_new (DiaBaseLineCP, 1);
      
      cp_data->factor = orig_cp_data->factor;
      /* now find the handle-index of the original handle that started
       * the segment. */
      for (n = 0; n < DIA_OBJECT (from)->handles->len; n++)
	{
	  if (DIA_OBJECT_GET_HANDLE (from, n) == orig_cp_data->handle)
	    {
	      cp_data->handle = DIA_OBJECT_GET_HANDLE (to, n);
	      break;
	    }
	}
      g_assert (cp_data->handle != NULL);
      
      DIA_OBJECT_GET_CP (to, i)->data = (gpointer) cp_data;
    }
}

static gint
is_empty (DiaObject* obj)
{
  return FALSE;
}

gboolean
dia_base_line_connect_handle (DiaBaseLine *line, DiaLayer *layer,
			      DiaHandle *handle)
{
  DiaObject *tmp_obj;
  Point tmp_pos;
  DiaConnectionPoint *tmp_cp;
  DiaConnectionPoint *conn_cp;
  gfloat dist;
  gboolean result = FALSE;
  
  dist = dia_layer_find_closest_cp (layer, handle,
				    &tmp_obj, &tmp_pos, &tmp_cp);

  if ((dist < DIA_CONNECTION_POINT_TRESHOLD)
      //&& !dia_object_is_connected_to (tmp_obj, DIA_OBJECT (line))
      && !dia_object_is_connected_to (DIA_OBJECT (line), tmp_obj))
    {
      conn_cp = dia_object_cp_connect (tmp_obj, handle, &tmp_pos);
      if (conn_cp)
	{
	  handle->pos = conn_cp->pos;
	  dia_object_calc_bounding_box_update (DIA_OBJECT (line));
	  DIA_OBJECT_SET_REQUEST (DIA_OBJECT (line), DIA_REQUEST_REDRAW
				  | DIA_REQUEST_EXTENTS
				  | DIA_REQUEST_UPDATE_CONNECTIONS);
	  result = TRUE;
	}
    }
  return result;
}

/* look for a connection point and "glue" the handle to it. */
static gboolean
find_cp (DiaLayer *layer, DiaHandle *handle, gboolean glue)
{
  DiaConnectionPoint *cp;
  DiaObject *obj;
  gfloat dist;
  Point con_pos;
  
  if (handle->is_connectable == FALSE)
    return FALSE;

  dist = dia_layer_find_closest_cp (layer, handle, &obj, &con_pos, &cp);

  if ((dist < DIA_CONNECTION_POINT_TRESHOLD)
      && !dia_object_is_connected_to (handle->object, obj)
      )//&& !dia_object_is_connected_to (obj, handle->object))
    {
      if (glue)/* do the "glue" effect! */
	handle->pos = con_pos;
      handle->could_be_connected = TRUE;
      //g_message ("found an object nearby!");
    }
  else
    {
      handle->could_be_connected = FALSE;
    }
  return handle->could_be_connected;
}

static gint
event (DiaObject *obj, DiaEvent *event, DiaLayer *layer)
{
  gboolean result = FALSE;
  gint i;

  switch (event->type)
    {
    case DIA_PLACE:
      dia_object_add_update (obj, &obj->bounding_box);
      
      dia_base_line_connect_handle (DIA_BASE_LINE (obj), layer,
				    DIA_OBJECT_GET_HANDLE (obj, 0));
	
      DIA_OBJECT_SET_FLAGS (obj, DIA_BASE_LINE_MOVE_HANDLE
			    | DIA_BASE_LINE_NEW);
      DIA_OBJECT_SET_REQUEST (obj, DIA_REQUEST_FOCUS | DIA_REQUEST_SELECT
			      | DIA_REQUEST_REDRAW | DIA_REQUEST_GRAB);
      DIA_BASE_LINE (obj)->handle_moving = 
	DIA_OBJECT_GET_HANDLE (obj, obj->handles->len - 1);
      DIA_BASE_LINE (obj)->handle_moving->pos = event->button.snap;
      
      result = TRUE;
      break;
    case DIA_BUTTON_PRESS:
      switch (event->button.button)
	{
	case 1:
	  //g_message ("DiaBaseLine::event(DIA_BUTTON_PRESS) %x", (int)obj);
	  obj->snap_pos = event->button.snap;
	  //g_message ("snap_pos = (%f, %f)", obj->snap_pos.x, obj->snap_pos.y);
	  
	  DIA_OBJECT_SET_REQUEST (obj, DIA_REQUEST_FOCUS | DIA_REQUEST_SELECT
				  | DIA_REQUEST_REDRAW | DIA_REQUEST_GRAB);
	  obj->update_box = obj->bounding_box;
	  if (event->button.handle
	      && event->button.handle->is_movable)
	    {
	      DIA_BASE_LINE (obj)->handle_moving = event->button.handle;
	      DIA_OBJECT_SET_FLAGS (obj, DIA_BASE_LINE_MOVE_HANDLE);
	      DIA_OBJECT_SET_REQUEST (obj, DIA_REQUEST_STOP_EMIT);
	      result = TRUE;
	    }
	  else
	    {
	      DIA_OBJECT_SET_FLAGS (obj, DIA_BASE_LINE_MOVE);
	      //point_sub (&obj->snap_pos, &obj->position);
	      result = TRUE;
	    }
	  break; 
	default:
	  break;
	} /* inner switch*/
      break; 
    case DIA_MOTION:
      if (obj->flags & DIA_BASE_LINE_MOVE)
	{
	  DiaHandle *handle;
	  Point delta;
  
	  /* unconnect handles */
	  for (i = 0; i < obj->handles->len; i++)
	    {
	      handle = DIA_OBJECT_GET_HANDLE (obj, i);
	      /* Only unconnect objects that are not selected (selected
	       * objects will also be dragged. */
	      if (!handle->keep_on_move && handle->connected_to &&
		  !(DIA_OBJECT_IS_SET (handle->connected_to->object,
				       DIA_OBJECT_STATE_SELECTED)))
		dia_handle_disconnect (handle);
	      handle->could_be_connected = FALSE;
	    }
	  
	  delta = event->motion.snap;
	  point_sub (&delta, &obj->snap_pos);
	  //g_print ("snap_pos = (%f, %f); ", obj->snap_pos.x, obj->snap_pos.y);
	  //g_print ("delta = (%f, %f)\n", delta.x, delta.y);

	  dia_object_move (obj, delta.x, delta.y);
	  //g_message ("Movin' line: %f, %f", delta.x, delta.y);
	  for (i = 0; i < obj->handles->len; i++)
	    find_cp (layer, DIA_OBJECT_GET_HANDLE (obj, i), FALSE);
	  
	  DIA_OBJECT_SET_REQUEST (obj, DIA_REQUEST_REDRAW
				  | DIA_REQUEST_EXTENTS
				  | DIA_REQUEST_UPDATE_CONNECTIONS);
	  result = TRUE;
	}
      else if (obj->flags & DIA_BASE_LINE_MOVE_HANDLE)
	{
	  /* disconnect if moved */
	  if ((DIA_BASE_LINE (obj)->handle_moving->connected_to))
	    {
	      dia_handle_disconnect (DIA_BASE_LINE (obj)->handle_moving);
	    }
	  
	  if (DIA_OBJECT_IS_SET (obj, DIA_BASE_LINE_NEW))
	    DIA_OBJECT_UNSET_FLAGS (obj, DIA_BASE_LINE_NEW);

	  /* define new position for the handle */
	  DIA_BASE_LINE (obj)->handle_moving->pos = event->motion.pos;
	  
	  if (!find_cp (layer, DIA_BASE_LINE (obj)->handle_moving, TRUE))
	    {
	      DIA_BASE_LINE (obj)->handle_moving->pos = event->motion.snap;
	      find_cp (layer, DIA_BASE_LINE (obj)->handle_moving, TRUE);
	    }
	  
	  dia_base_line_update_connection_points (DIA_BASE_LINE (obj),
						  DIA_BASE_LINE (obj)->handle_moving);
	  
	  dia_object_calc_bounding_box_update (obj);
	  
	  DIA_OBJECT_SET_REQUEST (obj, DIA_REQUEST_EXTENTS
				  | DIA_REQUEST_REDRAW
				  | DIA_REQUEST_UPDATE_CONNECTIONS
				  | DIA_REQUEST_STOP_EMIT);
	    
	  result = TRUE;
	}
      break;
    case DIA_BUTTON_RELEASE:
      //g_message ("DiaBaseLine::event(DIA_BUTTON_RELEASE)");
      switch (event->button.button)
	{
	case 1:
	  if (obj->flags & DIA_BASE_LINE_MOVE)
	    {
	      DIA_OBJECT_UNSET_FLAGS (obj, DIA_BASE_LINE_MOVE);
 
	      /* only do a connect if the handle is not already connected: */
	      if ((DIA_OBJECT_GET_HANDLE (obj, 0)->connected_to
		   ||dia_base_line_connect_handle (DIA_BASE_LINE (obj), layer,
						   DIA_OBJECT_GET_HANDLE (obj,
									  0)))
		  || (DIA_OBJECT_GET_HANDLE (obj, obj->handles->len - 1)->connected_to
		      || dia_base_line_connect_handle (DIA_BASE_LINE (obj), layer,
						       DIA_OBJECT_GET_HANDLE (obj, obj->handles->len - 1))))
		{
		  dia_object_calc_bounding_box_update (obj);
	  
		  DIA_OBJECT_SET_REQUEST (obj, DIA_REQUEST_EXTENTS
					  | DIA_REQUEST_REDRAW
					  | DIA_REQUEST_UPDATE_CONNECTIONS);
		}
	      DIA_OBJECT_SET_REQUEST (obj, DIA_REQUEST_UNGRAB);
	      
	      result = TRUE;
	    }
	  else if (obj->flags & DIA_BASE_LINE_MOVE_HANDLE)
	    {
	      DIA_OBJECT_UNSET_FLAGS (obj, DIA_BASE_LINE_MOVE_HANDLE);

	      //g_message ("DiaBaseLine::event(DIA_BUTTON_RELEASE)::MOVE_HANDLE");
	      if (DIA_OBJECT_IS_SET (obj, DIA_BASE_LINE_NEW)
		  && (DIA_OBJECT_GET_HANDLE (obj, obj->handles->len - 1)->is_movable))
		{
		  DIA_OBJECT_GET_HANDLE (obj, obj->handles->len - 1)->pos.x += 2.0;
		  DIA_OBJECT_GET_HANDLE (obj, obj->handles->len - 1)->pos.y += 2.0;
		  DIA_OBJECT_UNSET_FLAGS (obj,  DIA_BASE_LINE_NEW);
		  dia_object_calc_bounding_box (obj);
		  dia_object_add_update (obj, &obj->bounding_box);
		  DIA_OBJECT_SET_REQUEST (obj, DIA_REQUEST_REDRAW
					  | DIA_REQUEST_EXTENTS
					  | DIA_REQUEST_STOP_EMIT);
		}
	      if (!DIA_BASE_LINE (obj)->handle_moving->connected_to)
		dia_base_line_connect_handle (DIA_BASE_LINE (obj), layer,
					      DIA_BASE_LINE (obj)->handle_moving);
	      
	      DIA_BASE_LINE (obj)->handle_moving = NULL;
	      DIA_OBJECT_SET_REQUEST (obj, DIA_REQUEST_UNGRAB);
	      result = TRUE;
	    }
	    break;
	default:
	  break;
	} /* inner switch */
      break;
    case DIA_HANDLE_MOTION: /* explicit motion of a handle */
      /* define new position for the handle */
      event->handle_motion.handle->pos = event->handle_motion.pos;

      /* update the objects connection points */
      dia_base_line_update_connection_points (DIA_BASE_LINE (obj),
					      event->handle_motion.handle);
	  
      dia_object_calc_bounding_box_update (obj);
      
      DIA_OBJECT_SET_REQUEST (obj, DIA_REQUEST_EXTENTS
			      | DIA_REQUEST_REDRAW
			      | DIA_REQUEST_UPDATE_CONNECTIONS);
      result = TRUE;
      break;
    default:
      result = parent_ops->event (obj, event, layer);
      break;
    } /* switch */
  return result;
}

static gfloat
cp_distance (DiaObject *obj, Point *pos, Point *con_pos,
	     DiaConnectionPoint **cp)
{
  return parent_ops->cp_distance (obj, pos, con_pos, cp);
}

static DiaConnectionPoint*
cp_connect (DiaObject *obj, DiaHandle *h, Point *pos)
{
  return parent_ops->cp_connect (obj, h, pos);
}

static void
cp_disconnect (DiaObject *obj, DiaConnectionPoint *cp, DiaHandle *h)
{
  parent_ops->cp_disconnect (obj, cp, h);
}












