/* DiaCanvas -- A technical drawing canvas.
 * Copyright (C) 1999, Arjan Molenaar
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <gdk/gdk.h>
#include "diabasemodifytool.h"

static void free_base_modify_tool (DiaTool *tool);

static gint base_modify_tool_event_handler (DiaBaseModifyTool *tool,
					    DiaEvent *event, 
					    DiaDisplay *ddisp);

void
dia_base_modify_tool_init (DiaBaseModifyTool *tool)
{
  tool->tool.event = (DiaToolEventFunc) &base_modify_tool_event_handler;
  tool->tool.destroy = (DiaToolDestroyFunc) &free_base_modify_tool;
}

DiaTool*
dia_base_modify_tool_new ()
{
  DiaBaseModifyTool *tool;

  tool = g_new (DiaBaseModifyTool, 1);

  dia_base_modify_tool_init (tool);
  
  return (DiaTool*) tool;
}

static void
free_base_modify_tool (DiaTool *tool)
{
  g_free (tool);
}

static gboolean
handle_selected (DiaDisplay *ddisp, DiaEvent *event, DiaObject *not_me)
{
  GList *l = ddisp->selected;
  gboolean result = FALSE;
  
  while (l)
    {
      if (l->data != (gpointer) not_me)
	{
	  DIA_OBJECT (l->data)->request = DIA_REQUEST_NOTHING;
	  result |= DIA_OBJECT (l->data)->ops->event (DIA_OBJECT (l->data),
						      event,
						      ddisp->active_layer);
	}
      l = l->next;
    }
  l = ddisp->selected;
  while (l)
    {
      if (l->data != (gpointer) not_me)
	dia_base_modify_tool_handle_requests_2 (ddisp, DIA_OBJECT (l->data), event,
					   ddisp->active_layer);
      /* if object is empty and not in use, remove it from the canvas... */
      if ((ddisp->grab != DIA_OBJECT (l->data))
	  && dia_object_is_empty (DIA_OBJECT (l->data)))
	{
	  dia_diagram_remove_object (ddisp->diagram, ddisp->active_layer,
				     DIA_OBJECT (l->data));
	}
      l = l->next;
    }
  return result;
}

void
dia_base_modify_tool_handle_requests_1 (DiaDisplay *ddisp, DiaObject *object,
					DiaEvent *event, DiaLayer *layer)
{
  
  if (object->request & DIA_REQUEST_SELECT)
    {
      if (!dia_display_is_selected (ddisp, object))
	{
	  dia_display_select (ddisp, object,
			      event->any.modifier & DIA_SHIFT_MASK);
	  DIA_OBJECT_SET_FLAGS (object, DIA_OBJECT_STATE_SELECTED);
	}
    }
  if (object->request & DIA_REQUEST_UNSELECT)
    {
      dia_display_unselect (ddisp, object);
      DIA_OBJECT_UNSET_FLAGS (object, DIA_OBJECT_STATE_SELECTED);
    }
  if (object->request & DIA_REQUEST_FOCUS)
    {
      dia_display_set_focus (ddisp, object);
      DIA_OBJECT_SET_FLAGS (object, DIA_OBJECT_STATE_FOCUSED);
    }
  if (object->request & DIA_REQUEST_UNFOCUS)
    {
      if (dia_display_is_focused (ddisp, object))
	{
	  dia_display_set_focus (ddisp, NULL);
	  /* only remove the active flag: if it's selected it will
	     continue to be */
	  DIA_OBJECT_UNSET_FLAGS (object, DIA_OBJECT_STATE_FOCUSED);
	}
    }
  if (object->request & DIA_REQUEST_GRAB)
    {
      dia_display_grab (ddisp, object);
      DIA_OBJECT_SET_FLAGS (object, DIA_OBJECT_STATE_GRABBED);
    }
  if (object->request & DIA_REQUEST_UNGRAB)
    {
      dia_display_ungrab (ddisp, object);
      DIA_OBJECT_UNSET_FLAGS (object, DIA_OBJECT_STATE_GRABBED);
    }
  if (object->request & DIA_REQUEST_CURSOR)
    {
      dia_display_set_cursor (ddisp, object->cursor);
    }
  if (object->request & DIA_REQUEST_CURSOR_RESET)
    {
      dia_display_default_cursor (ddisp);
    }
}

/* Handle "non-focus" events.
 * This function is recursive, that's why it is imnportant to "unset"
 * flags when they are handled. */
void
dia_base_modify_tool_handle_requests_2 (DiaDisplay *ddisp, DiaObject *object,
					DiaEvent *event, DiaLayer *layer)
{
  DiaEvent new_event;
  DiaConnectionPoint *cp;
  
  if (object->request & DIA_REQUEST_REDRAW)
    {
      dia_diagram_add_update (ddisp->diagram, &object->update_box);
      /* clear update box: */
      object->update_box.top = object->update_box.bottom = 0.0;
      object->update_box.left = object->update_box.right = 0.0;
      DIA_OBJECT_UNSET_REQUEST (object, DIA_REQUEST_REDRAW);
    }
  if (object->request & DIA_REQUEST_DESTROY)
    {
      /* set the state to destroyed */
      //DIA_OBJECT_SET_FLAGS (object, DIA_OBJECT_STATE_DESTROYED);
  
      dia_diagram_remove_object (ddisp->diagram, ddisp->active_layer,
				 object);
      //g_message ("DiaModifyTool: Object destroyed.");
      /* Do not handle any more requests! */
      return;
    }
  if (object->request & DIA_REQUEST_EXTENTS)
    {
      dia_diagram_update_extents_fast (ddisp->diagram, ddisp->active_layer);
      DIA_OBJECT_UNSET_REQUEST (object, DIA_REQUEST_EXTENTS);
    }
  if (object->request & DIA_REQUEST_NORMAL)
    {
      if (dia_display_is_focused (ddisp, object))
	dia_display_set_focus (ddisp, NULL);
      if (dia_display_is_selected (ddisp, object))
	dia_display_unselect (ddisp, object);
      DIA_OBJECT_UNSET_REQUEST (object, DIA_REQUEST_NORMAL);
    }
  if (object->request & DIA_REQUEST_UPDATE_CONNECTIONS)
    {
      gint i;
      GList *handles;
      DiaObject *obj;
      DiaRequest orig_requests;

      DIA_OBJECT_UNSET_REQUEST (object, DIA_REQUEST_UPDATE_CONNECTIONS);
      
      for (i = 0; i < object->connections->len; i++)
	{
	  cp = (DiaConnectionPoint*)g_ptr_array_index (object->connections, i);
	  new_event.type = DIA_HANDLE_MOTION;
	  new_event.handle_motion.time = event->any.time;
	  new_event.handle_motion.modifier = event->any.modifier;
	  new_event.handle_motion.pos = cp->pos;
	  //point_add (&new_event.handle_motion.pos, &cp->object->position);
	  
	  handles = cp->connected;
	  while (handles)
	    {
	      obj = ((DiaHandle*)handles->data)->object;
	      new_event.handle_motion.handle = (DiaHandle*)handles->data;
	      /* Ensure that there are no requests are pending. */
	      dia_base_modify_tool_handle_requests_2 (ddisp, obj,
						      &new_event, layer);
	      orig_requests = obj->request;
	      obj->request = DIA_REQUEST_NOTHING;
	      dia_object_event (obj, &new_event, layer);
	      dia_base_modify_tool_handle_requests_2 (ddisp, obj,
						 &new_event, layer);
	      obj->request = orig_requests;
	      handles = g_list_next (handles);
	    }
	}
    }
  if (object->request & DIA_REQUEST_UPDATE_HANDLES)
    {
      gint i;
      DiaConnectionPoint *cp;
      DiaObject *obj;
      DiaRequest orig_requests;

      DIA_OBJECT_UNSET_REQUEST (object, DIA_REQUEST_UPDATE_HANDLES);
      
      for (i = 0; i < object->handles->len; i++)
	{
	  cp = DIA_OBJECT_GET_HANDLE (object, i)->connected_to;
	  if (cp)
	    {
	      new_event.type = DIA_UPDATE_CP;
	      new_event.update_cp.time = event->any.time;
	      new_event.update_cp.modifier = event->any.modifier;
	      new_event.update_cp.cp = cp;
	      /* object to send the event to: */
	      obj = cp->object;
	      
	      /* ensure that there are no requests pending: */
	      dia_base_modify_tool_handle_requests_2 (ddisp, obj,
						      &new_event, layer);
	      orig_requests = obj->request;
	      obj->request = DIA_REQUEST_NOTHING;
	      obj->ops->event (obj, &new_event, layer);
	      dia_base_modify_tool_handle_requests_2 (ddisp, obj,
						      &new_event, layer);
	      obj->request = orig_requests;
	    }
	}
    }
}

/* modify_tool_event_handler
 * Description: forward the event to the objects on the canvas in a
 *     predefined order:
 *      1a) if an object has grab, give that object the event
 *      1b) else give the event to the object under the mouse pointer
 *      2)  handle event and handle special requests (select, grab, etc.)
 *      3a) if object is selected, send event to all selected objects
 *      3b) else just handle the requests
 */
static gint
base_modify_tool_event_handler (DiaBaseModifyTool *tool, DiaEvent *event, 
				DiaDisplay *ddisp)
{
  GList *l;
  
  gboolean result = FALSE;
  DiaObject *object = NULL;
      
  /* First update all object states: */
  l = ddisp->active_layer->objects;
  while (l)
    {
      DIA_OBJECT_UNSET_FLAGS (l->data, DIA_OBJECT_STATE_MASK);
      DIA_OBJECT_SET_FLAGS (l->data,  DIA_OBJECT_STATE_ACTIVE);
      l = l->next;
    }
  l = ddisp->selected;
  while (l)
    {
      DIA_OBJECT_SET_FLAGS (l->data,  DIA_OBJECT_STATE_SELECTED);
      l = l->next;
    }
  if (ddisp->focus)
    DIA_OBJECT_SET_FLAGS (ddisp->focus, DIA_OBJECT_STATE_FOCUSED);
  if (ddisp->grab)
    DIA_OBJECT_SET_FLAGS (ddisp->grab, DIA_OBJECT_STATE_GRABBED);
  
  /* Now find an object: */
  if (ddisp->grab)
    object = ddisp->grab;
  else
    switch (event->type)
      {
      case DIA_MOTION:
	object = dia_layer_find_closest_object (ddisp->active_layer, 
						&event->motion.pos,
						0.1);
	break;
      case DIA_BUTTON_PRESS:
      case DIA_2BUTTON_PRESS:
      case DIA_3BUTTON_PRESS:
      case DIA_BUTTON_RELEASE:
	if (event->button.handle)
	  object = ((DiaHandle*) event->button.handle)->object;
	else
	  object = dia_layer_find_closest_object (ddisp->active_layer, 
						  &event->button.pos,
						  0.1);
	break;
      default:
	return FALSE;
      }
  
  /* forward event to object and handle some events */
  if (object)
    {
      //dia_display_update_object_state (ddisp, object);
      object->request = DIA_REQUEST_NOTHING;
      result = dia_object_event (object, event, ddisp->active_layer);
      //g_print ("DiaBaseModifyTool: result0 = %d\n", result);
      dia_base_modify_tool_handle_requests_1 (ddisp, object,event,
					      ddisp->active_layer);
    
      /* if object is selected, send it to all selected objects. Except
       * when the handle is selected! */
      //if (event->button.handle)
      //g_message ("button handle: %x", event->button.handle);
      
      if (DIA_OBJECT_IS_SET (object, DIA_OBJECT_STATE_SELECTED)
	  && !(object->request & DIA_REQUEST_STOP_EMIT))
	{
	  result |= handle_selected (ddisp, event, object);
	  /* handle remaining requests for the grabbed object */
	  dia_base_modify_tool_handle_requests_2 (ddisp, object, event,
						  ddisp->active_layer);
	}
      else if (result)
	  {
	    dia_base_modify_tool_handle_requests_2 (ddisp, object, event,
						    ddisp->active_layer);
	  }
  
      /* if object is empty and not in use, remove it from the canvas... */
      if ((ddisp->grab != object) && (dia_object_is_empty (object)))
	{
	  dia_diagram_remove_object (ddisp->diagram, ddisp->active_layer,
				     object);
	  result = TRUE;
	}
    }
  
  return result;
}











