/* DiaCanvas -- A technical drawing canvas.
 * Copyright (C) 1999, Arjan Molenaar
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <gdk/gdkkeysyms.h>
#include "diabasetext.h"
#include "diadisplay.h"
#include <glib.h>

static void destroy (DiaObject *object);
static void draw (DiaObject* obj, DiaRenderer* renderer);
static void draw_handles (DiaObject* obj, DiaRenderer* renderer);
static void draw_cps (DiaObject* obj, DiaRenderer* renderer);
static gfloat distance (DiaObject* obj, Point* point);
static void move (DiaObject *obj, gfloat dx, gfloat dy);
static void move_handle (DiaObject *obj, DiaHandle *handle,
			 gfloat dx, gfloat dy);
static void copy (DiaBaseText* from, DiaBaseText *to);
static gint is_empty (DiaObject* obj);
static void calc_bounding_box (DiaObject *obj);
static gint event (DiaObject *obj, DiaEvent *event, DiaLayer *layer);
static gfloat cp_distance (DiaObject *obj, Point *pos, Point *con_pos,
			   DiaConnectionPoint **cp);
static DiaConnectionPoint* cp_connect (DiaObject *obj, DiaHandle *h,
				       Point *pos);
static void cp_disconnect (DiaObject *obj, DiaConnectionPoint *cp,
			   DiaHandle *h);

static DiaObjectOps text_ops =
{
  destroy,
  draw,
  draw_handles,
  draw_cps,
  distance,
  move,
  move_handle,
  (DiaCopyFunc) copy,
  event,
  is_empty,
  calc_bounding_box,
  cp_distance,
  cp_connect,
  cp_disconnect
};

//static DiaObjectOps *parent_ops = NULL;
#define parent_ops (dia_base_text_get_type ()->parent_type ()->ops)

/* default values: */
static gchar* default_text = NULL;
static gfloat default_height = 1.5;
DiaFont *dia_base_text_standard_font = NULL;

DiaObjectType*
dia_base_text_get_type ()
{
  static DiaObjectType text_type =
  {
    "DiaBaseText",
    sizeof (DiaBaseText),
    (DiaInitFunc) dia_base_text_init,
    &text_ops,
    dia_object_get_type
  };
  return &text_type;
}

void
dia_base_text_init (DiaBaseText *text)
{
  //DiaObject *object = DIA_OBJECT (text);
  
  /* call parent */
  //dia_object_init (object);
  /* save parent options */
  if (!dia_base_text_standard_font)
    dia_base_text_standard_font = dia_font_get_font ("Helvetica");
  if (!default_text)
    default_text = g_strdup ("");
    
  
  /* overwrite old function declarations */
  //object->object_type = &text_type;
  //object->ops = &text_ops;

  /* add our own declarations: */
  text->lines = NULL;
  text->max_lines = -1;
  text->max_chars = -1;
  text->rowspacing = 0.9;
  text->editable = TRUE;
  //text->update_cursor = FALSE;
  //text->update_cursor_cr = FALSE;
  text->cursor_col = 0; //((GString*) (g_list_last (text->lines)->data))->len;
  text->cursor_row = 0; //g_list_length (text->lines);
  text->font = dia_base_text_standard_font;
  text->height = default_height;
  text->color = dia_color_black;
  text->halign = DIA_ALIGN_CENTER;//DIA_ALIGN_LEFT;
  //text->valign = DIA_ALIGN_BOTTOM;

  /* Set the string after all other values are initialized. */
  dia_base_text_set_string (text, default_text);
  
  dia_base_text_calculate_ascent_descent (text);
  
  //dia_object_calc_bounding_box (object);
  //text->width = 0.0;
}

DiaObject*
dia_base_text_new (Point *pos)
{
  DiaBaseText *new_text;
  
  g_return_val_if_fail (pos != NULL, NULL);
  
  new_text = DIA_BASE_TEXT (dia_object_create (dia_base_text_get_type ()));
  //g_new (DiaBaseText, 1);
  
  //dia_base_text_init (new_text);
  
  //dia_object_calc_bounding_box (DIA_OBJECT (new_text));
  
  dia_object_move (DIA_OBJECT (new_text), pos->x, pos->y);
  
  return DIA_OBJECT (new_text);
}

void
dia_base_text_set_default_text (gchar *string)
{
  g_return_if_fail (string != NULL);
  
  if (default_text)
    g_free (default_text);

  default_text = g_strdup (string);  
}

void
dia_base_text_set_default_height (gfloat height)
{
  g_return_if_fail (height > 0.0);
  
  default_height = height;
}

void
dia_base_text_set_string (DiaBaseText *text, gchar *string)
{
  gchar **lines;
  gint i = 0;
  GList *l;
  
  g_return_if_fail (text != NULL);
  g_return_if_fail (string != NULL);
  
  /* free the old string: */
  l = text->lines;
  while (l)
    {
      dia_text_line_free ((DiaTextLine*)l->data);
      l = l->next;
    }
  g_list_free (text->lines);
  text->lines = NULL;
  
  /* allocate the new string stuff: */
  lines = g_strsplit (string, "\n", text->max_chars);
  
  if (*lines == NULL)
    {
      text->lines = g_list_append (text->lines, dia_text_line_new (text, ""));
    }
  else
    while (lines[i])
      {
	text->lines = g_list_append (text->lines,
				     dia_text_line_new (text, lines[i]));
	i++;
      }
  g_free (lines);
  
  dia_base_text_calculate_ascent_descent (text);
  DIA_OBJECT (text)->ops->calc_bounding_box (DIA_OBJECT (text));

  /* FIXME: set the cursor to a nutral position */
}

DiaTextLine*
dia_base_text_get_line (DiaBaseText *text, gint row)
{
  GList *l;
  
  g_return_val_if_fail (text != NULL, NULL);
  g_return_val_if_fail (row >= 0, NULL);
  g_return_val_if_fail (row < g_list_length (text->lines), NULL);
  
  l = g_list_nth (text->lines, row);
  
  if (!l)
    g_error ("We're having an empty entry in the lines list");
  
  return DIA_TEXT_LINE (l->data);
}

void
dia_base_text_insert_char (DiaBaseText *text, gint row, gint col, gchar chr)
{
  DiaTextLine *line;
  
  g_return_if_fail (text != NULL);
  
  line = dia_base_text_get_line (text, row);
  g_assert (line != NULL);

  line->text = g_string_insert_c (line->text, col, chr);
  
  line->width = dia_base_text_string_width (text, line->text->str,
					    line->text->len);
}

void
dia_base_text_delete_char (DiaBaseText *text, gint row, gint col)
{
  DiaTextLine *line;
  
  g_return_if_fail (text != NULL);

  line = dia_base_text_get_line (text, row);

  /* merge two lines if the last character is deleted (the "enter" char): */
  if ((col == line->text->len) && (row < g_list_length (text->lines)))
    {
      dia_base_text_merge_lines (text, row);
    }
  else
    {
      line->text = g_string_erase (line->text, col, 1);
      line->width = dia_base_text_string_width (text, line->text->str,
						line->text->len);
    }
}

void
dia_base_line_split_line (DiaBaseText *text, gint row, gint col)
{
  DiaTextLine *line;
  DiaTextLine *new_line;
  GList *l;
  
  g_return_if_fail (text != NULL);
  //g_return_if_fail (line_to_split != NULL);
  
  //l = g_list_nth (text->lines, row + 1);
  line = dia_base_text_get_line (text, row);
  
  new_line = dia_text_line_new (text, &line->text->str[col]);
  new_line->pos = line->pos;
  
  g_string_truncate (line->text, col);
  line->width = dia_base_text_string_width (text, line->text->str,
					    line->text->len);

  text->lines = g_list_insert (text->lines, new_line, row + 1);  
  
  l = g_list_nth (text->lines, row + 1);
  while (l)
    {
      line = DIA_TEXT_LINE (l->data);
      line->pos.y += text->rowspacing;
      l = g_list_next (l);
    }
}

void
dia_base_text_merge_lines (DiaBaseText *text, gint firstrow)
{
  GList *link;
  DiaTextLine *line;
  DiaTextLine *next_line;

  g_return_if_fail (text != NULL);
  //g_return_if_fail (firstrow > 0);
  //g_return_if_fail (firstrow <= g_list_length (text->lines));
  
  //line = DIA_TEXT_LINE (g_list_nth (text->lines, firstrow)->data);
  line = dia_base_text_get_line (text, firstrow);
  
  link = g_list_nth (text->lines, firstrow + 1);
  if (link == NULL)
    return;
  
  next_line = (DiaTextLine*) link->data;

  line->text = g_string_append (line->text, next_line->text->str);
  line->width += next_line->width;
  
  /* ... and remove the next line of text */
  dia_text_line_free (next_line);
  text->lines = g_list_remove_link (text->lines, link);
  g_list_free (link);
  /* Now we must update the positions of the remaining lines: */
  link = g_list_nth (text->lines, firstrow + 1);
  while (link)
    {
      line = DIA_TEXT_LINE (link->data);
      line->pos.y -= text->rowspacing;
      link = g_list_next (link);
    }  
}

void
dia_base_text_set_height (DiaBaseText *text, gfloat height)
{
  g_return_if_fail (text != NULL);
  g_return_if_fail (height > 0.0);
  
  /* also adjust row spacing: */
  dia_base_text_set_row_spacing (text,
				 text->rowspacing * (height / text->height));

  text->height = height;

  /* Adjust the row spacing too: */
  dia_base_text_set_row_spacing (text, text->rowspacing * height);

  dia_base_text_calculate_ascent_descent (text);

  dia_object_calc_bounding_box (DIA_OBJECT (text));
}  

void
dia_base_text_set_font (DiaBaseText *text, DiaFont *font)
{
  g_return_if_fail (text != NULL);
  g_return_if_fail (font != NULL);
  
  text->font = font;
  dia_base_text_calculate_ascent_descent (text);
  dia_object_calc_bounding_box (DIA_OBJECT (text));
}

void
dia_base_text_set_halign (DiaBaseText *text, DiaHAlignment halign)
{
  gfloat max_width = 0.0;
  gfloat x = 0.0;
  GList *l;
  DiaTextLine *line;
  
  g_return_if_fail (text != NULL);
  
  if (text->halign == halign)
    return;
  
  text->halign = halign;
  
  /* find out what's the maximal width of a line: */
  l = text->lines;
  while (l)
    {
      line = DIA_TEXT_LINE (l->data);
      if (line->width > max_width)
	{
	  max_width = line->width;
	  x = line->pos.x;
	}
      l = g_list_next (l);
    }
  
  l = text->lines;
  switch (text->halign)
    {
    case DIA_ALIGN_LEFT:
      /* set all X'es to the most left point of the box: */
      while (l)
	{
	  DIA_TEXT_LINE (l->data)->pos.x = x;
	  l = g_list_next (l);
	}
      break;
    case DIA_ALIGN_CENTER:
      x = x + (max_width / 2);
      while (l)
	{
	  line = DIA_TEXT_LINE (l->data);
	  line->pos.x = x;// - (line->width / 2);
	  l = g_list_next (l);
	}
      break;
    case DIA_ALIGN_RIGHT:
      x += max_width;
      while (l)
	{
	  line = DIA_TEXT_LINE (l->data);
	  line->pos.x = x; // - line->width;
	  l = g_list_next (l);
	}      
      break;
    default:
      g_error ("Unknown hor. alignment type.");
      break;
    }
  dia_object_calc_bounding_box (DIA_OBJECT (text));
}

void
dia_base_text_set_row_spacing (DiaBaseText *text, gfloat spacing)
{
  GList *l;
  gfloat ypos;
  
  g_return_if_fail (text != NULL);
  
  l = text->lines;
  if (!l)
    g_error ("text->lines should at least contain one line!");
  
  ypos = DIA_TEXT_LINE (l->data)->pos.y;
  l = g_list_next (l);
  
  while (l)
    {
      ypos += spacing;
      DIA_TEXT_LINE (l->data)->pos.y = ypos;
      l = g_list_next (l);
    }
}

void
dia_base_text_calculate_ascent_descent (DiaBaseText *text)
{
  g_return_if_fail (text != NULL);

  text->ascent = dia_font_ascent (text->font, text->height);
  text->descent = dia_font_descent (text->font, text->height);
}

/* find a row that corresponds to the y position.
 * returns -1 if failed. Line 1 is the first line.
 */
gint
dia_base_text_find_cursor_row (DiaBaseText *text, gfloat ypos)
{
  GList *l;
  DiaTextLine *line;
  gint row = 0;
  
  g_return_val_if_fail (text != NULL, -1);
  
  l = text->lines;

  while (l)
    {
      line = DIA_TEXT_LINE (l->data);
      if ((ypos > line->pos.y - text->ascent) && (ypos < line->pos.y))
	return row;

      row++;
      l = g_list_next (l);
    }
  return -1;
}

gfloat
dia_base_text_get_line_begin_xpos (DiaBaseText *text, DiaTextLine *line)
{
  g_return_val_if_fail (text != NULL, 0.0);
  g_return_val_if_fail (line != NULL, 0.0);
  
  switch (text->halign)
    {
    case DIA_ALIGN_LEFT:
      return line->pos.x;
    case DIA_ALIGN_CENTER:
      return line->pos.x - (line->width / 2);
    case DIA_ALIGN_RIGHT:
      return line->pos.x - line->width;
    default:
      g_error ("Unknown horizontal align type");
    }
  return 0.0;
}

/* find the position in a text line where the cursor has to be placed.
 * pos 0 is before the first character, pos 1 after the first character etc.
 * row == 1 is the first row.
 * NOTE: Call this function only from the draw method!
 *       The right font has to be initialized!
 * returns -1 if the xpos is to much to the right
 * returns -2 if the xpos is to much to the left
 */
gint
dia_base_text_find_cursor_col (DiaBaseText *text,
			       gint row, gfloat xpos)
{
  gint pos = -1;
  gfloat rel_pos = 0.0;
  gfloat diff = 0.0;
  gfloat width = 0.0;
  DiaTextLine *line;
  gint i;
  
  g_return_val_if_fail (text != NULL, -1);

  line = dia_base_text_get_line (text, row);
  g_assert (line != NULL);
  
  rel_pos = dia_base_text_get_line_begin_xpos (text, line);
  rel_pos = xpos - rel_pos;
  
  diff = G_MAXFLOAT;

  if (rel_pos > line->width)
    return -1;
  else if (rel_pos < 0.0)
    return -2;
  
  for (i = 0; i <= line->text->len; i++)
    {
      width = dia_base_text_string_width (text, line->text->str, i);
      if (ABS (rel_pos - width) < diff)
	{
	  diff = ABS (rel_pos - width);
	  pos++;
	}
      else
	break;
    }
  return pos;
}


gfloat
dia_base_text_find_cursor_xpos (DiaBaseText *text,
				gint row, gint pos)
{
  gfloat xpos;
  DiaTextLine *line;
  gfloat width;
  
  g_return_val_if_fail (text != NULL, 0.0);

  line = dia_base_text_get_line (text, row);
  
  g_return_val_if_fail (pos <= line->text->len, 0.0);
  
  xpos = (gfloat) dia_base_text_get_line_begin_xpos (text, line);

  if (pos == 0)
    return xpos;

  width = dia_base_text_string_width (text, line->text->str, pos);
  
  return width + xpos;
}

gfloat
dia_base_text_find_cursor_ypos (DiaBaseText *text, gint row)
{
  gfloat ypos;

  g_return_val_if_fail (text != NULL, 0.0);
  //g_return_val_if_fail (row > 0, 0.0);
  
  ypos = dia_base_text_get_line (text, row)->pos.y
    - text->ascent;

  return ypos;
}

/* returns FALSE if the POS is not in some text Otherwise it sets both
 * cursor_row, cursor_col as well as cursor. */
gboolean
dia_base_text_find_cursor_pos (DiaBaseText *text,
			       Point *pos)
{
  g_return_val_if_fail (text != NULL, FALSE);
  //  g_return_val_if_fail (renderer != NULL, FALSE);
  g_return_val_if_fail (pos != NULL, FALSE);
  
  text->cursor_row = dia_base_text_find_cursor_row (text, pos->y);
  if (text->cursor_row < 0)
    return FALSE;
  
  text->cursor_col = dia_base_text_find_cursor_col (text,
						    text->cursor_row, 
						    pos->x);
  if (text->cursor_col < 0)
    return FALSE;

  /* update the cursor position for drawing */
  text->cursor.y = dia_base_text_find_cursor_ypos (text, text->cursor_row);
  text->cursor.x = dia_base_text_find_cursor_xpos (text,
						   text->cursor_row,
						   text->cursor_col);
  
  //g_message ("Cursor pos: (%d, %d) ~ (%f, %f)", text->cursor_col, text->cursor_row, text->cursor.x, text->cursor.y);

  return TRUE;
}

gfloat
dia_base_text_string_width (DiaBaseText *text, const gchar *string,
			    gint length)
{
  DiaRenderer *renderer;
  DiaDisplay *ddisp;
  gfloat flen;
  
  /* We add a little environment depending code here: we need to calculate some
   * values for which we need to know something about the environment. */
  ddisp = dia_display_get_last_edited_display ();
  if (ddisp)
    {
      renderer = DIA_RENDERER (ddisp->renderer);
      g_assert (renderer != NULL);
  
      renderer->ops->set_font (renderer, text->font, text->height);
      //g_message ("new way!");
      flen = renderer->interactive_ops->get_text_width (renderer,
							string, length);
    }
  else /* do it the old fashion way (less accurate) */
    {
      gchar *str = g_strdup (string);
      //g_message ("old way...");
      str[length] = '\0';
      flen = dia_font_string_width (str, text->font, text->height);
      g_free (str);
    }
  return flen;
}


/* "events" */
static void
destroy (DiaObject *object)
{
  //DiaBaseText *text = DIA_BASE_TEXT (object);
  GList *l;
  
  l = DIA_BASE_TEXT (object)->lines;
  while (l)
    {
      dia_text_line_free (DIA_TEXT_LINE (l->data));
      l = g_list_next (l);
    }
  
  /* text->font is a pointer to a font entry that is static! */

  parent_ops->destroy (object);
}

static void
draw_cursor (DiaBaseText* text, DiaRenderer* renderer)
{
  Point p;
  
  p.x = text->cursor.x;
  p.y = text->cursor.y + text->height;

  renderer->ops->set_linewidth (renderer, 0.05);
  
  renderer->ops->draw_line (renderer, &text->cursor, &p, &dia_color_black);
  
  //  g_message ("Drawing cursor from (%f, %f) to (%f, %f)",
  //     text->cursor.x, text->cursor.y, p.x, p.y);
}

static void
draw_line (DiaBaseText *text, DiaRenderer* renderer, DiaTextLine *line)
{
  Point pos;
  
  pos.x = dia_base_text_get_line_begin_xpos (text, line);
  pos.y = line->pos.y;
  
  renderer->ops->draw_string (renderer, line->text->str, &pos, &text->color);
  //g_message ("Cursor drawn");
}

static void
draw (DiaObject* obj, DiaRenderer* renderer)
{
  DiaBaseText *text;
  GList *l;
  Point p1, p2;
  
  text = DIA_BASE_TEXT (obj);
    
  renderer->ops->set_font (renderer, text->font, text->height);
  
  l = text->lines;
  while (l)
    {
      draw_line (text, renderer, DIA_TEXT_LINE (l->data));
      l = g_list_next (l);
    }

  p1.x = obj->bounding_box.left;
  p1.y = obj->bounding_box.top;
  p2.x = obj->bounding_box.right;
  p2.y = obj->bounding_box.bottom;
  
  //renderer->ops->draw_rect (renderer, &p1, &p2, &dia_color_black);
       
  /* Find the correct cursor position when the mouse has been clicked. */  
  if (renderer->is_interactive
      && DIA_OBJECT_IS_SET (obj, DIA_OBJECT_STATE_FOCUSED)
      && text->editable)
    draw_cursor (text, renderer);
  //g_message ("state is %x", obj->flags);
}

static void
draw_handles (DiaObject* obj, DiaRenderer* renderer)
{
  parent_ops->draw_handles (obj, renderer);
}

static void
draw_cps (DiaObject* obj, DiaRenderer* renderer)
{
  parent_ops->draw_cps (obj, renderer);
}

void
calc_bounding_box (DiaObject *obj)
{
  GList *l;
  gfloat max_width = 0.0;
  gfloat xpos;
  DiaBaseText *text;
  DiaTextLine *line;
  
  //g_return_if_fail (DIA_BASE_TEXT (obj)->lines != NULL);
  //g_return_if_fail (DIA_BASE_TEXT (obj)->lines->data != NULL);

  text = DIA_BASE_TEXT (obj);
  
  /* find the maximal width and the minimal x position of all text lines: */
  l = text->lines;
  if (!l)
    g_error ("no lines -> DiaBaseText should at least contain "
	     "ONE empty string!");
  xpos = DIA_TEXT_LINE (text->lines->data)->pos.x;
  max_width = DIA_TEXT_LINE (text->lines->data)->width;
  l = g_list_next (l);
  while (l)
    {
      line = DIA_TEXT_LINE (l->data);
      if (line->width > max_width)
	max_width = line->width;
      if (line->pos.x < xpos)
	xpos = line->pos.x;
          
      l = g_list_next (l);
    }
 
  obj->bounding_box.top =
    DIA_TEXT_LINE (text->lines->data)->pos.y - text->ascent;
  obj->bounding_box.bottom =
    DIA_TEXT_LINE (g_list_last (text->lines)->data)->pos.y + text->descent;

  
  switch (text->halign)
    {
    case DIA_ALIGN_LEFT:
      obj->bounding_box.left = xpos;
      obj->bounding_box.right = xpos + max_width;
      break;
    case DIA_ALIGN_CENTER:
      obj->bounding_box.left = xpos - (max_width / 2);
      obj->bounding_box.right = xpos + (max_width / 2);
      break;
    case DIA_ALIGN_RIGHT:
      obj->bounding_box.left = xpos - max_width;
      obj->bounding_box.right = xpos;
      break;
    }
}

static gfloat
distance (DiaObject* obj, Point* point)
{
  return  distance_rectangle_point (&obj->bounding_box, point);
}

static void
move (DiaObject *obj, gfloat dx, gfloat dy)
{
  GList *l;
  DiaBaseText *text = DIA_BASE_TEXT (obj);
  
  parent_ops->move (obj, dx, dy);

  l = text->lines;
  while (l)
    {
      DIA_TEXT_LINE (l->data)->pos.x += dx;
      DIA_TEXT_LINE (l->data)->pos.y += dy;
      l = g_list_next (l);
    }
}

static void
move_handle (DiaObject *obj, DiaHandle *handle, gfloat dx, gfloat dy)
{
  parent_ops->move_handle (obj, handle, dx, dy);
}

static void
copy (DiaBaseText *from, DiaBaseText *to)
{
  GList *l;
  DiaTextLine *line;
  DiaTextLine *new_line;
  
  parent_ops->copy (DIA_OBJECT (from), DIA_OBJECT (to));

  l = from->lines;
  while (l)
    {
      line = DIA_TEXT_LINE (l->data);
      new_line = dia_text_line_new (to, line->text->str);
      new_line->pos = line->pos;
      new_line->width = line->width;
      new_line->update_width = line->update_width;
      new_line->selection_start = line->selection_start;
      new_line->selection_end = line->selection_end;
    }
  
  to->max_lines = from->max_lines;
  to->max_chars = from->max_chars;
  to->rowspacing = from->rowspacing;
  to->editable = from->editable;
  to->cursor = from->cursor;
  to->cursor_col = from->cursor_col;
  to->cursor_row = from->cursor_row;
  to->font = from->font;
  to->height = from->height;
  to->color = from->color;
  to->halign = from->halign;
  to->ascent = from->ascent;
  to->descent = from->descent;
}

static gint
is_empty (DiaObject* obj)
{
  return ((g_list_length (DIA_BASE_TEXT (obj)->lines) == 1)
	  && (DIA_TEXT_LINE (DIA_BASE_TEXT (obj)->lines->data)->text->len == 0)
	  );
}
void
dia_base_text_calc_cursor (DiaBaseText *text)
{
  text->cursor.y = dia_base_text_find_cursor_ypos (text, text->cursor_row);
  text->cursor.x = dia_base_text_find_cursor_xpos (text, text->cursor_row,
						   text->cursor_col);
}

static gint
event (DiaObject *obj, DiaEvent *event, DiaLayer *layer)
{
  gboolean result = FALSE;
  DiaBaseText *text;
  
  text = DIA_BASE_TEXT (obj);
  
  switch (event->type)
    {
    case DIA_PLACE:
      DIA_OBJECT_SET_REQUEST (obj, DIA_REQUEST_FOCUS
			      | DIA_REQUEST_SELECT
			      | DIA_REQUEST_REDRAW);
		    
      obj->update_box = obj->bounding_box;
      obj->snap_pos = event->button.snap;
      text->cursor.x = dia_base_text_find_cursor_xpos (text, text->cursor_row,
						       text->cursor_col);
      text->cursor.y = dia_base_text_find_cursor_ypos (text, text->cursor_row);

      //dia_base_text_find_cursor_pos (DIA_BASE_TEXT (obj), &event->button.pos);
      
      if (text->editable)
	{
	  DIA_OBJECT_SET_REQUEST (obj, DIA_REQUEST_CURSOR);
	  obj->cursor = DIA_BASE_TEXT_TEXT_CURSOR;
	}
      
      result = TRUE;
      break;
    case DIA_BUTTON_PRESS:
	{
	  /* is pointer over some text (grab or ungrab)*/
	  if (!event->button.handle &&
	      //point_in_rectangle (&obj->bounding_box, &event->button.pos))
	      dia_base_text_find_cursor_pos (DIA_BASE_TEXT (obj),
					     &event->button.pos))
	    {
	      switch (event->button.button)
		{
		case 1:
		  //g_message ("Clicked inside text");
		  DIA_OBJECT_SET_REQUEST (obj, DIA_REQUEST_FOCUS
					  | DIA_REQUEST_SELECT
					  | DIA_REQUEST_REDRAW);
		  if (DIA_BASE_TEXT (obj)->editable)
		    DIA_OBJECT_SET_REQUEST (obj, DIA_REQUEST_GRAB
					    | DIA_REQUEST_CURSOR
					    | DIA_REQUEST_STOP_EMIT);
		    
		  obj->update_box = obj->bounding_box;
		  obj->snap_pos = event->button.snap;
		  obj->cursor = DIA_BASE_TEXT_TEXT_CURSOR;
		default:
		  break;
		}
	    }
	  else
	    {
	      //g_message ("Clicked outside text");
	      DIA_OBJECT_SET_REQUEST (obj, DIA_REQUEST_UNGRAB
				      | DIA_REQUEST_UNFOCUS
				      | DIA_REQUEST_REDRAW
				      | DIA_REQUEST_REENTER
				      | DIA_REQUEST_STOP_EMIT);
	      obj->update_box = obj->bounding_box;
	    }
	  result = TRUE;
	}
      break;
    case DIA_MOTION:
      if (DIA_OBJECT_IS_SET (obj, DIA_OBJECT_STATE_GRABBED))
	{
	  Point p;
	  p = event->motion.pos;
	  if (point_in_rectangle (&obj->bounding_box, &p))
	    {
	      DIA_OBJECT_SET_REQUEST (obj, DIA_REQUEST_CURSOR
				      | DIA_REQUEST_STOP_EMIT);
	      obj->cursor = DIA_BASE_TEXT_TEXT_CURSOR;
	    }
	  else
	    {
	      DIA_OBJECT_SET_REQUEST (obj, DIA_REQUEST_CURSOR_RESET);	      
	    }
	}
      break;
    case DIA_KEY_PRESS:
      DIA_OBJECT_SET_REQUEST (obj, DIA_REQUEST_STOP_EMIT);
      switch (event->key.keyval)
	{
	case GDK_Left:
	case GDK_KP_Left:
	  if (text->cursor_col > 0)
	    {
	      text->cursor_col--;
	      //text->update_cursor = TRUE;
	      text->cursor.x =
		dia_base_text_find_cursor_xpos (text, text->cursor_row,
						text->cursor_col);
	    }
	  else if (text->cursor_row > 0)
	    {
	      text->cursor_row--;
	      text->cursor_col = dia_base_text_get_line (text, text->cursor_row)->text->len;
	      //text->update_cursor = TRUE;
	      dia_base_text_calc_cursor (text);
	    }
	  
	  obj->update_box = obj->bounding_box;
	  DIA_OBJECT_SET_REQUEST (obj, DIA_REQUEST_REDRAW);
	  break;
	case GDK_Right:
	case GDK_KP_Right:
	  if (text->cursor_col < dia_base_text_get_line (text, text->cursor_row)->text->len)
	    {
	      text->cursor_col++;
	      //text->update_cursor = TRUE;
	      text->cursor.x =
		dia_base_text_find_cursor_xpos (text, text->cursor_row,
						text->cursor_col);
	    }
	  else if (text->cursor_row < g_list_length (text->lines) - 1)
	    {
	      text->cursor_row++;
	      text->cursor_col = 0;
	      dia_base_text_calc_cursor (text);
	    }
	  obj->update_box = obj->bounding_box;
	  DIA_OBJECT_SET_REQUEST (obj, DIA_REQUEST_REDRAW);
	  break;
	case GDK_Home:
	case GDK_KP_Home:
	  text->cursor_col = 0;
	  text->cursor.x =
	    dia_base_text_find_cursor_xpos (text, text->cursor_row,
					    text->cursor_col);
	  obj->update_box = obj->bounding_box;
	  DIA_OBJECT_SET_REQUEST (obj, DIA_REQUEST_REDRAW);
	  break;
	case GDK_End:
	case GDK_KP_End:
	  text->cursor_col = dia_base_text_get_line (text,
						     text->cursor_row)->text->len;
	  text->cursor.x =
	    dia_base_text_find_cursor_xpos (text, text->cursor_row,
					    text->cursor_col);
	  obj->update_box = obj->bounding_box;
	  DIA_OBJECT_SET_REQUEST (obj, DIA_REQUEST_REDRAW);
	  break;
	case GDK_Up:
	case GDK_KP_Up:
	  if (text->cursor_row > 0)
	    {
	      text->cursor_row--;
	      text->cursor_col =
		dia_base_text_find_cursor_col (text, text->cursor_row,
					       text->cursor.x);
	      if (text->cursor_col == -1)
		text->cursor_col = dia_base_text_get_line (text,
							   text->cursor_row)->text->len;
	      else if (text->cursor_col == -2)
		text->cursor_col = 0;

	      dia_base_text_calc_cursor (text);
	      
	      obj->update_box = obj->bounding_box;
	      DIA_OBJECT_SET_REQUEST (obj, DIA_REQUEST_REDRAW);
	    }
	  break;
	case GDK_Down:
	case GDK_KP_Down:
	  if (text->cursor_row < g_list_length (text->lines) - 1)
	    {
	      text->cursor_row++;
	      text->cursor_col =
		dia_base_text_find_cursor_col (text, text->cursor_row,
					       text->cursor.x);
	      if (text->cursor_col == -1)
		text->cursor_col = dia_base_text_get_line (text,
							   text->cursor_row)->text->len;
	      else if (text->cursor_col == -2)
		text->cursor_col = 0;

	      dia_base_text_calc_cursor (text);
	      
	      obj->update_box = obj->bounding_box;
	      DIA_OBJECT_SET_REQUEST (obj, DIA_REQUEST_REDRAW);
	    }
	  break;
	case GDK_Return:
	case GDK_KP_Enter:
	  obj->update_box = obj->bounding_box;
	  dia_base_line_split_line (text, text->cursor_row, text->cursor_col);
	  dia_object_calc_bounding_box (obj);
	  text->cursor_col = 0;
	  text->cursor_row++;
	  dia_base_text_calc_cursor (text);
	  rectangle_union (&obj->update_box, &obj->bounding_box);
	  DIA_OBJECT_SET_REQUEST (obj, DIA_REQUEST_REDRAW);
	  break;
	case GDK_BackSpace:
	  if (text->cursor_col > 0)
	    {
	      dia_base_text_delete_char (text, text->cursor_row,
					 --text->cursor_col);
	      text->cursor.x =
		dia_base_text_find_cursor_xpos (text, text->cursor_row,
						text->cursor_col);
	      obj->update_box = obj->bounding_box;
	      
	      dia_object_calc_bounding_box (obj);
	      DIA_OBJECT_SET_REQUEST (obj, DIA_REQUEST_REDRAW
				      | DIA_REQUEST_EXTENTS);
	    }
	  else if (text->cursor_row > 0)
	    {
	      obj->update_box = obj->bounding_box;
	      text->cursor_row--;
	      text->cursor_col = dia_base_text_get_line (text, text->cursor_row)->text->len;
	      dia_base_text_merge_lines (text, text->cursor_row);

	      dia_base_text_calc_cursor (text);

	      dia_object_calc_bounding_box (obj);
	      rectangle_union (&obj->update_box, &obj->bounding_box);
	      DIA_OBJECT_SET_REQUEST (obj, DIA_REQUEST_REDRAW
				      | DIA_REQUEST_EXTENTS);
	    }
	  break;
	case GDK_Delete:
	  if (text->cursor_col
	      < dia_base_text_get_line (text, text->cursor_row)->text->len)
	    {
	      dia_base_text_delete_char (text, text->cursor_row,
					 text->cursor_col);

	      text->cursor.x =
		dia_base_text_find_cursor_xpos (text, text->cursor_row,
						text->cursor_col);
	      obj->update_box = obj->bounding_box;
	      dia_object_calc_bounding_box (obj);
	      DIA_OBJECT_SET_REQUEST (obj, DIA_REQUEST_REDRAW
				      | DIA_REQUEST_EXTENTS);
	    }
	  else if (text->cursor_row < g_list_length (text->lines) - 1)
	    {
	      obj->update_box = obj->bounding_box;

	      dia_base_text_merge_lines (text, text->cursor_row);

	      dia_base_text_calc_cursor (text);

	      dia_object_calc_bounding_box (obj);
	      rectangle_union (&obj->update_box, &obj->bounding_box);
	      DIA_OBJECT_SET_REQUEST (obj, DIA_REQUEST_REDRAW
				      | DIA_REQUEST_EXTENTS);
	    }
	  break;
	default:
	  /* only add "normal" characters */
	  if ((event->key.length > 0) && (event->key.string[0] > 31))
	    {
	      obj->update_box = obj->bounding_box;
	      dia_base_text_insert_char (text, text->cursor_row,
					 text->cursor_col,
					 event->key.string[0]);
	      dia_object_calc_bounding_box (obj);
	      rectangle_union (&obj->update_box, &obj->bounding_box);

	      text->cursor_col++;
	      //text->update_cursor = TRUE;
	      text->cursor.x =
		dia_base_text_find_cursor_xpos (text, text->cursor_row,
						text->cursor_col);
	      DIA_OBJECT_SET_REQUEST (obj, DIA_REQUEST_REDRAW
				      | DIA_REQUEST_EXTENTS);
	    }
	} /* switch (event->key.keyval) */
      result = TRUE;
      break;
    default:
      result = parent_ops->event (obj, event, layer);
    } /* switch (event->type) */
  return result;
}

static gfloat
cp_distance (DiaObject *obj, Point *pos, Point *con_pos,
	     DiaConnectionPoint **cp)
{
  return parent_ops->cp_distance (obj, pos, con_pos, cp);
}

static DiaConnectionPoint*
cp_connect (DiaObject *obj, DiaHandle *h, Point *pos)
{
  return parent_ops->cp_connect (obj, h, pos);
}

static void
cp_disconnect (DiaObject *obj, DiaConnectionPoint *cp, DiaHandle *h)
{
  parent_ops->cp_disconnect (obj, cp, h);
}

/*
 * DiaTextLine
 */

DiaTextLine*
dia_text_line_new (DiaBaseText *text, gchar *str)
{
  DiaTextLine *line;
  
  g_return_val_if_fail (str != NULL, NULL);
  g_return_val_if_fail (text != NULL, NULL);
  
  line = g_new (DiaTextLine, 1);
  
  line->text = g_string_new (str);
  
  line->pos.x = 0.0;
  line->pos.y = 0.0;
  
  line->width = dia_base_text_string_width (text, line->text->str,
					    line->text->len);
  line->update_width = TRUE;
  return line;
}

void
dia_text_line_free (DiaTextLine *line)
{
  g_return_if_fail (line != NULL);
  
  g_string_free (line->text, TRUE);
  
  g_free (line);
}

/* recalc the width of the textline and return its length
 * (also set the line->width attribute) */
gfloat
dia_text_line_calc_width (DiaTextLine *line, DiaBaseText *text,
			  DiaRenderer *renderer)
{
  line->width = dia_base_text_string_width (text, line->text->str,
					    line->text->len);
  
  return line->width;
}









