/* DiaCanvas -- A technical drawing canvas.
 * Copyright (C) 1999, Arjan Molenaar
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include "diabox.h"

static void destroy (DiaObject *object);
static void draw (DiaObject* obj, DiaRenderer* renderer);
static void draw_handles (DiaObject* obj, DiaRenderer* renderer);
static void draw_cps (DiaObject* obj, DiaRenderer* renderer);
static gfloat distance (DiaObject* obj, Point* point);
static void move (DiaObject *obj, gfloat dx, gfloat dy);
static void move_handle (DiaObject *obj, DiaHandle *handle,
			 gfloat dx, gfloat dy);
static void copy (DiaBox *from, DiaBox *to);
static gint is_empty (DiaObject *obj);
static void calc_bounding_box (DiaObject *obj);
static gint event (DiaObject *obj, DiaEvent *event, DiaLayer *layer);
static gfloat cp_distance (DiaObject *obj, Point *pos, Point *con_pos,
			   DiaConnectionPoint **cp);
static DiaConnectionPoint* cp_connect (DiaObject *obj, DiaHandle *h,
				       Point *pos);
static void cp_disconnect (DiaObject *obj, DiaConnectionPoint *p,
			   DiaHandle *h);

static DiaObjectOps box_ops =
{
  destroy,
  draw,
  draw_handles,
  draw_cps,
  distance,
  move,
  move_handle,
  (DiaCopyFunc) copy,
  event,
  is_empty,
  calc_bounding_box,
  cp_distance,
  cp_connect,
  cp_disconnect
};

//static DiaObjectOps *parent_ops = NULL;
#define parent_ops (dia_box_get_type ()->parent_type ()->ops)

DiaObjectType*
dia_box_get_type ()
{
  static DiaObjectType box_type =
  {
    "DiaBox",
    sizeof (DiaBox),
    (DiaInitFunc) dia_box_init,
    &box_ops,
    dia_dyn_element_get_type /* parent */
  };
  return &box_type;
}

void
dia_box_init (DiaBox *box)
{
  DiaObject *obj;
  
  obj = DIA_OBJECT (box);
  
  /* call parent */
  //dia_dyn_element_init (DIA_DYN_ELEMENT (obj));
  /* save parent options */
  //if (!parent_ops)
  //parent_ops = obj->ops;
  
  /* overwrite old function declarations */
  //obj->object_type = &box_type;
  //obj->ops = &box_ops;
  
  box->line_color = dia_color_black;
  box->fill_color = dia_color_white;
  box->line_style = DIA_LINE_STYLE_SOLID;
  box->line_caps = DIA_LINE_CAPS_ROUND;
  box->line_join = DIA_LINE_JOIN_ROUND;
  box->line_width = 0.1;

  //calc_bounding_box (obj);
}

DiaObject*
dia_box_new (Point *pos)
{
  return dia_box_new_full (pos, 1.0, 1.0);
}

DiaObject*
dia_box_new_full (Point *pos, gfloat width, gfloat height)
{
  DiaObject *new_box;
  
  g_return_val_if_fail (pos != NULL, NULL);
  
  new_box = dia_object_create (dia_box_get_type ()); //g_new (DiaBox, 1);
  
  //dia_box_init (new_box);
  
  dia_object_move (new_box, pos->x, pos->y);
  
  dia_base_element_resize (DIA_BASE_ELEMENT (new_box), width, height);
  dia_object_calc_bounding_box (new_box);
  
  return new_box;
}

/* "events" */
static void
destroy (DiaObject *object)
{
  parent_ops->destroy (object);
}

static void
draw (DiaObject* obj, DiaRenderer* renderer)
{
  DiaBox *box = DIA_BOX (obj);
  Point p1, p2;
  
  renderer->ops->set_linestyle (renderer, box->line_style);
  renderer->ops->set_linewidth (renderer, box->line_width);
  renderer->ops->set_linecaps (renderer, box->line_caps);
  renderer->ops->set_linejoin (renderer, box->line_join);
  
  p1.x = DIA_BASE_ELEMENT (box)->size.left;
  p1.y = DIA_BASE_ELEMENT (box)->size.top;
  p2.x = DIA_BASE_ELEMENT (box)->size.right;
  p2.y = DIA_BASE_ELEMENT (box)->size.bottom;
  
  renderer->ops->fill_rect (renderer, &p1, &p2, &box->fill_color);
  renderer->ops->draw_rect (renderer, &p1, &p2, &box->line_color);
  
  parent_ops->draw (obj, renderer);
}

static void
draw_handles (DiaObject* obj, DiaRenderer* renderer)
{
  parent_ops->draw_handles (obj, renderer);
}

static void
draw_cps (DiaObject* obj, DiaRenderer* renderer)
{
  parent_ops->draw_cps (obj, renderer);
}

static gfloat
distance (DiaObject* obj, Point* point)
{
  return parent_ops->distance (obj, point);
}

static void
move (DiaObject *obj, gfloat dx, gfloat dy)
{
  parent_ops->move (obj, dx, dy);
}

static void
move_handle (DiaObject *obj, DiaHandle *handle, gfloat dx, gfloat dy)
{
  parent_ops->move_handle (obj, handle, dx, dy);
}

static void
copy (DiaBox *from, DiaBox *to)
{
  parent_ops->copy (DIA_OBJECT (from), DIA_OBJECT (to));

  to->line_color = from->line_color;
  to->fill_color = from->fill_color;
  to->line_style = from->line_style;
  to->line_width = from->line_width;
}

static gint
is_empty (DiaObject* obj)
{
  return FALSE;
}

static void
calc_bounding_box (DiaObject *obj)
{
  parent_ops->calc_bounding_box (obj);
  
  obj->bounding_box.left -= DIA_BOX (obj)->line_width / 2.0;
  obj->bounding_box.top -= DIA_BOX (obj)->line_width / 2.0 ;
  obj->bounding_box.right += DIA_BOX (obj)->line_width / 2.0;
  obj->bounding_box.bottom += DIA_BOX (obj)->line_width / 2.0;
}


static gint
event (DiaObject *obj, DiaEvent *event, DiaLayer *layer)
{
  return parent_ops->event (obj, event, layer);
}

static gfloat
cp_distance (DiaObject *obj, Point *pos, Point *con_pos, 
	     DiaConnectionPoint **cp)
{
  return parent_ops->cp_distance (obj, pos, con_pos, cp);
}

static DiaConnectionPoint*
cp_connect (DiaObject *obj, DiaHandle *h, Point *pos)
{
  return parent_ops->cp_connect (obj, h, pos);
}

static void
cp_disconnect (DiaObject *obj, DiaConnectionPoint *cp, DiaHandle *h)
{
  parent_ops->cp_disconnect (obj, cp, h);
}





