/* DiaCanvas -- A technical drawing canvas.
 * Copyright (C) 1999, Arjan Molenaar
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include "diaboxelement.h"

static void destroy (DiaObject *object);
static void draw (DiaObject* obj, DiaRenderer* renderer);
static void draw_handles (DiaObject* obj, DiaRenderer* renderer);
static void draw_cps (DiaObject* obj, DiaRenderer* renderer);
static gfloat distance (DiaObject* obj, Point* point);
static void move (DiaObject *obj, gfloat dx, gfloat dy);
static void move_handle (DiaObject *obj, DiaHandle *handle,
			 gfloat dx, gfloat dy);
static void copy (DiaBoxElement *from, DiaBoxElement *to);
static gint is_empty (DiaObject *obj);
static void calc_bounding_box (DiaObject *obj);
static gint event (DiaObject *obj, DiaEvent *event, DiaLayer *layer);
static gfloat cp_distance (DiaObject *obj, Point *pos, Point *con_pos,
			   DiaConnectionPoint **cp);
static DiaConnectionPoint* cp_connect (DiaObject *obj, DiaHandle *h,
				       Point *pos);
static void cp_disconnect (DiaObject *obj, DiaConnectionPoint *p,
			   DiaHandle *h);

static DiaObjectOps box_element_ops =
{
  destroy,
  draw,
  draw_handles,
  draw_cps,
  distance,
  move,
  move_handle,
  (DiaCopyFunc) copy,
  event,
  is_empty,
  calc_bounding_box,
  cp_distance,
  cp_connect,
  cp_disconnect
};

#define parent_ops (dia_box_element_get_type ()->parent_type ()->ops)

DiaObjectType*
dia_box_element_get_type ()
{
  static DiaObjectType box_element_type =
  {
    "DiaBoxElement",
    sizeof (DiaBoxElement),
    (DiaInitFunc) dia_box_element_init,
    &box_element_ops,
    dia_element_get_type
  };
  return &box_element_type;
}

void
dia_box_element_init (DiaBoxElement *box)
{  
  box->line_color = dia_color_black;
  box->fill_color = dia_color_white;
  box->line_style = DIA_LINE_STYLE_SOLID;
  box->line_caps = DIA_LINE_CAPS_ROUND;
  box->line_join = DIA_LINE_JOIN_ROUND;
  box->line_width = 0.1;
}

DiaObject*
dia_box_element_new (gfloat x, gfloat y, gfloat width, gfloat height)
{
  DiaObject *new_box;
  
  new_box = dia_object_create (dia_box_element_get_type ());
  
  dia_object_move (new_box, x, y);
  
  dia_base_element_resize (DIA_BASE_ELEMENT (new_box), width, height);
  dia_object_calc_bounding_box (new_box);
  
  return new_box;
}

/* elem: [in]
 * h: [in] handle that is inside the borders of the element
 * pos: [out] the position where the handle should be connected
 * return: TRUE if an intersectionpoint is found, FALSE otherwise.
 */
gboolean
dia_box_element_find_intersection (DiaBoxElement *elem,
				   DiaHandle *h, Point *pos)
{
  if (dia_object_check_type (h->object, dia_graph_get_type ()))
    {
      DiaHandle *next_h;
      Point *p;
      
      next_h = dia_graph_get_next_handle (DIA_GRAPH (h->object), h);
      
      if (line_rectangle_intersection (&h->pos, &next_h->pos,
				       &DIA_BASE_ELEMENT (elem)->size, &p))
	{
	  /* FIXME: return point that is closest to NEXT_H. */
	  *pos = *p;
	  g_free (p);
	  return TRUE;
	}
    }  
  return FALSE;
}

/* "events" */
static void
destroy (DiaObject *object)
{
  parent_ops->destroy (object);
}

static void
draw (DiaObject* obj, DiaRenderer* renderer)
{
  DiaBoxElement *box = DIA_BOX_ELEMENT (obj);
  Point p1, p2;
  
  renderer->ops->set_linestyle (renderer, box->line_style);
  renderer->ops->set_linewidth (renderer, box->line_width);
  renderer->ops->set_linecaps (renderer, box->line_caps);
  renderer->ops->set_linejoin (renderer, box->line_join);
  
  p1.x = DIA_BASE_ELEMENT (box)->size.left;
  p1.y = DIA_BASE_ELEMENT (box)->size.top;
  p2.x = DIA_BASE_ELEMENT (box)->size.right;
  p2.y = DIA_BASE_ELEMENT (box)->size.bottom;
  
  renderer->ops->fill_rect (renderer, &p1, &p2, &box->fill_color);
  renderer->ops->draw_rect (renderer, &p1, &p2, &box->line_color);
  
  //parent_ops->draw (obj, renderer);
}

static void
draw_handles (DiaObject* obj, DiaRenderer* renderer)
{
  parent_ops->draw_handles (obj, renderer);
}

static void
draw_cps (DiaObject* obj, DiaRenderer* renderer)
{
  parent_ops->draw_cps (obj, renderer);
}

static gfloat
distance (DiaObject* obj, Point* point)
{
  return parent_ops->distance (obj, point);
}

static void
move (DiaObject *obj, gfloat dx, gfloat dy)
{
  parent_ops->move (obj, dx, dy);
}

static void
move_handle (DiaObject *obj, DiaHandle *handle, gfloat dx, gfloat dy)
{
  parent_ops->move_handle (obj, handle, dx, dy);
}

static void
copy (DiaBoxElement *from, DiaBoxElement *to)
{
  parent_ops->copy (DIA_OBJECT (from), DIA_OBJECT (to));

  to->line_color = from->line_color;
  to->fill_color = from->fill_color;
  to->line_caps  = from->line_caps;
  to->line_join  = from->line_join;
  to->line_style = from->line_style;
  to->line_width = from->line_width;
}

static gint
is_empty (DiaObject* obj)
{
  return FALSE;
}

static void
calc_bounding_box (DiaObject *obj)
{
  parent_ops->calc_bounding_box (obj);
  
  obj->bounding_box.left -= DIA_BOX_ELEMENT (obj)->line_width / 2.0;
  obj->bounding_box.top -= DIA_BOX_ELEMENT (obj)->line_width / 2.0;
  obj->bounding_box.right += DIA_BOX_ELEMENT (obj)->line_width / 2.0;
  obj->bounding_box.bottom += DIA_BOX_ELEMENT (obj)->line_width / 2.0;
}

static gint
event (DiaObject *obj, DiaEvent *event, DiaLayer *layer)
{
  gint result = FALSE;
  
  switch (event->type)
    {
    case DIA_UPDATE_CP:
      if (g_list_length (event->update_cp.cp->connected) == 1)
	{
	  DiaConnectionPoint *cp;
	  DiaHandle *h;
	  Point p;
	  
	  cp = event->update_cp.cp;
	  h = (DiaHandle*) event->update_cp.cp->connected->data;
	  
	  if ((cp->pos.x != h->pos.x) || (cp->pos.y != h->pos.y))
	    {
	      if (dia_box_element_find_intersection (DIA_BOX_ELEMENT (obj),
						     h, &p))
		{
		  dia_object_move_handle_abs (h->object, h, p.x, p.y);
		  cp->pos = p;
		}
	      else
		{
		  dia_object_move_handle_abs (h->object, h, obj->bounding_box.left, obj->bounding_box.top);
		  cp->pos = h->pos;
		}
	    }
	  obj->update_box = obj->bounding_box;
	  DIA_OBJECT_SET_REQUEST (obj, DIA_REQUEST_REDRAW);
	  result = TRUE;
	}
      else
	g_warning ("Not exactly one handle connected (%d)",
		   g_list_length (event->update_cp.cp->connected));
      break;
    default:
      return parent_ops->event (obj, event, layer);
    }
  return result;
}

static gfloat
cp_distance (DiaObject *obj, Point *pos, Point *con_pos,
	     DiaConnectionPoint **cp)
{
  return parent_ops->cp_distance (obj, pos, con_pos, cp);
}

static DiaConnectionPoint*
cp_connect (DiaObject *obj, DiaHandle *h, Point *pos)
{
  DiaConnectionPoint *retval = NULL;
  Point p;
  
  if (dia_box_element_find_intersection (DIA_BOX_ELEMENT (obj), h, &p))
    {
      dia_object_move_handle_abs (h->object, h, p.x, p.y);
      //if (h->connected_to)
      //h->connected_to->pos = p;
      //else
      retval = parent_ops->cp_connect (obj, h, &p);
    }
  
  return retval;
}

static void
cp_disconnect (DiaObject *obj, DiaConnectionPoint *cp, DiaHandle *h)
{
  parent_ops->cp_disconnect (obj, cp, h);
}











