/* DiaCanvas -- A technical drawing canvas.
 * Copyright (C) 1999, Arjan Molenaar
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <gtk/gtkmain.h>
#include "diacreatetool.h"

static void free_create_tool (DiaTool *tool);

static gint create_tool_event_handler (DiaCreateTool *tool, DiaEvent *event, 
				       DiaDisplay *ddisp);

static DiaToolEventFunc parent_event;
static DiaToolDestroyFunc parent_destroy;

DiaTool*
dia_create_tool_new (DiaObjectType *object_type)
{
  DiaCreateTool *tool;
  
  tool = g_new (DiaCreateTool, 1);
  
  dia_base_modify_tool_init ((DiaBaseModifyTool*)tool);

  parent_event = ((DiaTool*) tool)->event;
  parent_destroy = ((DiaTool*) tool)->destroy;
  
  ((DiaTool*) tool)->event = (DiaToolEventFunc) &create_tool_event_handler;
  ((DiaTool*) tool)->destroy = (DiaToolDestroyFunc) &free_create_tool;

  tool->object_type = object_type;
  
  tool->added = 0;
  
  return (DiaTool*) tool;
}

static void
free_create_tool (DiaTool *tool)
{  
  parent_destroy (tool);
}

/* Catch the ungrab event if an object has been created and determine if the
 * creation was successfull by checking the DIA_REQUEST_DESTROY request. 
 * There is a possibility that the tool has been destroyed (for example
 * when selecting another option from the menu), that's why we do not use
 * the tool as parameter. We use the newly created object as descriminator.
 */
void
cb_ungrab_object (DiaDisplay *display, DiaObject *object, gpointer data)
{
  if (object->request & DIA_REQUEST_DESTROY)
    {
      gtk_signal_emit_by_name (GTK_OBJECT (display->diagram),
			       "add_item_fail", display->active_layer, object);
    }
  else
    {
      gtk_signal_emit_by_name (GTK_OBJECT (display->diagram),
			       "add_item_done", display->active_layer, object);
    }
  
  /* We do not want to catch any more signals. */
  //gtk_signal_disconnect (GTK_OBJECT (display), tool->signal_handler_id);  
  gtk_signal_disconnect_by_func (GTK_OBJECT (display),
				 GTK_SIGNAL_FUNC (cb_ungrab_object),
				 data);
}

/* create_tool_event_handler
 */
static gint
create_tool_event_handler (DiaCreateTool *tool, DiaEvent *event, 
			   DiaDisplay *ddisp)
{  
  gboolean result = FALSE;
  DiaObject *object = NULL;
  
  if (ddisp->grab)
    object = ddisp->grab;
  else
    if (event->type == DIA_BUTTON_PRESS)
      {
	/* create a new object */
	object = dia_object_create (tool->object_type);
	
	if (object)
	  {	  
	    dia_object_move (object, event->button.snap.x,
			     event->button.snap.y);
	    dia_diagram_add_object (ddisp->diagram, ddisp->active_layer,
				    object);
	    
	    dia_display_grab (ddisp, object);
	    tool->signal_handler_id =
	      gtk_signal_connect (GTK_OBJECT (ddisp), "ungrab_object",
				  GTK_SIGNAL_FUNC (cb_ungrab_object), object);
	    
	    dia_display_unselect_all (ddisp);
	    tool->added++;
	    
	    /* change event type to notify object it has just been created: */
	    event->type = DIA_PLACE;
	  }
	else
	  return FALSE;
      }
  
  result = parent_event ((DiaTool*)tool, event, ddisp);

  //return result;
  
  if (object && (object->request & DIA_REQUEST_REENTER))
    {
      /* We need to handle requests 'cause otherwise requesting a re-enter
       * might result in adding another object. */
      while (gtk_events_pending ())
	gtk_main_iteration ();
      
      result |= dia_tool_get_active ()->event (dia_tool_get_active (),
					       event, ddisp);
    }
  
  return result;
}











