/* DiaCanvas -- A technical drawing canvas.
 * Copyright (C) 1999, Arjan Molenaar
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */
/* Original license:
 * Dia -- an diagram creation/manipulation program
 * Copyright (C) 1998 Alexander Larsson
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "diadiagram.h"

enum {
  MODIFIED,
  ADD_ITEM,
  ADD_ITEM_DONE,
  ADD_ITEM_FAIL,
  REMOVE_ITEM,
  NAME_CHANGED,
  LAST_SIGNAL
};

static void dia_diagram_class_init          (DiaDiagramClass *class);
static void dia_diagram_init                (DiaDiagram *diagram);
static void dia_diagram_event_destroy       (GtkObject *object);

static void cb_add_item                     (DiaDiagram *dia, DiaLayer *layer, 
					     DiaObject *object);
static void cb_remove_item                  (DiaDiagram *dia, DiaLayer *layer,
					     DiaObject *obj);

static GtkObjectClass *parent_class;
static guint diagram_signals[LAST_SIGNAL] = { 0 };

GList *open_diagrams = NULL;

GtkType
dia_diagram_get_type ()
{
  static GtkType diagram_type = 0;

  if (!diagram_type)
    {
      GtkTypeInfo diagram_info =
      {
        "DiaDiagram",
        sizeof (DiaDiagram),
        sizeof (DiaDiagramClass),
        (GtkClassInitFunc) dia_diagram_class_init,
        (GtkObjectInitFunc) dia_diagram_init,
        /* reserved_1 */ NULL,
        /* reserved_2 */ NULL,
        (GtkClassInitFunc) NULL,
      };

      diagram_type = gtk_type_unique (gtk_object_get_type (), &diagram_info);
    }
 return diagram_type;
}


static void
dia_diagram_class_init (DiaDiagramClass  *class)
{
  GtkObjectClass *object_class;

  object_class = (GtkObjectClass*) class;

  object_class->destroy = dia_diagram_event_destroy;

  parent_class = gtk_type_class (gtk_object_get_type ());

  diagram_signals[MODIFIED] =
    gtk_signal_new ("modified",
                    GTK_RUN_FIRST,
                    object_class->type,
                    GTK_SIGNAL_OFFSET (DiaDiagramClass, modified),
                    gtk_signal_default_marshaller,
                    GTK_TYPE_NONE, 0);
  diagram_signals[ADD_ITEM] =
    gtk_signal_new ("add_item",
                    GTK_RUN_LAST,
                    object_class->type,
                    GTK_SIGNAL_OFFSET (DiaDiagramClass, add_item),
                    gtk_marshal_NONE__POINTER_POINTER,
                    GTK_TYPE_NONE, 2, GTK_TYPE_POINTER, GTK_TYPE_POINTER);
  diagram_signals[ADD_ITEM_DONE] =
    gtk_signal_new ("add_item_done",
                    GTK_RUN_LAST,
                    object_class->type,
                    GTK_SIGNAL_OFFSET (DiaDiagramClass, add_item_done),
                    gtk_marshal_NONE__POINTER_POINTER,
                    GTK_TYPE_NONE, 2, GTK_TYPE_POINTER, GTK_TYPE_POINTER);
  diagram_signals[ADD_ITEM_FAIL] =
    gtk_signal_new ("add_item_fail",
                    GTK_RUN_LAST,
                    object_class->type,
                    GTK_SIGNAL_OFFSET (DiaDiagramClass, add_item_fail),
                    gtk_marshal_NONE__POINTER_POINTER,
                    GTK_TYPE_NONE, 2, GTK_TYPE_POINTER, GTK_TYPE_POINTER);
  diagram_signals[REMOVE_ITEM] =
    gtk_signal_new ("remove_item",
                    GTK_RUN_LAST,
                    object_class->type,
                    GTK_SIGNAL_OFFSET (DiaDiagramClass, remove_item),
                    gtk_marshal_NONE__POINTER_POINTER,
                    GTK_TYPE_NONE, 2, GTK_TYPE_POINTER, GTK_TYPE_POINTER);
  diagram_signals[NAME_CHANGED] =
    gtk_signal_new ("name_changed",
                    GTK_RUN_FIRST,
                    object_class->type,
                    GTK_SIGNAL_OFFSET (DiaDiagramClass, name_changed),
                    gtk_signal_default_marshaller,
                    GTK_TYPE_NONE, 0);

  gtk_object_class_add_signals (object_class, diagram_signals, LAST_SIGNAL);

  class->modified = NULL;
  class->add_item = cb_add_item;
  class->add_item_done = NULL;
  class->add_item_fail = NULL;
  class->remove_item = cb_remove_item;
  class->name_changed = NULL;
}

static void
dia_diagram_init (DiaDiagram *dia)
{
  dia->name = NULL;
  dia->unsaved = TRUE;
  dia->modified = FALSE;
  dia->auto_resize = TRUE;
  dia->extents.top = 0.0;
  dia->extents.bottom = 10.0;
  dia->extents.left = 0.0;
  dia->extents.right = 10.0;
  dia->bg_color.red = 1.0;
  dia->bg_color.green = 1.0;
  dia->bg_color.blue = 1.0;
  dia->layers = NULL;
  dia->displays = NULL;
}

static void
dia_diagram_event_destroy (GtkObject *object)
{
  GSList *slist;
  GList *list;
  
  DiaDiagram *dia = (DiaDiagram*) object;
  
  if (dia->name)
    g_free (dia->name);
  
  list = dia->layers;
  while (list)
    {
      dia_layer_destroy ((DiaLayer*)list->data);
      list = g_list_next (list);
    }
  g_list_free (dia->layers);
  
  slist = dia->displays;
  while (slist)
    {
      gtk_object_destroy (GTK_OBJECT (slist->data));
      slist = g_slist_next (slist);
    }
  g_slist_free (dia->displays);
  
  if (GTK_OBJECT_CLASS (parent_class)->destroy)
    (* GTK_OBJECT_CLASS (parent_class)->destroy) (object);
}

static void
cb_add_item (DiaDiagram *dia, DiaLayer *layer, DiaObject *object)
{
  GSList *l;
  
  l = dia->displays;
  while (l)
    {
      dia_display_add_update_object (DIA_DISPLAY (l->data), object);
      //dia_display_flush (DIA_DISPLAY (l->data)); 
      l = l->next;
    }
}

static gint
idle_object_destroy (DiaObject *obj)
{
  dia_object_destroy (obj);
  return FALSE;
}

static void
cb_remove_item (DiaDiagram *dia, DiaLayer *layer, DiaObject *object)
{
  GSList *l;
  
  //DIA_OBJECT_SET_FLAGS (object, DIA_OBJECT_STATE_DESTROYED);
  
  l = dia->displays;
  while (l)
    {
      dia_display_add_update_object (DIA_DISPLAY (l->data), object);
      
      if (dia_display_is_selected (DIA_DISPLAY (l->data), object))
	dia_display_unselect (DIA_DISPLAY (l->data), object);
      
      if (dia_display_is_focused (DIA_DISPLAY (l->data), object))
	dia_display_set_focus (DIA_DISPLAY (l->data), NULL);
      
      dia_display_ungrab (DIA_DISPLAY (l->data), object);
      l = l->next;
    }
  dia_object_unconnect_all (object);
  //dia_object_destroy (object);
  dia_layer_remove_object (layer, object);
  dia_diagram_update_extents_fast (dia, layer);
  
  /* remove the object from memory after all DiaCanvas functions have ended */
  gtk_idle_add_priority (G_PRIORITY_HIGH, (GtkFunction)idle_object_destroy,
			 object);
  
  /* now update all displays */
/*   l = dia->displays; */
/*   while (l) */
/*     { */
/*       dia_display_flush (DIA_DISPLAY (l->data));   */
/*       l = l->next; */
/*     } */
}

DiaDiagram*
dia_diagram_new (gchar *name)
{
  DiaDiagram *dia;

  dia = gtk_type_new (dia_diagram_get_type ());

  if (name)
    dia->name = g_strdup (name);
  
  /* Append to list of open diagrams */
  open_diagrams = g_list_prepend (open_diagrams, dia);

  return dia;
}

void
dia_diagram_add_object (DiaDiagram *dia, DiaLayer *layer, DiaObject *obj)
{
  g_return_if_fail (dia != NULL);
  g_return_if_fail (DIA_IS_DIAGRAM (dia));
  g_return_if_fail (layer != NULL);
  g_return_if_fail (obj != NULL);
  g_return_if_fail (g_list_find (dia->layers, layer));
  
  dia_layer_add_object (layer, obj);
  dia_diagram_update_extents_fast (dia, layer);
  
  gtk_signal_emit (GTK_OBJECT (dia), diagram_signals[ADD_ITEM], layer, obj);
}

void
dia_diagram_add_objects (DiaDiagram *dia, DiaLayer *layer, GList *obj_list)
{
  GList *l;
  
  g_return_if_fail (dia != NULL);
  g_return_if_fail (DIA_IS_DIAGRAM (dia));
  g_return_if_fail (layer != NULL);
  g_return_if_fail (g_list_find (dia->layers, layer));
  
  l = obj_list;
  while (l)
    {
      dia_layer_add_object (layer, DIA_OBJECT (l->data));
  
      gtk_signal_emit (GTK_OBJECT (dia), diagram_signals[ADD_ITEM],
		       DIA_OBJECT (l->data));
      l = l->next;
    }
  dia_diagram_update_extents_fast (dia, layer);
}

void
dia_diagram_add_objects_first (DiaDiagram *dia, DiaLayer *layer,
			       GList *obj_list)
{
  GList *l;
  
  g_return_if_fail (dia != NULL);
  g_return_if_fail (DIA_IS_DIAGRAM (dia));
  g_return_if_fail (layer != NULL);
  g_return_if_fail (g_list_find (dia->layers, layer));
  
  l = obj_list;
  while (l)
    {
      dia_layer_add_object_first (layer, DIA_OBJECT (l->data));
  
      gtk_signal_emit (GTK_OBJECT (dia), diagram_signals[ADD_ITEM],
		       DIA_OBJECT (l->data));
      l = l->next;
    }
  dia_diagram_update_extents_fast (dia, layer);
}

void
dia_diagram_remove_object (DiaDiagram *dia, DiaLayer *layer, DiaObject *obj)
{
  g_return_if_fail (dia != NULL);
  g_return_if_fail (DIA_IS_DIAGRAM (dia));
  g_return_if_fail (layer != NULL);
  g_return_if_fail (obj != NULL);
  g_return_if_fail (g_list_find (dia->layers, layer));
  g_return_if_fail (g_list_find (layer->objects, obj));

  gtk_signal_emit (GTK_OBJECT (dia), diagram_signals[REMOVE_ITEM], layer, obj);
}

void
dia_diagram_set_modified (DiaDiagram *dia, gboolean is_modified)
{
  g_return_if_fail (dia != NULL);
  g_return_if_fail (DIA_IS_DIAGRAM (dia));
  
  dia->modified = is_modified;
}
     
gboolean
dia_diagram_modified_exist ()
{
  GList *list;
  
  list = open_diagrams;
  while (list)
    {
      if (DIA_DIAGRAM (list->data)->modified)
	return TRUE;
      list = g_list_next (list);
    }
  return FALSE;
}

void
dia_diagram_render(DiaDiagram *diagram, DiaRenderer *renderer,
		   DiaObjectRenderer obj_renderer /* Can be NULL */,
		   Rectangle *update_box, gpointer data)
{
  GList *list;
  
  g_return_if_fail (diagram != NULL);
  g_return_if_fail (DIA_IS_DIAGRAM (diagram));
  g_return_if_fail (renderer != NULL);
  //g_return_if_fail (update_box != NULL);
  
  (renderer->ops->begin_render) (renderer);
  
  if (update_box == NULL)
    {
      update_box = &diagram->extents;
      //g_message ("Empty update box");
    }
  
  list = diagram->layers;
  while (list)
    {
      //if (rectangle_intersects (update_box, &DIA_LAYER (list->data)->extents))
      if (DIA_LAYER (list->data)->visible)
	dia_layer_render ((DiaLayer*)list->data, renderer, obj_renderer,
			  update_box, data);
      list = list->next;
    }

  (renderer->ops->end_render) (renderer);
}

void
dia_diagram_set_name (DiaDiagram *diagram, gchar *name)
{
  g_return_if_fail (diagram != NULL);
  g_return_if_fail (DIA_IS_DIAGRAM (diagram));

  if (diagram->name)
    g_free (diagram->name);
  
  diagram->name = g_strdup (name);

  gtk_signal_emit (GTK_OBJECT (diagram), diagram_signals[NAME_CHANGED]);
}

gchar*
dia_diagram_get_name (DiaDiagram *diagram)
{
  g_return_val_if_fail (diagram != NULL, NULL);
  g_return_val_if_fail (DIA_IS_DIAGRAM (diagram), NULL);

  return diagram->name;
}

void
dia_diagram_set_auto_resize (DiaDiagram *diagram, gboolean auto_resize)
{
  g_return_if_fail (diagram != NULL);
  g_return_if_fail (DIA_IS_DIAGRAM (diagram));

  diagram->auto_resize = auto_resize;
  
  if (auto_resize)
    dia_diagram_update_extents (diagram);
/*   else */
/*     { */
/*       GList *l; */
      
/*       l = diagram->layers; */
/*       while (l) */
/* 	{ */
/* 	  ((DiaLayer*)l->data)->extents = diagram->extents; */
/* 	  l = l->next; */
/* 	} */
/*       dia_diagram_set_extents (diagram, */
/* 			       diagram->extents.left, diagram->extents.top, */
/* 			       diagram->extents.right, diagram->extents.bottom); */
/*     } */
}

void
dia_diagram_set_extents (DiaDiagram *diagram, gfloat left, gfloat top,
			 gfloat right, gfloat bottom)
{
  GSList *sl;
  GList *l;
  
  g_return_if_fail (diagram != NULL);
  g_return_if_fail (DIA_IS_DIAGRAM (diagram));
  g_return_if_fail (!diagram->auto_resize);
  
  diagram->extents.top = MIN (top, bottom);
  diagram->extents.left = MIN (left, right);
  diagram->extents.bottom = MAX (top, bottom);
  diagram->extents.right = MAX (left, right);

  l = diagram->layers;
  while (l)
    {
      ((DiaLayer*)l->data)->extents = diagram->extents;
      l = l->next;
    }
  
  /* update the scrollbars of the displays */
  sl = diagram->displays;
  while (sl)
    {
      dia_display_set_origo (DIA_DISPLAY (sl->data), left, top);
      //dia_display_update_scrollbars (DIA_DISPLAY (sl->data));
      dia_display_zoom (DIA_DISPLAY (sl->data), &DIA_DISPLAY (sl->data)->origo,
			1.0);
      sl = sl->next;
    }
}

gint
dia_diagram_update_extents (DiaDiagram *diagram)
{
  GList *l;
  GSList *sl;
  gint result = FALSE;
  
  g_return_val_if_fail (diagram != NULL, FALSE);
  g_return_val_if_fail (DIA_IS_DIAGRAM (diagram), FALSE);
 
  if (!diagram->auto_resize)
    return FALSE;
  
  l = diagram->layers;
  while (l)
    {
      result |= dia_layer_update_extents ((DiaLayer*) l->data);
      l = l->next;
    }
  
  /* Update scrollbars if extents were changed: */
  if (result)
    {
      DiaLayer *layer;
      
      l = diagram->layers;
      diagram->extents = ((DiaLayer*)l->data)->extents;
      l = l->next;
      while (l)
	{
	  layer = (DiaLayer*)l->data;
	  if (layer->extents.left != layer->extents.right ||
	      layer->extents.top != layer->extents.bottom)
	    rectangle_union (&diagram->extents, 
			     &layer->extents);
	  l = l->next;
	}
      
      sl = diagram->displays;
      while (sl)
	{
	  dia_display_update_scrollbars (DIA_DISPLAY (sl->data));
	  sl = sl->next;
	}
    }
  return result;
}

gint
dia_diagram_update_extents_fast (DiaDiagram *diagram, DiaLayer *layer)
{
  gint result;
  
  g_return_val_if_fail (diagram != NULL, FALSE);
  g_return_val_if_fail (DIA_IS_DIAGRAM (diagram), FALSE);
  g_return_val_if_fail (layer != NULL, FALSE);
  
  if (!diagram->auto_resize)
    return FALSE;
  
  result = dia_layer_update_extents (layer);
  
  /* Update scrollbars if extents were changed: */
  if (result)
    {
      GList *l;
      GSList *sl;
      DiaLayer *layer;
      
      l = diagram->layers;
      diagram->extents = ((DiaLayer*)l->data)->extents;
      l = l->next;
      while (l)
	{
	  layer = (DiaLayer*)l->data;
	  if (layer->extents.left != layer->extents.right ||
	      layer->extents.top != layer->extents.bottom)
	    rectangle_union (&diagram->extents, 
			     &layer->extents);
	  l = l->next;
	}
      
      sl = diagram->displays;
      while (sl)
	{
	  dia_display_update_scrollbars (DIA_DISPLAY (sl->data));
	  sl = sl->next;
	}
    }
  return result;
}

/*
 * DiaDisplay operations
 */

void
dia_diagram_add_display (DiaDiagram *diagram, DiaDisplay *display)
{
  g_return_if_fail (diagram != NULL);
  g_return_if_fail (DIA_IS_DIAGRAM (diagram));
  g_return_if_fail (display != NULL);
  g_return_if_fail (DIA_IS_DISPLAY (display));
  
  diagram->displays = g_slist_append (diagram->displays, display);
  
  /* Just give that thing an active layer... */
  if (diagram->layers)
    display->active_layer = diagram->layers->data;

}

void
dia_diagram_remove_display (DiaDiagram *diagram, DiaDisplay *display)
{
  g_return_if_fail (diagram != NULL);
  g_return_if_fail (DIA_IS_DIAGRAM (diagram));
  g_return_if_fail (display != NULL);
  g_return_if_fail (DIA_IS_DISPLAY (display));

  diagram->displays = g_slist_remove (diagram->displays, display);
  
  gtk_object_destroy (GTK_OBJECT (display));
}

/* update operations (wrappers for DIA_DISPLAY functions)
 */
void
dia_diagram_add_update (DiaDiagram *diagram, Rectangle *update)
{
  GSList *disps;

  g_return_if_fail (diagram != NULL);
  g_return_if_fail (DIA_IS_DIAGRAM (diagram));
  g_return_if_fail (update != NULL);

  disps = diagram->displays;
  while (disps)
    {
      dia_display_add_update (DIA_DISPLAY (disps->data), update);
      disps = g_slist_next (disps);
    }
}

void
dia_diagram_add_update_all (DiaDiagram *diagram)
{
  GSList *disps;

  g_return_if_fail (diagram != NULL);
  g_return_if_fail (DIA_IS_DIAGRAM (diagram));

  disps = diagram->displays;
  while (disps)
    {
      dia_display_add_update_all (DIA_DISPLAY (disps->data));
      disps = g_slist_next (disps);
    }
}

void
dia_diagram_add_update_object (DiaDiagram *diagram, DiaObject *object)
{
  GSList *disps;

  g_return_if_fail (diagram != NULL);
  g_return_if_fail (DIA_IS_DIAGRAM (diagram));
  g_return_if_fail (object != NULL);
  
  disps = diagram->displays;
  while (disps)
    {
      dia_display_add_update_object (DIA_DISPLAY (disps->data), object);
      disps = g_slist_next (disps);
    }
}	       

void
dia_diagram_add_update_list (DiaDiagram *diagram, GList *list_to_update)
{
  GSList *disps;

  g_return_if_fail (diagram != NULL);
  g_return_if_fail (DIA_IS_DIAGRAM (diagram));
  g_return_if_fail (list_to_update != NULL);
  
  disps = diagram->displays;
  while (disps)
    {
      dia_display_add_update_list (DIA_DISPLAY (disps->data), list_to_update);
      disps = g_slist_next (disps);
    }
}	       

/* This functions makes all displays redraw theirself */
void
dia_diagram_flush (DiaDiagram *diagram)
{
  GSList *disps;
  
  g_return_if_fail (diagram != NULL);
  g_return_if_fail (DIA_IS_DIAGRAM (diagram));

  disps = diagram->displays;
  while (disps)
    {
      dia_display_flush (DIA_DISPLAY (disps->data));
      disps = g_slist_next (disps);
    }
}

/* DiaLayer operations
 */
void
dia_diagram_add_layer (DiaDiagram *diagram, DiaLayer *layer)
{
  g_return_if_fail (diagram != NULL);
  g_return_if_fail (DIA_IS_DIAGRAM (diagram));
  g_return_if_fail (layer != NULL);
    
  diagram->layers = g_list_append (diagram->layers, layer);
  dia_diagram_update_extents (diagram);
}

void
dia_diagram_add_layer_at (DiaDiagram *diagram, DiaLayer *layer, gint pos)
{
  g_return_if_fail (diagram != NULL);
  g_return_if_fail (DIA_IS_DIAGRAM (diagram));
  g_return_if_fail (layer != NULL);

  diagram->layers = g_list_insert (diagram->layers, layer, pos);
  if (diagram->auto_resize)
    dia_diagram_update_extents (diagram);
  else
    layer->extents = diagram->extents;
}

void
dia_diagram_delete_layer (DiaDiagram *diagram, DiaLayer *layer)
{
  g_return_if_fail (diagram != NULL);
  g_return_if_fail (DIA_IS_DIAGRAM (diagram));
  g_return_if_fail (layer != NULL);
  g_return_if_fail (g_list_find (diagram->layers, layer));
  
  diagram->layers = g_list_remove (diagram->layers, layer);
  dia_layer_destroy (layer);
  dia_diagram_update_extents (diagram);
}

void
dia_diagram_delete_layer_at (DiaDiagram *diagram, gint pos)
{
  GList *list;
  
  g_return_if_fail (diagram != NULL);
  g_return_if_fail (DIA_IS_DIAGRAM (diagram));
  
  list = g_list_nth (diagram->layers, pos);
  if (!list)
    {
      g_warning ("dia_diagram_delete_layer_at: g_list_nth (diagram->layers, "
		 "pos), item not found.");
    }
  else
    {
      diagram->layers = g_list_remove (diagram->layers, list->data);
      dia_layer_destroy ((DiaLayer*)list->data);
      dia_diagram_update_extents (diagram);
    }
}





