/* DiaCanvas -- A technical drawing canvas.
 * Copyright (C) 1999, Arjan Molenaar
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */
/* Original license:
 * Dia -- an diagram creation/manipulation program
 * Copyright (C) 1998 Alexander Larsson
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <math.h>
#include <gtk/gtkmain.h>
#include <gtk/gtksignal.h>
#include <gdk/gdkkeysyms.h>
#include <glib.h>

#include "diadisplay.h"
#include "diacolor.h"
#include "diatool.h"
#include "diaobject.h"

/* For debugging it sometimes is convenient to disable the pointer grab: */
//#define NO_POINTER_GRAB

enum {
  SELECT_OBJECT,
  UNSELECT_OBJECT,
  FOCUS_OBJECT,
  UNFOCUS_OBJECT,
  GRAB_OBJECT,
  UNGRAB_OBJECT,
  ACTIVE_DISPLAY_CHANGE,
  LAST_SIGNAL
};

#define CANVAS_EVENT_MASK   \
         GDK_EXPOSURE_MASK | GDK_POINTER_MOTION_MASK | \
	 GDK_POINTER_MOTION_HINT_MASK | GDK_BUTTON_PRESS_MASK | \
	 GDK_BUTTON_RELEASE_MASK | GDK_STRUCTURE_MASK | \
	 GDK_ENTER_NOTIFY_MASK | GDK_KEY_PRESS_MASK |  \
	 GDK_KEY_RELEASE_MASK | GDK_FOCUS_CHANGE_MASK | GDK_ALL_EVENTS_MASK

/* set cursor values: */
static gint default_cursor = GDK_LEFT_PTR;
//static gint current_cursor = default_cursor;
static DiaDisplay *last_edited_display = NULL;

static void dia_display_class_init          (DiaDisplayClass *display_class);
static void dia_display_init                (DiaDisplay *ddisp);
static void dia_display_event_destroy       (GtkObject *object);
static void dia_display_active_display_change (DiaDisplay *ddisp);

static void dia_display_add_display_area (DiaDisplay *ddisp,
					  gint left, gint top,
					  gint right, gint bottom);
static void dia_display_add_update_pixels (DiaDisplay *ddisp, gint x, gint y,
					   gint pixel_width,
					   gint pixel_height);


static GtkTable *parent_class = NULL;
static guint display_signals[LAST_SIGNAL] = { 0 };

GtkType
dia_display_get_type (void)
{
  static GtkType display_type = 0;

  if (!display_type)
    {
      GtkTypeInfo display_info =
      {
        "DiaDisplay",
        sizeof (DiaDisplay),
        sizeof (DiaDisplayClass),
        (GtkClassInitFunc) dia_display_class_init,
        (GtkObjectInitFunc) dia_display_init,
        /* reserved_1 */ NULL,
        /* reserved_2 */ NULL,
        (GtkClassInitFunc) NULL,
      };

      display_type = gtk_type_unique (gtk_table_get_type (), &display_info);
    }
 return display_type;
}

static void
dia_display_class_init (DiaDisplayClass *display_class)
{
  GtkObjectClass *object_class;
  //GtkWidgetClass *widget_class;
  
  object_class = GTK_OBJECT_CLASS (display_class);
  //widget_class = (GtkWidgetClass*) display_class;

  display_signals[SELECT_OBJECT] =
    gtk_signal_new ("select_object",
                    GTK_RUN_FIRST, 
                    object_class->type, 
                    GTK_SIGNAL_OFFSET (DiaDisplayClass, select_object),
                    gtk_marshal_NONE__POINTER,
		    GTK_TYPE_NONE, 1, GTK_TYPE_POINTER);
  display_signals[UNSELECT_OBJECT] =
    gtk_signal_new ("unselect_object",
                    GTK_RUN_FIRST, 
                    object_class->type, 
                    GTK_SIGNAL_OFFSET (DiaDisplayClass, unselect_object),
                    gtk_marshal_NONE__POINTER,
		    GTK_TYPE_NONE, 1, GTK_TYPE_POINTER);
  display_signals[FOCUS_OBJECT] =
    gtk_signal_new ("focus_object",
                    GTK_RUN_FIRST, 
                    object_class->type, 
                    GTK_SIGNAL_OFFSET (DiaDisplayClass, focus_object),
                    gtk_marshal_NONE__POINTER,
		    GTK_TYPE_NONE, 1, GTK_TYPE_POINTER);
  display_signals[UNFOCUS_OBJECT] =
    gtk_signal_new ("unfocus_object",
                    GTK_RUN_FIRST, 
                    object_class->type, 
                    GTK_SIGNAL_OFFSET (DiaDisplayClass, unfocus_object),
                    gtk_marshal_NONE__POINTER,
		    GTK_TYPE_NONE, 1, GTK_TYPE_POINTER);
  display_signals[GRAB_OBJECT] =
    gtk_signal_new ("grab_object",
                    GTK_RUN_FIRST, 
                    object_class->type, 
                    GTK_SIGNAL_OFFSET (DiaDisplayClass, grab_object),
                    gtk_marshal_NONE__POINTER,
		    GTK_TYPE_NONE, 1, GTK_TYPE_POINTER);
  display_signals[UNGRAB_OBJECT] =
    gtk_signal_new ("ungrab_object",
                    GTK_RUN_FIRST, 
                    object_class->type, 
                    GTK_SIGNAL_OFFSET (DiaDisplayClass, ungrab_object),
                    gtk_marshal_NONE__POINTER,
		    GTK_TYPE_NONE, 1, GTK_TYPE_POINTER);
  display_signals[ACTIVE_DISPLAY_CHANGE] =
    gtk_signal_new ("active_display_change",
                    GTK_RUN_FIRST,
                    object_class->type, 
                    GTK_SIGNAL_OFFSET (DiaDisplayClass, active_display_change),
                    gtk_signal_default_marshaller,
		    GTK_TYPE_NONE, 0);
  gtk_object_class_add_signals (object_class, display_signals, LAST_SIGNAL);
  
  object_class->destroy = dia_display_event_destroy;

  //  widget_class->focus_in_event = dia_display_focus_in_event;
  //widget_class->grab_focus = dia_display_grab_focus;
  
  display_class->select_object = NULL;
  display_class->unselect_object = NULL;
  display_class->focus_object = NULL;
  display_class->unfocus_object = NULL;
  display_class->grab_object = NULL;
  display_class->ungrab_object = NULL;
  display_class->active_display_change = dia_display_active_display_change;
  
  parent_class = gtk_type_class (gtk_table_get_type ());
}

static void
dia_display_init (DiaDisplay *ddisp)
{
  ddisp->diagram = NULL;
  ddisp->active_layer = NULL;
  
  ddisp->redraw_handler = 0;
  
  ddisp->selected = NULL;
  ddisp->focus = NULL;
  ddisp->grab = NULL;
  ddisp->cursor = default_cursor;
  
  ddisp->origo.x = 0.0;
  ddisp->origo.y = 0.0;
  ddisp->zoom_factor = DIA_DISPLAY_NORMAL_ZOOM;

  dia_grid_init (&ddisp->grid);
  
  ddisp->autoscroll = TRUE;
  
  ddisp->renderer = dia_renderer_gdk_new (ddisp);
  
  ddisp->update_box.left = 0.0;
  ddisp->update_box.right = 0.0;
  ddisp->update_box.top = 0.0;
  ddisp->update_box.bottom = 0.0;
    
  ddisp->visible.top = 0;
  ddisp->visible.bottom = 0;
  ddisp->visible.left = 0;
  ddisp->visible.right = 0;
}

static void
dia_display_event_destroy (GtkObject *object)
{
  DiaDisplay *ddisp = DIA_DISPLAY (object);
  DiaDiagram *dia;

  if (last_edited_display == ddisp)
    last_edited_display = NULL;
  
  dia = ddisp->diagram;
  
  dia_diagram_remove_display (dia, ddisp);

  g_list_free (ddisp->selected);
  
  dia_renderer_gdk_destroy (ddisp->renderer);
  
  gtk_object_destroy (GTK_OBJECT (ddisp->canvas));
  gtk_object_destroy (GTK_OBJECT (ddisp->hsb));
  gtk_object_destroy (GTK_OBJECT (ddisp->vsb));
  gtk_object_destroy (GTK_OBJECT (ddisp->hrule));
  gtk_object_destroy (GTK_OBJECT (ddisp->vrule));
  gtk_object_destroy (GTK_OBJECT (ddisp->origin));
  gtk_object_destroy (GTK_OBJECT (ddisp->hsbdata));
  gtk_object_destroy (GTK_OBJECT (ddisp->vsbdata));
    
  if (GTK_OBJECT_CLASS (parent_class)->destroy)
    (* GTK_OBJECT_CLASS (parent_class)->destroy) (object);
}

static void
dia_display_active_display_change (DiaDisplay *ddisp)
{
  //g_message ("dia_display_active_display_change");
  
  last_edited_display = ddisp;
}

/* event on GtkDrawingArea */
gint
dia_display_canvas_events (GtkWidget *canvas,
			   GdkEvent  *gdk_event,
			   DiaDisplay *ddisp)
{
  GdkEventExpose *eevent;
  GdkEventKey *kevent;
  Rectangle *visible;
  Point middle;
  gint width, height;
  gint result = FALSE;
  DiaTool *active_tool;
  DiaEvent dia_event;
  gint tx, ty;
  GdkModifierType tmask;
  
  if (!canvas->window)
    return FALSE;

  gdk_window_get_pointer (canvas->window, &tx, &ty, &tmask);

  /* convert the motion-parameters to values relative to the canvas. */
  switch (gdk_event->type)
    {
    case GDK_MOTION_NOTIFY:
      {
	GdkEventMotion *mevent;

	mevent = (GdkEventMotion*) gdk_event;
	if (mevent->is_hint)
	  {
	    mevent->x = tx;
	    mevent->y = ty;
	    mevent->state = tmask;
	    mevent->is_hint = FALSE;
	  }
	
	ddisp->is_autoscrolled = dia_display_autoscroll (ddisp,
							 gdk_event->button.x,
							 gdk_event->button.y);
      }
      break;
#ifndef NO_POINTER_GRAB
    case GDK_BUTTON_PRESS:
      gdk_pointer_grab (canvas->window, FALSE,
       			GDK_POINTER_MOTION_HINT_MASK
       			| GDK_BUTTON_MOTION_MASK
			| GDK_BUTTON_PRESS_MASK
       			| GDK_BUTTON_RELEASE_MASK,
       			NULL, NULL, gdk_event->button.time);
      break;
    case GDK_BUTTON_RELEASE:
       gdk_pointer_ungrab (gdk_event->button.time); 
       break;
#endif /* NO_POINTER_GRAB */
    default:
      break;
    }

  /* send an display_change signal if somebody presses inside the canvas. */
  if ((last_edited_display != ddisp)
      && ((gdk_event->type  == GDK_BUTTON_PRESS)
	  || (gdk_event->type  == GDK_KEY_PRESS)))
    gtk_signal_emit (GTK_OBJECT (ddisp),
		     display_signals[ACTIVE_DISPLAY_CHANGE]);

  //g_print ("1st: %d\t", gdk_time_get () - tijd);
 
  dia_display_convert_gdk_event (ddisp, gdk_event, &dia_event);

  //g_print ("2nd: %d\t", gdk_time_get () - tijd);

  active_tool = dia_tool_get_active ();
  if (active_tool && active_tool->event && (dia_event.type != DIA_NONE))
    result = (*active_tool->event) (active_tool, &dia_event, ddisp);
  
  //g_print ("3rd: %d\t", gdk_time_get () - tijd);

  if (result == FALSE)
    {
      switch (gdk_event->type)
	{
	case GDK_EXPOSE:
	  //printf("GDK_EXPOSE\n");
	  /* If you wanna expose, first check if no redraw has to take place */
	  if (ddisp->redraw_handler)
	    dia_display_flush (ddisp); /* YES! explicit */
	  
	  eevent = (GdkEventExpose *) gdk_event;
	  dia_renderer_gdk_copy_to_window (ddisp->renderer,
					   ddisp->canvas->window,
					   eevent->area.x,
					   eevent->area.y,
					   eevent->area.width,
					   eevent->area.height);
	  break;
	case GDK_CONFIGURE:
	  //g_print ("GDK_CONFIGURE ");
	  width = ddisp->renderer->renderer.pixel_width;
	  height = ddisp->renderer->renderer.pixel_height;
	  if ((width != ddisp->canvas->allocation.width) ||
	      (height != ddisp->canvas->allocation.height))
	    {
	      dia_display_resize_canvas (ddisp,
					 ddisp->canvas->allocation.width,
					 ddisp->canvas->allocation.height);
	      /* Clean the whol region in one sweep, this will prevent
	       * us from getting 
	      ((DiaRenderer*) ddisp->renderer)->interactive_ops->fill_pixel_rect
		((DiaRenderer*) ddisp->renderer,
		 ddisp->canvas->allocation.x,
		 ddisp->canvas->allocation.y,
		 ddisp->canvas->allocation.width,
		 ddisp->canvas->allocation.height,
		 &ddisp->diagram->bg_color);*/
	      dia_display_update_scrollbars (ddisp);
	    }
	  //g_print ("done...\n");
	  break;
	case GDK_KEY_PRESS:
	  kevent = (GdkEventKey *) gdk_event;
	  //state = kevent->state;
	  result = TRUE;
	  //g_message ("A key is pressed");
	  switch(kevent->keyval)
	    {
	    case GDK_Up:
	      dia_display_scroll_up (ddisp);
	      //dia_display_flush (ddisp);
	      break;
	    case GDK_Down:
	      dia_display_scroll_down (ddisp);
	      //dia_display_flush (ddisp);
	      break;
	    case GDK_Left:
	      dia_display_scroll_left (ddisp);
	      //dia_display_flush (ddisp);
	      break;
	    case GDK_Right:
	      dia_display_scroll_right (ddisp);
	      //dia_display_flush (ddisp);
	      break;
	    case GDK_KP_Add:
	    case GDK_plus:
	    case GDK_equal: /* just like GIMP */
	      visible = &ddisp->visible;
	      middle.x = visible->left*0.5 + visible->right*0.5;
	      middle.y = visible->top*0.5 + visible->bottom*0.5;
	      dia_display_zoom(ddisp, &middle, M_SQRT2);
	      break;
	    case GDK_KP_Subtract:
	    case GDK_minus:
	      visible = &ddisp->visible;
	      middle.x = visible->left*0.5 + visible->right*0.5;
	      middle.y = visible->top*0.5 + visible->bottom*0.5;
	      dia_display_zoom(ddisp, &middle, M_SQRT1_2);
	      break;
	    default:
	      //g_message ("Unknown code: 0x%X", kevent->keyval);
	      result = FALSE;
	    }
	  break;
	default:
	  break;
	}
    }

  //g_print ("final: %d\n", gdk_time_get () - tijd);
  /* return FALSE on a motion event: the rulers have to update too! */
  return (result && (dia_event.type != DIA_MOTION));
}

gint
dia_display_hsb_update (GtkAdjustment *adjustment,
			DiaDisplay *ddisp)
{
  dia_display_set_origo (ddisp, adjustment->value, ddisp->origo.y);
  dia_display_add_update (ddisp, &ddisp->visible);
  //dia_display_flush (ddisp);
  return FALSE;
}

gint
dia_display_vsb_update (GtkAdjustment *adjustment,
			DiaDisplay *ddisp)
{
  dia_display_set_origo (ddisp, ddisp->origo.x, adjustment->value);
  dia_display_add_update (ddisp, &ddisp->visible);
  //dia_display_flush (ddisp);
  return FALSE;
}

/* Public functions
 */
GtkWidget*
dia_display_new (DiaDiagram *dia, gint width, gint height)
{
  DiaDisplay *ddisp;
  gint s_width, s_height;

  ddisp = gtk_type_new (dia_display_get_type ());

  ddisp->diagram = dia;

  s_width = gdk_screen_width ();
  s_height = gdk_screen_height ();
  if (width > s_width)
    width = s_width;
  if (height > s_height)
    width = s_width;

  /*  The adjustment data  */
  ddisp->hsbdata = GTK_ADJUSTMENT (gtk_adjustment_new (0, 0, width, 1, 1, width-1));
  ddisp->vsbdata = GTK_ADJUSTMENT (gtk_adjustment_new (0, 0, height, 1, 1, height-1));

  /* creating widget 
   */
  /* table (parent) */
  gtk_table_resize (GTK_TABLE (ddisp), 3, 3);
  gtk_table_set_homogeneous (GTK_TABLE (ddisp), FALSE);
  gtk_table_set_col_spacing (GTK_TABLE (ddisp), 0, 1);
  gtk_table_set_col_spacing (GTK_TABLE (ddisp), 1, 2);
  gtk_table_set_row_spacing (GTK_TABLE (ddisp), 0, 1);
  gtk_table_set_row_spacing (GTK_TABLE (ddisp), 1, 2);
  gtk_container_set_border_width (GTK_CONTAINER (ddisp), 2);
  
  /* frame that fills the upper left corner */
  ddisp->origin = gtk_frame_new (NULL);
  gtk_frame_set_shadow_type (GTK_FRAME (ddisp->origin), GTK_SHADOW_OUT);
  gtk_widget_show (ddisp->origin);
  
  /* Horizontal ruler */
  ddisp->hrule = gtk_hruler_new ();
  gtk_widget_show (ddisp->hrule);
  
  /* vertical ruler */
  ddisp->vrule = gtk_vruler_new ();
  gtk_widget_show (ddisp->vrule);
  
  /* horizontal scroll bar */
  ddisp->hsb = gtk_hscrollbar_new (ddisp->hsbdata);
  GTK_WIDGET_UNSET_FLAGS (ddisp->hsb, GTK_CAN_FOCUS);
  gtk_signal_connect (GTK_OBJECT (ddisp->hsbdata), "value_changed",
		      (GtkSignalFunc) dia_display_hsb_update,
		      ddisp);
  gtk_widget_show (ddisp->hsb);
  
  /* vertical scroll bar */
  ddisp->vsb = gtk_vscrollbar_new (ddisp->vsbdata);
  GTK_WIDGET_UNSET_FLAGS (ddisp->vsb, GTK_CAN_FOCUS);
  gtk_signal_connect (GTK_OBJECT (ddisp->vsbdata), "value_changed",
		      (GtkSignalFunc) dia_display_vsb_update,
		      ddisp);
  gtk_widget_show (ddisp->vsb);

  /* the _real_ canvas */
  ddisp->canvas = gtk_drawing_area_new ();
  gtk_widget_show (ddisp->canvas);
  gtk_drawing_area_size (GTK_DRAWING_AREA (ddisp->canvas), width, height);
  gtk_widget_set_events (ddisp->canvas, CANVAS_EVENT_MASK);
  GTK_WIDGET_SET_FLAGS (ddisp->canvas, GTK_CAN_FOCUS | GTK_HAS_FOCUS);

  gtk_signal_connect (GTK_OBJECT (ddisp->canvas), "event",
		      (GtkSignalFunc) dia_display_canvas_events,
		      ddisp);
  gtk_signal_connect_object (GTK_OBJECT (ddisp->canvas), "motion_notify_event",
                             (GtkSignalFunc) GTK_WIDGET_CLASS (GTK_OBJECT (ddisp->hrule)->klass)->motion_notify_event,
                             GTK_OBJECT (ddisp->hrule));

  gtk_signal_connect_object (GTK_OBJECT (ddisp->canvas), "motion_notify_event",
                             (GtkSignalFunc) GTK_WIDGET_CLASS (GTK_OBJECT (ddisp->vrule)->klass)->motion_notify_event,
                             GTK_OBJECT (ddisp->vrule));

  //ddisp->accel_group = gtk_accel_group_new ();
  
  //dia_display_set_cursor (ddisp, ddisp->cursor);
  
 /*  pack all the widgets  */
  gtk_table_attach (GTK_TABLE (ddisp), ddisp->origin, 0, 1, 0, 1,
                    GTK_FILL, GTK_FILL, 0, 0);
  gtk_table_attach (GTK_TABLE (ddisp), ddisp->hrule, 1, 2, 0, 1,
                    GTK_EXPAND | GTK_SHRINK | GTK_FILL, GTK_FILL, 0, 0);
  gtk_table_attach (GTK_TABLE (ddisp), ddisp->vrule, 0, 1, 1, 2,
                    GTK_FILL, GTK_EXPAND | GTK_SHRINK | GTK_FILL, 0, 0);
  gtk_table_attach (GTK_TABLE (ddisp), ddisp->hsb, 0, 2, 2, 3,
                    GTK_EXPAND | GTK_SHRINK | GTK_FILL, GTK_FILL, 0, 0);
  gtk_table_attach (GTK_TABLE (ddisp), ddisp->vsb, 2, 3, 0, 2,
                    GTK_FILL, GTK_EXPAND | GTK_SHRINK | GTK_FILL, 0, 0);
  gtk_table_attach (GTK_TABLE (ddisp), ddisp->canvas, 1, 2, 1, 2,
                    GTK_EXPAND | GTK_SHRINK | GTK_FILL,
                    GTK_EXPAND | GTK_SHRINK | GTK_FILL, 0, 0);


  /* Apply DiaDisplay to the diagram (now the connection is bi-directional!) */
  dia_diagram_add_display (dia, DIA_DISPLAY(ddisp));

  return GTK_WIDGET (ddisp);
}

GtkWidget*
dia_display_new_in_window (DiaDiagram *dia, gint width, gint height,
			   gchar *title)
{
  GtkWidget *display;
  GtkWidget *window;
  
  g_return_val_if_fail (dia != NULL, NULL);
  
  display = dia_display_new (dia, width, height);
  gtk_widget_show (display);

  window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
  gtk_window_set_policy (GTK_WINDOW (window), TRUE, TRUE, TRUE);
  gtk_widget_set_events (window,
			 GDK_POINTER_MOTION_MASK |
			 GDK_POINTER_MOTION_HINT_MASK |
			 GDK_FOCUS_CHANGE_MASK);
  
  if (title)
    gtk_window_set_title (GTK_WINDOW (window), title);

  gtk_container_add (GTK_CONTAINER (window), display);

  gtk_object_set_data (GTK_OBJECT (window), "display", display);
  gtk_object_set_data (GTK_OBJECT (window), "canvas",
		       DIA_DISPLAY (display)->canvas);
  
  /* never forget to set the focus, otherwise the canvas can't
   * recieve key events. */
  gtk_widget_grab_focus (DIA_DISPLAY (display)->canvas);  
  
  return window;
}

void
dia_display_select (DiaDisplay *ddisp, DiaObject *object, gboolean add)
{
  g_return_if_fail (ddisp != NULL);
  g_return_if_fail (DIA_IS_DISPLAY (ddisp));
  g_return_if_fail (object != NULL);

  if (dia_display_is_selected (ddisp, object))
    return;
  
  if (!add)
    dia_display_unselect_all (ddisp);
  
  ddisp->selected = g_list_append (ddisp->selected, object);
  
  gtk_signal_emit (GTK_OBJECT (ddisp), display_signals[SELECT_OBJECT], object);

  dia_display_add_update_object (ddisp, object);
}

gboolean
dia_display_is_selected (DiaDisplay *ddisp, DiaObject *object)
{
  g_return_val_if_fail (ddisp != NULL, FALSE);
  g_return_val_if_fail (DIA_IS_DISPLAY (ddisp), FALSE);
  g_return_val_if_fail (object != NULL, FALSE);
 
  return g_list_find (ddisp->selected, object) ? TRUE : FALSE;
}

void
dia_display_unselect (DiaDisplay *ddisp, DiaObject *object)
{
  g_return_if_fail (ddisp != NULL);
  g_return_if_fail (DIA_IS_DISPLAY (ddisp));
  g_return_if_fail (object != NULL);
  
  if (g_list_find (ddisp->selected, object))
    {
      ddisp->selected = g_list_remove (ddisp->selected, object);
  
      gtk_signal_emit (GTK_OBJECT (ddisp),
		       display_signals[UNSELECT_OBJECT], object);

      dia_display_add_update_object (ddisp, object);
    }
  else
    g_warning ("dia_display_unselect: you try to unselected a not selected"
	       " object!");
}

void
dia_display_unselect_all (DiaDisplay *ddisp)
{
  GList *list;
  
  g_return_if_fail (ddisp != NULL);
  g_return_if_fail (DIA_IS_DISPLAY (ddisp));

  list = ddisp->selected;
  while (list)
    {
      gtk_signal_emit (GTK_OBJECT (ddisp),
		       display_signals[UNSELECT_OBJECT],
		       DIA_OBJECT (list->data));

      dia_display_add_update_object (ddisp, DIA_OBJECT (list->data));
      list = g_list_next (list);
    }
  g_list_free (ddisp->selected);
  ddisp->selected = NULL;
}

void
dia_display_set_focus (DiaDisplay *ddisp, DiaObject *object)
{
  g_return_if_fail (ddisp != NULL);
  g_return_if_fail (DIA_IS_DISPLAY (ddisp));
  //g_return_if_fail (object != NULL);

  if (ddisp->focus)
    {
      if (object == ddisp->focus)
	return; /* fast way, no event! */
      
      gtk_signal_emit (GTK_OBJECT (ddisp),
		       display_signals[UNFOCUS_OBJECT], ddisp->focus);
      dia_display_add_update_object (ddisp, ddisp->focus);
      ddisp->focus = NULL;
    }
  
  if (object)
    {
      ddisp->focus = object;
  
      gtk_signal_emit (GTK_OBJECT (ddisp),
		       display_signals[FOCUS_OBJECT], object);

      //dia_display_add_update_object (ddisp, object);
    }
}

DiaObject*
dia_display_get_focus (DiaDisplay *ddisp)
{
  g_return_val_if_fail (ddisp != NULL, NULL);
  g_return_val_if_fail (DIA_IS_DISPLAY (ddisp), NULL);

  return (ddisp->focus);
}

gboolean
dia_display_is_focused (DiaDisplay *ddisp, DiaObject *object)
{
  g_return_val_if_fail (ddisp != NULL, FALSE);
  g_return_val_if_fail (DIA_IS_DISPLAY (ddisp), FALSE);
  g_return_val_if_fail (object != NULL, FALSE);

  return (ddisp->focus == object) ? TRUE : FALSE;
}

gboolean
dia_display_grab (DiaDisplay *ddisp, DiaObject *object)
{
  g_return_val_if_fail (ddisp != NULL, FALSE);
  g_return_val_if_fail (DIA_IS_DISPLAY (ddisp), FALSE);
  g_return_val_if_fail (object != NULL, FALSE);
  
  if (!ddisp->grab)
    {
      ddisp->grab = object;
      gtk_signal_emit (GTK_OBJECT (ddisp),
		       display_signals[GRAB_OBJECT], object);
   
      return TRUE;
    }
  return FALSE;
}

void
dia_display_ungrab (DiaDisplay *ddisp, DiaObject *object)
{
  g_return_if_fail (ddisp != NULL);
  g_return_if_fail (DIA_IS_DISPLAY (ddisp));
  
  if (ddisp->grab == object)
    {
      gtk_signal_emit (GTK_OBJECT (ddisp),
		       display_signals[UNGRAB_OBJECT], ddisp->grab);
      ddisp->grab = NULL;
    }
}

void
dia_display_transform_coords (DiaDisplay *ddisp,
			      gfloat x, gfloat y,
			      gint *xi, gint *yi)
{
  Rectangle *visible;
  gint width;
  gint height;

  g_return_if_fail (ddisp != NULL);
  g_return_if_fail (DIA_IS_DISPLAY (ddisp));
  g_return_if_fail (ddisp->renderer != NULL);

  visible = &ddisp->visible;
  width = ddisp->renderer->renderer.pixel_width;
  height = ddisp->renderer->renderer.pixel_height;
  
  *xi = ROUND ( (x - visible->left)  * (gfloat)width /
		(visible->right - visible->left) );
  *yi = ROUND ( (y - visible->top)  * (gfloat)height /
		(visible->bottom - visible->top) );
}

/* Takes real length and returns pixel length */
gint
dia_display_transform_length(DiaDisplay *ddisp, gfloat len)
{
  g_return_val_if_fail (ddisp != NULL, 0);
  g_return_val_if_fail (DIA_IS_DISPLAY (ddisp), 0);
  
  return (gint)(len * ddisp->zoom_factor);
}

/* Takes pixel length and returns real length */
gfloat
dia_display_untransform_length(DiaDisplay *ddisp, gint len)
{
  g_return_val_if_fail (ddisp != NULL, 0.0);
  g_return_val_if_fail (DIA_IS_DISPLAY (ddisp), 0.0);

  return (gfloat)len / ddisp->zoom_factor;
}

void
dia_display_untransform_coords (DiaDisplay *ddisp,
				gint xi, gint yi,
				gfloat *x, gfloat *y)
{
  Rectangle *visible;
  gint width;
  gint height;

  g_return_if_fail (ddisp != NULL);
  g_return_if_fail (DIA_IS_DISPLAY (ddisp));
  g_return_if_fail (ddisp->renderer != NULL);
  
  visible = &ddisp->visible;
  width = ddisp->renderer->renderer.pixel_width;
  height = ddisp->renderer->renderer.pixel_height;
  
  *x = visible->left + xi*(visible->right - visible->left) / (gfloat)width;
  *y = visible->top +  yi*(visible->bottom - visible->top) / (gfloat)height;
}

void
dia_display_add_update_list (DiaDisplay *ddisp, GList *list_to_update)
{
  GList *l;
  
  l = list_to_update;
  while (l)
    {
      dia_display_add_update (ddisp, &DIA_OBJECT (l->data)->bounding_box);
      l = g_list_next (l);
    }
}

/* This function is used to add update regions in pixels. */
static void
dia_display_add_update_pixels (DiaDisplay *ddisp, gint x, gint y,
			       gint pixel_width, gint pixel_height)
{
  Rectangle rect;
  //gfloat size_x, size_y;

  g_return_if_fail (ddisp != NULL);
  g_return_if_fail (DIA_IS_DISPLAY (ddisp));
  
  //size_x = dia_display_untransform_length(ddisp, pixel_width+1);
  //size_y = dia_display_untransform_length(ddisp, pixel_height+1);

  dia_display_untransform_coords (ddisp, x - (pixel_width + 1)/2,
				  y - (pixel_width + 1)/2,
				  &rect.left, &rect.top);
  dia_display_untransform_coords (ddisp, x + (pixel_width + 1)/2,
				  y + (pixel_width + 1)/2,
				  &rect.right, &rect.bottom);

  dia_display_add_update (ddisp, &rect);
}

void
dia_display_add_update_all (DiaDisplay *ddisp)
{
  g_return_if_fail (ddisp != NULL);
  g_return_if_fail (DIA_IS_DISPLAY (ddisp));

  dia_display_add_update(ddisp, &ddisp->visible);
}

void
dia_display_add_update_object (DiaDisplay *ddisp, DiaObject *object)
{
  g_return_if_fail (ddisp != NULL);
  g_return_if_fail (DIA_IS_DISPLAY (ddisp));
  g_return_if_fail (object != NULL);
  
  dia_display_add_update (ddisp, &object->bounding_box);
}

static gint
dia_display_idle_redraw (DiaDisplay *ddisp)
{
  //gtk_idle_remove (ddisp->redraw_handler);

  ddisp->redraw_handler = 0;
  
  dia_display_flush (ddisp);
  
  return FALSE; /* stop calling this function */
}

void
dia_display_add_update (DiaDisplay *ddisp, Rectangle *rect)
{
  g_return_if_fail (ddisp != NULL);
  g_return_if_fail (DIA_IS_DISPLAY (ddisp));
  g_return_if_fail (rect != NULL);

  /* no currect update box is selected */
  if ((ddisp->update_box.left == ddisp->update_box.right)
      && (ddisp->update_box.top == ddisp->update_box.bottom))
    {
      ddisp->update_box = *rect;
    }
  else
    rectangle_union (&ddisp->update_box, rect);

  if (ddisp->redraw_handler == 0)
    ddisp->redraw_handler = gtk_idle_add_priority (GTK_PRIORITY_REDRAW, (GtkFunction) dia_display_idle_redraw, ddisp);
}

void
dia_display_add_display_area (DiaDisplay *ddisp,
			      gint left, gint top,
			      gint right, gint bottom)
{
  Rectangle rect;
  
  g_return_if_fail (ddisp != NULL);
  g_return_if_fail (DIA_IS_DISPLAY (ddisp));
  
  dia_display_untransform_coords (ddisp, left, top,
				  &rect.left, &rect.top);
  dia_display_untransform_coords (ddisp, right, bottom,
				  &rect.right, &rect.bottom);

  dia_display_add_update (ddisp, &rect);
}

void
dia_display_flush (DiaDisplay *ddisp)
{
  GdkRectangle ir;
  DiaRenderer *renderer;
  Point pos = {0.0, 0.0};
  Rectangle handle_bb;
  Rectangle cp_bb;
  gint x,y;  
  
  g_return_if_fail (ddisp != NULL);
  g_return_if_fail (DIA_IS_DISPLAY (ddisp));
  g_return_if_fail (ddisp->renderer != NULL);

  /* Renders updates to pixmap + copies display_areas to canvas(screen) */
  
  /* We're about to render, so remove the handler */
  if (ddisp->redraw_handler != 0)
    {
      gtk_idle_remove (ddisp->redraw_handler);
      ddisp->redraw_handler = 0;
    }
  
  if (((ddisp->update_box.left >= ddisp->update_box.right)
       && (ddisp->update_box.top >= ddisp->update_box.bottom))
      || !rectangle_intersects (&ddisp->update_box, &ddisp->visible))
    return;
  
  renderer = &ddisp->renderer->renderer;

  rectangle_intersection (&ddisp->update_box, &ddisp->visible);

  /* Enlarge update box so the handles can also be drawn. */
  renderer->interactive_ops->get_handle_bb (renderer, &pos, &handle_bb);
  renderer->interactive_ops->get_cp_bb (renderer, &pos, &cp_bb);
  rectangle_union (&handle_bb, &cp_bb);
  ddisp->update_box.left += handle_bb.left;
  ddisp->update_box.right += handle_bb.right;
  ddisp->update_box.top += handle_bb.top;
  ddisp->update_box.bottom += handle_bb.bottom;
/*   g_message ("handle-box: (%f, %f), (%f, %f).", handle_bb.left, */
/* 	     handle_bb.top, handle_bb.right, */
/* 	     handle_bb.bottom); */

  /* if the diagram has a fixed size, do not draw anything outside the
   * diagram size. */
  if (!ddisp->diagram->auto_resize)
    rectangle_intersection (&ddisp->update_box, &ddisp->diagram->extents);
    
  /* calculate the region on the screen that has to be updated (pixels) */
  dia_display_transform_coords (ddisp, ddisp->update_box.left,
				ddisp->update_box.top, &x, &y);

  ir.x = x;
  ir.y = y;
  
  ir.width = dia_display_transform_length (ddisp, ddisp->update_box.right - ddisp->update_box.left) + 1;
  ir.height = dia_display_transform_length (ddisp, ddisp->update_box.bottom - ddisp->update_box.top) + 1;
  
  dia_display_render_pixmap (ddisp, &ir);
  
  if (ddisp->drawing_box)
    {
      Point p1, p2;
            
      p1.x = MIN (ddisp->start_box.x, ddisp->end_box.x);
      p1.y = MIN (ddisp->start_box.y, ddisp->end_box.y);
      p2.x = MAX (ddisp->start_box.x, ddisp->end_box.x);
      p2.y = MAX (ddisp->start_box.y, ddisp->end_box.y);

      renderer->ops->set_linestyle (renderer, DIA_LINE_STYLE_DOTTED);
      renderer->ops->set_linewidth (renderer, 0.0);
      renderer->ops->draw_rect (renderer, &p1, &p2, &dia_color_black);
    }
  
  dia_renderer_gdk_copy_to_window (ddisp->renderer, ddisp->canvas->window,
				   ir.x, ir.y, ir.width, ir.height);

  /* "Clear" the update box */
  ddisp->update_box.left = ddisp->update_box.right =
    ddisp->update_box.top = ddisp->update_box.bottom = 0.0;
}

void
dia_display_update_object_state (DiaDisplay *disp, DiaObject *obj)
{
  g_return_if_fail (obj != NULL);
  g_return_if_fail (disp != NULL);
  
  /* determine the state for this object */
  /* first reset the state flags (bit 0 and 1) to normal mode */
  DIA_OBJECT_UNSET_FLAGS (obj, DIA_OBJECT_STATE_MASK);
    
  if (obj == disp->focus)
    DIA_OBJECT_SET_FLAGS (obj, DIA_OBJECT_STATE_FOCUSED);
  if (g_list_find (disp->selected, obj))
    DIA_OBJECT_SET_FLAGS (obj, DIA_OBJECT_STATE_SELECTED);
  if (disp->active_layer && g_list_find (disp->active_layer->objects, obj))
    DIA_OBJECT_SET_FLAGS (obj, DIA_OBJECT_STATE_ACTIVE);
  if (obj == disp->grab)
    DIA_OBJECT_SET_FLAGS (obj, DIA_OBJECT_STATE_GRABBED);    
}

static void
dia_display_obj_render (DiaObject *obj, DiaRenderer *renderer,
			gpointer data)
{
  dia_display_update_object_state ((DiaDisplay *)data, obj);
  
  dia_object_draw (obj, renderer);
}


void
dia_display_render_pixmap (DiaDisplay *ddisp, GdkRectangle *ir)
{
  DiaRenderer *renderer;
  GList *l;
  Rectangle update_box;
  
  g_return_if_fail (ddisp != NULL);
  g_return_if_fail (DIA_IS_DISPLAY (ddisp));
  g_return_if_fail (ddisp->renderer != NULL);
  
  renderer = &ddisp->renderer->renderer;

  /* untransform the pixel rectangle since values might change a little as
   * they are converted. */
  dia_display_untransform_coords (ddisp, ir->x, ir->y,
				  &update_box.left, &update_box.top);
  dia_display_untransform_coords (ddisp, ir->x + ir->width + 1,
				  ir->y + ir->height + 1,
				  &update_box.right, &update_box.bottom);

  /* clear the region */
  renderer->interactive_ops->clip_region_clear (renderer);
  
  renderer->interactive_ops->clip_region_add_rect (renderer, &update_box);
  
/*   g_message ("update-box (1): (%f, %f), (%f, %f).", ddisp->update_box->left, */
/* 	     ddisp->update_box->top, ddisp->update_box->right, */
/* 	     ddisp->update_box->bottom); */
/*   g_message ("update-box (2): (%f, %f), (%f, %f).", update_box.left, */
/* 	     update_box.top, update_box.right, */
/* 	     update_box.bottom); */

  renderer->interactive_ops->fill_pixel_rect (renderer,
					      ir->x, ir->y,
					      ir->width,
					      ir->height,
					      &ddisp->diagram->bg_color);
      
  /* Draw grid */
  dia_grid_draw (ddisp, &update_box);

  dia_diagram_render (ddisp->diagram, (DiaRenderer *)ddisp->renderer,
		      dia_display_obj_render, &update_box,
		      (gpointer) ddisp);

  /* now draw the handles and connection points: */
  l = ddisp->active_layer->objects;
  while (l)
    {
      dia_object_draw_cps (DIA_OBJECT (l->data), renderer);
      l = g_list_next (l);
    }

  l = ddisp->selected;
  while (l)
    {
      dia_object_draw_handles (DIA_OBJECT (l->data), renderer);
      l = g_list_next (l);
    }
}

void
dia_display_update_scrollbars (DiaDisplay *ddisp)
{
  Rectangle extents;
  Rectangle *visible;
  GtkAdjustment *hsbdata, *vsbdata;
  gfloat extra_border_x, extra_border_y;

  g_return_if_fail (ddisp != NULL);
  g_return_if_fail (DIA_IS_DISPLAY (ddisp));

  extents = ddisp->diagram->extents;
  visible = &ddisp->visible;
  
  if (ddisp->diagram->auto_resize)
    {
      extra_border_x = (visible->right - visible->left);
      extra_border_y = (visible->bottom - visible->top);
    }
  else
    {
      extra_border_x = 0.0;
      extra_border_y = 0.0;
    }
  
  /* Set the scrollbars when there are no objects placed: */
  if (extents.left == extents.right &&
      extents.top == extents.bottom)
    {
      extents.right = visible->right;
      extents.bottom = visible->bottom;
    }
   
  hsbdata = ddisp->hsbdata;
  /* Horizontal: */
  hsbdata->lower = MIN (extents.left - extra_border_x, visible->left);
  hsbdata->upper = MAX (extents.right + extra_border_x, visible->right);
  hsbdata->page_size = visible->right - visible->left - 0.0001;
  /* remove some to fix strange behaviour in gtk_range_adjustment_changed */
  hsbdata->page_increment = (visible->right - visible->left) / 2.0;
  hsbdata->step_increment = (visible->right - visible->left) / 10.0;
  hsbdata->value = visible->left;

  gtk_signal_emit_by_name (GTK_OBJECT (ddisp->hsbdata), "changed");

  /* Vertical: */
  vsbdata = ddisp->vsbdata;
  vsbdata->lower = MIN (extents.top - extra_border_y, visible->top);
  vsbdata->upper = MAX (extents.bottom + extra_border_x, visible->bottom);
  vsbdata->page_size = visible->bottom - visible->top - 0.00001;
  /* remove some to fix strange behaviour in gtk_range_adjustment_changed */
  vsbdata->page_increment = (visible->bottom - visible->top) / 2.0;
  vsbdata->step_increment = (visible->bottom - visible->top) / 10.0;
  vsbdata->value = visible->top;

  gtk_signal_emit_by_name (GTK_OBJECT (ddisp->vsbdata), "changed");
}

void
dia_display_set_origo (DiaDisplay *ddisp, gfloat x, gfloat y)
{
  Rectangle *extents;
  Rectangle *visible;
  gint width;
  gint height;
    
  g_return_if_fail (ddisp != NULL);
  g_return_if_fail (DIA_IS_DISPLAY (ddisp));

  extents = &ddisp->diagram->extents;
  visible = &ddisp->visible;
  width = ddisp->renderer->renderer.pixel_width;
  height = ddisp->renderer->renderer.pixel_height;

  if (!ddisp->diagram->auto_resize)
    {
      gint tw, th;
     
      tw = dia_display_transform_length (ddisp,
					 extents->right - extents->left);
      th = dia_display_transform_length (ddisp,
					 extents->bottom - extents->top);

      /* make the canvas look nice, even if it is smaller than the display: */
      if (tw < width)
	x = extents->left -
	  dia_display_untransform_length (ddisp, (width - tw) / 2);
      else
	if (x < extents->left)
	  x = extents->left;
	else
	  if ((x + width/ddisp->zoom_factor) > extents->right)
	    x = extents->right - width/ddisp->zoom_factor;
      
      if (th < height)
	y = extents->top -
	  dia_display_untransform_length (ddisp, (height - th) / 2);
      else
	if (y < extents->top)
	  y = extents->top;
	else
	  if ((y + height/ddisp->zoom_factor) > extents->bottom)
	    y = extents->bottom - height/ddisp->zoom_factor;
    }

  ddisp->origo.x = x;
  ddisp->origo.y = y;
  
  if (ddisp->zoom_factor < DIA_DISPLAY_MIN_ZOOM)
    ddisp->zoom_factor = DIA_DISPLAY_MIN_ZOOM;
  
  if (ddisp->zoom_factor > DIA_DISPLAY_MAX_ZOOM)
    ddisp->zoom_factor = DIA_DISPLAY_MAX_ZOOM;

  visible->left = ddisp->origo.x;
  visible->top = ddisp->origo.y;
  visible->right = ddisp->origo.x + width/ddisp->zoom_factor;
  visible->bottom = ddisp->origo.y + height/ddisp->zoom_factor;

  gtk_ruler_set_range  (GTK_RULER (ddisp->hrule),
			visible->left,
			visible->right,
			0.0f /* position*/,
			MAX(extents->right, visible->right)/* max_size*/);
  gtk_ruler_set_range  (GTK_RULER (ddisp->vrule),
			visible->top,
			visible->bottom,
			0.0f /*        position*/,
			MAX(extents->bottom, visible->bottom)/* max_size*/);
}

void
dia_display_zoom (DiaDisplay *ddisp, Point *point, gfloat magnify)
{
  Rectangle *visible;
  gfloat width, height;

  g_return_if_fail (ddisp != NULL);
  g_return_if_fail (DIA_IS_DISPLAY (ddisp));

  visible = &ddisp->visible;

  width = (visible->right - visible->left)/magnify;
  height = (visible->bottom - visible->top)/magnify;

  if ((ddisp->zoom_factor <= DIA_DISPLAY_MIN_ZOOM) && (magnify<=1.0))
    return;
  if ((ddisp->zoom_factor >= DIA_DISPLAY_MAX_ZOOM) && (magnify>=1.0))
    return;

  ddisp->zoom_factor *= magnify;

  dia_display_set_origo(ddisp, point->x - width/2.0, point->y - height/2.0);
  
  dia_display_update_scrollbars (ddisp);
  dia_display_add_update_all (ddisp);

  /* Do a window clean here.. After a set-origo they problably wanna
   * redraw too... (Can't do it in the renderer 'cause it will mess up
   * when updating only regions) */
  gtk_style_apply_default_background (ddisp->canvas->style,
				      ddisp->canvas->window,
				      TRUE, GTK_STATE_NORMAL, NULL,
				      0,0, ddisp->canvas->allocation.width,
				      ddisp->canvas->allocation.height);

  //dia_display_flush (ddisp);
}

void
dia_display_zoom_box (DiaDisplay *ddisp, Point *origo, gfloat width,
		      gfloat height)
{
  gfloat mag;
  Point p;
  
  mag = MIN ((ddisp->visible.right - ddisp->visible.left) / width,
	     (ddisp->visible.bottom - ddisp->visible.top) / height);
  p.x = origo->x + width/2.0;
  p.y = origo->y + height/2.0;
  
  dia_display_zoom (ddisp, &p, mag);
}

gboolean
dia_display_autoscroll (DiaDisplay *ddisp, gint x, gint y)
{
  guint16 width, height;
  Point scroll;
  
  if (!ddisp->autoscroll)
    return FALSE;

  scroll.x = scroll.y = 0.0;

  width = GTK_WIDGET (ddisp->canvas)->allocation.width;
  height = GTK_WIDGET (ddisp->canvas)->allocation.height;

  if (x < 0)
    scroll.x = x;
  else if ( x > width)
    scroll.x = x - width;
  
  if (y < 0)
    scroll.y = y;
  else if (y > height)
    scroll.y = y - height;
  

  if ((scroll.x != 0.0) || (scroll.y != 0.0))
    {
      scroll.x = dia_display_untransform_length (ddisp, scroll.x);
      scroll.y = dia_display_untransform_length (ddisp, scroll.y);

      return dia_display_scroll (ddisp, &scroll);
    //dia_display_flush (ddisp);

      //return TRUE;
    }
  else
    return FALSE;
}

gboolean
dia_display_scroll (DiaDisplay *ddisp, Point *delta)
{
  Point new_origo;
  Rectangle extents;
  Rectangle *visible;
  gfloat width, height;
  
  g_return_val_if_fail (ddisp != NULL, FALSE);
  g_return_val_if_fail (DIA_IS_DISPLAY (ddisp), FALSE);

  visible = &ddisp->visible;

  new_origo = ddisp->origo;
  point_add (&new_origo, delta);

  width = visible->right - visible->left;
  height = visible->bottom - visible->top;
  
  extents = ddisp->diagram->extents;
  rectangle_union (&extents, visible);
  
  if (new_origo.x < extents.left)
    new_origo.x = extents.left;

  if (new_origo.x+width > extents.right)
    new_origo.x = extents.right - width;

  if (new_origo.y < extents.top)
    new_origo.y = extents.top;
  
  if (new_origo.y+height > extents.bottom)
    new_origo.y = extents.bottom - height;

  if ((ddisp->origo.x != new_origo.x)
      || (ddisp->origo.y != new_origo.y))
    {
      dia_display_set_origo (ddisp, new_origo.x, new_origo.y);
      dia_display_update_scrollbars (ddisp);
      dia_display_add_update_all (ddisp);
      return TRUE;
    }
  return FALSE;
}

void
dia_display_scroll_up (DiaDisplay *ddisp)
{
  Point delta;

  g_return_if_fail (ddisp != NULL);
  g_return_if_fail (DIA_IS_DISPLAY (ddisp));

  delta.x = 0;
  delta.y = -(ddisp->visible.bottom - ddisp->visible.top)/4.0;
  
  dia_display_scroll(ddisp, &delta);
}

void
dia_display_scroll_down (DiaDisplay *ddisp)
{
  Point delta;

  g_return_if_fail (ddisp != NULL);
  g_return_if_fail (DIA_IS_DISPLAY (ddisp));

  delta.x = 0;
  delta.y = (ddisp->visible.bottom - ddisp->visible.top)/4.0;
  
  dia_display_scroll (ddisp, &delta);
}

void
dia_display_scroll_left (DiaDisplay *ddisp)
{
  Point delta;

  g_return_if_fail (ddisp != NULL);
  g_return_if_fail (DIA_IS_DISPLAY (ddisp));

  delta.x = -(ddisp->visible.right - ddisp->visible.left)/4.0;
  delta.y = 0;
  
  dia_display_scroll(ddisp, &delta);
}

void
dia_display_scroll_right (DiaDisplay *ddisp)
{
  Point delta;

  g_return_if_fail (ddisp != NULL);
  g_return_if_fail (DIA_IS_DISPLAY (ddisp));

  delta.x = (ddisp->visible.right - ddisp->visible.left)/4.0;
  delta.y = 0;
  
  dia_display_scroll (ddisp, &delta);
}

void
dia_display_resize_canvas (DiaDisplay *ddisp,
			   gint width,  gint height)
{  
  g_return_if_fail (ddisp != NULL);
  g_return_if_fail (DIA_IS_DISPLAY (ddisp));
  g_return_if_fail (ddisp->renderer != NULL);

  dia_renderer_gdk_set_size (ddisp->renderer, ddisp->canvas->window,
			     width, height);

  dia_display_set_origo (ddisp, ddisp->origo.x, ddisp->origo.y);

  dia_display_add_update_all (ddisp);
  //dia_display_flush (ddisp);
}

void
dia_display_set_default_cursor (gint cursor)
{
  default_cursor = cursor;

  dia_display_set_all_cursor (cursor);
}

void
dia_display_set_all_cursor (int cursor)
{
  DiaDiagram *dia;
  DiaDisplay *ddisp;
  GList *list;
  GSList *slist;

  list = open_diagrams;
  while (list != NULL)
    {
      dia = (DiaDiagram *) list->data;

      slist = dia->displays;
      while (slist != NULL)
	{
	  ddisp = (DiaDisplay *) slist->data;
	  
	  dia_display_set_cursor (ddisp, cursor);
	  
	  slist = g_slist_next(slist);
	}
      
      list = g_list_next(list);
    }
}

void
dia_display_set_cursor (DiaDisplay *ddisp, gint cursor)
{
  GdkCursor *c;
  
  g_return_if_fail (ddisp != NULL);
  g_return_if_fail (DIA_IS_DISPLAY (ddisp));
  
  if (cursor == ddisp->cursor)
    return;
  
  c = gdk_cursor_new (cursor);
  if (!c)
    {
      g_warning ("dia_display_set_cursor: Cursor with value %d "
		 "does not exist!!!", cursor);
      return;
    }
  
  ddisp->cursor = cursor;
  
  gdk_window_set_cursor (ddisp->canvas->window, c);
  gdk_cursor_destroy (c);
}

void
dia_display_default_cursor (DiaDisplay *ddisp)
{
  g_return_if_fail (ddisp != NULL);
  g_return_if_fail (DIA_IS_DISPLAY (ddisp));
  
  dia_display_set_cursor (ddisp, default_cursor);
}


void
dia_display_set_rulers_visibility (DiaDisplay *ddisp, gboolean visible)
{
  g_return_if_fail (ddisp != NULL);
  g_return_if_fail (DIA_IS_DISPLAY (ddisp));

  if (visible)
    {
      gtk_widget_show (ddisp->hrule);
      gtk_widget_show (ddisp->vrule);
      gtk_widget_show (ddisp->origin);
    }
  else
    {
      gtk_widget_hide (ddisp->hrule);
      gtk_widget_hide (ddisp->vrule);
      gtk_widget_hide (ddisp->origin);
    }
}

gboolean
dia_display_rulers_visible (DiaDisplay *ddisp)
{
  g_return_val_if_fail (ddisp != NULL, FALSE);
  g_return_val_if_fail (DIA_IS_DISPLAY (ddisp), FALSE);
  
  return GTK_WIDGET_VISIBLE (ddisp->hrule) &&
    GTK_WIDGET_VISIBLE (ddisp->vrule) &&
    GTK_WIDGET_VISIBLE (ddisp->origin);
}

void
dia_display_convert_gdk_event (DiaDisplay *disp, GdkEvent *gdk_event, 
			       DiaEvent *dia_event)
{
  Point p;
  gfloat d;
  DiaHandle *handle;

  g_return_if_fail (gdk_event != NULL);
  g_return_if_fail (dia_event != NULL);
  
  /* conversion of the event type */
  switch (gdk_event->type)
    {
    case GDK_MOTION_NOTIFY:
      dia_event->type = DIA_MOTION;
      break;
    case GDK_BUTTON_PRESS:
      dia_event->type = DIA_BUTTON_PRESS;
      break;
    case GDK_2BUTTON_PRESS:
      dia_event->type = DIA_2BUTTON_PRESS;
       break;
   case GDK_3BUTTON_PRESS:
      dia_event->type = DIA_3BUTTON_PRESS;
      break;
    case GDK_BUTTON_RELEASE:
      dia_event->type = DIA_BUTTON_RELEASE;
      break;
    case GDK_KEY_PRESS:
      dia_event->type = DIA_KEY_PRESS;
      break;
    case GDK_KEY_RELEASE:
      dia_event->type = DIA_KEY_RELEASE;
      break;
    default:
      dia_event->type = DIA_NONE;
      break;
    }

  switch (dia_event->type)
    {
    case DIA_MOTION:
      {
	DiaEventMotion *event = (DiaEventMotion*) dia_event;
       	
	dia_display_untransform_coords (disp, gdk_event->motion.x,
					gdk_event->motion.y,
					&p.x, &p.y);
	event->modifier = gdk_event->motion.state;
	event->time = gdk_event->motion.time;
	event->pos.x = event->snap.x = p.x;
	event->pos.y = event->snap.y = p.y;
	dia_grid_snap (&disp->grid, &event->snap.x, &event->snap.y);
	event->pressure = gdk_event->motion.pressure;
      }
      break;
    case DIA_BUTTON_PRESS:
    case DIA_2BUTTON_PRESS:
    case DIA_3BUTTON_PRESS:
    case DIA_BUTTON_RELEASE:
      {
	DiaEventButton *event = (DiaEventButton*) dia_event;
	dia_display_untransform_coords (disp, gdk_event->button.x,
					gdk_event->button.y,
					&p.x, &p.y);
	event->modifier = gdk_event->button.state;
	event->time = gdk_event->button.time;
	event->pos.x = event->snap.x = p.x;
	event->pos.y = event->snap.y = p.y;
	dia_grid_snap (&disp->grid, &event->snap.x, &event->snap.y);
	event->pressure = gdk_event->button.pressure;
	event->button = gdk_event->button.button;

	d = dia_display_untransform_length (disp, (gint)DIA_HANDLE_SIZE);

	//if (dia_event->type != DIA_BUTTON_RELEASE)
	//{
	/* find a handle in the selected objects list: */
	if ((disp->focus)
	    && (dia_object_find_closest_handle (disp->focus, &p,
						&handle) < d))
	  event->handle = handle;
	else if ((disp->selected)
		 && (dia_find_closest_handle_from_list (disp->selected, &p,
							&handle) < d))
	  event->handle = handle;
	else if (dia_layer_find_closest_handle (disp->active_layer, &p,
						&handle) < d)
	  event->handle = handle;
	else
	  event->handle = NULL;
	//}
      }
      break;
    case DIA_KEY_PRESS:
    case DIA_KEY_RELEASE:
      {
	DiaEventKey *event = (DiaEventKey*) dia_event;
	event->modifier = gdk_event->key.state;
	event->time = gdk_event->key.time;
	event->keyval = gdk_event->key.keyval;
	event->length = gdk_event->key.length;
	event->string = gdk_event->key.string;
      }
      break;
    default:
      break;
    }
}

DiaDisplay*
dia_display_get_last_edited_display ()
{
  return last_edited_display;
}









